;-*-Mode:Lisp; Package:FILE-SYSTEM; Base: 8 -*-

;; Enable file connections to this machine.

;If you don't specify the local host,
;we will use whichever local host name the remote end asks us to use,
;which is normally the right thing to do.
(DEFUN ENABLE-FILE-SERVER (&OPTIONAL (LOCAL-HOST LFS-HOST))
  (ADD-INITIALIZATION "LMFILE"
		      `(PROCESS-RUN-FUNCTION
			 "LMFILE server" 'FILE-SERVE ',LOCAL-HOST)
		      NIL
		      'CHAOS:SERVER-ALIST))

(DEFUN DISABLE-FILE-SERVER ()
  (SETQ CHAOS:SERVER-ALIST
	(DELQ (ASSOC "LMFILE" CHAOS:SERVER-ALIST) CHAOS:SERVER-ALIST))
  NIL)

(DEFVAR FILE-SERVER-TIMEOUT 10000000.)

;While processing a file connection, this variable
;holds the connection being handled.
(DEFVAR FILE-CONN)

;This is T if we should be transmitting data from the file
;without having to be specifically asked for it.
;This is used on what the user end would call an "input" channel.
;This variable can also be a positive fixnum, in which
;case we should transmit that many bytes (decrementing the count as we go)
;and then stop and set the variable back to nil.
(DEFVAR FILE-TRANSMIT-DATA)

;This is the stream through which the local file is accessed.
(DEFVAR FILE-STREAM)

;This is either FILE-CHARACTER-OPCODE or FILE-BINARY-OPCODE,
;depending on the byte size in use.
(DEFVAR FILE-DATA-OPCODE)

;This is stream-type for the file-stream.
(DEFVAR FILE-STREAM-TYPE)

(DEFVAR LMFILE-SERVER-CONNECTION-RECORD NIL)

;; This is the top level function of the file server process.
;; Protocol is LMFILE<sp><user-id><sp><command>
(DEFUN FILE-SERVE (READ-PATHNAME-HOST
		   &AUX (SERVER-LOCAL-HOST NIL)
		   FILE-CONN PKT-STRING ARGSTRING (USER-ID "File-Server") TEM
		   CURRENT-FILE-AREA)
  (SETQ FILE-CONN (CHAOS:LISTEN "LMFILE"))
  (COND ((CHAOS:WAIT FILE-CONN 'CHAOS:LISTENING-STATE FILE-SERVER-TIMEOUT "Connection wait")
	 (COND ((EQ (CHAOS:STATE FILE-CONN) 'CHAOS:RFC-RECEIVED-STATE)
		(SETQ PKT-STRING (CHAOS:PKT-STRING (CHAOS:READ-PKTS FILE-CONN))
		      USER-ID (SUBSTRING PKT-STRING 7
					 (SETQ TEM (STRING-SEARCH-CHAR #\SP PKT-STRING 7)))
		      ARGSTRING (SUBSTRING PKT-STRING (1+ TEM)))
		(WITHOUT-INTERRUPTS
		  (PUSH (LIST USER-ID
			      (STRING (SI:GET-HOST-FROM-ADDRESS
					(CHAOS:FOREIGN-ADDRESS FILE-CONN)
					':CHAOS))
			      (TIME:PRINT-CURRENT-TIME NIL))
			LMFILE-SERVER-CONNECTION-RECORD))
		(AND (FBOUNDP 'GET-FILE-AREA)
		     (PROGN (SETQ CURRENT-FILE-AREA (GET-FILE-AREA USER-ID))
			    (FILE-AREA-OPEN-SERVER CURRENT-FILE-AREA CURRENT-PROCESS)))
		;; This could avoid a string copy by not returnning the RFC
		;; pkt immediately...
		(CHAOS:ACCEPT FILE-CONN)
		(FUNCALL TV:WHO-LINE-FILE-STATE-SHEET ':ADD-SERVER FILE-CONN "LMFILE")
		(*CATCH 'CONNECTION-CLOSED
		  (CONDITION-BIND
		    ((SYS:BAD-CONNECTION-STATE
		      'FILE-SERVER-CONDITION-HANDLER)
		     (MORE-ROOM
		      'FILE-SERVER-DISK-UNFULL-HANDLER)
		     (NO-MORE-ROOM
		      'FILE-SERVER-DISK-FULL-HANDLER))
		    (UNWIND-PROTECT
		      (FILE-SERVE-1 FILE-CONN ARGSTRING)
		      (CHAOS:CLOSE-CONN FILE-CONN)
		      (FUNCALL TV:WHO-LINE-FILE-STATE-SHEET ':DELETE-SERVER FILE-CONN)
		      (AND (FBOUNDP 'GET-FILE-AREA)
			   (FILE-AREA-CLOSE-SERVER CURRENT-FILE-AREA CURRENT-PROCESS))
		      (IF (FBOUNDP 'MAYBE-FREE-SOME-FILE-AREAS)
			  (MAYBE-FREE-SOME-FILE-AREAS))))))
	       (T (CHAOS:CLOSE-CONN FILE-CONN
				    (FORMAT NIL "Connection went into ~S after listening."
					    (CHAOS:STATE FILE-CONN))))))
	(T (CHAOS:CLOSE-CONN FILE-CONN "Timeout on Listen."))))

(DEFUN FILE-SERVER-CONDITION-HANDLER (&REST IGNORE)
       (*THROW 'CONNECTION-CLOSED NIL))

(DEFUN FILE-SERVER-DISK-FULL-HANDLER (CONDITION &AUX VOLID-STRING)
  (SETQ VOLID-STRING
	(OR (SEND CONDITION ':SEND-IF-HANDLES ':VOLUME)
	    (SEND CONDITION ':SEND-IF-HANDLES ':PATHNAME)))
  (FILE-SEND-MESSAGE FILE-CONN FILE-LOSSAGE-OPCODE
	     `(DISK-FULL 11 (SEND CONDITION ':REPORT-STRING) ,VOLID-STRING))
  ':NO-ACTION)

(DEFUN FILE-SERVER-DISK-UNFULL-HANDLER (&REST IGNORE)
  (FILE-SEND-MESSAGE FILE-CONN FILE-WINNAGE-OPCODE NIL)
  ':NO-ACTION)

(defvar last-file-command nil)

(DEFUN FILE-SERVE-1 (FILE-CONN ARGSTRING)
  (LET (FILE-STREAM FILE-TRANSMIT-DATA
	FILE-CONTINUED-PKTS
	FILE-DATA-OPCODE
	FILE-STREAM-TYPE
	last-file-command
	USE-STANDARD-SYMBOL-ENCODING
	(USER-ID USER-ID))
    (SETQ FILE-CONTINUED-PKTS (NCONC FILE-CONTINUED-PKTS (LIST ARGSTRING)))
    (UNWIND-PROTECT
     (PROGN (FILE-PROCESS-COMMAND)
	    (DO () (())
	      (PROCESS-WAIT "CHAOS" 'FILE-WAIT-CONDITION FILE-CONN FILE-TRANSMIT-DATA)
	      (FILE-SERVER-QUANTUM)))
     (AND FILE-STREAM (CLOSE FILE-STREAM)))))

;This function returns T if there is work that can be done for this
;connection right now.
(DEFUN FILE-WAIT-CONDITION (FILE-CONN FILE-TRANSMIT-DATA)
  (OR (NEQ (CHAOS:STATE FILE-CONN) 'CHAOS:OPEN-STATE)
      (CHAOS:DATA-AVAILABLE FILE-CONN)
      (AND FILE-TRANSMIT-DATA (CHAOS:MAY-TRANSMIT FILE-CONN))))

;Process one input packet, or send one output data packet (output files only).
;Does not hang if there is no input or no room in the window for output.
(DEFUN FILE-SERVER-QUANTUM (&AUX PKT TEM)
  (COND ((NEQ (CHAOS:STATE FILE-CONN) 'CHAOS:OPEN-STATE)
	 (*THROW 'CONNECTION-CLOSED NIL))
	((SETQ PKT (CHAOS:GET-NEXT-PKT FILE-CONN T))
	 (SELECT (CHAOS:PKT-OPCODE PKT)
	   (FILE-CHARACTER-OPCODE
	    (FUNCALL FILE-STREAM ':STRING-OUT (CHAOS:PKT-STRING PKT))
	    (CHAOS:RETURN-PKT PKT))
	   (FILE-BINARY-OPCODE
	    (FUNCALL FILE-STREAM ':STRING-OUT PKT
		     CHAOS:FIRST-DATA-WORD-IN-PKT
		     (+ CHAOS:FIRST-DATA-WORD-IN-PKT
			(TRUNCATE (CHAOS:PKT-NBYTES PKT) 2)))
	    (CHAOS:RETURN-PKT PKT))
	   (FILE-COMMAND-OPCODE
	    (SETQ FILE-CONTINUED-PKTS (NCONC FILE-CONTINUED-PKTS (LIST PKT)))
	    (FILE-PROCESS-COMMAND))
  	   (FILE-CONTINUED-OPCODE
	    (SETQ FILE-CONTINUED-PKTS (NCONC FILE-CONTINUED-PKTS (LIST PKT))))
	   (:OTHERWISE (FILE-CONNECTION-LOSSAGE PKT))))
	((EQ FILE-TRANSMIT-DATA 0) (SETQ FILE-TRANSMIT-DATA NIL))
	((AND FILE-TRANSMIT-DATA (CHAOS:MAY-TRANSMIT FILE-CONN))
	 (SETQ PKT (CHAOS:GET-PKT))
	 ;; Compute the maximum number of bytes we want to and can send now.
	 (IF (= FILE-DATA-OPCODE FILE-CHARACTER-OPCODE)
	     (SETF (CHAOS:PKT-NBYTES PKT)
		   (FUNCALL FILE-STREAM ':STRING-IN NIL
			    (CHAOS:PKT-STRING PKT) 0
			    (IF (EQ FILE-TRANSMIT-DATA T)
				CHAOS:MAX-DATA-BYTES-PER-PKT
				(MIN CHAOS:MAX-DATA-BYTES-PER-PKT
				     FILE-TRANSMIT-DATA))))
	     (SETF (CHAOS:PKT-NBYTES PKT)
		   (* (- (FUNCALL FILE-STREAM ':STRING-IN NIL
				  PKT CHAOS:FIRST-DATA-WORD-IN-PKT
				  (+ CHAOS:FIRST-DATA-WORD-IN-PKT
				     (IF (EQ FILE-TRANSMIT-DATA T)
					 CHAOS:MAX-DATA-WORDS-PER-PKT
				       (MIN CHAOS:MAX-DATA-WORDS-PER-PKT
					    (TRUNCATE FILE-TRANSMIT-DATA 2)))))
			 CHAOS:FIRST-DATA-WORD-IN-PKT)
		      2)))
	 ;; Count off the bytes we are now transmitting.
	 (OR (EQ FILE-TRANSMIT-DATA T)
	     (SETQ FILE-TRANSMIT-DATA (- FILE-TRANSMIT-DATA (CHAOS:PKT-NBYTES PKT))))
	 (IF (AND (ZEROP (CHAOS:PKT-NBYTES PKT))
		  (SETQ TEM (FUNCALL FILE-STREAM ':SEND-IF-HANDLES
				   ':NEXT-NODE)))
	     ;; If no data but the file has another node,
	     ;; put the node name in the pipeline
	     ;; after we prepare to send more data.
	     (FILE-SEND-MESSAGE FILE-CONN FILE-NEXT-NODE-OPCODE TEM)
	   ;; Send the packet, but if we got no data (which can only be due to EOF)
	   ;; make it an EOF packet instead, and stop trying to transmit.
	   (CHAOS:SEND-PKT FILE-CONN PKT
			   (COND ((ZEROP (CHAOS:PKT-NBYTES PKT))
				  (SETQ FILE-TRANSMIT-DATA NIL)
				  FILE-EOF-OPCODE)
				 (T FILE-DATA-OPCODE))))
	 T)))

(DEFUN FILE-CONNECTION-LOSSAGE (PKT)
  (CHAOS:CLOSE-CONN FILE-CONN
		    (FORMAT NIL "Packet with opcode ~O received" (CHAOS:PKT-OPCODE PKT)))
  (CHAOS:RETURN-PKT PKT)
  (*THROW 'CONNECTION-CLOSED NIL))

(DEFUN FILE-PROCESS-COMMAND ()
  (LET (COMMANDLIST TEM)
    (ERRSET (SETQ COMMANDLIST (FILE-EXTRACT-LIST)) NIL)
    (setq last-file-command commandlist)
    (IF (ATOM COMMANDLIST)		;Don't bomb out totally if poorly formed.
	(PROGN (CHAOS:CLOSE-CONN FILE-CONN "Unreadable argument present in file command
/(probably an array or instance)")
	       (*THROW 'CONNECTION-CLOSED NIL))
	;; The file command can throw here to send no reply.
	(*CATCH 'COMMAND-NO-REPLY
	  (LET ((SEQNUM (CAR COMMANDLIST))
		(COMMAND (CADR COMMANDLIST))
		VALUES)
	    (COND ((SETQ TEM (AND (SYMBOLP COMMAND) (GET COMMAND 'FILE-COMMAND)))
		   (SETQ VALUES
			 (*CATCH 'COMMAND-FAILURE
			   (CONDITION-BIND
			     ((FILE-ERROR 'FILE-COMMAND-ERROR-HANDLER))
			     (MULTIPLE-VALUE-LIST
			       (APPLY TEM (CDDR COMMANDLIST)))))))
		  (T (SETQ VALUES (LIST 'ERROR
					(FORMAT NIL "XXX ERROR UFC F ~S undefined file command" COMMAND)
					0))))
	    (FILE-SEND-REPLY SEQNUM
			     (CONS COMMAND;command included for debugging ease.
				   VALUES)))))))

;; This is a condition-handler for FILE-ERROR conditions going off in file commands.
(DEFUN FILE-COMMAND-ERROR-HANDLER (ERROR)
  (LET* ((PN (SEND ERROR ':SEND-IF-HANDLES ':PATHNAME))
	 (PNN (IF PN (SEND PN ':STRING-FOR-PRINTING)))
	 (EN (SEND ERROR ':REPORT-STRING)))
    ;; DROP A PERIOD OFF THE END OF THE ERROR MESSAGE.
    (AND (CHAR-EQUAL (AREF EN (1- (LENGTH EN))) #/.)
	 (SETQ EN (SUBSTRING EN 0 (1- (LENGTH EN)))))
    ;; DROP " FOR LM: FOO" OFF THE END.
    (AND PNN
	 (STRING-EQUAL EN PNN (- (LENGTH EN) (LENGTH PNN)))
	 (SETQ EN (SUBSTRING EN 0 (- (LENGTH EN) (LENGTH PNN) 5))))
    (*THROW 'COMMAND-FAILURE
	    (LIST 'ERROR
		  (STRING-APPEND "XXX ERROR "
				 (DOLIST (CN (SEND ERROR ':CONDITION-NAMES))
				   (WHEN (GET CN 'FILE-ERROR)
				     (RETURN (GET CN 'FILE-ERROR))))
				 " F "
				 EN)
		  ;; Determine an ITS error code to return as well.
		  (DOLIST (CN (SEND ERROR ':CONDITION-NAMES) 0)
		    (WHEN (GET CN 'FILE-PDP10-ERROR)
		      (RETURN (GET CN 'FILE-PDP10-ERROR))))))))

;send a reply.  Automatically uses multiple packets if the reply is very long.
(DEFUN FILE-SEND-REPLY (SEQNUM LIST)
 (FILE-SEND-MESSAGE FILE-CONN FILE-REPLY-OPCODE (CONS SEQNUM LIST)))

;File commands.

;A file command is defined by creating a FILE-COMMAND property on the symbol
;which is the name of the command, in the FS package.

;File commands can occur in command packets or in the RFC packet.
;Opening the file stream is done by commands that are usually called
;in the RFC packet.

;The arguments passed to the file command function
;are the arguments in the command string which follow the command name.
;The function should make all its args optional and allow a &rest arg
;so that too many or few args does not cause a Lisp error.

;All values returned will be sent back in the reply packet.  
;The first value is one of T, STREAM, or ERROR.
;  A value of T indicates that a normal return is taking place, and the second
;  value is a VALUES-LIST which should be returned at the remote end.
;
;  A value of STREAM indicates that a successful open has taken place, and a 
;  stream now exists.   The second value is a stream-type.  Stream-types are
;  defined in RMDEFS.  The remaining values depend on the file command.
;
;  A value of ERROR indicates that an error has taken place, and the second value
;  is a string describing it.  The third value is a number which is a pdp10
;  error code.  This is only useful for pdp10s.
;  Signaling a FILE-ERROR condition has the same effect as returning ERROR.

;A few file commands may decide to send no reply.
;This is done by START-DATA-TRANSMISSION, because why use up
;net bandwidth to ack that when the data requested does it just as well.
;Commands do this by throwing to COMMAND-NO-REPLY

;Commands that can't win if no file has been open should call this function.
;This is so that they complain to the user rather than getting Lisp errors.
(DEFUN REQUIRE-OPEN-FILE ()
  (OR FILE-STREAM (*THROW 'COMMAND-FAILURE '(ERROR "XXX ERROR NFO F No file open" 0))))

(DEFUN FILE-COMMAND-RETURN-VALUES (&REST VALUES)
  (VALUES T (COPYLIST VALUES)))

(DEFUN FILE-COMMAND-THROW-ERROR (ERROR-STRING)
  (*THROW 'COMMAND-FAILURE (LIST 'ERROR ERROR-STRING 0)))

;These stream operations of the local file stream,
;plus all standard stream operations, do not get mentioned
;in the WHICH-OPERATIONS list we send to the remote system
;because they are either implemented specially over there
;or are not supposed to be available there.
(DEFCONST SERVER-UNMENTIONED-STREAM-OPS
	  `(:INIT
	    :GET :GETL :GET-LOCATION :PUTPROP :REMPROP :PUSH-PROPERTY
	    :PLIST :PROPERTY-LIST :SET-PROPERTY-LIST :SETPLIST
	    :INFO :PATHNAME :GENERIC-PATHNAME :STATUS
	    :DELETE :RENAME :UNDELETE :EXPUNGE :OPEN :CHANGE-PROPERTIES
	    :ALL-DIRECTORIES :DIRECTORY-LIST :DIRECTORY-LIST-STREAM :PEEK-FILE-SYSTEM
	    :SET-BYTE-SIZE :BYTE-SIZE :CREATION-DATE :LENGTH
	    ;; SOME LMFILE-ONLY ONES.
	    :AUTHOR :PDP10-FORMAT :MUST-EXPLICITLY-CLOSE
	    :FORCE-CLOSE :NODE :ITS-DIRECTORY-STREAM
	    . ,STANDARD-STREAM-OPS))

;Takes filename and list of options as arguments.
;If the file can be opened, we set FILE-STREAM to a stream,
;and set FILE-DATA-OPCODE according to the byte size.
;Otherwise we leave FILE-STREAM nil and report failure,
;passing along the error message as our reply string.

;A successful open returns the values 'STREAM, the stream type,
;  a list of nonstandard operations supported,
;  the STREAM-PLIST, an unused value (the file's plist, if known, for compatibility),
;  our stream's PLIST, and a flag saying whether the data on the stream are characters.
;The last one is for compatibility, as the same info is now
;the :characters property on the stream-plist.
(DEFUN (OPEN FILE-COMMAND) (&OPTIONAL PATHNAME KEYWORD-ARGS &REST IGNORE
			    &AUX CHAOS-BYTE-SIZE
			    FILE-STREAM-SAVED)
  (AND FILE-STREAM (FILE-COMMAND-THROW-ERROR "XXX ERROR FAO F Another file already open"))
  ;; Override any :error keyword specified with T
  (PUTPROP (LOCF KEYWORD-ARGS) T ':ERROR)
  ;; Override any :AUTO-NEXT-NODE arg with NIL.
  (PUTPROP (LOCF KEYWORD-ARGS) NIL ':AUTO-NEXT-NODE)
  (SETQ FILE-STREAM-SAVED
	(LEXPR-FUNCALL 'OPEN PATHNAME KEYWORD-ARGS))
  (SETQ FILE-STREAM FILE-STREAM-SAVED)
  (SETQ CHAOS-BYTE-SIZE (FUNCALL FILE-STREAM ':BYTE-SIZE))
  (FUNCALL FILE-STREAM ':SET-NOTIFY-ON-CLOSE
	   (CLOSURE '(FILE-CONN)
		    #'(LAMBDA (IGNORE IGNORE)
			(CHAOS:CLOSE-CONN FILE-CONN "FC maintenance"))))
  (SETQ FILE-DATA-OPCODE
	(COND (( CHAOS-BYTE-SIZE 8)
	       FILE-CHARACTER-OPCODE)
	      (( CHAOS-BYTE-SIZE 16.)
	       FILE-BINARY-OPCODE)
	      (T (FILE-COMMAND-THROW-ERROR
		   (FORMAT NIL "XXX ERROR IBS F Byte size ~D. too large"
			   CHAOS-BYTE-SIZE)))))
  (SETQ FILE-STREAM-TYPE (DETERMINE-FILE-STREAM-TYPE FILE-STREAM))
  (SETQ FILE-TRANSMIT-DATA (GET FILE-STREAM-TYPE 'INITIAL-TRANSMIT-DATA))
  (IF (EQ FILE-STREAM-TYPE 'PROBE)
      (PROGN (CLOSE FILE-STREAM)
	     (SETQ FILE-STREAM NIL)))
  (VALUES 'STREAM FILE-STREAM-TYPE
	  (SUBSET #'(LAMBDA (ELT) (NOT (MEMQ ELT SERVER-UNMENTIONED-STREAM-OPS)))
		  (FUNCALL FILE-STREAM-SAVED ':WHICH-OPERATIONS))
	  (GET-STREAM-PLIST FILE-STREAM-SAVED FILE-STREAM-TYPE)
	  (OR (FUNCALL FILE-STREAM-SAVED ':SEND-IF-HANDLES ':FILE-CONTENTS-PLIST)
	      (FUNCALL FILE-STREAM-SAVED ':SEND-IF-HANDLES ':FILE-PLIST))  ;Obsolete name.
	  (FUNCALL FILE-STREAM-SAVED ':PLIST)
	  (FUNCALL FILE-STREAM-SAVED ':CHARACTERS)))

;OPEN-FOR-PDP10 returns only the data that the PDP10 wants to see,
;in a fixed order, so that the PDP10 doesn't have to manipulate plists.
;The order is  'STREAM, stream-type, creation date (univ. time as number),
;stream byte size, file-byte size, truename (a pathname),
;author (a string), qfaslp, length in bytes,
;reference date (universal time as number), backup-info (a string or NIL),
;ascii flag (T or NIL), pdp-10 format data flag (T or NIL).
;If unsuccessful, it returns ERROR and an error string.
(DEFUN (OPEN-FOR-PDP10 FILE-COMMAND) (&OPTIONAL PATHNAME OPTIONS &REST IGNORE)
  (MULTIPLE-VALUE-BIND (NIL STREAM-TYPE NIL STREAM-PLIST NIL PLIST)
         (IF (STRING-EQUAL (CAR (FUNCALL PATHNAME ':NAME)) ".FILE. (DIR)")
	     (OPEN-ITS-DIRECTORY PATHNAME OPTIONS)
	     (FUNCALL (GET 'OPEN 'FILE-COMMAND) PATHNAME OPTIONS))
    (SETQ FILE-TRANSMIT-DATA NIL)
    (VALUES 'STREAM STREAM-TYPE
	    (GET (LOCF PLIST) ':CREATION-DATE)
	    (GET (LOCF STREAM-PLIST) ':BYTE-SIZE)
	    (GET (LOCF PLIST) ':BYTE-SIZE)
	    (GET (LOCF STREAM-PLIST) ':TRUENAME)
	    (GET (LOCF PLIST) ':AUTHOR)
	    (GET (LOCF STREAM-PLIST) ':QFASLP)
	    (GET (LOCF PLIST) ':LENGTH-IN-BYTES)
	    (GET (LOCF PLIST) ':REFERENCE-DATE)
	    (GET (LOCF PLIST) ':DUMPED)
	    (GET (LOCF STREAM-PLIST) ':CHARACTERS)
	    (OR (GET (LOCF STREAM-PLIST) ':PDP10-FORMAT)
		(FUNCALL FILE-STREAM ':SEND-IF-HANDLES ':PDP10-FORMAT)))))

;Reading a directory, in ITS format.
(DEFUN OPEN-ITS-DIRECTORY (&OPTIONAL PATH OPTIONS)
  OPTIONS
  (LET (VAL)
    (SETQ PATH (FUNCALL PATH ':NEW-PATHNAME ':NAME ':WILD ':TYPE ':WILD ':VERSION ':WILD))
    (SETQ VAL (FUNCALL PATH ':ITS-DIRECTORY-STREAM T))
    (SETQ FILE-STREAM VAL)
    (SETQ FILE-STREAM-TYPE 'INPUT
	  FILE-DATA-OPCODE FILE-CHARACTER-OPCODE)
    (VALUES T 'INPUT '(:PDP10-FORMAT) '(:BYTE-SIZE 7 :CHARACTERS T :PDP10-FORMAT T)
	    () () T)))

;Cons up a stream-plist, which holds misc. info that is not
;on the file's plist (often because it depends on the stream
;and not just on the file that the stream is talking to).
;As of now, for compatibility, we put on some info that
;old versions of REMOTE expected to find here.
(DEFUN GET-STREAM-PLIST (STREAM STREAM-TYPE)
      `(:QFASLP ,(FUNCALL STREAM ':SEND-IF-HANDLES ':QFASLP)
	:CHARACTERS ,(FUNCALL STREAM ':SEND-IF-HANDLES ':CHARACTERS)
	:TRUENAME ,(FUNCALL STREAM ':SEND-IF-HANDLES ':TRUENAME)
	:BYTE-SIZE ,(FUNCALL STREAM ':SEND-IF-HANDLES ':BYTE-SIZE)
	:INFO ,(FUNCALL STREAM ':SEND-IF-HANDLES ':INFO)
	:LENGTH ,(FUNCALL STREAM ':SEND-IF-HANDLES ':LENGTH)
	:MUST-EXPLICITLY-CLOSE ,(IF (FUNCALL STREAM ':OPERATION-HANDLED-P ':MUST-EXPLICITLY-CLOSE)
				    (FUNCALL STREAM ':MUST-EXPLICITLY-CLOSE)
				    (EQ STREAM-TYPE 'OUTPUT))))

;Use the CLOSE command after transmitting the data of a file you are writing.
;If you are reading the file, don't bother; just drop the connection.
(DEFUN (CLOSE FILE-COMMAND) (&OPTIONAL ABORT-P &REST IGNORE)
  (COND (FILE-STREAM
	 (FUNCALL FILE-STREAM ':CLOSE ABORT-P)
	 (LET ((TEM FILE-STREAM))
	   (SETQ FILE-STREAM NIL)
	   (FILE-COMMAND-RETURN-VALUES
	     (FUNCALL TEM ':TRUENAME)
	     (FUNCALL TEM ':INFO))))
	(T (FILE-COMMAND-RETURN-VALUES))))

;Use the FINISH command to make sure that the data you have sent so far
;is in final form on the disk.  When you get the reply, all the
;data sent before the FINISH command is guaranteed to be there
;if the system crashes afterward.
(DEFUN (FINISH FILE-COMMAND) (&REST IGNORE)
  (REQUIRE-OPEN-FILE)
  (FUNCALL FILE-STREAM ':FINISH)
  T)

;No-op command for use in RFC packet when actual desired command
;is too long to fit in that packet.
(DEFUN (NIL FILE-COMMAND) (&REST IGNORE)
  T)

;Use the START-DATA-TRANSMISSION command to ask to receive
;the data of a file you are reading.
;The argument is either T, meaning transmit an unlimited amount,
;or a number, meaning transmit that many bytes
;(in addition to any bytes previously requested in this way).
(DEFUN (START-DATA-TRANSMISSION FILE-COMMAND) (&OPTIONAL N-BYTES &REST IGNORE)
  (REQUIRE-OPEN-FILE)
  (SETQ FILE-TRANSMIT-DATA
	(COND ((AND (NUMBERP N-BYTES) (PLUSP N-BYTES))
	       (+ N-BYTES (IF (NUMBERP FILE-TRANSMIT-DATA)
			      FILE-TRANSMIT-DATA
			      0)))
	      (T)))
  (*THROW 'COMMAND-NO-REPLY NIL))

;Use the STOP-DATA-TRANSMISSION command to tell the file server to
;stop transmitting the contents of the file.
;This does not get rid of the data in the pipeline already.
;You just have to discard it.
(DEFUN (STOP-DATA-TRANSMISSION FILE-COMMAND) (&REST IGNORE)
  (REQUIRE-OPEN-FILE)
  (SETQ FILE-TRANSMIT-DATA NIL)
  T)

(DEFUN (NEXT-NODE FILE-COMMAND) (&OPTIONAL PATHNAME &REST IGNORE)
  (REQUIRE-OPEN-FILE)
  (LET ((VAL (FUNCALL FILE-STREAM ':NEXT-NODE PATHNAME)))
    (IF (ERRORP VAL)
	(SIGNAL-CONDITION VAL)
      (FILE-COMMAND-RETURN-VALUES VAL))))

(DEFUN (DELETE FILE-COMMAND) (&OPTIONAL PATHNAME &REST IGNORE)
  (FILE-COMMAND-RETURN-VALUES
    (FUNCALL PATHNAME ':DELETE T)))

(DEFUN (DELETE-STREAM FILE-COMMAND) (&REST IGNORE)
  (REQUIRE-OPEN-FILE)
  (FILE-COMMAND-RETURN-VALUES
    (FUNCALL FILE-STREAM ':DELETE T)))

(DEFUN (RENAME FILE-COMMAND) (&OPTIONAL PATH1 PATH2 &REST IGNORE)
  (FUNCALL PATH1 ':RENAME PATH2)
  (FILE-COMMAND-RETURN-VALUES T))

(DEFUN (RENAME-STREAM FILE-COMMAND) (&OPTIONAL PATH &REST IGNORE)
  (REQUIRE-OPEN-FILE)
  (FUNCALL FILE-STREAM ':RENAME PATH)
  (FILE-COMMAND-RETURN-VALUES (FUNCALL FILE-STREAM ':TRUENAME)))

(DEFUN (MAKE-LINK FILE-COMMAND) (&OPTIONAL PATH1 PATH2 &REST IGNORE)
  (FUNCALL PATH1 ':MAKE-LINK PATH2)
  (FILE-COMMAND-RETURN-VALUES))

(DEFUN (UNDELETE FILE-COMMAND) (&OPTIONAL PATHNAME &REST IGNORE)
  (FILE-COMMAND-RETURN-VALUES (FUNCALL PATHNAME ':UNDELETE)))

(DEFUN (SET-POINTER FILE-COMMAND) (&OPTIONAL (NEW-POINTER 0) TRANSMIT-DATA &REST IGNORE)
  (REQUIRE-OPEN-FILE)
  (FUNCALL FILE-STREAM ':SET-POINTER NEW-POINTER)
  (SETQ FILE-TRANSMIT-DATA TRANSMIT-DATA)
  T)

(DEFUN (SET-BYTE-SIZE FILE-COMMAND) (&OPTIONAL NEW-BYTE-SIZE TRANSMIT-DATA &REST IGNORE)
  (REQUIRE-OPEN-FILE)
  (OR (AND (NUMBERP NEW-BYTE-SIZE)
	   (FIXP NEW-BYTE-SIZE)
	   (PLUSP NEW-BYTE-SIZE)
	   (<= NEW-BYTE-SIZE 16.))
      (FILE-COMMAND-THROW-ERROR
	(FORMAT NIL "XXX ERROR IBS F Byte size ~D. too large" NEW-BYTE-SIZE)))
  ;; Restore pointer to standard place.
  (FUNCALL FILE-STREAM ':SET-POINTER 0)
  (FUNCALL FILE-STREAM ':SET-BYTE-SIZE NEW-BYTE-SIZE)
  (SETQ FILE-DATA-OPCODE
	(COND ((<= NEW-BYTE-SIZE 8)
	       FILE-CHARACTER-OPCODE)
	      ((<= NEW-BYTE-SIZE 16.)
	       FILE-BINARY-OPCODE)))
  (SETQ FILE-TRANSMIT-DATA TRANSMIT-DATA)
  T)

;Handle a random stream operation.  The remote checks it against the
;WHICH-OPERATIONS list that we supplied at open-time, but this
;ought to be fixed to handle and report errors that occur
;in the execution of the operation.
(DEFUN (STREAM-OPERATION FILE-COMMAND) (&OPTIONAL OPERATION &REST ARGS)
  (REQUIRE-OPEN-FILE)
  (LEXPR-FUNCALL 'FILE-COMMAND-RETURN-VALUES
		 (MULTIPLE-VALUE-LIST (LEXPR-FUNCALL FILE-STREAM OPERATION ARGS))))

(DEFUN (DIRECTORY-STREAM FILE-COMMAND) (&OPTIONAL PATH OPTIONS &REST IGNORE)
  (IF (SYMBOLP OPTIONS) (SETQ OPTIONS (LIST OPTIONS)))
  (SETQ OPTIONS (DELQ ':NOERROR OPTIONS))
  (LET ((STR (OR (FUNCALL PATH ':SEND-IF-HANDLES ':DIRECTORY-LIST-STREAM
			  OPTIONS)
		 (MAKE-DIRECTORY-LIST-STREAM PATH OPTIONS))))
    (FILE-SEND-MESSAGE FILE-CONN FILE-REPLY-OPCODE (LIST 0 'DIRECTORY-STREAM 'STREAM))
    (LET ((CONN-STR (MAKE-INSTANCE 'CHAOS-PATHNAME-PRINTING-STREAM ':CONNECTION FILE-CONN)))
      (LOOP FOR ELEMENT = (FUNCALL STR ':ENTRY)
	    UNTIL (NULL ELEMENT)
	    DO
	    (PROGN
	      (FUNCALL CONN-STR ':LINE-OUT
		       (IF (CAR ELEMENT) (FUNCALL (CAR ELEMENT) ':STRING-FOR-HOST) ""))
	      (DO ((NEXT (CDR ELEMENT) (CDDR NEXT))) ((NULL NEXT))
		(FUNCALL CONN-STR ':STRING-OUT (STRING (CAR NEXT)))
		(TYO #/  CONN-STR)
		(DO ((XX *KNOWN-DIRECTORY-PROPERTIES* (CDR XX)))
		    ((NULL XX) (PRINT-RANDOM-SEXP (CADR NEXT) CONN-STR))
		  (IF (MEMQ (CAR NEXT) (CDR (CAR XX)))
		      (RETURN (FUNCALL (CADAR (CAR XX)) (CADR NEXT) CONN-STR))))
		(TERPRI CONN-STR))
	      (FUNCALL CONN-STR ':LINE-OUT "")))
      (FUNCALL CONN-STR ':FINISH)
      (FILE-SEND-MESSAGE FILE-CONN FILE-EOF-OPCODE NIL)
      (CHAOS:FINISH FILE-CONN))))

(DEFUN MAKE-DIRECTORY-LIST-STREAM (PATH OPTIONS)
  (LET ((DIRLIST (FUNCALL PATH ':DIRECTORY-LIST OPTIONS)))
    (IF (ERRORP DIRLIST)
	DIRLIST
      (LET-CLOSED ((DIRLIST1 DIRLIST))
	#'(LAMBDA (&OPTIONAL OP IGNORE)
	    (COND ((EQ OP ':CLOSE) NIL)
		  (T (POP DIRLIST1))))))))

(DEFFLAVOR CHAOS-PATHNAME-PRINTING-STREAM ()
	   (CHAOS:OUTPUT-CHARACTER-STREAM))

(DEFMETHOD (CHAOS-PATHNAME-PRINTING-STREAM :PRINT) (ARG1 &REST REST)
  (COND ((CONSP ARG1)
	 (SI:PRINT-LIST ARG1 (FIRST REST)
			SELF '(:TYO :PRINT))
	 T)
	((TYPEP ARG1 'PATHNAME)
	 (LET ((READTABLE FILE-READTABLE))
	   (FUNCALL-SELF ':TYO #/#)
	   (FUNCALL-SELF ':TYO #/!)
	   (PRIN1 (OR SERVER-LOCAL-HOST
		      (FUNCALL (FUNCALL ARG1 ':HOST) ':NAME))
		  SELF)
	   (SI:PRINT-OBJECT (FUNCALL ARG1 ':STRING-FOR-HOST)
			    (FIRST REST) 
			    SELF NIL))
	 T)
	((AND (SYMBOLP ARG1)
	      USE-STANDARD-SYMBOL-ENCODING
	      (SETQ TEM (FIND-POSITION-IN-LIST ARG1 STANDARD-SYMBOL-OUTPUT-LIST)))
	 (LET ((READTABLE FILE-READTABLE))
	   (FUNCALL-SELF ':TYO #/#)
	   (FUNCALL-SELF ':TYO #/?)
	   (PRIN1 TEM SELF))
	 T)
	))

(COMMENT ;NOT USED FOR NOW
(DEFUN (DIRECTORY-LIST FILE-COMMAND) (&OPTIONAL PATH OPTIONS
				      &REST IGNORE)
  (FILE-COMMAND-RETURN-VALUES
    (FUNCALL PATH ':DIRECTORY-LIST (DELQ ':NOERROR OPTIONS))))

(DEFUN (DIRECTORY-LIST-STREAM FILE-COMMAND) (&OPTIONAL PATH OPTIONS &REST IGNORE)
  (LET ((STR (FUNCALL PATH ':DIRECTORY-LIST-STREAM (DELQ ':NOERROR OPTIONS))))
    (FILE-SEND-MESSAGE FILE-CONN FILE-REPLY-OPCODE (LIST 0 'DIRECTORY-LIST-STREAM 'STREAM))
    (LET ((CONN-STR (MAKE-INSTANCE 'CHAOS-PATHNAME-PRINTING-STREAM ':CONNECTION FILE-CONN)))
      (LOOP FOR ELEMENT = (FUNCALL STR)
	    UNTIL (NULL ELEMENT)
	    DO (FILE-BIND-FOR-WRITE (PRINT ELEMENT CONN-STR)))
      (FUNCALL CONN-STR ':FINISH)
      (FILE-SEND-MESSAGE FILE-CONN FILE-EOF-OPCODE NIL)
      (CHAOS:FINISH-CONN FILE-CONN))))
)

(DEFUN (EXPUNGE FILE-COMMAND) (&OPTIONAL PATH &REST IGNORE)
  (FILE-COMMAND-RETURN-VALUES (FUNCALL PATH ':EXPUNGE)))

(DEFUN (HOMEDIR FILE-COMMAND) (&OPTIONAL PATH &REST IGNORE)
  (FILE-COMMAND-RETURN-VALUES (USER-HOMEDIR (FUNCALL PATH ':HOST))))

(DEFUN (ALL-DIRECTORIES FILE-COMMAND) (&OPTIONAL PATH &REST IGNORE)
  (FILE-COMMAND-RETURN-VALUES (ALL-DIRECTORIES PATH)))

(DEFUN (MULTIPLE-FILE-PROPERTY-LISTS FILE-COMMAND) (&OPTIONAL PATH PATHNAMES OPTIONS
						    &REST IGNORE)
  (FILE-COMMAND-RETURN-VALUES
    (FUNCALL PATH ':MULTIPLE-FILE-PLISTS PATHNAMES OPTIONS)))

(DEFUN (PROPERTIES FILE-COMMAND) (&OPTIONAL PATH ERROR-P)
  ERROR-P
  (FILE-COMMAND-RETURN-VALUES (FUNCALL PATH ':PROPERTIES)))

(DEFUN (CHANGE-PROPERTIES FILE-COMMAND) (&OPTIONAL PATH ERROR-P PROPERTIES)
  ERROR-P
  (LEXPR-FUNCALL PATH ':CHANGE-PROPERTIES NIL PROPERTIES))

(DEFUN (DELETE-MULTIPLE-FILES FILE-COMMAND) (&OPTIONAL PATH PATHNAMES &REST IGNORE)
  (FILE-COMMAND-RETURN-VALUES
    (FUNCALL PATH ':DELETE-MULTIPLE-FILES NIL PATHNAMES)))

(DEFUN (UNDELETE-MULTIPLE-FILES FILE-COMMAND) (&OPTIONAL PATH PATHNAMES &REST IGNORE)
  (FILE-COMMAND-RETURN-VALUES
    (FUNCALL PATH ':UNDELETE-MULTIPLE-FILES NIL PATHNAMES)))

;Return a list (completed-string success-flag) to the remote pathname.
(DEFUN (COMPLETE-STRING FILE-COMMAND) (PATH STRING OPTIONS)
  (MULTIPLE-VALUE-BIND (STRING FLAG)
      (FUNCALL PATH ':COMPLETE-STRING STRING OPTIONS)
    (FILE-COMMAND-RETURN-VALUES
      (LIST
	(LET ((HOSTEND (STRING-SEARCH-CHAR #/: STRING)))
	  (IF HOSTEND
	      (SUBSTRING STRING (+ 2 HOSTEND))
	      STRING))
	FLAG))))


(DEFUN PRINT-SERVER-CONNECTIONS ()
  (LOOP FOR ELEM IN LMFILE-SERVER-CONNECTION-RECORD DOING
	(LEXPR-FUNCALL #'FORMAT T "~%~A at ~A~40T~A" ELEM)))

;; Send a message to all people who have used the file computer within MINUTES minutes.
;; If MINUTES is not a number, all users since cold-booting are notified.
(DEFUN FILE-QSEND (&OPTIONAL (MINUTES 60.))
  (LET* ((CURRENT-TIME (TIME:GET-UNIVERSAL-TIME))
	 (CUTOFF-TIME (IF (NUMBERP MINUTES)
			  (- CURRENT-TIME (* 60. MINUTES))
			0))
	 (MESSAGE (PROGN (FORMAT T "~&Message (Type End to send it):~%")
			 (ZWEI:QSEND-GET-MESSAGE))))
    (LOOP FOR (USER HOST TIME) IN FS:LMFILE-SERVER-CONNECTION-RECORD
	  DO (SETQ TIME (TIME:PARSE-UNIVERSAL-TIME TIME))
	  UNTIL (< TIME CUTOFF-TIME)
	  WHEN (NOT (MEMBER USER USERS-NOTIFIED))
	  COLLECT USER INTO USERS-NOTIFIED
	  AND COLLECT (LIST USER HOST (CHAOS:OPEN-STREAM HOST (STRING-APPEND "SEND " USER)
							 ':DIRECTION ':OUTPUT
							 ':ERROR NIL))
	  INTO STREAMS
	  FINALLY (LOOP FOR (USER HOST STREAM) IN STREAMS
			DO (FORMAT T "~%~A@~A:  " USER HOST)
			WHEN (STRINGP STREAM)
			  DO (FUNCALL STANDARD-OUTPUT ':STRING-OUT STREAM)
			ELSE DO (FORMAT STREAM "Notice from the FileComputer:~2%~A" MESSAGE)
			        (FUNCALL STREAM ':CLOSE)
				(FUNCALL STANDARD-OUTPUT ':STRING-OUT
					 "Message sent successfully")))))

(GLOBALIZE 'FILE-QSEND)
(GLOBALIZE 'PRINT-SERVER-CONNECTIONS)


;Function for bringing up the FC machine.

;NOTE: this is called, with argument T, for both warm and cold boots;
;but warm booting is handled elsewhere,
;so we don't really want to do anything for them.
;So, we do nothing if FOR-WARM-BOOT is T and FILE-SYSTEM-RUNNING is T.
;For a cold boot, FILE-SYSTEM-RUNNING will be NIL.
;It would be simpler to put this function on the cold init list,
;but it cannot work there, because it wants to access files on AI.

(defvar user-timeout (* 60. 60. 5.)
  "Time to wait before doing the canonical things if the user doesn't answer the question.")

(defvar *additional-filecomputer-server-files-loaded* ()
  "Hook to load up additional modules (more servers, etc.)")

(add-initialization "Reset files-loaded flag"
		    '(setq *additional-filecomputer-server-files-loaded* ())
		    ':before-cold)

(DEFUN BRING-UP-SERVER (&OPTIONAL FOR-WARM-BOOT)
  (DISABLE-FILE-SERVER)
  (DISABLE-MAIL-SERVER)
  (AND FILE-SYSTEM-RUNNING (NOT FOR-WARM-BOOT)
       (STOP-FILE-SYSTEM))
  (COND ((AND (NOT FILE-SYSTEM-RUNNING)
	      (OR (NOT FOR-WARM-BOOT)
		  (SI:WITH-TIMEOUT
		    (user-timeout #'(lambda ()
				      (format t "~&No answer after 5 minutes, Bring up Server.~&")
				      t)) ;;return t
		    (y-or-n-p "Bring up LMFile? "))))
	 (COND ((CHAOS:HOST-UP-P (or (si:get-site-option ':sys-host) si:associated-machine))
		(SI:WITH-SYS-HOST-ACCESSIBLE
		  (LOAD-PATCHES ':NOSELECTIVE)
		  (or *additional-filecomputer-server-files-loaded*
		      (condition-case (result) (probef "SYS: FILE2; EXTRA-FILES LISP")
			(fs:file-error t)
			(:no-error (not result)))
		      (format t "~&Loading more modules into server.~%") ; Returns ()
		      (progn (load "SYS: FILE2; EXTRA-FILES LISP" "USER" () T T)
			     (setq *additional-filecomputer-server-files-loaded* t))))
		)
	       (T (FORMAT T "~%Cannot connect to SYS host ~A to load the latest patches.
When it is up again, please call BRING-UP-SERVER."  (si:get-site-option ':sys-host))))

	 ;;Note: at a WARM boot, as opposed to a cold boot,
	 ;;this does not run, but LMFILE-WARM-BOOT handles everything.
	 (FORMAT T "~%Booting LMFILE on partition SRVR...")
	 (FS:CONSIDER-UNIT 2 "SRVR")
	 (FS:START-FILE-SYSTEM)
	 (UNLESS (GET 'LMFILE-CHAOS-HOST 'SI:FLAVOR)
	   (FS:ADD-LFS-HOST "FC"))
	 (FORMAT T "~%LMFILE booted and running.")))

  (COND (FILE-SYSTEM-RUNNING
	 (ENABLE-FILE-SERVER)
	 (ENABLE-MAIL-SERVER)))
	 
  ;; Random initializations.
  ;; Allow typeout to continue past the end of screen without hanging.
  (SETQ TV:MORE-PROCESSING-GLOBAL-ENABLE NIL)
  (SETQ USER-ID "LMFile")
  (fs:file-host-user-id "LMFILE" (si:parse-host "mc"))
  (setq fs:user-personal-name-first-name-first "The AI File Server")

  (PKG-GOTO 'FS))

(GLOBALIZE 'BRING-UP-SERVER)

(DEFUN FILE-DISPLAY ()
  (FUNCALL STANDARD-OUTPUT ':CLEAR-SCREEN)
  (SI:PRINT-HERALD)
  (FORMAT T "~%~%MIT AI Lab File Computer using this console.~%
Please use another LispMachine if one is available.")
  (TIME:PRINT-CURRENT-DATE)
  (FORMAT T "~%~%")
  T)

(GLOBALIZE 'FILE-DISPLAY)

(add-initialization "start up LMFILE" '(bring-up-server T) '(WARM))

