;;; -*- Mode:LISP; Package:FS;Base:8 -*-
;;; Tapemaster Tape Drive Controller support.
;;; Documentation for the controller can be found in Computer Products Corporation
;;; document #21010011. This code is based on the July 81 (C) edition.
;;; (c) 1983 Lisp Machine Incorporated
;;; This file contains the command and buffer transfer functions
;;;
;;; This version does not do anything tricky with DMA.
;;; Nor does it support separate processes using the same controller.

;;; And now, finally, for some functions that actually DO something . . .
;;; Hardware path dependent functions.  ``Real'' microcoded ones are in COMPILER:
;;; Compile this on a Lambda for best results

(eval-when (load)
  (if-in-cadr (ferror () "This won't work on the CADR."))
  )

(defun =multibus-read-16 (addr)
  (%logdpb (compiler:%multibus-read-8 (1+ addr)) 1010 (compiler:%multibus-read-8 addr)))

(defun =multibus-write-16 (addr data)
  (compiler:%multibus-write-8 addr (%logldb 0010 data))
  (compiler:%multibus-write-8 (1+ addr) (%logldb 1010 data)))

(defun %multibus-io-write-8 (addr data)
  "Works only for small addresses (no mapping done), and writes only a byte."
  (compiler:%io-space-write (+ 377000 addr) data) ; *** this doesn't work, ask RG
  )

(defsubst %multibus-write-16-x2 (addr low-data high-data)
  "Uses COMPILER:%MULTIBUS-WRITE-32"
  (compiler:%multibus-write-32 addr (dpb high-data 2020 low-data)))

(defun tapemaster-read-iopb-16 (address-within-iopb)
  (=multibus-read-16 (tm+ :multibus-iopb address-within-iopb)))

(defun tapemaster-write-iopb-16 (address-within-iopb data)
  (=multibus-write-16 (tm+ :multibus-iopb address-within-iopb) data))

;;; Less hardware dependent, mostly command level
(defSUBST tapemaster-status ()
  (tapemaster-read-iopb-16 %tapemaster-status))

(defsubst tapemaster-return-count ()
  (tapemaster-read-iopb-16 %tapemaster-return-count))

(defun tapemaster-unit-status (unit)
  (tapemaster-execute-command tapemaster-drive-status
			      (dpb unit %%tapemaster-tape-select 0)
			      0
			      0
			      T
			      ())
  (tapemaster-status))

(DEFUN TAPEMASTER-DETERMINE-BLOCK-SIZE (UNIT)
  (TAPEMASTER-EXECUTE-COMMAND TAPEMASTER-READ-FOREIGN-TAPE (TAPEMASTER-UNIT-FIELD UNIT) 0 0 T NIL)
  (LOGIOR
    (TAPEMASTER-READ-IOPB-16 %TAPEMASTER-RETURN-COUNT)
    (ASH (TAPEMASTER-READ-IOPB-16 %TAPEMASTER-BUFFER-SIZE) 20)))

(defun tm-print-unit-status (&optional (unit 0))
  (tapemaster-print-status (tapemaster-unit-status unit)))

;DO NOT USE THIS AS A PROCESS-WAIT FUNCTION!!
(defun tapemaster-unit-ready (unit)
  (let ((status  (tapemaster-unit-status unit)))
    (and (ldb-test %%tapemaster-ready status) (ldb-test %%tapemaster-online status))))
       

(defun tapemaster-controller-ready ()
  (zerop (%multibus-read-8 (tm+ :channel-control-block 1)))) ; just multibus ??

(defun tapemaster-wait-controller-ready-with-timeout (time)
  "Returns non-() if the controller is ready after TIME."
  (process-wait-with-timeout "MagTape" time 'tapemaster-controller-ready))

(defun tapemaster-wait-controller-ready ()
  (process-wait *wait-controller-whostate* 'tapemaster-controller-ready))

(defun tapemaster-wait-unit-ready (&optional (unit 0))
  ;THIS CANNOT BE DONE WITH A PROCESS-WAIT, BECAUSE THE COMMAND MUST ACTUALLY BE ISSUED REPEATEDLY
  ; TO THE CONTROLLER.  IT DOES NOT WIN TO TRY TO ISSUE THE COMMAND FROM THE SCHEDULER LEVEL!!
  (do () (())
    (cond ((tapemaster-unit-ready unit)
	   (return t)))
    (process-sleep 5 "MT Unit")))

;; (deff wait-ready 'tapemaster-wait-controller-ready)
;; (deff wait-unit-ready 'tapemaster-wait-unit-ready)

;;; This seems to take up too much of the processor -- it should be changed to use
;;; PROCESS-WAIT

(DEFUN TAPEMASTER-WAIT-CONTROLLER-RESTARTABLE
       (TIMEOUT message)
  (ERROR-RESTART (GATE-TIME-OUT message)
    (if (not (tapemaster-wait-controller-ready-with-timeout (* timeout 60.)))
	(FERROR 'GATE-TIME-OUT
		"Tapemaster command failed to finish after ~D seconds" timeout))))


(defun tapemaster-build-iopb (command control buffer-size buffer-address)
  (tapemaster-write-iopb-16 %tapemaster-command command)
  (tapemaster-write-iopb-16 %tapemaster-command-zero 0)	;high word of command
  (tapemaster-write-iopb-16 %tapemaster-control
			    (DPB 1 %%TAPEMASTER-WIDTH control))
  (tapemaster-write-iopb-16 %tapemaster-return-count 0)	;return count
  (tapemaster-write-iopb-16 %tapemaster-buffer-size buffer-size)
  (tapemaster-write-iopb-16 %tapemaster-records 0)		;records/overrun
  (tapemaster-write-iopb-16 %tapemaster-source buffer-address);low 16 bits of buffer-address
  (tapemaster-write-iopb-16 %tapemaster-source-high
			    (dpb (ldb 2004 buffer-address) ;high 4 bits of buffer address
				 1404
				 0))
  (tapemaster-write-iopb-16 %tapemaster-status 0)		;status
  (tapemaster-write-iopb-16 %tapemaster-link 0)		;first word of link field
  (tapemaster-write-iopb-16 %tapemaster-link-high 0)	;second word of link field
  )

(DEFUN TAPEMASTER-PRINT-CONTROL (CONTROL &OPTIONAL (S TERMINAL-IO))
  (FORMAT S "~&Tape Select ~D, Bank Select ~D, ~[Forward~;Reverse~] mode."
	  (LDB %%TAPEMASTER-TAPE-SELECT CONTROL)
	  (LDB %%TAPEMASTER-BANK-SELECT CONTROL)
	  (LDB %%TAPEMASTER-REVERSE CONTROL))
  (FORMAT S "~&Interrupt type: ~[non-vectored~;mailbox~]." (LDB %%TAPEMASTER-MAILBOX CONTROL))
  (FORMAT S "~&Interrupts ~[off~;on~], Link Bit ~[off~;on~], Bus ~[un~;~]locked."
	  (LDB %%TAPEMASTER-INTERRUPTS CONTROL)
	  (LDB %%TAPEMASTER-LINK CONTROL)
	  (LDB %%TAPEMASTER-BUS-LOCK CONTROL))
  (FORMAT S "~&Speed (Density): ~[low (high)~;high (low)~], with ~[non-~;~]continous movement."
	  (LDB %%TAPEMASTER-DENSITY CONTROL)
	  (LDB %%TAPEMASTER-CONTINUOUS CONTROL))
  (FORMAT S "~&Bus Width: ~D." (IF (LDB-TEST %%TAPEMASTER-WIDTH CONTROL) 16. 8.)))

(DEFUN TAPEMASTER-PRINT-STATUS (STATUS &OPTIONAL (S TERMINAL-IO))
  (FORMAT S "~&Write ring is ~[there~;not there~]." (ldb %%tapemaster-write-protect status))
  (FORMAT S "~&Formatter is ~[idle~;busy~]." (ldb %%tapemaster-ready status))
  (FORMAT S "~&The selected drive is ~[~;not ~]ready and "
	  (ldb %%tapemaster-formatter-busy status))
  (IF (NOT (ldb-TEST %%tapemaster-online status))
      (SEND S ':STRING-OUT "offline.")
    (SEND S ':STRING-OUT "online, ")
    (if (ldb-test %%tapemaster-end-of-tape status) (FORMAT S "at the end of tape."))
    (if (ldb-test %%tapemaster-load-point status) (FORMAT S "at beginning of tape.")))
  (if (ldb-test %%tapemaster-file-mark status)
      (FORMAT S "~&A file mark was detected on this operation"))
  (let ((error-number (ldb %%tapemaster-error status)))
    (cond ((not (zerop error-number))
	   (FORMAT S "~%ERROR ~D.: ~A" ERROR-NUMBER (TAPEMASTER-ERROR-AS-STRING ERROR-NUMBER)))))
  (IF (LDB-TEST %%TAPEMASTER-RETRY STATUS)
      (FORMAT S "~&This command had to be retried."))
  (IF (LDB-TEST %%TAPEMASTER-COMPLETE STATUS)
      (FORMAT S "~&Command complete."))
  (if (ldb-test %%tapemaster-entered status)
      (FORMAT S "~&Command received by controller.")))

(defun tapemaster-print-iopb (&optional (s terminal-io))
  (FORMAT S "~&Command: ~A"
	  (tapemaster-command-as-string (tapemaster-read-iopb-16 %tapemaster-command)))
  (cond ((not (zerop (tapemaster-read-iopb-16 %tapemaster-command-zero)))
	 (FORMAT S "~&WARNING: Second word of command not zero")))
  (tapemaster-print-control (tapemaster-read-iopb-16 %tapemaster-control) s)
  (FORMAT S "~&Return count: ~O~:* (~D.)"
	  (TAPEMASTER-READ-IOPB-16 %TAPEMASTER-RETURN-COUNT))
  (FORMAT S "~&Buffer size: ~O~:* (~D.)"
	  (TAPEMASTER-READ-IOPB-16 %TAPEMASTER-BUFFER-SIZE))
  (FORMAT S "~&Records//Overrun ~O~:* (~D.)"
	  (TAPEMASTER-READ-IOPB-16 %TAPEMASTER-RECORDS))
  (FORMAT S "~&Buffer Address: ~O" (+ (tapemaster-read-iopb-16 %tapemaster-source)
				      (dpb (tapemaster-read-iopb-16
						    %tapemaster-source-high)
					   0420
					   0)))
  (tapemaster-print-status (tapemaster-read-iopb-16 %tapemaster-status) s)
  (IF (OR (NOT (ZEROP (TAPEMASTER-READ-IOPB-16 %TAPEMASTER-LINK)))
	  (NOT (ZEROP (TAPEMASTER-READ-IOPB-16 %TAPEMASTER-LINK-HIGH))))
      (FORMAT S "~&WARNING: Interrupt//Link field is not zero")))

(DEFUN TAPEMASTER-INITIALIZE (&optional quiet-p)
  ; Set system configuration-pointer
  (%MULTIBUS-IO-WRITE-8 (TM+ :IO 1) 0)	;reset
  (COMPILER:%MULTIBUS-WRITE-8 tapemaster-system-configuration-pointer-address 1);a 16 bit bus
  (COMPILER:%MULTIBUS-WRITE-8 (tm+ :configuration-pointer 1) 0)	;unused byte
  (COMPILER:%MULTIBUS-WRITE-8 (tm+ :configuration-pointer 2)
		       (ldb 0010 tapemaster-system-configuration-block-address));low 8 bits
  (COMPILER:%MULTIBUS-WRITE-8 (tm+ :configuration-pointer 3)
		       (ldb 1010 tapemaster-system-configuration-block-address));next 8 bits
  (COMPILER:%MULTIBUS-WRITE-8 (tm+ :configuration-pointer 4)
		       (ldb 2010 tapemaster-system-configuration-block-address))
  (COMPILER:%MULTIBUS-WRITE-8 (tm+ :configuration-pointer 5)
		       (ldb 3010 tapemaster-system-configuration-block-address))
  ;set up system configuration block
  (=MULTIBUS-WRITE-16 tapemaster-system-configuration-block-address 3)
  (=MULTIBUS-WRITE-16 (tm+ :configuration-block 2)
		     tapemaster-channel-control-block-address)
  (=MULTIBUS-WRITE-16 (tm+ :configuration-block 4)
		     (ldb 2020 tapemaster-channel-control-block-address))
  ;set up channel control block
  (=MULTIBUS-WRITE-16 tapemaster-channel-control-block-address #xff11) ; ccw and gate
  (tapemaster-channel-attention)		;Send channel attention
  (tapemaster-wait-controller-restartable 500 "Keep waiting for init to finish")
  (unless quiet-p (format t "~&Now configuring...~%"))
  (tapemaster-execute-command tapemaster-configure 0 0 0 T NIL)
  (unless quiet-p (tapemaster-print-iopb standard-output))
  t)

;;; Wakeup the controller at warm boot.  (Should be modified when a shared peripheral
;;; scheme is hacked).
(comment ; Let pace take a look at this
(add-initialization "Initialize TapeMaster Controller" '(tapemaster-initialize t))
(defsubst tapemaster-channel-attention ()
  (%multibus-io-write-8 tapemaster-io-address 0))
)

;;; Strategies for execute commands (look at SYS:FILE;MTDEFS)
;;; [1] A low level function that waits until the controller is ready, and then writes the
;;; command to the controller hardware.
;;; [2] Two functions that transfer data via RQBs, using something like MT-RUN.
;;; [3] A function for spacing operations, something like MT-RUN-SIMPLE.

(defun tapemaster-execute-command (command control buffer-size buffer-address
				   &optional hang-p (wait-unit-p t))
  (and wait-unit-p (tapemaster-wait-unit-ready (ldb %%tapemaster-tape-select control)))
  (TAPEMASTER-WAIT-CONTROLLER-RESTARTABLE 3 "Keep waiting for gate to open")
  (COMPILER:%MULTIBUS-WRITE-8 (tm+ :channel-control-block 1) #xff)
  (tapemaster-build-iopb command control buffer-size buffer-address)
  (=multibus-write-16 (tm+ :channel-control-block 2)
		      tapemaster-multibus-iopb-address)
  (=multibus-write-16 (tm+ :channel-control-block 4) 0)
  (tapemaster-channel-attention)
  (and (or hang-p *await-command-completion*)
       (tapemaster-wait-controller-restartable
	 (if (eq command tapemaster-rewind) 120. 5.)
	 "keep waiting for tape command to finish")))

(DEFUN TAPEMASTER-EXECUTE-SIMPLE-COMMAND
       (COMMAND CONTROL COUNT REVERSE-BIT UNIT &OPTIONAL RQB)
  "Useful for tape motion.  If RQB is given, then store the resulting IOPB in that."
  (tapemaster-wait-unit-ready unit)
  (TAPEMASTER-WAIT-CONTROLLER-READY)
  (TAPEMASTER-WRITE-IOPB-16 %TAPEMASTER-COMMAND COMMAND)
  (TAPEMASTER-WRITE-IOPB-16 %TAPEMASTER-CONTROL
			    (DPB REVERSE-BIT %%TAPEMASTER-REVERSE
				 (DPB UNIT %%TAPEMASTER-TAPE-SELECT CONTROL)))
  (TAPEMASTER-WRITE-IOPB-16 %TAPEMASTER-RECORDS COUNT)
  (tapemaster-write-iopb-16 %tapemaster-command-zero 0)	;high word of command
  (tapemaster-write-iopb-16 %tapemaster-return-count 0)	;return count
  (tapemaster-write-iopb-16 %tapemaster-buffer-size 0)
  (tapemaster-write-iopb-16 %tapemaster-source 0)
  (tapemaster-write-iopb-16 %tapemaster-source-high 0)
  (tapemaster-write-iopb-16 %tapemaster-status 0)		;status
  (tapemaster-write-iopb-16 %tapemaster-link 0)		;first word of link field
  (tapemaster-write-iopb-16 %tapemaster-link-high 0)	;second word of link field
  (tapemaster-channel-attention)
  (TAPEMASTER-WAIT-CONTROLLER-READY)
  (WHEN RQB (tm-aset-iopb rqb))) ; get status info
