;;; -*- Mode: Lisp; Package: ZWEI -*-
;;; ** (c) Copyright 1980 Massachusetts Institute of Technology **

;;; This file contains the interface between the editor and the screen system.
;;; A ZWEI-window is the editor's data structure.  A window-sheet is the screen
;;; system's active data structure.

;;; Most of the functions in this file do not run in the regular editor
;;; environment and may not use the editor global variables.  The sheet
;;; methods are only allowed to touch the "wrapper" of a window: the outline,
;;; the label, and the blinkers.  Any operation involving the window per se
;;; or other editor data structures is done by sending a request to the
;;; editor process that owns the window.  A request is a "character" in
;;; the input stream which is actually a list.  There is a slight exception
;;; to this in changing the edges of a window, the ZWEI-window data-structure is
;;; altered immediately.

;;; The following are the requests to the editor process.  Note that if these
;;; are seen by other than the command-loop, some should be ignored and some should
;;; be executed anyway.
;;; (REDISPLAY) - wakes up the editor process so that it will do a redisplay.
;;;	The degree is not passed in this request, but stored directly into the
;;;	window, so that if the editor does a redisplay on its own before processing
;;;	the request the right things will happen.  If the window system tells the window
;;;	to redisplay with DIS-ALL, we want to make sure the window knows it is munged
;;;	if the editor displays on it, and we want to avoid redisplaying twice.
;;; (SELECT-WINDOW window) - causes Zmacs to select this window as the current window.
;;;	This is used to reflect window-selection by mouse to the editor, mainly
;;;	for the case of one editor that knows about more than one window.
;;; (SCROLL window line-number) - put line-number line of the buffer at the top of the window.
;;;	The argument is not guaranteed to be in range.
;;;	This does not deal in continuation lines.
;;; (MOUSE window mouse-char window-relative-X window-relative-Y) - mouse command.

;;; Modifier to certain methods telling them not to send a request back to the editor.
(DEFVAR *EDITOR-ALREADY-KNOWS* NIL)

;;; Functions called by the Editor

;;; Called when a window is selected by editor command, to inform the
;;; sheet.  Note that this must NOT send a SELECT-WINDOW request back to
;;; the editor!  (Consider interaction with macros, type-ahead)
(DEFUN SELECT-WINDOW (ZWEI-WINDOW &AUX SHEET)
  (SETQ SHEET (WINDOW-SHEET ZWEI-WINDOW))
  (COND ((EQ ZWEI-WINDOW *MINI-BUFFER-WINDOW*)
	 (OR (MEMQ ZWEI-WINDOW *WINDOW-LIST*)
	     (PUSH ZWEI-WINDOW *WINDOW-LIST*))))
  (AND (NEQ SHEET TV:SELECTED-WINDOW)
       (LET ((*EDITOR-ALREADY-KNOWS* T))
	 (FUNCALL SHEET ':SELECT)))
  (COMTAB-MOUSE-PROMPT *COMTAB* (WINDOW-WHO-LINE-DOCUMENTATION-STRING ZWEI-WINDOW)))

;;; Called when the label of the window might have changed
(DEFUN CHANGE-WINDOW-LABEL (ZWEI-WINDOW)
  (FUNCALL (WINDOW-SHEET ZWEI-WINDOW) ':CHANGE-LABEL
	   (BUFFER-NAME (WINDOW-INTERVAL ZWEI-WINDOW))))

(DEFUN CLOBBER-WINDOW-LABEL (ZWEI-WINDOW &AUX (SHEET (WINDOW-SHEET ZWEI-WINDOW)))
  (CHANGE-WINDOW-LABEL ZWEI-WINDOW)
  (LET ((LINES-HEIGHT (* (TV:SHEET-LINE-HEIGHT SHEET) (WINDOW-N-PLINES ZWEI-WINDOW)))
	(INSIDE-HEIGHT (TV:SHEET-INSIDE-HEIGHT SHEET)))
      (OR (= LINES-HEIGHT INSIDE-HEIGHT)
	  (TV:SHEET-FORCE-ACCESS (SHEET)
	    (TV:%DRAW-RECTANGLE (TV:SHEET-INSIDE-WIDTH SHEET) (- INSIDE-HEIGHT LINES-HEIGHT)
				(TV:SHEET-INSIDE-LEFT SHEET)
				(+ (TV:SHEET-INSIDE-TOP SHEET) LINES-HEIGHT)
				(TV:SHEET-ERASE-ALUF SHEET) SHEET)))))

;;; Call before redisplaying a window.
;;; This deactivates the window's typeout stream.
(DEFUN PREPARE-WINDOW-FOR-REDISPLAY (ZWEI-WINDOW)
  (LET ((SHEET (WINDOW-SHEET ZWEI-WINDOW))
	(*EDITOR-ALREADY-KNOWS* T))
    (OR (TV:SHEET-EXPOSED-P SHEET)
	(FUNCALL SHEET ':EXPOSE))
    (LET ((TYPEOUT-WINDOW (TV:ESSENTIAL-WINDOW-WITH-TYPEOUT-MIXIN-TYPEOUT-WINDOW SHEET)))
      (COND ((TV:BASIC-TYPEOUT-WINDOW-BOTTOM-REACHED TYPEOUT-WINDOW)
	     (LET ((BOTTOM-REACHED (FUNCALL TYPEOUT-WINDOW ':BOTTOM-REACHED))
		   (N-PLINES (WINDOW-N-PLINES ZWEI-WINDOW))
		   (LINE-HEIGHT (TV:SHEET-LINE-HEIGHT SHEET)))
	       (FUNCALL TYPEOUT-WINDOW ':DEACTIVATE)
	       (LET ((N-PLINES-CLOBBERED (MIN (1+ (// (+ BOTTOM-REACHED (1- LINE-HEIGHT))
						      LINE-HEIGHT))
					      N-PLINES)))
		 (SETF (WINDOW-REDISPLAY-DEGREE ZWEI-WINDOW)
		       (MAX DIS-TEXT (WINDOW-REDISPLAY-DEGREE ZWEI-WINDOW)))
		 ;; Force redisplay of he lines that were used for typeout.
		 (DOTIMES (I N-PLINES-CLOBBERED)
		   (SETF (PLINE-TICK ZWEI-WINDOW I) -1)))
	       (LET ((LINES-HEIGHT (* LINE-HEIGHT N-PLINES))
		     (INSIDE-HEIGHT (TV:SHEET-INSIDE-HEIGHT SHEET)))
		 (OR ( BOTTOM-REACHED LINES-HEIGHT)
		     (= LINES-HEIGHT INSIDE-HEIGHT)
		     (TV:SHEET-FORCE-ACCESS (SHEET)
		       (TV:%DRAW-RECTANGLE (TV:SHEET-INSIDE-WIDTH SHEET)
					   (- INSIDE-HEIGHT LINES-HEIGHT)
					   (TV:SHEET-INSIDE-LEFT SHEET)
					   (+ (TV:SHEET-INSIDE-TOP SHEET) LINES-HEIGHT)
					   (TV:SHEET-ERASE-ALUF SHEET) SHEET))))))))
    (AND (TV:DELAYED-REDISPLAY-LABEL-MIXIN-LABEL-NEEDS-UPDATING SHEET)
	 (FUNCALL SHEET ':UPDATE-LABEL))))

;;; This is called to see if the window should not be redisplayed, because it
;;; is de-exposed, or because it is covered by typeout and not selected (this
;;; allows you to continue to look at typeout while typing in the mini-buffer.)
;;; If the window is going to be exposed and selected when it is redisplayed,
;;; then it is done here.
(DEFUN WINDOW-READY-P (ZWEI-WINDOW &OPTIONAL (CURRENT-WINDOW-SPECIAL T) &AUX SHEET)
  (SETQ SHEET (WINDOW-SHEET ZWEI-WINDOW))
  (AND (NOT (TV:SHEET-OUTPUT-HELD-P SHEET))
       (OR (AND CURRENT-WINDOW-SPECIAL (EQ ZWEI-WINDOW *WINDOW*))
	   (NOT (TV:BASIC-TYPEOUT-WINDOW-BOTTOM-REACHED
		  (TV:ESSENTIAL-WINDOW-WITH-TYPEOUT-MIXIN-TYPEOUT-WINDOW SHEET))))))

;;; Find out if the window is on the screen in any way, even if covered by typeout
(DEFUN WINDOW-EXPOSED-P (ZWEI-WINDOW)
  (NOT (DO SHEET (WINDOW-SHEET ZWEI-WINDOW) (TV:SHEET-SUPERIOR SHEET) (NULL SHEET)
	   (OR (TV:SHEET-EXPOSED-P SHEET) (RETURN T)))))

;;; Return the typeout stream associated with the given window
(DEFUN WINDOW-TYPEOUT-WINDOW (ZWEI-WINDOW)
  (FUNCALL (WINDOW-SHEET ZWEI-WINDOW) ':TYPEOUT-WINDOW))

(DEFUN TYPEOUT-WINDOW-INCOMPLETE-P (TYPEOUT-WINDOW)
  (IF (TYPEP TYPEOUT-WINDOW 'TV:BASIC-TYPEOUT-WINDOW)
      (TV:BASIC-TYPEOUT-WINDOW-INCOMPLETE-P TYPEOUT-WINDOW)
      (FUNCALL TYPEOUT-WINDOW ':INCOMPLETE-P)))

(DEFUN WINDOW-IO-BUFFER (ZWEI-WINDOW)
  (FUNCALL (WINDOW-SHEET ZWEI-WINDOW) ':IO-BUFFER))

;;; This returns the string which can be clobbered to the new documentation
;;; when the mouse command-table changes.
(DEFUN WINDOW-WHO-LINE-DOCUMENTATION-STRING (ZWEI-WINDOW)
  (FUNCALL (WINDOW-SHEET ZWEI-WINDOW) ':MOUSE-DOCUMENTATION-STRING))

(DEFUN DISAPPEAR-MINI-BUFFER-WINDOW ()
  ;; Bring the typein window back, and hence get rid of the mini-buffer
  (TV:BLINKER-SET-VISIBILITY (WINDOW-POINT-BLINKER *MINI-BUFFER-WINDOW*) NIL)
  (DO L (WINDOW-SPECIAL-BLINKER-LIST *MINI-BUFFER-WINDOW*) (CDR L) (NULL L)
    (TV:BLINKER-SET-VISIBILITY (CDAR L) NIL))
  (FUNCALL *TYPEIN-WINDOW* ':EXPOSE)
  (FUNCALL (WINDOW-SHEET *MINI-BUFFER-WINDOW*) ':DEACTIVATE)
  (FUNCALL *MODE-LINE-WINDOW* ':DONE-WITH-MODE-LINE-WINDOW))

(LOCAL-DECLARE ((SPECIAL PROCESS))
(DEFUN FLUSH-PROCESS-WINDOWS (PROCESS)
  (TV:MAP-OVER-SHEETS #'(LAMBDA (W)
			  (AND (GET-HANDLER-FOR W ':PROCESS)
			       (EQ (FUNCALL W ':PROCESS) PROCESS)
			       (FUNCALL W ':DEACTIVATE))))))

(DEFVAR *ZMACS-SHOULD-BURY-ON-EXIT* T)

(DEFMETHOD (ZMACS-TOP-LEVEL-EDITOR :EXIT) ()
  ;; Go back to the window in which (ZED) was last done.
  (FUNCALL (OR *ZMACS-CALLER-WINDOW* (TV:IDLE-LISP-LISTENER)) ':SELECT NIL)
  (AND *ZMACS-SHOULD-BURY-ON-EXIT*
       (FUNCALL (WINDOW-FRAME *WINDOW*) ':BURY))
  (SI:PROCESS-WAIT-FOREVER))

(DEFUN FLUSH-PROCESS (PROCESS)
  (FUNCALL PROCESS ':FLUSH))

(DEFUN RESET-PROCESS (PROCESS)
  (FUNCALL PROCESS ':RESET))

(DEFUN NOTIFY-SCROLL-BAR (ZWEI-WINDOW)
  (FUNCALL (WINDOW-SHEET ZWEI-WINDOW) ':NEW-SCROLL-POSITION))

;;; If text has changed, make sure the mouse blinker knows about it
(DEFUN MOUSE-RETHINK (ZWEI-WINDOW &AUX SHEET)
  (AND (EQ (SETQ SHEET (WINDOW-SHEET ZWEI-WINDOW)) TV:MOUSE-WINDOW)
       (TV:MOUSE-WAKEUP)))

(DEFUN GET-WINDOW-EDGES (ZWEI-WINDOW)
  (FUNCALL (WINDOW-SHEET ZWEI-WINDOW) ':EDGES))

(DEFUN DEACTIVATE-WINDOW (ZWEI-WINDOW)
  (FUNCALL (WINDOW-SHEET ZWEI-WINDOW) ':DEACTIVATE))

(DEFUN WINDOW-BACKSPACE-OVERPRINTING-FLAG (ZWEI-WINDOW)
  (ZEROP (TV:SHEET-BACKSPACE-NOT-OVERPRINTING-FLAG (WINDOW-SHEET ZWEI-WINDOW))))

(DEFUN REDEFINE-WINDOW-BACKSPACE-FLAG (ZWEI-WINDOW BACKSPACE-OVERPRINTING-FLAG)
  (LET* ((SHEET (WINDOW-SHEET ZWEI-WINDOW))
	 (OLD (TV:SHEET-BACKSPACE-NOT-OVERPRINTING-FLAG SHEET))
	 (NEW (IF BACKSPACE-OVERPRINTING-FLAG 0 1)))
    (COND (( OLD NEW)
	   (SETF (TV:SHEET-BACKSPACE-NOT-OVERPRINTING-FLAG (WINDOW-SHEET ZWEI-WINDOW)) NEW)
	   (MUNG-LINES-WITH-CHAR #\BS ZWEI-WINDOW)))))

(DEFUN WINDOW-TAB-NCHARS (ZWEI-WINDOW)
  (TV:SHEET-TAB-NCHARS (WINDOW-SHEET ZWEI-WINDOW)))

(DEFUN REDEFINE-WINDOW-TAB-NCHARS (ZWEI-WINDOW TAB-NCHARS)
  (LET* ((SHEET (WINDOW-SHEET ZWEI-WINDOW))
	 (OLD (TV:SHEET-TAB-NCHARS SHEET)))
    (COND (( OLD TAB-NCHARS)
	   (SETF (TV:SHEET-TAB-NCHARS (WINDOW-SHEET ZWEI-WINDOW)) TAB-NCHARS)
	   (MUNG-LINES-WITH-CHAR #\TAB ZWEI-WINDOW)))))

(DEFUN MUNG-LINES-WITH-CHAR (CH ZWEI-WINDOW)
  (DO ((I 0 (1+ I))
       (NPLINES (WINDOW-N-PLINES ZWEI-WINDOW))
       (LINE)
       (FLAG NIL))
      (( I NPLINES)
       (AND FLAG (SETF (WINDOW-REDISPLAY-DEGREE ZWEI-WINDOW)
		       (MAX (WINDOW-REDISPLAY-DEGREE ZWEI-WINDOW) DIS-TEXT))))
    (COND ((AND (SETQ LINE (PLINE-LINE ZWEI-WINDOW I))
		(STRING-SEARCH-CHAR CH LINE (PLINE-FROM-INDEX ZWEI-WINDOW I)
				    (MIN (LINE-LENGTH LINE)
					 (PLINE-TO-INDEX ZWEI-WINDOW I))))
	   (SETF (PLINE-TICK ZWEI-WINDOW I) 0)
	   (SETQ FLAG T)))))

(DEFUN TURN-ON-MINI-BUFFER-COMPLETION-BLINKER (ZWEI-WINDOW ON-P)
  (LET ((SHEET (WINDOW-SHEET ZWEI-WINDOW)))
    (LET ((POINT-BLINKER (FUNCALL SHEET ':POINT-BLINKER))
	  (COMPLETION-BLINKER (FUNCALL SHEET ':COMPLETION-BLINKER)))
      (COND ((EQ (WINDOW-POINT-BLINKER ZWEI-WINDOW)
		 (IF ON-P POINT-BLINKER COMPLETION-BLINKER))
	     (TV:BLINKER-SET-VISIBILITY (WINDOW-POINT-BLINKER ZWEI-WINDOW) NIL)
	     (SETF (WINDOW-POINT-BLINKER ZWEI-WINDOW) 
		   (IF ON-P COMPLETION-BLINKER POINT-BLINKER))
	     (MUST-REDISPLAY ZWEI-WINDOW DIS-BPS)
	     T)))))

;;; Two window stuff, takes two windows (structures, not sheets)
;;; and makes them share the area originally occupied by the first of the two.
;;; Returns the edges of the area which they now share.
(DEFUN TWO-WINDOWS (ZWEI-WINDOW-1 ZWEI-WINDOW-2)
  (REDISPLAY ZWEI-WINDOW-1 ':NONE)
  (LET ((W1 (WINDOW-SHEET ZWEI-WINDOW-1))
	(W2 (WINDOW-SHEET ZWEI-WINDOW-2))
	(FRAME (WINDOW-FRAME ZWEI-WINDOW-1)))
    (MULTIPLE-VALUE-BIND (LEFT TOP RIGHT BOTTOM)
	(FUNCALL FRAME ':INSIDE-EDGES-WITHOUT-MODE-LINE-WINDOW)
      (TV:DELAYING-SCREEN-MANAGEMENT
	(FUNCALL W1 ':DEEXPOSE)
	(FUNCALL W2 ':DEEXPOSE)
	(LET ((HEIGHT (// (- BOTTOM TOP) 2)))
	  (FUNCALL W1 ':SET-EDGES LEFT TOP RIGHT (+ TOP HEIGHT))
	  (FUNCALL W2 ':SET-EDGES LEFT (+ TOP HEIGHT) RIGHT BOTTOM))
	(FUNCALL W1 ':EXPOSE)		;Make sure they are both there
	(FUNCALL W2 ':EXPOSE)))
    (FUNCALL FRAME ':UPDATE-LABELS)))

;;; Grow a window, shrinking the other one
(DEFUN GROW-WINDOW (ZWEI-WINDOW-1 ZWEI-WINDOW-2 NLINES)
  (LET ((W1 (WINDOW-SHEET ZWEI-WINDOW-1))
	(W2 (WINDOW-SHEET ZWEI-WINDOW-2)))
    (LET ((HEIGHT (* NLINES (TV:SHEET-LINE-HEIGHT W1)))
	  LEFT TOP RIGHT BOTTOM BOTTOM1 TOP2)
      (MULTIPLE-VALUE (LEFT TOP RIGHT BOTTOM1)
	  (FUNCALL W1 ':EDGES))
      (MULTIPLE-VALUE (NIL TOP2 NIL BOTTOM)
	(FUNCALL W2 ':EDGES))
      (SETQ BOTTOM1 (+ BOTTOM1 HEIGHT))
      (SETQ TOP2 (+ TOP2 HEIGHT))
      (AND (OR (< BOTTOM1 TOP) (> BOTTOM1 BOTTOM)
	       (< TOP2 TOP) (> TOP2 BOTTOM))
	   (BARF))
      (COND ((> HEIGHT 0)
	     (FUNCALL W2 ':SET-EDGES LEFT TOP2 RIGHT BOTTOM)
	     (FUNCALL W1 ':SET-EDGES LEFT TOP RIGHT BOTTOM1))
	    (T
	     (FUNCALL W1 ':SET-EDGES LEFT TOP RIGHT BOTTOM1)
	     (FUNCALL W2 ':SET-EDGES LEFT TOP2 RIGHT BOTTOM))))))

;;; This puts the specified number of lines in the top window
(DEFUN SPLIT-SCREEN-BETWEEN-TWO-WINDOWS (TOP-ZWEI-WINDOW BOTTOM-ZWEI-WINDOW NLINES)
  (LET ((W1 (WINDOW-SHEET TOP-ZWEI-WINDOW))
	(W2 (WINDOW-SHEET BOTTOM-ZWEI-WINDOW))
	(FRAME (WINDOW-FRAME TOP-ZWEI-WINDOW)))
    (MULTIPLE-VALUE-BIND (LEFT TOP RIGHT BOTTOM MIDDLE)
	(FUNCALL FRAME ':INSIDE-EDGES-WITHOUT-MODE-LINE-WINDOW)
      (SETQ MIDDLE (+ TOP (* NLINES (TV:SHEET-LINE-HEIGHT W1))
		      (TV:SHEET-TOP-MARGIN-SIZE W1) (TV:SHEET-BOTTOM-MARGIN-SIZE W1)))
      (OR (AND (TV:SHEET-EXPOSED-P W1)
	       (TV:SHEET-EXPOSED-P W2)
	       (MULTIPLE-VALUE-BIND (LF TP RT BT)
		   (FUNCALL W1 ':EDGES)
		 (AND (= LF LEFT) (= TP TOP) (= RT RIGHT) (= BT MIDDLE)))
	       (MULTIPLE-VALUE-BIND (LF TP RT BT)
		   (FUNCALL W2 ':EDGES)
		 (AND (= LF LEFT) (= TP MIDDLE) (= RT RIGHT) (= BT BOTTOM))))
	  (LET (SELECT-1-P SELECT-2-P)
	    (DO SHEET TV:SELECTED-WINDOW (TV:SHEET-SUPERIOR SHEET) (NULL SHEET)
	      (AND (EQ SHEET W1) (SETQ SELECT-1-P T))
	      (AND (EQ SHEET W2) (SETQ SELECT-2-P T)))
	    (TV:DELAYING-SCREEN-MANAGEMENT
	      (FUNCALL W1 ':DEEXPOSE)
	      (FUNCALL W2 ':DEEXPOSE)
	      (FUNCALL W1 ':SET-EDGES LEFT TOP RIGHT MIDDLE)
	      (FUNCALL W2 ':SET-EDGES LEFT MIDDLE RIGHT BOTTOM)
	      (FUNCALL W1 (IF SELECT-1-P ':SELECT ':EXPOSE))
	      (FUNCALL W2 (IF SELECT-2-P ':SELECT ':EXPOSE)))
	    (FUNCALL FRAME ':UPDATE-LABELS))))))

(DEFUN MAKE-WINDOW-FULL-SCREEN (ZWEI-WINDOW &AUX FRAME LEFT TOP RIGHT BOTTOM)
  (SETQ FRAME (WINDOW-FRAME ZWEI-WINDOW))
  (MULTIPLE-VALUE (LEFT TOP RIGHT BOTTOM)
    (FUNCALL FRAME ':INSIDE-EDGES-WITHOUT-MODE-LINE-WINDOW))
  (FUNCALL (WINDOW-SHEET ZWEI-WINDOW) ':SET-EDGES LEFT TOP RIGHT BOTTOM)
  (FUNCALL FRAME ':UPDATE-LABELS))

(DEFFLAVOR MENU-COMMAND-MENU-MIXIN () ()
  (:INCLUDED-FLAVORS TV:MENU))

(DEFMETHOD (MENU-COMMAND-MENU-MIXIN :WHO-LINE-DOCUMENTATION-STRING) ()
  (AND (BOUNDP 'TV:CURRENT-ITEM)
       (OR (GET (CDR TV:CURRENT-ITEM) ':WHO-LINE-DOCUMENTATION)
	   (GET (CDR TV:CURRENT-ITEM) ':DOCUMENTATION))))

(DEFFLAVOR MENU-COMMAND-MOMENTARY-MENU () (MENU-COMMAND-MENU-MIXIN TV:MOMENTARY-MENU))

;;; Menu stuff
;;; Return a command that does a pop-up menu from the given command-list
;;; We actually make a menu rather than calling TV:MENU-CHOOSE to provide faster
;;; pop up and remembering of the last-selected item in that menu (per screen).

;;; This is the one resource for all of these menus; the item-list parameter
;;; is used to distinguish among them.
(DEFWINDOW-RESOURCE MENU-COMMAND-MENU (ITEM-LIST)
	:MAKE-WINDOW (MENU-COMMAND-MOMENTARY-MENU :ITEM-LIST ITEM-LIST
						  :SAVE-BITS ':DELAYED)
	:INITIAL-COPIES 0
	:REUSABLE-WHEN :DEACTIVATED)

(DEFUN MAKE-MENU-COMMAND (COMMAND-LIST)
  (LET-CLOSED ((ITEM-LIST (MAKE-COMMAND-ALIST COMMAND-LIST)))
    'MAKE-MENU-COMMAND-DRIVER))

(DEFUN MAKE-MENU-COMMAND-DRIVER (&AUX COMMAND)
  (LOCAL-DECLARE ((SPECIAL ITEM-LIST))
    (USING-RESOURCE (MENU MENU-COMMAND-MENU ITEM-LIST)
      (SETQ COMMAND (FUNCALL MENU ':CHOOSE)))
    (IF COMMAND (FUNCALL COMMAND) DIS-NONE)))

(DEFUN MENU-COMMAND-P (X)
  (AND (CLOSUREP X)
       (EQ (CLOSURE-FUNCTION X) 'MAKE-MENU-COMMAND-DRIVER)))

(DEFUN GET-MENU-COMMAND-COMMANDS (X)
  (SYMEVAL-IN-CLOSURE X 'ITEM-LIST))

;;; The window-sheet definitions

(DEFFLAVOR EDITOR-TYPEOUT-WINDOW ()
	   (TV:ANY-TYI-MIXIN TV:TYPEOUT-WINDOW-WITH-MOUSE-SENSITIVE-ITEMS))

(DEFMETHOD (EDITOR-TYPEOUT-WINDOW :MORE-TYI) ()
  (DO ((CH)) (NIL)
    (AND (OR (NUMBERP (SETQ CH (FUNCALL-SELF ':ANY-TYI)))
	     (AND (LISTP CH) (EQ (CAR CH) ':TYPEOUT-EXECUTE)))
	 (RETURN CH))))

(DEFMETHOD (EDITOR-TYPEOUT-WINDOW :MORE-EXCEPTION) (&AUX CH)
  (COND ((NOT (ZEROP (TV:SHEET-MORE-FLAG)))
	 (SETQ CH (TV:SHEET-MORE-HANDLER ':MORE-TYI))
	 (COND ((OR (NOT (NUMBERP CH)) ( CH #\SP))
		(OR (MEMQ CH '(#\RUBOUT #\CR))
		    (FUNCALL STANDARD-INPUT ':UNTYI CH))
		(LET ((EPF (TV:SHEET-END-PAGE-FLAG)))
		  (SETF (TV:SHEET-END-PAGE-FLAG) 0)
		  (TV:SHEET-STRING-OUT SELF "**FLUSHED**")
		  (SETF (TV:SHEET-END-PAGE-FLAG) EPF))
		(AND (BOUNDP '*CURRENT-COMMAND*)
		     (*THROW (IF *INSIDE-BREAK* 'SYS:COMMAND-LEVEL 'ZWEI-COMMAND-LOOP)
			     T)))))))

(DEFFLAVOR ZWEI
       (ZWEI-WINDOW			;The corresponding editor data structure
	(DELAYED-SELECT-PENDING NIL)	;T means this window wants to be exposed and
					;selected, but is waiting until typeahead
					;has been absorbed before popping up.
	(GLITCH-AT-END-OF-PAGE T)	;Should scroll to handle wraparound
	(MOUSE-DOCUMENTATION-STRING (MAKE-ARRAY 100. ':TYPE 'ART-STRING
						     ':LEADER-LIST '(0)))
	)
       ()
  (:INCLUDED-FLAVORS TV:ESSENTIAL-WINDOW TV:ESSENTIAL-MOUSE TV:STREAM-MIXIN TV:LABEL-MIXIN
		     TV:BASIC-SCROLL-BAR)
  (:INITABLE-INSTANCE-VARIABLES ZWEI-WINDOW)
  (:GETTABLE-INSTANCE-VARIABLES ZWEI-WINDOW MOUSE-DOCUMENTATION-STRING)
  (:SETTABLE-INSTANCE-VARIABLES GLITCH-AT-END-OF-PAGE)
  (:DEFAULT-INIT-PLIST :MORE-P NIL :SAVE-BITS ':DELAYED :RIGHT-MARGIN-CHARACTER-FLAG 1
		       :BACKSPACE-NOT-OVERPRINTING-FLAG 1
		       :MINIMUM-WIDTH 40 :MINIMUM-HEIGHT 40) ;for creation with mouse
  (:DOCUMENTATION :MIXIN "Editor windows"))

(DEFMETHOD (ZWEI :AFTER :EXPOSE) (&REST IGNORE)
  (SETQ DELAYED-SELECT-PENDING NIL))	;We have appeared now

(DEFMETHOD (ZWEI :START-DELAYED-SELECT) ()
  (SETQ DELAYED-SELECT-PENDING (NOT TV:EXPOSED-P)))

(DEFMETHOD (ZWEI :FLUSH-DELAYED-SELECT) ()
  (SETQ DELAYED-SELECT-PENDING NIL))

(DEFMETHOD (ZWEI :FINISH-DELAYED-SELECT) ()
  (AND DELAYED-SELECT-PENDING (NOT TV:EXPOSED-P)
       (SELECT-WINDOW ZWEI-WINDOW))
  NIL)

(DEFMETHOD (ZWEI :CHANGE-LABEL) (NEW-LABEL)
  (FUNCALL-SELF ':DELAYED-SET-LABEL NEW-LABEL))

(DEFMETHOD (ZWEI :BEFORE :END-OF-PAGE-EXCEPTION) ()
  (COND (GLITCH-AT-END-OF-PAGE
	 (RECENTER-WINDOW-RELATIVE ZWEI-WINDOW (// (WINDOW-N-PLINES ZWEI-WINDOW) 3))
	 (COND (( (+ TV:CURSOR-Y TV:LINE-HEIGHT) (TV:SHEET-INSIDE-BOTTOM))
		(SETF (TV:SHEET-END-PAGE-FLAG) 0)  ;Continuation lines can prevent glitching
		(SIGNAL 'END-OF-PAGE-GLITCH))))))

(DEFMETHOD (ZWEI :WHO-LINE-DOCUMENTATION-STRING) ()
  (OR *GLOBAL-MOUSE-CHAR-BLINKER-DOCUMENTATION-STRING*
      MOUSE-DOCUMENTATION-STRING))

(DEFFLAVOR ZWEI-WITH-TYPEOUT ()
       (ZWEI TV:ANY-TYI-MIXIN TV:WINDOW-WITH-TYPEOUT-MIXIN TV:DELAYED-REDISPLAY-LABEL-MIXIN
	TV:WINDOW)
  (:INIT-KEYWORDS :ITEM-TYPE-ALIST)
  (:DEFAULT-INIT-PLIST :ITEM-TYPE-ALIST NIL)
  (:DOCUMENTATION :COMBINATION "An editor window with a typeout window too"))

(DEFFLAVOR ZWEI-WINDOW ()
       (ZWEI-WITH-TYPEOUT TV:FLASHY-SCROLLING-MIXIN TV:BASIC-SCROLL-BAR)
  (:DOCUMENTATION :COMBINATION "A non-ZMACS editor window"))

(DEFFLAVOR ZMACS-WINDOW
	((TV:PROCESS *ZMACS-WINDOW-PROCESS*))
	(TV:PROCESS-MIXIN ZWEI-WINDOW ZWEI-MACRO-MIXIN)
  (:SETTABLE-INSTANCE-VARIABLES TV:PROCESS)
  (:DEFAULT-INIT-PLIST :ITEM-TYPE-ALIST *TYPEOUT-COMMAND-ALIST*)
  (:DOCUMENTATION :MIXIN "A window associated with the ZMACS editor"))

(DEFMETHOD (ZWEI-WITH-TYPEOUT :AFTER :INIT) (INIT-PLIST)
  (OR TV:TYPEOUT-WINDOW
      (SETQ TV:TYPEOUT-WINDOW
	    (TV:MAKE-WINDOW 'EDITOR-TYPEOUT-WINDOW
			    ':ITEM-TYPE-ALIST (GET INIT-PLIST ':ITEM-TYPE-ALIST)
			    ':IO-BUFFER TV:IO-BUFFER
			    ':SUPERIOR SELF))))

(DEFMETHOD (ZMACS-WINDOW :PRINT-NOTIFICATION) (TIME STRING WINDOW-OF-INTEREST)
  WINDOW-OF-INTEREST  ;ignored
  (WINDOW-TYPEIN-NOTIFICATION ZWEI-WINDOW TIME STRING))

(DEFUN WINDOW-TYPEIN-NOTIFICATION (ZWEI-WINDOW TIME STRING)
  (MULTIPLE-VALUE-BIND (*MODE-LINE-WINDOW* *TYPEIN-WINDOW* *MINI-BUFFER-WINDOW*)
      (WINDOW-MODE-LINE-WINDOWS ZWEI-WINDOW)
    (BEEP)
    (TYPEIN-LINE "[")
    (TIME:PRINT-BRIEF-UNIVERSAL-TIME TIME *TYPEIN-WINDOW*)
    (TYPEIN-LINE-MORE " ~A]" STRING)
    (FUNCALL *TYPEIN-WINDOW* ':TYPEOUT-STAYS)))

(DEFMETHOD (ZMACS-WINDOW :NAME-FOR-SELECTION) ()
  (AND (WINDOW-INTERVAL ZWEI-WINDOW)	;Can be NIL if has never been used
       (FORMAT NIL "Edit: ~A" (BUFFER-NAME (WINDOW-INTERVAL ZWEI-WINDOW)))))

(DEFMETHOD (ZWEI-WITH-TYPEOUT :SCREEN-MANAGE) (&REST IGNORE))

(DEFMETHOD (ZWEI-WITH-TYPEOUT :TURN-OFF-BLINKERS-FOR-TYPEOUT) ()
  (TV:BLINKER-SET-VISIBILITY *MOUSE-BLINKER* NIL)
  (TV:MOUSE-STANDARD-BLINKER))

(DEFMETHOD (ZWEI-WITH-TYPEOUT :TURN-ON-BLINKERS-FOR-TYPEOUT) ()
  (TV:MOUSE-SET-BLINKER-DEFINITION ':CHARACTER *MOUSE-X-OFFSET* *MOUSE-Y-OFFSET* ':ON
				   ':SET-CHARACTER *MOUSE-FONT-CHAR*))

;This is for the various mini-buffers, so that you cannot select it with the mouse
(DEFFLAVOR ZWEI-WITH-TYPEOUT-UNSELECTABLE ()
	   (TV:DONT-SELECT-WITH-MOUSE-MIXIN ZWEI-WITH-TYPEOUT)
  (:DOCUMENTATION :COMBINATION))

(DEFFLAVOR ZWEI-MINI-BUFFER
	(POINT-BLINKER
	 COMPLETION-BLINKER)
	(ZWEI-WITH-TYPEOUT-UNSELECTABLE)
  (:GETTABLE-INSTANCE-VARIABLES POINT-BLINKER COMPLETION-BLINKER))

(DEFMETHOD (ZWEI-MINI-BUFFER :AFTER :INIT) (IGNORE)
  (SETQ POINT-BLINKER (CAR (LAST TV:BLINKER-LIST))
	COMPLETION-BLINKER (TV:MAKE-BLINKER SELF 'TV:REVERSE-CHARACTER-BLINKER
					    ':CHARACTER #/c ':VISIBILITY NIL)))

;;; This is the top-level function for *ZMACS-WINDOW-PROCESS*.
(DEFUN ZMACS-WINDOW-TOP-LEVEL ()
  (FUNCALL *ZMACS-STREAM* ':MACRO-ERROR)	;Halt runaway keyboard macro
  (AND (TYPEP TV:SELECTED-WINDOW 'EDITOR-TYPEOUT-WINDOW)
       (FUNCALL TV:SELECTED-WINDOW ':MAKE-COMPLETE))
  (FUNCALL *ZMACS-COMMAND-LOOP* ':EDIT)
  (FUNCALL *ZMACS-COMMAND-LOOP* ':EXIT))

(DEFMACRO COMMAND-BUFFER-PUSH (THING)
  `(TV:IO-BUFFER-PUT TV:IO-BUFFER ,THING))

(DEFMETHOD (ZMACS-WINDOW :BEFORE :INIT) (IGNORE)
  (AND (BOUNDP '*ZMACS-COMMAND-LOOP*)
       (SETQ TV:IO-BUFFER (FUNCALL *ZMACS-COMMAND-LOOP* ':IO-BUFFER))))

;;; When a new zwei-window-class instance is made, this is called to initialize it.
(DEFMETHOD (ZWEI :AFTER :INIT) (IGNORE)
  ;; If we weren't pre-supplied with a ZWEI data structure, make one.
  ;; Tell it about us so it doesn't make a new sheet itself.
  (OR (AND (BOUNDP 'ZWEI-WINDOW) ZWEI-WINDOW)
      (SETQ ZWEI-WINDOW (CREATE-WINDOW SELF))))

;;; This makes the redisplay mechanism forget everything it knows.
(DEFMETHOD (ZWEI :AFTER :REFRESH) (&OPTIONAL TYPE)
  (AND (OR (NOT TV:RESTORED-BITS-P) (EQ TYPE ':SIZE-CHANGED))
       (TELL-EDITOR-TO-REDISPLAY ZWEI-WINDOW DIS-ALL)))

(DEFMETHOD (ZWEI :AFTER :SELECT) (&OPTIONAL IGNORE)
  (OR (MEMQ (TV:BLINKER-VISIBILITY (WINDOW-POINT-BLINKER ZWEI-WINDOW)) '(:BLINK :ON))
      (SETF (WINDOW-REDISPLAY-DEGREE ZWEI-WINDOW)	;Make sure the blinkers are correct
	    (MAX (WINDOW-REDISPLAY-DEGREE ZWEI-WINDOW) DIS-BPS)))
  (OR *EDITOR-ALREADY-KNOWS*
      (COMMAND-BUFFER-PUSH `(SELECT-WINDOW ,ZWEI-WINDOW))))

;;; If the window has changed completely, don't bring back any bits
(DEFWRAPPER (ZWEI :EXPOSE) (IGNORE . BODY)
  `(PROGN
     (COND (( (WINDOW-REDISPLAY-DEGREE ZWEI-WINDOW) DIS-ALL)
	    (AND (< (LENGTH SI:.DAEMON-CALLER-ARGS.) 3)
		 (SETQ SI:.DAEMON-CALLER-ARGS. (LIST (CAR SI:.DAEMON-CALLER-ARGS.)
						     (CADR SI:.DAEMON-CALLER-ARGS.)
						     NIL)))
	    (AND (NEQ (CADDR SI:.DAEMON-CALLER-ARGS.) ':NOOP)
		 (SETF (CADDR SI:.DAEMON-CALLER-ARGS.) ':CLEAN))))
     . ,BODY))

(DEFMETHOD (ZMACS-WINDOW :BEFORE :EXPOSE) (&REST IGNORE)
  (COND ((NOT TV:EXPOSED-P)
	 (OR (WINDOW-INTERVAL ZWEI-WINDOW)
	     (SET-WINDOW-INTERVAL ZWEI-WINDOW (CREATE-ONE-BUFFER-TO-GO)))
	 (FUNCALL *ZMACS-COMMAND-LOOP* ':ADD-WINDOW ZWEI-WINDOW))))

(DEFMETHOD (ZMACS-WINDOW :AFTER :EXPOSE) (&REST IGNORE)
  (COMMAND-BUFFER-PUSH '(CONFIGURATION-CHANGED)))

(DEFVAR *PACKAGE-RECURSION* NIL)

(DEFMETHOD (ZMACS-WINDOW :AFTER :SELECT) (&REST IGNORE)
  (OR *PACKAGE-RECURSION*
      (LET ((*PACKAGE-RECURSION* T))
	(FUNCALL *ZMACS-COMMAND-LOOP* ':COMPUTE-PACKAGE PACKAGE))))

(DEFMETHOD (ZWEI :VERIFY-NEW-EDGES) (NEW-LEFT NEW-TOP NEW-WIDTH NEW-HEIGHT)
  NEW-LEFT NEW-TOP NEW-WIDTH
  (AND (< (- NEW-HEIGHT TV:TOP-MARGIN-SIZE TV:BOTTOM-MARGIN-SIZE)
	  TV:LINE-HEIGHT)
       "Zero line editor windows do not work"))

(DEFMETHOD (ZWEI :AFTER :CHANGE-OF-SIZE-OR-MARGINS) ZWEI-INSIDE-CHANGED)

(DEFMETHOD (ZWEI :AFTER :SET-FONT-MAP) ZWEI-INSIDE-CHANGED)

(DECLARE-FLAVOR-INSTANCE-VARIABLES (ZWEI)
(DEFUN ZWEI-INSIDE-CHANGED (&REST IGNORE &AUX INSIDE-HEIGHT)
  (COND ((BOUNDP 'ZWEI-WINDOW)				;May not be during initialization
	 (SETQ INSIDE-HEIGHT (TV:SHEET-INSIDE-HEIGHT))
	 (LET ((N-PLINES (// INSIDE-HEIGHT TV:LINE-HEIGHT)))
	   ;; If number of screen lines has changed, we must change the window array.
	   (COND (( (WINDOW-N-PLINES ZWEI-WINDOW) N-PLINES)
		  (LET ((MAX-N-PLINES (ARRAY-DIMENSION-N 2 ZWEI-WINDOW)))
		    (IF (< MAX-N-PLINES N-PLINES)
			;; Too small, grow it
			(ARRAY-GROW ZWEI-WINDOW (ARRAY-DIMENSION-N 1 ZWEI-WINDOW)
				    (MAX (FIX (* 1.5s0 MAX-N-PLINES))
					 N-PLINES))
			;; Too big, forget old information
			(DO I N-PLINES (1+ I) ( I MAX-N-PLINES)
			  (SETF (PLINE-LINE ZWEI-WINDOW I) NIL))))
		  (SETF (WINDOW-N-PLINES ZWEI-WINDOW) N-PLINES)))
	   (LET ((LINES-HEIGHT (* TV:LINE-HEIGHT N-PLINES)))
	     (OR (= LINES-HEIGHT INSIDE-HEIGHT)
		 (TV:SHEET-FORCE-ACCESS (SELF)
		   (TV:%DRAW-RECTANGLE (TV:SHEET-INSIDE-WIDTH) (- INSIDE-HEIGHT LINES-HEIGHT)
				       (TV:SHEET-INSIDE-LEFT)
				       (+ (TV:SHEET-INSIDE-TOP) LINES-HEIGHT)
				       TV:ERASE-ALUF SELF)))))))))

;;; This function tells the editor to redisplay
(DEFUN TELL-EDITOR-TO-REDISPLAY (ZWEI-WINDOW DEGREE)
  (AND (SETF (WINDOW-REDISPLAY-DEGREE ZWEI-WINDOW)
	     (MAX (WINDOW-REDISPLAY-DEGREE ZWEI-WINDOW) DEGREE)))
  (LOCAL-DECLARE ((SPECIAL TV:IO-BUFFER))
    (COMMAND-BUFFER-PUSH '(REDISPLAY))))

;;;Get a handle on the text in the window
(DEFMETHOD (ZWEI :STREAM) ()
  (MAKE-EDITOR-STREAM-FROM-WINDOW ZWEI-WINDOW))

(DEFMETHOD (ZWEI :INTERVAL-STRING) ()
  (STRING-INTERVAL (WINDOW-INTERVAL ZWEI-WINDOW)))

(DEFMETHOD (ZWEI :INTERVAL-STREAM) ()
  (INTERVAL-STREAM (WINDOW-INTERVAL ZWEI-WINDOW)))

(DEFMETHOD (ZWEI :SET-INTERVAL-STRING) (STRING &AUX (INTERVAL (WINDOW-INTERVAL ZWEI-WINDOW)))
  (DELETE-INTERVAL INTERVAL)
  (INSERT (INTERVAL-LAST-BP INTERVAL) STRING)
  (TELL-EDITOR-TO-REDISPLAY ZWEI-WINDOW DIS-ALL))

(DEFMETHOD (ZWEI :ADD-TO-INTERVAL-STRING) (STRING
					   &AUX (INTERVAL (WINDOW-INTERVAL ZWEI-WINDOW)))
  (INSERT (INTERVAL-LAST-BP INTERVAL) STRING)
  (TELL-EDITOR-TO-REDISPLAY ZWEI-WINDOW DIS-TEXT))

(DEFMETHOD (ZWEI :ADD-STRING-AT-POINT) (STRING)
  (INSERT-MOVING (WINDOW-POINT ZWEI-WINDOW) STRING)
  (TELL-EDITOR-TO-REDISPLAY ZWEI-WINDOW DIS-TEXT))

;;; Mouse screen primitives

(DECLARE (SPECIAL *MOUSE-P* *MOUSE-BLINKER*
		  *GLOBAL-MOUSE-CHAR-BLINKER*
		  *GLOBAL-MOUSE-CHAR-BLINKER-HANDLER*))

(DEFVAR *MOUSE-FONT-CHAR* 31)
(DEFVAR *MOUSE-X-OFFSET* 8)
(DEFVAR *MOUSE-Y-OFFSET* 0)

;;; Called when the mouse enters a ZWEI window.  Change the shape of the blinker,
;;; then call the standard mouse tracker, telling it we have a scroll bar.
(DEFMETHOD (ZWEI :HANDLE-MOUSE) ()
  (LET-GLOBALLY ((*MOUSE-P* T))
    (TV:MOUSE-SET-BLINKER-DEFINITION ':CHARACTER *MOUSE-X-OFFSET* *MOUSE-Y-OFFSET* ':ON
				     ':SET-CHARACTER *MOUSE-FONT-CHAR*)
    (TV:MOUSE-DEFAULT-HANDLER SELF (FUNCALL-SELF ':SCROLL-BAR-P))
    (TV:BLINKER-SET-VISIBILITY *MOUSE-BLINKER* NIL)))

(DEFMETHOD (ZWEI :SCROLL-BAR-P) () T)

;; Handle a movement of the mouse within an editor window.
;; Update the blinker which flashes the character being pointed at.
(DEFMETHOD (ZWEI :MOUSE-MOVES) (NEW-X NEW-Y &AUX CHAR CHAR-X CHAR-Y LINE INDEX WIDTH)
  NEW-X NEW-Y
  (TV:MOUSE-SET-BLINKER-CURSORPOS)
  (AND (OR (EDITOR-WINDOW-SELECTED-P SELF) *MOUSE-CLICK-ALWAYS-SELECTS*)
       ( NEW-X (TV:SHEET-INSIDE-LEFT)) (< NEW-X (TV:SHEET-INSIDE-RIGHT))
       (MULTIPLE-VALUE (CHAR CHAR-X CHAR-Y LINE INDEX WIDTH)
	 (MOUSE-CHAR ZWEI-WINDOW)))
  (AND CHAR
       ;;There is a timing problem if the editor's process can disable the global blinker
       ;;handler while we are inside it, it will turn on the blinker after the editor has
       ;;just turned it off.
       (WITHOUT-INTERRUPTS
	 (COND (*GLOBAL-MOUSE-CHAR-BLINKER-HANDLER*
		(FUNCALL *GLOBAL-MOUSE-CHAR-BLINKER-HANDLER*
			 *GLOBAL-MOUSE-CHAR-BLINKER* ZWEI-WINDOW
			 CHAR CHAR-X CHAR-Y LINE INDEX)
		(SETQ CHAR NIL)))))		;Only have one blinker
  (COND (CHAR
	 (TV:BLINKER-SET-SHEET *MOUSE-BLINKER* SELF)
	 (LET ((FONT (AREF TV:FONT-MAP (LDB %%CH-FONT CHAR)))
	       (CH (LDB %%CH-CHAR CHAR)))
	   (COND ((TYPEP *MOUSE-BLINKER* 'TV:CHARACTER-BLINKER)
		  (LET ((LKT (FONT-LEFT-KERN-TABLE FONT)))
		    (AND LKT ( CH (ARRAY-ACTIVE-LENGTH LKT))
			 (SETQ CHAR-X (- CHAR-X (AREF LKT CH)))))
		  (SHEET-SET-BLINKER-CURSORPOS SELF *MOUSE-BLINKER* CHAR-X CHAR-Y)
		  (TV:BLINKER-SET-CHARACTER *MOUSE-BLINKER* FONT
					    ;; Non printing characters get blinking underscore
					    (IF (OR (= CH #\SP) ( CH 200)) #/_ CH))
		  (TV:BLINKER-SET-VISIBILITY *MOUSE-BLINKER* ':BLINK))
		 ((OR ( CH #\CR)
		      (AND (< NEW-X (+ (TV:SHEET-INSIDE-LEFT) CHAR-X
				       (SETQ WIDTH TV:CHAR-WIDTH)))
			   (< NEW-Y (+ (TV:SHEET-INSIDE-TOP) CHAR-Y TV:LINE-HEIGHT))))
		  (SHEET-SET-BLINKER-CURSORPOS SELF *MOUSE-BLINKER* CHAR-X CHAR-Y)
		  (TV:BLINKER-SET-SIZE *MOUSE-BLINKER* WIDTH (FONT-BLINKER-HEIGHT FONT))
		  (TV:BLINKER-SET-VISIBILITY *MOUSE-BLINKER* T))
		 (T
		  (TV:BLINKER-SET-VISIBILITY *MOUSE-BLINKER* NIL)))))
	(T
	 (TV:BLINKER-SET-SHEET *MOUSE-BLINKER* SELF)
	 (TV:BLINKER-SET-VISIBILITY *MOUSE-BLINKER* NIL))))

(DEFUN SHEET-SET-BLINKER-CURSORPOS (SHEET BLINKER X Y)
  (MULTIPLE-VALUE-BIND (XOFF YOFF)
      (TV:SHEET-CALCULATE-OFFSETS SHEET (TV:BLINKER-SHEET BLINKER))
    (TV:BLINKER-SET-CURSORPOS BLINKER (+ X XOFF) (+ Y YOFF))))

(DEFVAR *MOUSE-CLICK-ALWAYS-SELECTS* NIL)

(DEFMETHOD (ZWEI :MOUSE-CLICK) (BUTTON X Y &AUX HANDLED-P)
  (COND ((AND (NOT (EDITOR-WINDOW-SELECTED-P SELF))
	      (OR (= BUTTON #\MOUSE-1-1)
		  *MOUSE-CLICK-ALWAYS-SELECTS*))
	 (PROCESS-RUN-FUNCTION "Select" SELF ':SELECT)
	 (SETQ HANDLED-P T)))
  (COND ((OR *MOUSE-CLICK-ALWAYS-SELECTS*
	     (NOT HANDLED-P))
	 (COMMAND-BUFFER-PUSH `(:MOUSE ,ZWEI-WINDOW ,BUTTON ,X ,Y))))
  T)

(DEFUN EDITOR-WINDOW-SELECTED-P (SHEET &AUX MODE-LINE-WINDOW)
  (SETQ MODE-LINE-WINDOW (FUNCALL SHEET ':MODE-LINE-WINDOW))
  (OR (DO ((SH TV:SELECTED-WINDOW (TV:SHEET-SUPERIOR SH)))
	  ((NULL SH) NIL)
	(AND (EQ SHEET SH) (RETURN T)))
      (EQ TV:SELECTED-WINDOW MODE-LINE-WINDOW)
      (EQ TV:SELECTED-WINDOW (FUNCALL MODE-LINE-WINDOW ':TYPEIN-WINDOW))
      (EQ TV:SELECTED-WINDOW (WINDOW-SHEET (FUNCALL MODE-LINE-WINDOW ':MINI-BUFFER-WINDOW)))
      (EQ TV:SELECTED-WINDOW (WINDOW-SHEET (FUNCALL MODE-LINE-WINDOW
						    ':SEARCH-MINI-BUFFER-WINDOW)))))

(DEFWRAPPER (ZMACS-WINDOW :MOUSE-CLICK) ((BUTTON) . BODY)
  `(PROG1 (PROGN . ,BODY)
	  (AND ( BUTTON #\MOUSE-3-2)
	       (SYMEVAL-IN-STACK-GROUP '*INSIDE-BREAK* (PROCESS-STACK-GROUP TV:PROCESS))
	       (FUNCALL TV:PROCESS ':RESET))))

;;; Scrolling

;;; Returns 2 values: current line#, total #lines
(DEFMETHOD (ZWEI :SCROLL-POSITION) ()
  (PROG (INTERVAL TOP-BP TOP-LINE-NUMBER)
    (SETQ INTERVAL (WINDOW-INTERVAL ZWEI-WINDOW)
	  TOP-BP (WINDOW-START-BP ZWEI-WINDOW))
    (SETQ TOP-LINE-NUMBER (1- (COUNT-LINES (INTERVAL-FIRST-BP INTERVAL) TOP-BP T)))
    (RETURN TOP-LINE-NUMBER
	    (+ TOP-LINE-NUMBER (COUNT-LINES TOP-BP (INTERVAL-LAST-BP INTERVAL) T))
	    TV:LINE-HEIGHT)))

(DEFMETHOD (ZWEI :SCROLL-MORE-ABOVE) ()
  (NOT (BP-= (WINDOW-START-BP ZWEI-WINDOW)
	     (INTERVAL-FIRST-BP (WINDOW-INTERVAL ZWEI-WINDOW)))))

; Valid only if redisplay up to date
(DEFMETHOD (ZWEI :SCROLL-MORE-BELOW) ()
  (AREF ZWEI-WINDOW 0 (1- (WINDOW-N-PLINES ZWEI-WINDOW))))

;;; Scroll the window.
(DEFMETHOD (ZWEI :SCROLL-TO) (POS TYPE)
  (OR (AND (EQ TYPE ':RELATIVE) (ZEROP POS))		;Don't ask not to scroll
      (COMMAND-BUFFER-PUSH `(SCROLL ,ZWEI-WINDOW ,POS ,TYPE)))
  POS)

;;; The mini buffer and stuff like that

;;;These windows pop up like a temporary window if typed on while deexposed
(DEFFLAVOR DEEXPOSED-TEMPORARY-WINDOW (DEEXPOSED-TEMPORARY-BIT-ARRAY) ()
  (:INCLUDED-FLAVORS TV:ESSENTIAL-WINDOW))

(DEFMETHOD (DEEXPOSED-TEMPORARY-WINDOW :AFTER :INIT) (IGNORE)
  (SETQ DEEXPOSED-TEMPORARY-BIT-ARRAY
	(MAKE-ARRAY NIL 'ART-1B (LIST (LOGAND -40 (+ 37 TV:WIDTH)) TV:HEIGHT))))

(DEFMETHOD (DEEXPOSED-TEMPORARY-WINDOW :AFTER :CHANGE-OF-SIZE-OR-MARGINS) (&REST IGNORE)
  (SETQ DEEXPOSED-TEMPORARY-BIT-ARRAY
	(TV:GROW-BIT-ARRAY DEEXPOSED-TEMPORARY-BIT-ARRAY TV:WIDTH TV:HEIGHT)))

(DEFMETHOD (DEEXPOSED-TEMPORARY-WINDOW :TEMPORARY-EXPOSE) (&REST IGNORE)
  (COND ((NOT TV:EXPOSED-P)
	 (IF (TV:SHEET-EXPOSED-P TV:SUPERIOR)
	     (SETQ TV:TEMPORARY-BIT-ARRAY DEEXPOSED-TEMPORARY-BIT-ARRAY)
	     (FUNCALL TV:SUPERIOR ':TEMPORARY-EXPOSE))
	 (FUNCALL-SELF ':EXPOSE))))

(DEFMETHOD (DEEXPOSED-TEMPORARY-WINDOW :TEMPORARY-DEEXPOSE) ()
  (FUNCALL-SELF ':DEEXPOSE)
  (SETQ TV:TEMPORARY-BIT-ARRAY NIL))

(DEFFLAVOR FIXED-HEIGHT-WINDOW-MIXIN () ()
  (:INCLUDED-FLAVORS TV:ESSENTIAL-SET-EDGES TV:ESSENTIAL-WINDOW)
  (:DOCUMENTATION :MIXIN "Does not allow changing the height of the window after init"))

(DEFMETHOD (FIXED-HEIGHT-WINDOW-MIXIN :VERIFY-NEW-EDGES)
	   (NEW-LEFT NEW-TOP NEW-WIDTH NEW-HEIGHT)
  NEW-LEFT NEW-TOP NEW-WIDTH
  (AND ( NEW-HEIGHT TV:HEIGHT)
       "Attempt to change height"))

(DEFMETHOD (FIXED-HEIGHT-WINDOW-MIXIN :AFTER :CHANGE-OF-SIZE-OR-MARGINS) (&REST IGNORE)
  (DOLIST (INFERIOR TV:INFERIORS)
    (FUNCALL INFERIOR ':SET-SIZE (TV:SHEET-INSIDE-WIDTH) (TV:SHEET-HEIGHT INFERIOR))))

(DEFFLAVOR DONT-SCREEN-MANAGE-MIXIN () ()
  (:INCLUDED-FLAVORS TV:ESSENTIAL-WINDOW)
  (:DOCUMENTATION :MIXIN "Don't do screen management on itself"))

(DEFMETHOD (DONT-SCREEN-MANAGE-MIXIN :SCREEN-MANAGE) (&REST IGNORE) NIL)

(DEFFLAVOR ECHO-AREA-WINDOW () (TV:DONT-SELECT-WITH-MOUSE-MIXIN TV:ANY-TYI-MIXIN
				DEEXPOSED-TEMPORARY-WINDOW DONT-SCREEN-MANAGE-MIXIN
				TV:STREAM-MIXIN TV:SELECT-MIXIN TV:NOTIFICATION-MIXIN
				FIXED-HEIGHT-WINDOW-MIXIN TV:MINIMUM-WINDOW)
  (:DEFAULT-INIT-PLIST :MORE-P NIL))

;This is used to make a *TYPEIN-WINDOW*; it will acquire a blinker
;if anyone tries to read input from it.  This window can be the value of
;QUERY-IO.
(DEFFLAVOR ECHO-AREA-QUERY-WINDOW () (ECHO-AREA-WINDOW))

(DEFWRAPPER (ECHO-AREA-QUERY-WINDOW :RUBOUT-HANDLER) (IGNORE . BODY)
  `(IF (AND (EQ (TV:BLINKER-VISIBILITY (CAR TV:BLINKER-LIST)) ':OFF)
	    ;; Next line is because selection doesn't work right yet
	    (TV:SHEET-ME-OR-MY-KID-P TV:SELECTED-WINDOW (TV:SHEET-SUPERIOR TV:SUPERIOR)))
       (TV:WINDOW-CALL (SELF :DESELECT)
	 . ,BODY)
       (PROGN . ,BODY)))

(DEFWRAPPER (ECHO-AREA-QUERY-WINDOW :TYI) (IGNORE . BODY)
  `(IF (AND (EQ (TV:BLINKER-VISIBILITY (CAR TV:BLINKER-LIST)) ':OFF)
	    ;; Next line is because selection doesn't work right yet
	    (TV:SHEET-ME-OR-MY-KID-P TV:SELECTED-WINDOW (TV:SHEET-SUPERIOR TV:SUPERIOR)))
       (TV:WINDOW-CALL (SELF :DESELECT)
	 . ,BODY)
       (PROGN . ,BODY)))

;;; This is the flavor used to make a *TYPEIN-WINDOW*.
(DEFFLAVOR TYPEIN-WINDOW
	((TYPEIN-STATUS ':CLEAR))		;:CLEAR, :IN-USE, :USED
	(ECHO-AREA-QUERY-WINDOW)
  (:SETTABLE-INSTANCE-VARIABLES TYPEIN-STATUS)
  (:OUTSIDE-ACCESSIBLE-INSTANCE-VARIABLES TYPEIN-STATUS))

(DEFMETHOD (TYPEIN-WINDOW :AFTER :REFRESH) (&REST IGNORE)
  (OR TV:RESTORED-BITS-P (SETQ TYPEIN-STATUS ':CLEAR)))

(DEFMETHOD (TYPEIN-WINDOW :AFTER :CLEAR-SCREEN) ()
  (SETQ TYPEIN-STATUS ':CLEAR))

(DEFMETHOD (TYPEIN-WINDOW :COMMAND-LOOP-REDISPLAY) ()
  (SELECTQ TYPEIN-STATUS
    (:IN-USE
     (SETQ TYPEIN-STATUS ':USED))
    (:IN-USE-STAYS
     (SETQ TYPEIN-STATUS ':USED-STAYS))
    (:USED
     (IF TV:EXPOSED-P
	 (FUNCALL-SELF ':CLEAR-SCREEN)
	 (SETQ TYPEIN-STATUS ':CLEAR)))))
  
(DEFMETHOD (TYPEIN-WINDOW :PREPARE-FOR-TYPEOUT) ()
  (LET* ((MINI-BUFFER-SHEET (WINDOW-SHEET (FUNCALL TV:SUPERIOR ':MINI-BUFFER-WINDOW)))
	 (MINI-BUFFER-IN-USE (TV:SHEET-EXPOSED-P MINI-BUFFER-SHEET)))
    (FUNCALL-SELF ':EXPOSE)
    (AND MINI-BUFFER-IN-USE (FUNCALL MINI-BUFFER-SHEET ':START-DELAYED-SELECT)))
  (IF (MEMQ TYPEIN-STATUS '(:IN-USE :IN-USE-STAYS))
      NIL					;Do a :FRESH-LINE
      (OR (EQ TYPEIN-STATUS ':CLEAR)
	  (FUNCALL-SELF ':CLEAR-SCREEN))
      (SETQ TYPEIN-STATUS ':IN-USE)
      T))

(DEFMETHOD (TYPEIN-WINDOW :PREPARE-FOR-MORE-TYPEOUT) ()
  (OR (MEMQ TYPEIN-STATUS '(:IN-USE :IN-USE-STAYS))
      (SETQ TYPEIN-STATUS ':IN-USE)))

(DEFMETHOD (TYPEIN-WINDOW :TYPEOUT-STAYS) ()
  (AND (EQ TYPEIN-STATUS ':IN-USE) (SETQ TYPEIN-STATUS ':IN-USE-STAYS)))

(DECLARE-FLAVOR-INSTANCE-VARIABLES (TYPEIN-WINDOW)
(DEFUN MAKE-INCOMPLETE (&REST IGNORE)
  (AND (EQ TYPEIN-STATUS ':CLEAR) (SETQ TYPEIN-STATUS ':IN-USE))))

(DEFMETHOD (TYPEIN-WINDOW :BEFORE :TYO) MAKE-INCOMPLETE)
(DEFMETHOD (TYPEIN-WINDOW :BEFORE :STRING-OUT) MAKE-INCOMPLETE)
(DEFMETHOD (TYPEIN-WINDOW :BEFORE :LINE-OUT) MAKE-INCOMPLETE)
(DEFMETHOD (TYPEIN-WINDOW :BEFORE :FRESH-LINE) MAKE-INCOMPLETE)

;;; This improves interaction with FQUERY
(DEFMETHOD (TYPEIN-WINDOW :MAKE-COMPLETE) ()
  (AND (EQ TYPEIN-STATUS ':IN-USE) (SETQ TYPEIN-STATUS ':USED)))

;;; This knows how to display the editor's mode line properly
(DEFFLAVOR MODE-LINE-WINDOW-MIXIN ((PREVIOUS-MODE-LINE NIL)) ()
  (:INCLUDED-FLAVORS TV:ESSENTIAL-WINDOW)
  (:DEFAULT-INIT-PLIST :TRUNCATE-LINE-OUT-FLAG 1
		       :CR-NOT-NEWLINE-FLAG 1))

(DEFMETHOD (MODE-LINE-WINDOW-MIXIN :AFTER :REFRESH) (&REST IGNORE)
  (OR TV:RESTORED-BITS-P (SETQ PREVIOUS-MODE-LINE NIL)))

(DEFMETHOD (MODE-LINE-WINDOW-MIXIN :CLOBBER) ()
  (SETQ PREVIOUS-MODE-LINE NIL))

;;; Update the mode line if necessary, FORCE says really do it
;;; MODE-LINE-LIST is a list of things to be displayed, whose elements can be:
;;;  a constant string
;;;  a symbol, which is evaluated to either a string or NIL, and printed in the former case
;;;  a list, the CAR of which should be an atom, which is evaluated and the rest of the
;;;    list handled as strings or symbols as above if it is non-NIL (up to any :ELSE), or
;;;    if NIL, anything after a :ELSE in the list.
;;;  eg ("FOOMACS" "(" MODE-NAME ")" (BUFFER-NAMED-P BUFFER-NAME :ELSE "(Null buffer)")
;;;      (FILE-NAME-P FILE-NAME))
;;; As a special hack, if MODE-LINE-LIST is NIL, then the mode line is not changed,
;;;  this is appropriate for things that want to typeout on the prompt-line and then
;;;  invoke the mini-buffer.
;;; PREVIOUS-MODE-LINE is a list of strings that make up the line, since nothing we do
;;;  generates new guys for this, EQness is used to determine if the mode-line has changed
(DEFMETHOD (MODE-LINE-WINDOW-MIXIN :REDISPLAY) (MODE-LINE-LIST &OPTIONAL FORCE)
  (AND FORCE					;If we are going to type things out
       MODE-LINE-LIST				;unless suppressed
       (SETQ PREVIOUS-MODE-LINE NIL))
  (DO ((MODES MODE-LINE-LIST)
       (PREV PREVIOUS-MODE-LINE)
       (L)
       (THING))
      (NIL)
      (COND (L					;Still more to go on a list
	     (POP L THING)
	     (AND (EQ THING ':ELSE)
		  (SETQ L NIL THING NIL)))
	    ((NULL MODES)			;All done with MODE-LINE-LIST
	     (AND PREV (NOT FORCE) (FUNCALL-SELF ':REDISPLAY MODE-LINE-LIST T))
	     (RETURN NIL))
	    (T					;Get next object from MODE-LINE-LIST
	     (POP MODES THING)
	     (COND ((SYMBOLP THING)
		    (SETQ THING (SYMEVAL THING))
		    (AND (LISTP THING)		;If value is a list, dont check CAR
			 (SETQ L THING THING NIL)))
		   ((LISTP THING)		;It's a list,
		    (SETQ L THING)
		    (POP L THING)		;check its CAR
		    (COND ((NULL (SYMEVAL THING))
			   (DO ()		;Failing conditional, look for :ELSE
			       ((NULL L))
			       (POP L THING)
			       (AND (EQ THING ':ELSE)
				    (RETURN NIL)))))
		    (SETQ THING NIL)))))	;And get stuff next pass
      (AND (SYMBOLP THING) (SETQ THING (SYMEVAL THING)))
      (COND ((NULL THING))
	    ;;THING is now the next string to be put into the mode line
	    (FORCE				;Put it in if consing new one
	     (PUSH THING PREVIOUS-MODE-LINE))
	    ((AND PREV (EQ THING (POP PREV))))	;Still matching?
	    (T					;Different thing,
	     (FUNCALL-SELF ':REDISPLAY MODE-LINE-LIST T)	;do it right this time!
	     (RETURN NIL))))
  (COND (FORCE
	 (SETQ PREVIOUS-MODE-LINE (NREVERSE PREVIOUS-MODE-LINE))
	 (COND (TV:EXPOSED-P
		(TV:SHEET-HOME SELF)
		(TV:SHEET-CLEAR-EOL SELF)
		(*CATCH 'MODE-LINE-OVERFLOW
		  (DOLIST (STR PREVIOUS-MODE-LINE)
		    (FUNCALL-SELF ':STRING-OUT STR))))))))

(DEFMETHOD (MODE-LINE-WINDOW-MIXIN :BEFORE :END-OF-LINE-EXCEPTION) ()
  (OR (ZEROP (TV:SHEET-TRUNCATE-LINE-OUT-FLAG))
      (*THROW 'MODE-LINE-OVERFLOW T)))

(DEFMETHOD (MODE-LINE-WINDOW-MIXIN :DONE-WITH-MODE-LINE-WINDOW) ())

(DEFFLAVOR ZMACS-MODE-LINE-WINDOW-MIXIN (*BUFFER-MODIFIED-P*) (MODE-LINE-WINDOW-MIXIN))

(DEFMETHOD (ZMACS-MODE-LINE-WINDOW-MIXIN :BEFORE :REDISPLAY) (&REST IGNORE &AUX INT-TICK)
  (SETQ INT-TICK (NODE-TICK *INTERVAL*))
  (SETQ *BUFFER-MODIFIED-P* (COND ((EQ INT-TICK ':READ-ONLY) " (RO)")
				  ((NOT (TYPEP *INTERVAL* 'FILE-BUFFER)) NIL)
				  ((< (BUFFER-TICK *INTERVAL*) INT-TICK) " *")
				  (T NIL))))

(DEFFLAVOR MODE-LINE-SUPERIOR-MIXIN
	(TYPEIN-WINDOW MINI-BUFFER-WINDOW SEARCH-MINI-BUFFER-WINDOW)
	()
  (:INCLUDED-FLAVORS TV:STREAM-MIXIN TV:MINIMUM-WINDOW)
  (:INIT-KEYWORDS :NUMBER-OF-MINI-BUFFER-LINES)
  (:GETTABLE-INSTANCE-VARIABLES TYPEIN-WINDOW)
  (:DEFAULT-INIT-PLIST :BLINKER-DESELECTED-VISIBILITY ':OFF
		       :DEEXPOSED-TYPEOUT-ACTION '(:TEMPORARY-EXPOSE)))

(DEFFLAVOR MODE-LINE-WINDOW ()
	   (MODE-LINE-SUPERIOR-MIXIN MODE-LINE-WINDOW-MIXIN ECHO-AREA-WINDOW))

(DEFFLAVOR ZMACS-MODE-LINE-WINDOW ()
	   (MODE-LINE-SUPERIOR-MIXIN ZMACS-MODE-LINE-WINDOW-MIXIN ECHO-AREA-WINDOW))

(DEFMETHOD (MODE-LINE-SUPERIOR-MIXIN :MINI-BUFFER-WINDOW) ()
  (FUNCALL MINI-BUFFER-WINDOW ':ZWEI-WINDOW))

(DEFMETHOD (MODE-LINE-SUPERIOR-MIXIN :SEARCH-MINI-BUFFER-WINDOW) ()
  (FUNCALL SEARCH-MINI-BUFFER-WINDOW ':ZWEI-WINDOW))

(DEFMETHOD (MODE-LINE-SUPERIOR-MIXIN :BEFORE :INIT) (PLIST &AUX NLINES)
  (AND (SETQ NLINES (GET PLIST ':NUMBER-OF-MINI-BUFFER-LINES))
       (SETQ TV:HEIGHT (+ 4 TV:TOP-MARGIN-SIZE TV:BOTTOM-MARGIN-SIZE
			    (* (1+ NLINES)
			       (+ (FONT-CHAR-HEIGHT
				    TV:(SCREEN-DEFAULT-FONT (SHEET-GET-SCREEN SUPERIOR)))
				  2))))))

(DEFMETHOD (MODE-LINE-SUPERIOR-MIXIN :AFTER :INIT) (IGNORE)
  (SETQ TYPEIN-WINDOW (TV:MAKE-WINDOW 'TYPEIN-WINDOW
				      ':TOP TV:(+ (SHEET-INSIDE-TOP) LINE-HEIGHT)
				      ':IO-BUFFER TV:IO-BUFFER ':SUPERIOR SELF
				      ':MORE-P NIL ':ACTIVATE-P T
				      ':BLINKER-DESELECTED-VISIBILITY ':OFF)
	MINI-BUFFER-WINDOW (TV:MAKE-WINDOW 'ZWEI-MINI-BUFFER
					   ':TOP TV:(+ (SHEET-INSIDE-TOP) LINE-HEIGHT)
					   ':IO-BUFFER TV:IO-BUFFER ':SUPERIOR SELF
					   ':LABEL NIL ':SAVE-BITS NIL ':ACTIVATE-P T)
	SEARCH-MINI-BUFFER-WINDOW (TV:MAKE-WINDOW 'ZWEI-WITH-TYPEOUT-UNSELECTABLE
						  ':FONT-MAP '(FONTS:CPTFONT FONTS:SEARCH)
						  ':TOP TV:(+ (SHEET-INSIDE-TOP) LINE-HEIGHT)
						  ':IO-BUFFER TV:IO-BUFFER ':SUPERIOR SELF
						  ':BLINKER-FLAVOR
						    'TV:REVERSE-CHARACTER-BLINKER
						  ':BLINKER-P '(:CHARACTER #/s)
						  ':LABEL NIL ':SAVE-BITS NIL ':ACTIVATE-P T))
  (SET-WINDOW-INTERVAL (FUNCALL MINI-BUFFER-WINDOW ':ZWEI-WINDOW) (CREATE-INTERVAL))
  (SET-WINDOW-INTERVAL (FUNCALL SEARCH-MINI-BUFFER-WINDOW ':ZWEI-WINDOW)
		       (LET* ((LINE (CREATE-LINE 'ART-FAT-STRING 50. NIL))
			      (INT (CREATE-INTERVAL (CREATE-BP LINE 0 ':NORMAL)
						    (CREATE-BP LINE 0 ':MOVES))))
			 (SETF (LINE-NODE LINE) INT)
			 INT)))

(DEFMETHOD (MODE-LINE-SUPERIOR-MIXIN :SET-IO-BUFFER) (NEW-IO-BUFFER)
  (SETQ TV:IO-BUFFER NEW-IO-BUFFER)
  (FUNCALL TYPEIN-WINDOW ':SET-IO-BUFFER TV:IO-BUFFER)
  (FUNCALL MINI-BUFFER-WINDOW ':SET-IO-BUFFER TV:IO-BUFFER)
  (FUNCALL (FUNCALL MINI-BUFFER-WINDOW ':TYPEOUT-WINDOW) ':SET-IO-BUFFER TV:IO-BUFFER)
  (FUNCALL SEARCH-MINI-BUFFER-WINDOW ':SET-IO-BUFFER TV:IO-BUFFER)
  (FUNCALL (FUNCALL SEARCH-MINI-BUFFER-WINDOW ':TYPEOUT-WINDOW) ':SET-IO-BUFFER TV:IO-BUFFER))

(DEFMETHOD (MODE-LINE-SUPERIOR-MIXIN :AFTER :CHANGE-OF-SIZE-OR-MARGINS) (&REST IGNORE)
  (LET ((LEFT (TV:SHEET-INSIDE-LEFT))
	(TOP (+ (TV:SHEET-INSIDE-TOP) TV:LINE-HEIGHT))
	(RIGHT (TV:SHEET-INSIDE-RIGHT))
	(BOTTOM (TV:SHEET-INSIDE-BOTTOM)))
    (FUNCALL TYPEIN-WINDOW ':SET-EDGES LEFT TOP RIGHT BOTTOM)
    (FUNCALL MINI-BUFFER-WINDOW ':SET-EDGES LEFT TOP RIGHT BOTTOM)
    (FUNCALL SEARCH-MINI-BUFFER-WINDOW ':SET-EDGES LEFT TOP RIGHT BOTTOM)))

(DEFVAR *GLOBAL-MODE-LINE-WINDOW*)

(DEFMETHOD (ZWEI :MODE-LINE-WINDOW) ()
  *GLOBAL-MODE-LINE-WINDOW*)

(DEFUN WINDOW-MODE-LINE-WINDOWS (ZWEI-WINDOW &AUX MODE-LINE-WINDOW)
  (DECLARE (RETURN-LIST MODE-LINE-WINDOW TYPEIN-WINDOW MINI-BUFFER-WINDOW))
  (SETQ MODE-LINE-WINDOW (FUNCALL (WINDOW-SHEET ZWEI-WINDOW) ':MODE-LINE-WINDOW))
  (PROG () (RETURN MODE-LINE-WINDOW
		   (FUNCALL MODE-LINE-WINDOW ':TYPEIN-WINDOW)
		   (FUNCALL MODE-LINE-WINDOW ':MINI-BUFFER-WINDOW))))

(DEFUN INITIALIZE-MINI-BUFFER-WINDOW (&OPTIONAL (NLINES 3) (SUPERIOR TV:DEFAULT-SCREEN)
						(MODE-LINE-FLAVOR 'ZMACS-MODE-LINE-WINDOW)
						IO-BUFFER
				      &REST OPTIONS)
  (SETQ *GLOBAL-MODE-LINE-WINDOW*
	(LEXPR-FUNCALL #'TV:MAKE-WINDOW MODE-LINE-FLAVOR 
		       ':NUMBER-OF-MINI-BUFFER-LINES NLINES
		       ':SUPERIOR SUPERIOR ':IO-BUFFER IO-BUFFER
		       OPTIONS)))

(DEFMETHOD (TOP-LEVEL-EDITOR :BEFORE :EDIT) (&REST IGNORE)
  (FUNCALL *MODE-LINE-WINDOW* ':SET-IO-BUFFER TV:IO-BUFFER)
  (TURN-ON-USER-MODES))

(DEFVAR SYN-TYPEIN-WINDOW-IO (MAKE-SYN-STREAM '*TYPEIN-WINDOW*))

;;; This ensures that streams are bound to the right thing inside the :EDIT method
;;; Some of these may be instance variables, in which case an extra binding is
;;; generated, but that is ok.
;;; Should return TERMINAL-IO, STANDARD-INPUT, STANDARD-OUTPUT, and QUERY-IO
(DEFMETHOD (TOP-LEVEL-EDITOR :TERMINAL-STREAMS) ()
  (VALUES *TYPEOUT-WINDOW* SI:SYN-TERMINAL-IO SI:SYN-TERMINAL-IO SYN-TYPEIN-WINDOW-IO))

(DEFWRAPPER (TOP-LEVEL-EDITOR :EDIT) (IGNORE . BODY)
  `(MULTIPLE-VALUE-BIND (TERMINAL-IO STANDARD-INPUT STANDARD-OUTPUT QUERY-IO)
       (FUNCALL-SELF ':TERMINAL-STREAMS)
     . ,BODY))

(DEFFLAVOR OWN-STANDARD-INPUT-EDITOR-MIXIN
	(STANDARD-INPUT)
	()
  (:INCLUDED-FLAVORS TOP-LEVEL-EDITOR)
  (:INITABLE-INSTANCE-VARIABLES STANDARD-INPUT))

(DEFMETHOD (OWN-STANDARD-INPUT-EDITOR-MIXIN :TERMINAL-STREAMS) ()
  (VALUES *TYPEOUT-WINDOW* STANDARD-INPUT SI:SYN-TERMINAL-IO SYN-TYPEIN-WINDOW-IO))

(DEFFLAVOR ZWEI-WITH-POP-UP-MINI-BUFFER-MIXIN () ()
  (:INCLUDED-FLAVORS ZWEI))

(DEFMETHOD (ZWEI-WITH-POP-UP-MINI-BUFFER-MIXIN :BEFORE :FINISH-DELAYED-SELECT)
	   (&AUX (SHEET (WINDOW-SHEET *WINDOW*)))
  (AND (LISTP (TV:SHEET-LOCK SHEET)) (MEMQ *MODE-LINE-WINDOW* (TV:SHEET-LOCK SHEET))
       (TV:WINDOW-CALL (*MODE-LINE-WINDOW* :TEMPORARY-DEEXPOSE)
	 (PROMPT-LINE "Type any char to flush: ")
	 (LET ((CH (FUNCALL *MODE-LINE-WINDOW* ':ANY-TYI)))
	   (COND ((NOT (= CH #\SP))
		  (FUNCALL STANDARD-INPUT ':UNTYI CH)))))))

(DEFFLAVOR EDITOR-WITH-POP-UP-MINI-BUFFER-MIXIN (*GLOBAL-MODE-LINE-WINDOW*) ()
  (:INCLUDED-FLAVORS TOP-LEVEL-EDITOR)
  (:INIT-KEYWORDS :MODE-LINE-WINDOW))

(DEFMETHOD (EDITOR-WITH-POP-UP-MINI-BUFFER-MIXIN :AFTER :INIT) (PLIST)
  (OR (SETQ *GLOBAL-MODE-LINE-WINDOW* (GET PLIST ':MODE-LINE-WINDOW))
      (LET ((SHEET (IF (BOUNDP '*WINDOW*) (WINDOW-SHEET *WINDOW*) SELF)))
	(INITIALIZE-MINI-BUFFER-WINDOW
	  3
	  (TV:SHEET-SUPERIOR SHEET)
	  'POP-UP-MODE-LINE-WINDOW
	  TV:IO-BUFFER)
	(SETQ *MODE-LINE-WINDOW* *GLOBAL-MODE-LINE-WINDOW*
	      *TYPEIN-WINDOW* (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':TYPEIN-WINDOW)
	      *MINI-BUFFER-WINDOW* (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':MINI-BUFFER-WINDOW))
	(SETF (TV:SHEET-DEEXPOSED-TYPEOUT-ACTION *TYPEIN-WINDOW*) '(:TEMPORARY-EXPOSE))
	(SETF (TV:SHEET-DEEXPOSED-TYPEOUT-ACTION *MODE-LINE-WINDOW*) '(:TEMPORARY-EXPOSE)))))

(DEFFLAVOR EDITOR-WINDOW-WITH-POP-UP-MINI-BUFFER-MIXIN ()
	   (TOP-LEVEL-EDITOR EDITOR-WITH-POP-UP-MINI-BUFFER-MIXIN
	    ZWEI-WITH-POP-UP-MINI-BUFFER-MIXIN))

(DEFMETHOD (EDITOR-WINDOW-WITH-POP-UP-MINI-BUFFER-MIXIN :AFTER :INIT)
	   EDITOR-WINDOW-WITH-POP-UP-MINI-BUFFER-MIXIN-POSITION-MODE-LINE-WINDOW)

(DEFMETHOD (EDITOR-WINDOW-WITH-POP-UP-MINI-BUFFER-MIXIN :AFTER :SET-EDGES)
	   EDITOR-WINDOW-WITH-POP-UP-MINI-BUFFER-MIXIN-POSITION-MODE-LINE-WINDOW)

(DECLARE-FLAVOR-INSTANCE-VARIABLES (EDITOR-WINDOW-WITH-POP-UP-MINI-BUFFER-MIXIN)
(DEFUN EDITOR-WINDOW-WITH-POP-UP-MINI-BUFFER-MIXIN-POSITION-MODE-LINE-WINDOW (&REST IGNORE)
  (LET* ((BOTTOM (+ TV:Y-OFFSET TV:HEIGHT))
	 (HEIGHT (TV:SHEET-HEIGHT *GLOBAL-MODE-LINE-WINDOW*))
	 (DESIRED-TOP (- BOTTOM HEIGHT))
	 (TOP (MAX DESIRED-TOP (TV:SHEET-INSIDE-TOP
				 (TV:SHEET-SUPERIOR *GLOBAL-MODE-LINE-WINDOW*)))))
    (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':SET-EDGES
	     TV:X-OFFSET TOP (+ TV:X-OFFSET TV:WIDTH) (+ TOP HEIGHT)))))

(DEFFLAVOR POP-UP-MODE-LINE-WINDOW () (MODE-LINE-WINDOW))

(DEFMETHOD (POP-UP-MODE-LINE-WINDOW :DONE-WITH-MODE-LINE-WINDOW) ()
  (FUNCALL-SELF ':TEMPORARY-DEEXPOSE)
  (FUNCALL-SELF ':DEACTIVATE))

(DEFFLAVOR ZWEI-MACRO-MIXIN () ())

(DEFMETHOD (ZWEI-MACRO-MIXIN :MACRO-TERMINATE) ()
  (LET ((PT (POINT)))
    (OR (BP-= PT (INTERVAL-FIRST-BP *INTERVAL*))
	(BP-= PT (INTERVAL-LAST-BP *INTERVAL*)))))

(DEFMETHOD (ZWEI-MACRO-MIXIN :SET-MACRO-LEVEL) (LEVEL)
  (SETQ *MACRO-LEVEL* LEVEL)
  (REDISPLAY-MODE-LINE))

(DEFMETHOD (ZWEI-MACRO-MIXIN :READ-MACRO-LINE) (PROMPT
						&AUX (*CURRENT-COMMAND* 'READ-MACRO-LINE))
  (TYPEIN-LINE-READLINE PROMPT))

(DEFFLAVOR BASIC-EDITOR-WINDOW () ()
  (:INCLUDED-FLAVORS TOP-LEVEL-EDITOR ZWEI-WINDOW))

(DEFMETHOD (BASIC-EDITOR-WINDOW :AFTER :INIT) (IGNORE)
  (SETQ *WINDOW* ZWEI-WINDOW
	*INTERVAL* (CREATE-INTERVAL NIL NIL T))
  (SET-WINDOW-INTERVAL *WINDOW* *INTERVAL*)  
  (PUSH *WINDOW* *WINDOW-LIST*))

(DEFFLAVOR EDITOR-WINDOW () (TOP-LEVEL-EDITOR BASIC-EDITOR-WINDOW ZWEI-WINDOW))

(DEFMETHOD (EDITOR-WINDOW :EDIT-STRING) (STRING)
  (EDSTRING STRING SELF))

(DEFUN EDSTRING (STRING &OPTIONAL WINDOW-OR-LIST-OR-LEFT TOP RIGHT BOTTOM COMTAB
			&AUX FLA SUP OPTS)
  (COND ((NOT (TYPEP WINDOW-OR-LIST-OR-LEFT 'TV:SHEET))
	 (SETQ FLA (COND ((SYMBOLP WINDOW-OR-LIST-OR-LEFT) WINDOW-OR-LIST-OR-LEFT)
			 ((LISTP WINDOW-OR-LIST-OR-LEFT)
			  (SETQ SUP (GET WINDOW-OR-LIST-OR-LEFT ':SUPERIOR)
				OPTS (CDR WINDOW-OR-LIST-OR-LEFT))
			  (CAR WINDOW-OR-LIST-OR-LEFT))))
	 (OR FLA (SETQ FLA 'STANDALONE-EDITOR-WINDOW))
	 (OR SUP (SETQ SUP TV:DEFAULT-SCREEN))
	 (OR (NUMBERP WINDOW-OR-LIST-OR-LEFT) (SETQ WINDOW-OR-LIST-OR-LEFT 0))
	 (OR TOP (SETQ TOP 0))
	 (OR RIGHT (SETQ RIGHT (TV:SHEET-WIDTH SUP)))
	 (OR BOTTOM (SETQ BOTTOM (* 4 (TV:SHEET-LINE-HEIGHT SUP))))
	 (OR COMTAB (SETQ COMTAB *STANDALONE-COMTAB*))
	 (SETQ WINDOW-OR-LIST-OR-LEFT (LEXPR-FUNCALL #'TV:MAKE-WINDOW FLA
						     ':SUPERIOR SUP
						     ':LEFT WINDOW-OR-LIST-OR-LEFT ':TOP TOP
						     ':RIGHT RIGHT ':BOTTOM BOTTOM ':LABEL NIL
						     ':*COMTAB* COMTAB
						     OPTS))))
  (FUNCALL WINDOW-OR-LIST-OR-LEFT ':SET-INTERVAL-STRING STRING)
  (TV:WINDOW-CALL (WINDOW-OR-LIST-OR-LEFT :DEACTIVATE)
    (FUNCALL WINDOW-OR-LIST-OR-LEFT ':EDIT))
  (VALUES (FUNCALL WINDOW-OR-LIST-OR-LEFT ':INTERVAL-STRING) WINDOW-OR-LIST-OR-LEFT))

;;; This flavor is useful for the COMPLETING-READ function
(DEFFLAVOR ZWEI-WINDOW-WITH-TOP-OUTSIDE-LABEL () (TV:TOP-LABEL-MIXIN ZWEI-WINDOW)
  (:DOCUMENTATION :COMBINATION))

;;; This is useful for standalone small editing tasks
(DEFFLAVOR STANDALONE-EDITOR-WINDOW
	((*COMTAB* *STANDALONE-COMTAB*))
	(EDITOR-WINDOW-WITH-POP-UP-MINI-BUFFER-MIXIN EDITOR-WINDOW))

(DEFFLAVOR STANDALONE-EDITOR-PANE () (TV:PANE-MIXIN STANDALONE-EDITOR-WINDOW))

(DEFFLAVOR TEMPORARY-MODE-LINE-WINDOW
	((BACKGROUND-TYPEOUT-WINDOW NIL)
	 BACKGROUND-TYPEOUT-STREAM)
	(MODE-LINE-SUPERIOR-MIXIN MODE-LINE-WINDOW-MIXIN
	 TV:DONT-SELECT-WITH-MOUSE-MIXIN TV:ANY-TYI-MIXIN
	 TV:TEMPORARY-WINDOW-MIXIN DONT-SCREEN-MANAGE-MIXIN
	 TV:STREAM-MIXIN TV:SELECT-MIXIN
	 FIXED-HEIGHT-WINDOW-MIXIN TV:MINIMUM-WINDOW)
  (:SETTABLE-INSTANCE-VARIABLES BACKGROUND-TYPEOUT-WINDOW)
  (:GETTABLE-INSTANCE-VARIABLES BACKGROUND-TYPEOUT-STREAM))

(DEFMETHOD (TEMPORARY-MODE-LINE-WINDOW :AFTER :INIT) (IGNORE)
  (SETQ BACKGROUND-TYPEOUT-STREAM (LET-CLOSED ((*GLOBAL-MODE-LINE-WINDOW* SELF))
				    'BACKGROUND-TYPEOUT-STREAM)))

;;; Perhaps this should go someplace else
(DEFMETHOD (TEMPORARY-MODE-LINE-WINDOW :MOVE-NEAR-WINDOW) (WINDOW &OPTIONAL (EXPOSE-P T))
  (FUNCALL-SELF ':SET-SUPERIOR (TV:SHEET-SUPERIOR WINDOW))
  (MULTIPLE-VALUE-BIND (LEFT TOP RIGHT BOTTOM)
      (FUNCALL WINDOW ':EDGES)
      ;If it won't fit below try putting it above
      (AND (> (+ BOTTOM TV:HEIGHT) TV:(SHEET-INSIDE-BOTTOM SUPERIOR))
	   (SETQ BOTTOM (MAX (- TOP TV:HEIGHT) TV:(SHEET-INSIDE-TOP SUPERIOR))))
      ;Put it there
      (FUNCALL-SELF ':SET-EDGES LEFT BOTTOM RIGHT (+ BOTTOM TV:HEIGHT))
      (AND EXPOSE-P (FUNCALL-SELF ':EXPOSE))))

(DEFMETHOD (TEMPORARY-MODE-LINE-WINDOW :AFTER :REDISPLAY) (IGNORE &OPTIONAL FORCE)
  (AND FORCE (NOT TV:EXPOSED-P)
       (LET ((LEN (LOOP FOR STR IN PREVIOUS-MODE-LINE
			SUM (TV:SHEET-STRING-LENGTH SELF STR))))
	 (AND (> LEN (TV:SHEET-INSIDE-WIDTH))
	      (FUNCALL-SELF ':SET-SIZE
			    (MIN (+ TV:LEFT-MARGIN-SIZE LEN TV:RIGHT-MARGIN-SIZE)
				 (TV:SHEET-INSIDE-WIDTH TV:SUPERIOR))
			    TV:HEIGHT))
	 (AND (> (+ TV:X-OFFSET TV:WIDTH) (TV:SHEET-INSIDE-RIGHT TV:SUPERIOR))
	      (FUNCALL-SELF ':SET-POSITION (- (TV:SHEET-INSIDE-RIGHT TV:SUPERIOR) TV:WIDTH)
			    TV:Y-OFFSET)))))

(DEFMETHOD (TEMPORARY-MODE-LINE-WINDOW :AFTER :DEACTIVATE) ()
  (COND (BACKGROUND-TYPEOUT-WINDOW
	 (FUNCALL BACKGROUND-TYPEOUT-WINDOW ':DEACTIVATE)
	 (SETQ BACKGROUND-TYPEOUT-WINDOW BACKGROUND-TYPEOUT-STREAM))))

(DEFFLAVOR TEMPORARY-MODE-LINE-WINDOW-WITH-BORDERS ()
	   (TV:BORDERS-MIXIN TEMPORARY-MODE-LINE-WINDOW))

(DEFFLAVOR EDITOR-FOR-TEMPORARY-MINI-BUFFER
	(*GLOBAL-MODE-LINE-WINDOW*
	 (*NUMERIC-ARG* NIL) (*NUMERIC-ARG-P* 1))
	()
  (:INCLUDED-FLAVORS TOP-LEVEL-EDITOR)
  (:INITABLE-INSTANCE-VARIABLES *GLOBAL-MODE-LINE-WINDOW*))

(DEFRESOURCE EDITOR-FOR-TEMPORARY-MINI-BUFFER-RESOURCE ()
  :CONSTRUCTOR (MAKE-EDITOR-FOR-TEMPORARY-MINI-BUFFER)
  :INITIAL-COPIES 0)	;Due to bootstrapping

(DEFUN MAKE-EDITOR-FOR-TEMPORARY-MINI-BUFFER (&OPTIONAL (SUPERIOR TV:DEFAULT-SCREEN)
							IO-BUFFER
					      &AUX *GLOBAL-MODE-LINE-WINDOW*)
  (INITIALIZE-MINI-BUFFER-WINDOW 1 SUPERIOR
				 'TEMPORARY-MODE-LINE-WINDOW-WITH-BORDERS
				 IO-BUFFER)
  (MAKE-COMMAND-LOOP *STANDARD-COMTAB*
		     (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':MINI-BUFFER-WINDOW)
		     'EDITOR-FOR-TEMPORARY-MINI-BUFFER
		     ':*GLOBAL-MODE-LINE-WINDOW* *GLOBAL-MODE-LINE-WINDOW*))

(DEFMETHOD (EDITOR-FOR-TEMPORARY-MINI-BUFFER :AFTER :INIT) (IGNORE)
  (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':SET-IO-BUFFER TV:IO-BUFFER)
  (PUSH (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':MINI-BUFFER-WINDOW) *WINDOW-LIST*)
  (SETF (TV:SHEET-DEEXPOSED-TYPEOUT-ACTION *GLOBAL-MODE-LINE-WINDOW*) ':EXPOSE))

(DEFMETHOD (EDITOR-FOR-TEMPORARY-MINI-BUFFER :TERMINAL-STREAMS) ()
  (VALUES (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':BACKGROUND-TYPEOUT-STREAM)
	  *GLOBAL-MODE-LINE-WINDOW*
	  SI:SYN-TERMINAL-IO SYN-TYPEIN-WINDOW-IO))

(DEFMETHOD (EDITOR-FOR-TEMPORARY-MINI-BUFFER :CALL-MINI-BUFFER-NEAR-WINDOW)
	   (WINDOW FUNCTION &REST ARGS &AUX OLD-SELECTED-WINDOW *TYPEOUT-WINDOW*)
  (SETQ *TYPEOUT-WINDOW* 'BACKGROUND-TYPEOUT-STREAM)
  (COND ((EQ WINDOW ':MOUSE)
	 (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':SET-SIZE 700
		  (TV:SHEET-HEIGHT *GLOBAL-MODE-LINE-WINDOW*))
	 (TV:EXPOSE-WINDOW-NEAR *GLOBAL-MODE-LINE-WINDOW* '(:MOUSE) NIL NIL))
	(T
	 (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':MOVE-NEAR-WINDOW WINDOW NIL)))
  (UNWIND-PROTECT
    (MULTIPLE-VALUE-BIND (TERMINAL-IO STANDARD-INPUT)
	(FUNCALL-SELF ':TERMINAL-STREAMS)
      (SETQ OLD-SELECTED-WINDOW TV:SELECTED-WINDOW)
      ;; This is a crock.  If the user has managed to typeahead the whole thing, we can
      ;; avoid exposing the pop-up window.  Take hardware input and force it in.
      TV:(LET ((SELECTED-IO-BUFFER IO-BUFFER))
	   (KBD-SNARF-INPUT IO-BUFFER))
      (*CATCH 'TOP-LEVEL
	(APPLY FUNCTION ARGS)))
    (AND OLD-SELECTED-WINDOW (FUNCALL OLD-SELECTED-WINDOW ':SELECT))
    (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':DEACTIVATE)
    (FUNCALL (WINDOW-SHEET (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':MINI-BUFFER-WINDOW))
	     ':DEACTIVATE)))

(DEFUN TYPEIN-LINE-READLINE-NEAR-WINDOW (WINDOW CTL-STRING &REST ARGS)
  (USING-RESOURCE (EDITOR EDITOR-FOR-TEMPORARY-MINI-BUFFER-RESOURCE)
    (LEXPR-FUNCALL EDITOR ':CALL-MINI-BUFFER-NEAR-WINDOW WINDOW
		   #'TYPEIN-LINE-READLINE CTL-STRING ARGS)))

(DEFUN READ-DEFAULTED-PATHNAME-NEAR-WINDOW (WINDOW PROMPT
					    &OPTIONAL (DEFAULTS (PATHNAME-DEFAULTS))
						      SPECIAL-TYPE)
  (USING-RESOURCE (EDITOR EDITOR-FOR-TEMPORARY-MINI-BUFFER-RESOURCE)
    (FUNCALL EDITOR ':CALL-MINI-BUFFER-NEAR-WINDOW WINDOW
	     #'READ-DEFAULTED-PATHNAME PROMPT DEFAULTS SPECIAL-TYPE)))

(DEFUN READ-BUFFER-NAME-NEAR-WINDOW (WINDOW PROMPT DEFAULT &OPTIONAL IMPOSSIBLE-IS-OK-P)
  (USING-RESOURCE (EDITOR EDITOR-FOR-TEMPORARY-MINI-BUFFER-RESOURCE)
    (FUNCALL EDITOR ':CALL-MINI-BUFFER-NEAR-WINDOW WINDOW
	     #'READ-BUFFER-NAME PROMPT DEFAULT IMPOSSIBLE-IS-OK-P)))

(DEFVAR *BACKGROUND-TYPEOUT-WHICH-OPERATIONS*)

(DEFUN BACKGROUND-TYPEOUT-STREAM (OP &REST ARGS)
  (SELECTQ OP
    (:WHICH-OPERATIONS 
     (OR (BOUNDP '*BACKGROUND-TYPEOUT-WHICH-OPERATIONS*)
	 (USING-RESOURCE (WINDOW BACKGROUND-TYPEOUT-WINDOWS)
	   (LET ((WO (FUNCALL WINDOW ':WHICH-OPERATIONS)))
	     (SETQ *BACKGROUND-TYPEOUT-WHICH-OPERATIONS*
		   (IF (MEMQ ':BEEP WO) WO (CONS ':BEEP WO))))))
     *BACKGROUND-TYPEOUT-WHICH-OPERATIONS*)
    (:BEEP
     (LET ((W (WITHOUT-INTERRUPTS
		TV:(IF SELECTED-WINDOW
		       (SHEET-GET-SCREEN SELECTED-WINDOW)
		       DEFAULT-SCREEN))))
       (LEXPR-FUNCALL W ':BEEP ARGS)))
    ((:INCOMPLETE-P :MAKE-COMPLETE) NIL)
    (OTHERWISE
     (SETQ TERMINAL-IO (ALLOCATE-RESOURCE 'BACKGROUND-TYPEOUT-WINDOWS
					  (TV:SHEET-SUPERIOR *GLOBAL-MODE-LINE-WINDOW*)))
     (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':SET-BACKGROUND-TYPEOUT-WINDOW TERMINAL-IO)
     (TV:SHEET-FORCE-ACCESS (TERMINAL-IO :NO-PREPARE)
       (FUNCALL TERMINAL-IO ':SET-LABEL (STRING-APPEND (FUNCALL *GLOBAL-MODE-LINE-WINDOW*
								':NAME)
						       " Background Typeout Window"))
       (FUNCALL TERMINAL-IO ':SET-PROCESS CURRENT-PROCESS)
       (LOCAL-DECLARE ((SPECIAL TV:IO-BUFFER))
	 (FUNCALL TERMINAL-IO ':SET-IO-BUFFER TV:IO-BUFFER))
       (FUNCALL TERMINAL-IO ':CLEAR-SCREEN))
     (LET ((OLD-SEL TV:SELECTED-WINDOW))
       (TV:WITH-SHEET-DEEXPOSED (*GLOBAL-MODE-LINE-WINDOW*)
	 (FUNCALL *GLOBAL-MODE-LINE-WINDOW* ':MOVE-NEAR-WINDOW TERMINAL-IO NIL)
	 (FUNCALL TERMINAL-IO ':EXPOSE))
       (AND OLD-SEL (FUNCALL OLD-SEL ':SELECT NIL)))
     (SETQ *TYPEOUT-WINDOW* TERMINAL-IO)
     (LEXPR-FUNCALL TERMINAL-IO OP ARGS))))

(DEFFLAVOR BACKGROUND-TYPEOUT-WINDOW
	()
	TV:(ANY-TYI-MIXIN BASIC-MOUSE-SENSITIVE-ITEMS WINDOW))

(DEFMETHOD (BACKGROUND-TYPEOUT-WINDOW :INCOMPLETE-P) () NIL)

(DEFMETHOD (BACKGROUND-TYPEOUT-WINDOW :MAKE-COMPLETE) () NIL)

(DEFWINDOW-RESOURCE BACKGROUND-TYPEOUT-WINDOWS ()
	:MAKE-WINDOW (BACKGROUND-TYPEOUT-WINDOW
			 :HEIGHT (// TV:(SHEET-HEIGHT DEFAULT-SCREEN) 3))
	:REUSABLE-WHEN :DEACTIVATED
	:INITIAL-COPIES 0)
  
(DEFWINDOW-RESOURCE SPLIT-SCREEN-MENU ()
	:MAKE-WINDOW (TV:POP-UP-MENU :NAME "Split Screen" :LABEL "Split screen buffer:"
				     :COLUMNS 2)
	:REUSABLE-WHEN :DEEXPOSED
	:INITIAL-COPIES 0)

(DEFCONST *SPLIT-SCREEN-AMONG-BUFFERS-FIXED-ITEMS*
     '(("New buffer" :VALUE "New buffer"
	:DOCUMENTATION "Create a new, empty buffer.  Prompt for its name.")
       ("Find file" :VALUE "Find file" 
	:DOCUMENTATION "Do a Find File command and put the resulting buffer in a window.")
       ("Do It" :VALUE "Do It"
	:FONT FONTS:MEDFNB
	:DOCUMENTATION "Complete the selection and set up the windows as specified.")
       ("Abort" :VALUE "Abort"
	:FONT FONTS:MEDFNB
	:DOCUMENTATION "Abort the Split Screen command.")))

(DEFUN SPLIT-SCREEN-AMONG-BUFFERS-VIA-MENUS (FRAME BUFFER-ALIST)
  (USING-RESOURCE (MENU SPLIT-SCREEN-MENU)
    (FUNCALL MENU ':SET-ITEM-LIST (APPEND BUFFER-ALIST
					  (IF (ODDP (LENGTH BUFFER-ALIST))
					      '(("" :NO-SELECT T)))
					  '(("" :NO-SELECT T)) '(("" :NO-SELECT T))
					  *SPLIT-SCREEN-AMONG-BUFFERS-FIXED-ITEMS*))
    (TV:EXPOSE-WINDOW-NEAR MENU '(:MOUSE))
    (USING-RESOURCE (LAYWIN TV:SPLIT-SCREEN-LAYOUT-WINDOW)
      (FUNCALL LAYWIN ':CLEAR-FROBS)
      (UNWIND-PROTECT
	(DO ((BUFFER-LIST NIL)
	     (N-WINDOWS 0)
	     (RES))
	    (NIL)
	  (AND (= N-WINDOWS 1)
	       (FUNCALL LAYWIN ':MOVE-NEAR-WINDOW MENU
			(MULTIPLE-VALUE-BIND (WIDTH HEIGHT)
			    (FUNCALL FRAME ':INSIDE-SIZE)
			  (CONS WIDTH HEIGHT))))
	  (SETQ RES (FUNCALL MENU ':CHOOSE))
	  (AND (EQUAL RES "New buffer")
	       (SETQ RES (READ-BUFFER-NAME-NEAR-WINDOW MENU "New buffer:" NIL T)))
	  (COND ((SYMBOLP RES))
		((NOT (STRINGP RES))
		 (PUSH RES BUFFER-LIST)
		 (FUNCALL LAYWIN ':ADD-FROB (BUFFER-NAME RES))
		 (SETQ N-WINDOWS (1+ N-WINDOWS)))
		((STRING-EQUAL RES "Abort") (RETURN NIL))
		((STRING-EQUAL RES "Find file")
		 (SETQ RES (READ-DEFAULTED-PATHNAME-NEAR-WINDOW MENU "Find file:"))
		 (COND ((TYPEP RES 'FS:PATHNAME)
			(PUSH RES BUFFER-LIST)
			(FUNCALL LAYWIN ':ADD-FROB (FUNCALL RES ':STRING-FOR-EDITOR))
			(SETQ N-WINDOWS (1+ N-WINDOWS)))))
		(T (RETURN (NREVERSE BUFFER-LIST)))))
	(TV:DELAYING-SCREEN-MANAGEMENT
	  (FUNCALL LAYWIN ':DEACTIVATE)
	  (FUNCALL MENU ':DEACTIVATE))))))

(DEFUN SPLIT-SCREEN-AMONG-BUFFERS-DO-IT (FRAME BUFFER-LIST &AUX N-COLUMNS N-ROWS WIDTH HEIGHT
					       FRAME-LEFT FRAME-TOP FRAME-RIGHT FRAME-BOTTOM
					       WINDOW-LIST)
  (LET ((N-WINDOWS (LENGTH BUFFER-LIST)))
    (IF (< N-WINDOWS 4)
	(SETQ N-COLUMNS 1 N-ROWS N-WINDOWS)
	(SETQ N-COLUMNS 2 N-ROWS (// (1+ N-WINDOWS) 2))))
  (MULTIPLE-VALUE (FRAME-LEFT FRAME-TOP FRAME-RIGHT FRAME-BOTTOM)
    (FUNCALL FRAME ':INSIDE-EDGES-WITHOUT-MODE-LINE-WINDOW))
  (SETQ WIDTH (// (- FRAME-RIGHT FRAME-LEFT) N-COLUMNS)
	HEIGHT (// (- FRAME-BOTTOM FRAME-TOP) N-ROWS))
  (SETQ WINDOW-LIST (FUNCALL FRAME ':N-EDITOR-WINDOWS (LENGTH BUFFER-LIST)))
  (TV:DELAYING-SCREEN-MANAGEMENT
    (DO ((BL BUFFER-LIST (CDR BL))
	 (WL WINDOW-LIST (CDR WL))
	 (I 0 (1+ I))
	 (LEFT) (RIGHT) (TOP) (BOTTOM) (WINDOW))
	((NULL BL))
      (SETQ LEFT (+ FRAME-LEFT (* (\ I N-COLUMNS) WIDTH))
	    RIGHT (+ LEFT WIDTH)
	    TOP (+ FRAME-TOP (* (// I N-COLUMNS) HEIGHT))
	    BOTTOM (+ TOP HEIGHT))
      ;; The bottom-most window is wider if there are an odd number of them
      (AND (NULL (CDR BL)) (SETQ RIGHT FRAME-RIGHT))
      (SETQ WINDOW (WINDOW-SHEET (CAR WL)))
      (FUNCALL WINDOW ':SET-EDGES LEFT TOP RIGHT BOTTOM)
      (FUNCALL WINDOW ':EXPOSE NIL ':CLEAN)
      (AND (ZEROP I) (FUNCALL WINDOW ':SELECT NIL))))
  (FUNCALL FRAME ':UPDATE-LABELS)
  WINDOW-LIST)

(DEFUN WINDOW-FRAME (ZWEI-WINDOW)
  (TV:SHEET-SUPERIOR (WINDOW-SHEET ZWEI-WINDOW)))

(DEFFLAVOR ZWEI-FRAME
	((TV:IO-BUFFER NIL)
	 (MODE-LINE-WINDOW 'MODE-LINE-WINDOW))
	(TV:INITIALLY-INVISIBLE-MIXIN TV:BASIC-FRAME)
  (:INITABLE-INSTANCE-VARIABLES TV:IO-BUFFER MODE-LINE-WINDOW)
  (:GETTABLE-INSTANCE-VARIABLES MODE-LINE-WINDOW)
  (:INIT-KEYWORDS :NUMBER-OF-MINI-BUFFER-LINES)
  (:DEFAULT-INIT-PLIST :NUMBER-OF-MINI-BUFFER-LINES 3))

(DEFMETHOD (ZWEI-FRAME :AFTER :INIT) (INIT-PLIST)
  (AND (SYMBOLP MODE-LINE-WINDOW)
       (LET (*GLOBAL-MODE-LINE-WINDOW*)
	 (INITIALIZE-MINI-BUFFER-WINDOW (GET INIT-PLIST ':NUMBER-OF-MINI-BUFFER-LINES)
					SELF MODE-LINE-WINDOW TV:IO-BUFFER
					':BOTTOM (TV:SHEET-INSIDE-BOTTOM)
					':EXPOSE-P T)
	 (SETQ MODE-LINE-WINDOW *GLOBAL-MODE-LINE-WINDOW*
	       TV:IO-BUFFER (FUNCALL MODE-LINE-WINDOW ':IO-BUFFER))
	 (FUNCALL (FUNCALL MODE-LINE-WINDOW ':TYPEIN-WINDOW) ':EXPOSE))))

(DEFMETHOD (ZWEI-FRAME :CREATE-WINDOW) (TYPE &REST OPTIONS)
  (LEXPR-FUNCALL #'CREATE-WINDOW TYPE
		 ':SUPERIOR SELF ':BOTTOM (TV:SHEET-Y-OFFSET MODE-LINE-WINDOW)
		 ':IO-BUFFER TV:IO-BUFFER
		 OPTIONS))

(DEFMETHOD (ZWEI-FRAME :EDITOR-WINDOW) ()
  (DO ((L TV:INFERIORS (CDR L)))
      ((NULL L) (FERROR NIL "No inferiors"))
    (OR (EQ (CAR L) MODE-LINE-WINDOW)
	(RETURN (FUNCALL (CAR L) ':ZWEI-WINDOW)))))

(DEFMETHOD (ZWEI-FRAME :SELECTABLE-WINDOWS) ()
  (LET ((SELECTABLE-WINDOWS NIL))
    (DOLIST (I TV:EXPOSED-INFERIORS)
      (OR (EQ I MODE-LINE-WINDOW)
	  (LET ((STRING (FUNCALL I ':NAME-FOR-SELECTION)))
	    (AND STRING
		 (PUSH (LIST STRING SELF) SELECTABLE-WINDOWS)))))
    (NREVERSE SELECTABLE-WINDOWS)))

(DEFMETHOD (ZWEI-FRAME :INSIDE-EDGES-WITHOUT-MODE-LINE-WINDOW) ()
  (VALUES (TV:SHEET-INSIDE-LEFT) (TV:SHEET-INSIDE-TOP) (TV:SHEET-INSIDE-RIGHT)
	  (- (TV:SHEET-INSIDE-BOTTOM) (TV:SHEET-HEIGHT MODE-LINE-WINDOW))))

(DEFWRAPPER (ZWEI-FRAME :CHANGE-OF-SIZE-OR-MARGINS) (IGNORE . BODY)
  `(LOCAL-DECLARE ((SPECIAL OLD-EXPOSED-INFERIORS
			    OLD-INSIDE-LEFT OLD-INSIDE-TOP
			    OLD-INSIDE-RIGHT OLD-INSIDE-BOTTOM))
     (LET ((OLD-EXPOSED-INFERIORS (COPYLIST TV:EXPOSED-INFERIORS)))
       (MULTIPLE-VALUE-BIND (OLD-INSIDE-LEFT OLD-INSIDE-TOP
			     OLD-INSIDE-RIGHT OLD-INSIDE-BOTTOM)
	   (FUNCALL-SELF ':INSIDE-EDGES-WITHOUT-MODE-LINE-WINDOW)
	 . ,BODY))))

(DEFMETHOD (ZWEI-FRAME :AFTER :CHANGE-OF-SIZE-OR-MARGINS)
	   (&REST IGNORE &AUX OLD-INSIDE-WIDTH OLD-INSIDE-HEIGHT
			      NEW-INSIDE-LEFT NEW-INSIDE-TOP
			      NEW-INSIDE-RIGHT NEW-INSIDE-BOTTOM
			      NEW-INSIDE-WIDTH NEW-INSIDE-HEIGHT)
  (DECLARE (SPECIAL OLD-EXPOSED-INFERIORS
		    OLD-INSIDE-LEFT OLD-INSIDE-TOP
		    OLD-INSIDE-RIGHT OLD-INSIDE-BOTTOM))
  (SETQ OLD-INSIDE-WIDTH (- OLD-INSIDE-RIGHT OLD-INSIDE-LEFT)
	OLD-INSIDE-HEIGHT (- OLD-INSIDE-BOTTOM OLD-INSIDE-TOP))
  (MULTIPLE-VALUE (NEW-INSIDE-LEFT NEW-INSIDE-TOP NEW-INSIDE-RIGHT NEW-INSIDE-BOTTOM)
    (FUNCALL-SELF ':INSIDE-EDGES-WITHOUT-MODE-LINE-WINDOW))
  (SETQ NEW-INSIDE-WIDTH (- NEW-INSIDE-RIGHT NEW-INSIDE-LEFT)
	NEW-INSIDE-HEIGHT (- NEW-INSIDE-BOTTOM NEW-INSIDE-TOP))
  (TV:WITH-SHEET-DEEXPOSED (SELF)
    (DO ((WL (COPYLIST TV:INFERIORS) (CDR WL))
	 (WINDOW)
	 (OLD-LEFT) (OLD-TOP) (OLD-RIGHT) (OLD-BOTTOM)
	 (NEW-LEFT) (NEW-TOP) (NEW-RIGHT) (NEW-BOTTOM))
	((NULL WL))
      (SETQ WINDOW (CAR WL))
      (MULTIPLE-VALUE (OLD-LEFT OLD-TOP OLD-RIGHT OLD-BOTTOM)
	(FUNCALL WINDOW ':EDGES))
      (IF (EQ WINDOW MODE-LINE-WINDOW)
	  (SETQ NEW-LEFT NEW-INSIDE-LEFT
		NEW-TOP NEW-INSIDE-BOTTOM
		NEW-RIGHT NEW-INSIDE-RIGHT
		NEW-BOTTOM (+ NEW-INSIDE-BOTTOM (- OLD-BOTTOM OLD-TOP)))
	  (SETQ NEW-LEFT (IF (= OLD-LEFT OLD-INSIDE-LEFT) NEW-INSIDE-LEFT
			     (// (* OLD-LEFT NEW-INSIDE-WIDTH) OLD-INSIDE-WIDTH))
		NEW-TOP (IF (= OLD-TOP OLD-INSIDE-TOP) NEW-INSIDE-TOP
			    (// (* OLD-TOP NEW-INSIDE-HEIGHT) OLD-INSIDE-HEIGHT))
		NEW-RIGHT (IF (= OLD-RIGHT OLD-INSIDE-RIGHT) NEW-INSIDE-RIGHT
			      (// (* OLD-RIGHT NEW-INSIDE-WIDTH) OLD-INSIDE-WIDTH))
		NEW-BOTTOM (IF (= OLD-BOTTOM OLD-INSIDE-BOTTOM) NEW-INSIDE-BOTTOM
			       (// (* OLD-BOTTOM NEW-INSIDE-HEIGHT) OLD-INSIDE-HEIGHT))))
      (FUNCALL WINDOW ':SET-EDGES NEW-LEFT NEW-TOP NEW-RIGHT NEW-BOTTOM)
      (AND (MEMQ WINDOW OLD-EXPOSED-INFERIORS)
	   (FUNCALL WINDOW ':EXPOSE)))))

(DEFFLAVOR ZWEI-PANE () (TV:PANE-MIXIN ZWEI))

(DEFMETHOD (ZWEI-PANE :MODE-LINE-WINDOW) ()
  (FUNCALL TV:SUPERIOR ':MODE-LINE-WINDOW))

(DEFFLAVOR ZWEI-WINDOW-PANE () (ZWEI-PANE ZWEI-WINDOW))

(DEFFLAVOR ZMACS-WINDOW-PANE () (ZWEI-PANE ZMACS-WINDOW)
  (:DEFAULT-INIT-PLIST :SAVE-BITS NIL))

(DEFMETHOD (ZMACS-WINDOW-PANE :CHANGE-LABEL) (NEW-LABEL)
  (FUNCALL TV:SUPERIOR ':CHANGE-PANE-LABEL SELF NEW-LABEL))

(DEFFLAVOR ZMACS-FRAME () (ZWEI-FRAME)
  (:DEFAULT-INIT-PLIST :IO-BUFFER (FUNCALL *ZMACS-COMMAND-LOOP* ':IO-BUFFER)
		       :MODE-LINE-WINDOW 'ZMACS-MODE-LINE-WINDOW
		       :SAVE-BITS T :NUMBER-OF-MINI-BUFFER-LINES 2))

(DEFMETHOD (ZMACS-FRAME :AFTER :INIT) (IGNORE)
  (SETQ TV:SELECTED-PANE (WINDOW-SHEET
			   (FUNCALL-SELF ':CREATE-WINDOW 'ZMACS-WINDOW-PANE
					 ':ACTIVATE-P T ':LABEL NIL))))

(DEFMETHOD (ZMACS-FRAME :CHANGE-PANE-LABEL) (PANE NEW-LABEL)
  (AND ( (LENGTH TV:EXPOSED-INFERIORS) 3)
       (FUNCALL PANE ':DELAYED-SET-LABEL NEW-LABEL)))

(DEFMETHOD (ZMACS-FRAME :UPDATE-LABELS) ()
  (IF ( (LENGTH TV:EXPOSED-INFERIORS) 3)
      (DOLIST (W TV:EXPOSED-INFERIORS)
	(OR (EQ W MODE-LINE-WINDOW)
	    (FUNCALL W ':DELAYED-SET-LABEL
		     (BUFFER-NAME (WINDOW-INTERVAL (FUNCALL W ':ZWEI-WINDOW))))))
      (DOLIST (W TV:EXPOSED-INFERIORS)
	(OR (EQ W MODE-LINE-WINDOW)
	    (FUNCALL W ':SET-LABEL NIL)))))

(DEFMETHOD (ZMACS-FRAME :TWO-EDITOR-WINDOWS) ()
  (DO ((L TV:INFERIORS (CDR L))
       (WINDOW)
       (TOP-WINDOW) (BOTTOM-WINDOW))
      ((NULL L)
       (VALUES (FUNCALL TOP-WINDOW ':ZWEI-WINDOW)
	       (FUNCALL-SELF ':CREATE-WINDOW 'ZMACS-WINDOW-PANE)))
    (COND ((EQ (SETQ WINDOW (CAR L)) MODE-LINE-WINDOW))
	  ((NULL TOP-WINDOW)
	   (SETQ TOP-WINDOW WINDOW))
	  (T
	   (SETQ BOTTOM-WINDOW WINDOW)
	   (AND (TV:SHEET-EXPOSED-P BOTTOM-WINDOW)
		(< (TV:SHEET-Y-OFFSET BOTTOM-WINDOW) (TV:SHEET-Y-OFFSET TOP-WINDOW))
		(PSETQ TOP-WINDOW BOTTOM-WINDOW BOTTOM-WINDOW TOP-WINDOW))
	   (RETURN (FUNCALL TOP-WINDOW ':ZWEI-WINDOW)
		   (FUNCALL BOTTOM-WINDOW ':ZWEI-WINDOW))))))

(DEFMETHOD (ZMACS-FRAME :N-EDITOR-WINDOWS) (N &AUX LIST)
  (DO ((L TV:INFERIORS (CDR L))
       (I 0 (1+ I)))
      ((OR (NULL L) ( I N)))
    (OR (EQ (CAR L) MODE-LINE-WINDOW)
	(PUSH (FUNCALL (CAR L) ':ZWEI-WINDOW) LIST)))
  (DOTIMES (I (- N (LENGTH LIST)))
    (PUSH (FUNCALL-SELF ':CREATE-WINDOW 'ZMACS-WINDOW-PANE) LIST))
  LIST)

(DEFMETHOD (ZMACS-FRAME :PANE-TYPES-ALIST) ()
  '(("Edit" . ZMACS-WINDOW-PANE)))
