/*
 * fputest.c
 * test for SIGFPE delivery under Darwin
 */

/* set up FPU to trap floating point exceptions */
extern void u_fpu_setup(void);

#include <architecture/ppc/fp_regs.h>
#include <mach/mach.h>
#include <pthread.h>

static void *fpu_fpe_enable(void *arg);
#define FE0_MASK (1<<11)
#define FE1_MASK (1<<8)
/* FE0 FE1 exceptions enabled if either FE0 or FE1 set
 * 0 0 -- floating-point exceptions disabled
 * 0 1 -- floating-point imprecise nonrecoverable
 * 1 0 -- floating-point imprecise recoverable
 * 1 1 -- floating-point precise mode
 */

/* a thread cannot get or set its own MSR bits */
static void *
fpu_fpe_enable(void *arg)
{
	thread_t t = *(thread_t *)arg;
	struct ppc_thread_state state;
	unsigned int state_size = PPC_THREAD_STATE_COUNT;
	if (thread_get_state(t, PPC_THREAD_STATE,
			     (natural_t *)&state, &state_size) == KERN_SUCCESS) {
		state.srr1 |= FE1_MASK;
		thread_set_state(t, PPC_THREAD_STATE, (natural_t *)&state, state_size);
	}
	return 0;
}

void
u_fpu_setup(void)
{
	static volatile int looping = 0;
	thread_t self = mach_thread_self();
	pthread_t enabler;
	ppc_fp_scr_t r = get_fp_scr();
/* turn off exception bits to prevent immediate re-fault */
r.fx = r.fex = r.vx = r.ox = r.ux = r.zx = r.xx = r.vx_snan = r.vx_isi =
	r.vx_idi = r.vx_zdz = r.vx_imz = r.vx_xvc = r.vx_cvi = r.vx_soft = 0;
/* these only have to be set once, but may as well set anyway */
 r.ve = 1; /* invalid */
 r.oe = 1; /* overflow */
 r.ue = 0; /* underflow */
 r.ze = 1; /* zero divide */
 r.xe = 0; /* inexact */
 if (!looping) {
	 looping++;
	 set_fp_scr(r);
/* MSR bit reset after every fault, set it again */
	 if (!pthread_create(&enabler, 0, fpu_fpe_enable, &self))
		 pthread_join(enabler, 0);
 }
 looping = 0;
}

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include <setjmp.h>
static jmp_buf u_jmp_target;

extern void u_sigfpe(int sig);
extern double reciprocal(double x);
extern double quotient(double x, double y);
extern double powpow(double x, int n);
extern int decrement(int *ip);
static int always_true = 0;

int
main(int argc, char *argv[])
{
	int i = 5;
	double zero = (double)(argc>1000);
	double huge = (argc>1000)? 2.0 : 1.e20;
	always_true = !(argc>1000);

/* signal *ought* to be enough to get SIGFPE delivered
 * -- but it never is -- see README.fpu */
	signal(SIGFPE, &u_sigfpe);

	setjmp(u_jmp_target);
	u_fpu_setup();
/* need to make sure that loop index i actually decrements
 * despite interrupt */
	while (decrement(&i)) {
		printf("SIGFPE handling failed on pass %d, 1./0. = %g\n", 5-i,
		       reciprocal(zero));
		if (i) break;
	}
	if (!i) {
		i = 5;
		setjmp(u_jmp_target);
		u_fpu_setup();
		while (decrement(&i)) {
			printf("SIGFPE handling failed on pass %d, 0./0. = %g\n", 5-i,
			       quotient(zero, zero));
			if (i) break;
		}
		if (!i) {
			i = 5;
			setjmp(u_jmp_target);
			u_fpu_setup();
			while (decrement(&i)) {
				printf("SIGFPE handling failed on pass %d, 10.^20480 = %g\n", 5-i,
				       powpow(huge, 10));
				if (i) break;
			}
			if (!i) {
				if (setjmp(u_jmp_target)) {
					puts("SIGFPE improperly generated on underflow");
					i = 11;
				} else {
					double x = powpow(1./huge, 10);
					if (x != 0.0)
						printf("SIGFPE handling works, but 10.^-20480 = %g\n", x);
					else
						puts("SIGFPE handling works properly");
				}
			}
		}
	}
	exit(i? 2 : 0);
}

void
u_sigfpe(int sig)
{
	if (sig==SIGFPE) {
		signal(SIGFPE, &u_sigfpe);
		longjmp(u_jmp_target, 1);
	} else {
		puts("u_sigfpe called, but with bad parameter");
	}
	exit(1);
}

int
decrement(int *ip)
{
	int i = *ip;
	if (always_true) i--;
	else i -= 2;
	*ip = i;
	return i>0;
}

double
reciprocal(double x)
{
	return 1./x;
}

double
quotient(double x, double y)
{
	return x/y;
}

double
powpow(double x, int n)
{
	double y = x;
	while (n--) y *= y;
	return y;
}
