/*
 * lsim.c
 * l-machine macroinstruction simulator
 *
 * this version is not designed to be fast...  the goal is to make
 * it work first!
 *
 * Copyright (C) 2004 Brad Parker <brad@heeltoe.com>
 * $Id: lsim.c,v 1.2 2004/06/09 15:14:00 brad Exp $
 */

#include <stdio.h>
#include <stdlib.h>

#define a28(a)		((a) & 0x0fffffff)

int sim_running;
u_long sim_pc;
u_long sim_cycles;

int trace_mem;
int trace_vm;

typedef struct {
	u_long word;
	u_char tag;
} w36;

/* tags */
enum {
#if 0
  DTP_NULL = 0,
  DTP_NIL = 1,
  DTP_SYMBOL = 2,
  DTP_EXTENDED_NUMBER = 3,
  DTP_EXTERNAL_VALUE_CELL_POINTER = 4,
  DTP_LOCATIVE = 5,
  DTP_LIST = 6,
  DTP_COMPILED_FUNCTION = 7,

  DTP_ARRAY = 8,
  DTP_CLOSURE,
  DTP_ENTITY,
  DTP_LEXICAL_CLOSURE,
  DTP_SELECT_METHOD,
  DTP_INSTANCE,
  DTP_HEADER_P,
  DTP_HEADER_I,

  DTP_FIX = 16,
  DTP_FLOAT = 32,

  DTP_EVEN_PC = 48,
  DTP_GC_FORWARD,
  DTP_ONE_Q_FORWARD,
  DTP_HEADER_FORWARD,
  DTP_BODY_FORWARD,
  DTP_65,
  DTP_66,
  DTP_67,
  DTP_ODD_PC,
  DTP_71,
  DTP_72,
  DTP_73,
  DTP_74,
  DTP_75,
  DTP_76,
  DTP_77
#else
  DTP_NULL = 0,
  DTP_NIL = 1,
  DTP_SYMBOL = 2,
  DTP_EXTENDED_NUMBER = 3,
  DTP_LIST = 4,
  DTP_LOCATIVE = 5,
  DTP_COMPILED_FUNCTION = 6,
  DTP_ARRAY = 7,
  DTP_CLOSURE,
  DTP_CHARACTER,
  DTP_INSTANCE,
  DTP_LEXICAL_CLOSURE,
  DTP_LOGIC_VARIABLE,
  DTP_GENERIC_FUNCTION,
  DTP_16,
  DTP_17,

  DTP_FIX = 16,
  DTP_FLOAT = 32,

  DTP_EVEN_PC = 48,
  DTP_EXTERNAL_VALUE_CELL_POINTER,
  DTP_ONE_Q_FORWARD,
  DTP_HEADER_FORWARD,
  DTP_64,
  DTP_GC_FORWARD,
  DTP_HEADER_P,
  DTP_HEADER_I,
  DTP_ODD_PC,

  DTP_MONITOR_FORWARD,
  DTP_ELEMENT_FORWARD,

  DTP_73,
  DTP_74,
  DTP_75,
  DTP_76,
  DTP_77
#endif
};

#define dtp_fix_p(tag)	((tag & 0xf0) == DTP_FIX)

char *tag_names[] = {
	"DTP_NULL",
	"DTP_NIL",
	"DTP_SYMBOL",
	"DTP_EXTENDED_NUMBER",
	"DTP_LIST",
	"DTP_LOCATIVE",
	"DTP_COMPILED_FUNCTION",
	"DTP_ARRAY",
	"DTP_CLOSURE",
	"DTP_CHARACTER",
	"DTP_INSTANCE",
	"DTP_LEXICAL_CLOSURE",
	"DTP_LOGIC_VARIABLE",
	"DTP_GENERIC_FUNCTION",
	"DTP_16",
	"DTP_17",

	"DTP_FIX", "DTP_FIX", "DTP_FIX", "DTP_FIX",
	"DTP_FIX", "DTP_FIX", "DTP_FIX", "DTP_FIX",
	"DTP_FIX", "DTP_FIX", "DTP_FIX", "DTP_FIX",
	"DTP_FIX", "DTP_FIX", "DTP_FIX", "DTP_FIX",

	"DTP_FLOAT", "DTP_FLOAT", "DTP_FLOAT", "DTP_FLOAT",
	"DTP_FLOAT", "DTP_FLOAT", "DTP_FLOAT", "DTP_FLOAT",
	"DTP_FLOAT", "DTP_FLOAT", "DTP_FLOAT", "DTP_FLOAT",
	"DTP_FLOAT", "DTP_FLOAT", "DTP_FLOAT", "DTP_FLOAT",

	"DTP_EVEN_PC",
	"DTP_EXTERNAL_VALUE_CELL_POINTER",
	"DTP_ONE_Q_FORWARD",
	"DTP_HEADER_FORWARD",
	"DTP_64",
	"DTP_GC_FORWARD",
	"DTP_HEADER_P",
	"DTP_HEADER_I",
	"DTP_ODD_PC",

	"DTP_MONITOR_FORWARD",
	"DTP_ELEMENT_FORWARD",

	"DTP_73",
	"DTP_74",
	"DTP_75",
	"DTP_76",
	"DTP_77"
};

char *array_dispatch_names[] = {
	"%ARRAY-DISPATCH-1-BIT",
	"%ARRAY-DISPATCH-2-BIT",
	"%ARRAY-DISPATCH-4-BIT",
	"%ARRAY-DISPATCH-8-BIT",
	"%ARRAY-DISPATCH-16-BIT",
	"%ARRAY-DISPATCH-WORD",
	"%ARRAY-DISPATCH-SHORT-INDIRECT",
	"%ARRAY-DISPATCH-FIXNUM",
	"%ARRAY-DISPATCH-BOOLEAN",
	"%ARRAY-DISPATCH-LEADER",
	"%ARRAY-DISPATCH-SHORT-2D",
	"%ARRAY-DISPATCH-CHARACTER",
	"%ARRAY-DISPATCH-14",
	"%ARRAY-DISPATCH-LONG",
	"%ARRAY-DISPATCH-LONG-MULTIDIMENSIONAL",
	"%ARRAY-DISPATCH-FAT-CHARACTER"
};

enum {
	ARRAY_DISPATCH_1_BIT = 0,
	ARRAY_DISPATCH_2_BIT,
	ARRAY_DISPATCH_4_BIT,
	ARRAY_DISPATCH_8_BIT,
	ARRAY_DISPATCH_16_BIT,
	ARRAY_DISPATCH_WORD,
	ARRAY_DISPATCH_SHORT_INDIRECT,
	ARRAY_DISPATCH_FIXNUM,
	ARRAY_DISPATCH_BOOLEAN,
	ARRAY_DISPATCH_LEADER,
	ARRAY_DISPATCH_SHORT_2D,
	ARRAY_DISPATCH_CHARACTER,
	ARRAY_DISPATCH_14,
	ARRAY_DISPATCH_LONG,
	ARRAY_DISPATCH_LONG_MULTIDIMENSIONAL,
	ARRAY_DISPATCH_FAT_CHARACTER
};

char *array_type_names[] = {
	"ART-1B",
	"ART-2B",
	"ART-4B",
	"ART-8B",
	"ART-16B",
	"ART-Q",
	"ART-Q-LIST",
	"%ARRAY-TYPE-7",
	"ART-BOOLEAN",
	"%ARRAY-TYPE-11",
	"%ARRAY-TYPE-12",
	"%ARRAY-TYPE-13",
	"%ARRAY-TYPE-14",
	"ART-STRING",
	"ART-FAT-STRING",
	"ART-FIXNUM"
};

/* low memory (virtual), starting at 0 */
char *fep_communication_area_names[] = {
	"FEP-VERSION-NUMBER",
	"KLUDGE-INPUT-CHARACTER",
	"KLUDGE-OUTPUT-CHARACTER",
	"KLUDGE-MINI-BUFFER-NUMBER",
	"KLUDGE-MINI-BUFFER-FULL",
	"MACHINE-CONFIGURATION-TABLE",
	"LOAD-MAP-SIZE",
	"LOAD-MAP-ADDRESS",
	"LOAD-MAP-DPN-ADDRESS",
	"SWAP-MAP-SIZE",
	"SWAP-MAP-ADDRESS",
	"SWAP-MAP-DPN-ADDRESS",
	"BAD-MEMORY-PAGES-SIZE",
	"BAD-MEMORY-PAGES-ADDRESS",
	"NET-ADDRESS-1",
	"NET-ADDRESS-2",
	"CHAOS-ADDRESS",
	"%UNWIRED-VIRTUAL-ADDRESS-LOW",
	"%UNWIRED-VIRTUAL-ADDRESS-HIGH",
	"%UNWIRED-PHYSICAL-ADDRESS-LOW",
	"COLD-LOAD-STREAM-OUTPUT-CHARACTER",
	"COLD-LOAD-STREAM-INPUT-CHARACTER",
	"CART-STATUS",
	"CART-COMMAND",
	"CART-EXT-STATUS-1",
	"CART-EXT-STATUS-2",
	"CART-MAP-TABLE-1",
	"CART-MAP-COUNT-1",
	"CART-MAP-TABLE-2",
	"CART-MAP-COUNT-2",
	"CART-LISP-COUNTER",
	"CART-FEP-COUNTER",
	"CART-LAST-BUF-OFFSET",
	"FEP-CHANNELS",
	"NEW-FEP-CHANNEL",
	"TASK-3-REQUESTS",
	"TASK-3-ENABLES",
	"SCREEN-DATA-OFFSET",
	"SCREEN-LOCATIONS-PER-LINE",
	"SCREEN-RASTER-HEIGHT",
	"SCREEN-TOP-FAKE-LINES",
	"SCREEN-BOTTOM-FAKE-LINES",
	"SCREEN-TOP-MARGIN-LINES",
	"SCREEN-BOTTOM-MARGIN-LINES",
	"SCREEN-RASTER-WIDTH",
	"SCREEN-LEFT-MARGIN-PIXELS",
	"SCREEN-RIGHT-MARGIN-PIXELS",
	"FEP-PRELOADED-ADDRESS",
	"FEP-PRELOADED-SIZE",
	"MACHINE-ID-PROMS",
	"CALENDAR-CLOCK-COMMAND",
	"CALENDAR-CLOCK-DATA-1",
	"CALENDAR-CLOCK-DATA-2",
	"REQUESTING-LISP-TO-STOP",
	"FEP-COMMAND-STRING",
	"FEP-BUS-COMMAND",
	"FEP-BUS-ADDRESS",
	"FEP-BUS-DATA",
	"FEP-BUS-ADDRESS-2",
	"FEP-BUS-CONDITIONS",
	"CONSOLE-BRIGHTNESS",
	"CONSOLE-VOLUME",
	"CONSOLE-COMMAND",
	"%HARDWARE-CONFIGURATION",
	"%SOFTWARE-CONFIGURATION",
	"*CRASH-DATA-REQUEST*",
	"*CONSOLE-SWITCHES*",
	"*CONSOLE-SLOT*"
};

void write_amem_36(int addr, w36 *v);
void read_amem_36(int addr, w36 *v);

/* cdr codes (top 2 bits of tag) */
enum {
	CDR_NEXT = 0,
	CDR_NIL,
	CDR_NORMAL,
	CDR_SPARE
};

/*
 * simple mmu
 * implements basic 3-level page table
 */

/* single page entry */
struct page_s {
	int fpn;
	int paged_in;
	u_long *mem_addr;
	u_char *tag_addr;
};

/* directory of pages */
struct page_dir_s {
	struct page_s pages[256];
};

static struct page_dir_s *page_root;
static void *world_stream;

struct page_dir_s *
paging_new_dir(void)
{
	struct page_dir_s *d;
	int i;

	d = (struct page_dir_s *)malloc(sizeof(struct page_dir_s));
	if (d == 0)
		return 0;

	memset((char *)d, 0, sizeof(struct page_dir_s));

	for (i = 0; i < 256; i++)
	  d->pages[i].fpn = -1;

	return d;
}

int
paging_init(void)
{
	/* create root directory */
	page_root = paging_new_dir();
	if (page_root == 0)
		return -1;

	return 0;
}

/*
 *
 * 28 bit virtual address mapping
 *
 *  x1 22 33 44
 *                                       
 *         top 4 bits (x1)  8 bits (22) 8 bits (33)  8 bits (44)
 * root -> root dir[0]
 *         root dir[1]   -> dir[0]
 *         ...              dir[1]   -> dir[0]
 *         root dir[255]    ...         dir[1]   ->  mem/tag [44]
 *                          dir[255]    ...          
 *                                      dir[255]     
 */

struct page_s *
paging_find_entry(u_long vma)
{
	int p1, p2, p3;
	struct page_dir_s *dir = page_root;
	struct page_s *p;

	p1 = vma >> 24;
	p = &page_root->pages[p1];

	if (!p->paged_in) {
		page_root->pages[p1].paged_in = 1;
		page_root->pages[p1].mem_addr = (u_long *)paging_new_dir();

		if (0) printf("new p1 %d\n", p1);
	}

	p2 = (vma >> 16) & 0xff;
	dir = (struct page_dir_s *)p->mem_addr;
	p = &dir->pages[p2];

	if (!p->paged_in) {
		dir->pages[p2].paged_in = 1;
		dir->pages[p2].mem_addr = (u_long *)paging_new_dir();

		if (0) printf("new p2 %d\n", p2);
	}

	p3 = (vma >> 8) & 0xff;
	dir = (struct page_dir_s *)p->mem_addr;
	p = &dir->pages[p3];

	return p;
}

int
paging_map_to_pma(u_long vma, u_long **pma, u_char **pta)
{
	int p3;
	struct page_s *p;

	p = paging_find_entry(vma);
	if (p == 0)
		return -1;

	if (!p->paged_in) {
		int offset;

		/* xxx use a heap */
		p->mem_addr = malloc(256*4);
		p->tag_addr = malloc(256);
		memset((char *)p->mem_addr, 0, 256*4);
		memset((char *)p->tag_addr, 0, 256);

		if (p->fpn >= 0) {
		  offset = p->fpn * 1152;  /* (256/2)*9 */

		  if (trace_vm) {
			  printf("paging_map_to_pma() new page; vma=0%o, pfn=0x%x\n",
				 vma, p->fpn);
		  }

		  read36_page(world_stream, offset, p->mem_addr, p->tag_addr);
		} else {
			if (trace_vm) {
				printf("paging_map_to_pma() new page; vma=0%o (zeros)\n",
				       vma);
			}

		  memset(p->mem_addr, 0, 256*4);
		  memset(p->tag_addr, 0, 256);
		}

		p->paged_in = 1;
	}

	*pma = p->mem_addr;
	*pta = p->tag_addr;

	return 0;
}

int
paging_add_vma_entry(u_long vma, int fpn)
{
	struct page_s *p;

	p = paging_find_entry(vma);
	if (p == 0)
		return -1;

	if (0) printf("vma 0x%x, page entry %p, fpn %x\n", vma, p, fpn);

	p->fpn = fpn;
	return 0;
}

int
fetch_isn(u_long spc, int *pisn)
{
	u_long pc, isn;
	u_long *pma;
	u_char *pta;
	int offset;

	pc = spc >> 1;
	if (paging_map_to_pma(a28(pc), &pma, &pta))
		return -1;

	offset = pc % 256;

	printf("fetch_isn() %08x tag %x ", pma[offset], pta[offset]);

	if (spc & 1) {
		isn = pma[offset] >> 16;
		if (pta[offset] & 0x80)
			isn |= 0x10000;
	} else {
		isn = pma[offset] & 0xffff;
		if (pta[offset] & 0x40)
			isn |= 0x10000;
	}

	*pisn = isn;

	return 0;
}

void
read_mem_36(u_long vma, w36 *v)
{
	u_long *pma;
	u_char *pta;
	int offset;

	if ((vma >> 12) == 0x5ffff) {

		read_amem_36(vma & 0x0fff, v);

		if (trace_mem) {
			char *sym;
			extern char *find_sym_by_val(int, int);

			sym = find_sym_by_val(1, vma & 0x0fff);
		
			printf("read_mem_36(vma=0%o) -> amem %02x:%08x %o",
			       vma, v->tag, v->word, v->word);
			if (sym) {
				printf(" %s", sym);
			}
			printf("\n");
		}

		return;
	}

	if (paging_map_to_pma(a28(vma), &pma, &pta))
		;

	offset = vma % 256;


	if (trace_mem) {
		char *sym = 0;

		if (a28(vma) < sizeof(fep_communication_area_names))
			sym = fep_communication_area_names[a28(vma)];

		printf("read_mem_36(vma=0%o) -> %02x:%08x %o",
		       vma, pta[offset], pma[offset], pma[offset]);
		if (sym) {
			printf(" %s", sym);
		}
		printf("\n");
	}

	v->word = pma[offset];
	v->tag = pta[offset];
}

void
write_mem_36(u_long vma, w36 *v)
{
	u_long *pma;
	u_char *pta;
	int offset;

	if ((vma >> 12) == 0x5ffff) {
		write_amem_36(vma & 0x0fff, v);

		if (trace_mem) {
			printf("write_mem_36(vma=0%o) <- amem %02x:%08x %o\n",
			       vma, v->tag, v->word, v->word);
		}

		return;
	}

	if (paging_map_to_pma(a28(vma), &pma, &pta))
		;

	offset = vma % 256;

	if (trace_mem) {
		printf("write_mem_36(vma=0%o) <- %02x:%08x\n",
		       vma, v->tag, v->word);
	}

	pma[offset] = v->word;
	pta[offset] = v->tag;
}


w36 top_of_stack;
int stack_pointer;
int stack_low;
int stack_limit;
int frame_pointer;
int a_pclsr_top_of_stack;

w36 quote_nil;
w36 quote_t;

extern u_long amem[];
extern u_char amem_tag[];

void
write_amem_36(int addr, w36 *v)
{
	amem[addr & 0x0fff] = v->word;
	amem_tag[addr & 0x0fff] = v->tag;
}

void
read_amem_36(int addr, w36 *v)
{
	v->word = amem[addr & 0x0fff];
	v->tag = amem_tag[addr & 0x0fff];
}

void
set_cdr(w36 *v, int cdr_code)
{
	v->tag = (v->tag & 0x3f) | (cdr_code << 6);
}

void
set_tag(w36 *v, int tag)
{
	v->tag = tag;
}

void
set_type(w36 *v, int tag)
{
	v->tag = tag;
	v->word = (v->word & 0x0fffffff) | ((v->tag & 0xf) << 28);
}

void
pushval(w36 *v)
{
	set_cdr(v, CDR_NEXT);
	stack_pointer++;
	top_of_stack = *v;
	printf("pushval: (%o) %011o %08x\n", stack_pointer, v->word, v->word);
	write_amem_36(stack_pointer, v);
}

void
pushval_with_cdr(w36 *v)
{
	stack_pointer++;
	top_of_stack = *v;
	printf("pushval: (%o) %11o %08x\n", stack_pointer, v->word, v->word);
	write_amem_36(stack_pointer, v);
}

void
popval(w36 *v)
{
	*v = top_of_stack;
	printf("popval: (%o) %x:%08x %x:%o\n",
	       stack_pointer,
	       top_of_stack.tag, top_of_stack.word,
	       top_of_stack.tag, top_of_stack.word);
	stack_pointer--;
	read_amem_36(stack_pointer, &top_of_stack);
}

void
pop2push(w36 *v)
{
	/* like doing two popval's and then pushval */
	set_cdr(v, CDR_NEXT);
	write_amem_36(stack_pointer - 1, v);
	top_of_stack = *v;
	stack_pointer--;
	printf("pop2push: (%o) %x:%08x\n", stack_pointer,
	       top_of_stack.tag, top_of_stack.word);
}

void
newtop(w36 *v)
{
	set_cdr(v, CDR_NEXT);
	write_amem_36(stack_pointer, v);
	top_of_stack = *v;
	printf("newtop: (%o) %x:%08x %x:%o\n", stack_pointer,
	       top_of_stack.tag, top_of_stack.word,
	       top_of_stack.tag, top_of_stack.word);
}

int vma;

void
popmem(void)
{
	/* read mem to get tag; popstack, write data but merge tag */
	w36 v, v1;
	read_mem_36(vma, &v);
	popval(&v1);
	v.word = v1.word;
	v.tag = (v.tag & 0x3f) | v1.tag & 0xc0;
	write_mem_36(vma, &v);
}

void
pushmem(void)
{
	/* read mem to get tag; popstack, write data but merge tag */
	w36 v;
	printf("pushmem: vma 0x%x\n", vma);
	read_mem_36(vma, &v);
	pushval(&v);
}

void
popmemind(void)
{
	w36 v;
	printf("popmemind: vma %x\n", vma);
	/* indirect version of popmem */
	read_mem_36(vma, &v);
	vma = v.word;
	popmem();
}

void
pushmemind(void)
{
	w36 v;
	/* indirect version of popmem */
	printf("pushmemind: vma 0x%x\n", vma);
	read_mem_36(vma, &v);
	vma = v.word;
	pushmem();
}

u_long
frame_function(void)
{
	printf("frame_function() fp-1 %o %08x %o\n",
	       frame_pointer-1,
	       amem[frame_pointer - 1], amem[frame_pointer - 1]);

	return amem[frame_pointer - 1];
}

void
pop_indirect(int operand)
{
	vma = frame_function() - operand - 1;
	popmemind();
}

#define FRAME_CLEANUP_BITS	(0xff00)
#define FRAME_LEXPR_CALLED	(1<<25)
#define FRAME_INSTANCE_CALLED	(1<<26)
#define FRAME_FUNCALLED		(1<<27)

u_long
frame_misc_data(void)
{
	return amem[frame_pointer - 2];
}

int
frame_funccalled(void)
{
  return frame_misc_data() & FRAME_FUNCALLED;
}

int
frame_lexpr_called(void)
{
  return frame_misc_data() & FRAME_LEXPR_CALLED;
}

void
dump_stack_frame(char *who)
{
  int i;
  w36 v;

  printf("%s: stack:\n", who);
  printf("stack-pointer %o, frame-pointer %o\n",
	 stack_pointer, frame_pointer);

  for (i = stack_low; i < stack_pointer + 10; i++) {
    read_amem_36(i, &v);
    printf("%o: %02x:%11o (0x%x)\n", i, v.tag, v.word, v.word);
  }
}

int
sign_extend_7(int operand)
{
	/* assumes 32 bit machine */
	if (operand & 0x80)
		return operand | 0xffffff80;
	return operand & 0x7f;
}

int
unsigned_7(int operand)
{
	return operand & 0x7f;
}

char *arg_dispatch_names[] = {
	"slow case",
	"no arguments",
	"no required, 1 optional",
	"1 required",
	"no required, 2 optional",
	"1 required, 1 optional",
	"2 required",
	"no required, 3 optional",
	"1 required, 2 optional",
	"2 required, 1 optional",
	"3 required",
	"no required, 4 optional",
	"1 required, 3 optional",
	"2 required, 2 optional",
	"3 required, 1 optional",
	"4 required"
};

void
show_pname(int loc)
{
	w36 v;
	int i, j, len;
	char str[256];

	read_mem_36(loc, &v);
	len = v.word & 0x0fff;

	j = 0;
	for (i = 0; i < (len/4)+1; i++) {
		read_mem_36(++loc, &v);
		str[j++] = v.word;
		str[j++] = v.word >> 8;
		str[j++] = v.word >> 16;
		str[j++] = v.word >> 24;
	}
	str[len] = 0;

	printf("%s", str);
}

void
_show_taged(char *str, w36 *v, int follow)
{
	w36 p;
	int ptr, cdr_code, i;

	printf("%s %s %x:%x ", str, tag_names[v->tag & 0x3f], v->tag, a28(v->word));
	ptr = v->word;

	cdr_code = v->tag >> 6;

	switch (v->tag & 0x3f) {
	case DTP_SYMBOL:
//		if (!follow) {
//			printf("\n");
//			return;
//		}

		printf("\n");
		ptr = v->word;
		for (i = 0; i < 5; i++) {
			char str[32];
			w36 s;
			read_mem_36(ptr+i, &s);
			sprintf(str, " [%d]:", i);
			_show_taged(str, &s, 0);

			if (!follow) break;
		}
		break;

	case DTP_ARRAY:
		printf("\n");
		ptr = v->word;
		for (i = 0; i < 10; i++) {
			char str[32];
			w36 s;
			read_mem_36(ptr+i, &s);
			sprintf(str, "   [%d]:", i);
			_show_taged(str, &s, 0);
#if 0
			if ((s.tag & 0x3f) == DTP_LOCATIVE) {
				w36 l;
				int j;
				ptr = s.word;

				for (j = 0; j < 32; j++) {
					read_mem_36(ptr+j, &l);
					sprintf(str, " [%d]->:", j);
					_show_taged(str, &l, 0);
				}
			}
#endif
		}
		break;

	case DTP_HEADER_I:
		switch (cdr_code) {
		case 0: printf("%%header-type-compiled-function "); break;
		case 1: printf("%%header-type-array "); break;
		case 2: printf("%%header-type-number "); break;
		default: printf("cdr_code %d\n", cdr_code);
		}
//		show_pname(ptr);
		break;
	case DTP_HEADER_P:
		switch (cdr_code) {
		case 0: printf("%%header-type-symbol "); break;
		case 1: printf("%%header-type-instance "); break;
		}

		show_pname(v->word);
		break;
	}

	printf("\n");
}

void
show_taged(char *str, w36 *v)
{
	_show_taged(str, v, 1);
}

void
show_array(int loc)
{
	w36 v, hp, hi;
	int ptr, i;

	printf("array: %x\n", loc);
	read_mem_36(loc+0, &hp);
	show_taged("+0:", &hp);

	read_mem_36(loc+1, &hi);
	show_taged("+1:", &hi);

	read_mem_36(loc+2, &hi);
	show_taged("+2:", &hi);

	read_mem_36(loc+3, &hi);
	show_taged("+3:", &hi);

	read_mem_36(loc+4, &hi);
	show_taged("+4:", &hi);
}

/*
 *  interpret the header before a DTP-COMPILED-FUNCTION
 */
void
show_compiled_function(int loc)
{
	w36 v;
	int ptr, len, i, j;
	char name[256];
	int table_size, arg_dispatch;
	int compiled_function_table_size,
		compiled_function_total_size,
		compiled_function_args_info,
		cca_current_definition_p,
		compiled_function_language_index;

	read_mem_36(loc, &v);
	table_size = v.word & 0xff;
	arg_dispatch = (v.word >> 8) & 0xf;

	/* COMPILED-FUNCTION-HEADER-WORD-2 */
	read_mem_36(loc - table_size - 4, &v);
	if (v.tag != DTP_HEADER_I)
		return;

	compiled_function_table_size = v.word & 0xff; /* # of ext refs and constants */
	compiled_function_total_size = (v.word >> 8) & 0xffff; /* # of words total */

	/* DTP-HEADER-I, %HEADER-TYPE-COMPILED-FUNCTION */
	read_mem_36(loc - table_size - 3, &v);
	if (!dtp_fix_p(v.tag))
		return;

	compiled_function_args_info = v.word & 0x3ffff; /* %ARGS-INFO data */
	cca_current_definition_p = v.word & 0x40000;
	compiled_function_language_index = (v.word >> 28) & 0xf;

	/* COMPILED-FUNCTION-EXTRA-INFO */
	read_mem_36(loc - table_size - 2, &v);
	ptr = v.word;
	printf("ptr %x:%x\n", v.tag, v.word);
	if (v.tag != DTP_LIST)
		return;

	/* extra-info is a cons of the function name and debug-info a-list */

	read_mem_36(ptr, &v);
	ptr = v.word;
	printf("ptr2 %x:%x\n", v.tag, v.word);

	read_mem_36(ptr, &v);
	ptr = v.word;
	printf("ptr3 %x:%x\n", v.tag, v.word);

	/* pname string */

	read_mem_36(ptr, &v);
	printf("string %x:%x\n", v.tag, v.word);
	len = v.word & 0x0fff;

	if (v.tag == DTP_HEADER_P) {
		ptr = v.word;
		read_mem_36(ptr, &v);
		printf("string %x:%x\n", v.tag, v.word);
		len = v.word & 0x0fff;
	}

	j = 0;
	for (i = 0; i < (len/4)+1; i++) {
		read_mem_36(++ptr, &v);
		name[j++] = v.word;
		name[j++] = v.word >> 8;
		name[j++] = v.word >> 16;
		name[j++] = v.word >> 24;
	}
	name[len] = 0;

	printf("function: %s\n", name);
	printf("table-size %d, arg-dispatch %d %s\n",
	       table_size, arg_dispatch,
	       arg_dispatch_names[arg_dispatch]);

	{
		int max_args, min_args, rest_arg, quoted, interpreted;

		max_args = compiled_function_args_info & 0xff;
		min_args = (compiled_function_args_info >> 8) & 0xff;
		rest_arg = compiled_function_args_info & 0x1000 ? 1 : 0;
		quoted = compiled_function_args_info & 0x2000 ? 1 : 0;
		interpreted = compiled_function_args_info & 0x4000 ? 1 : 0;

		printf("max-args %d, min-args %d, rest-arg %s, quoted %s, interpreted %s\n", 
		       max_args, min_args,
		       rest_arg ? "t" : "nil",
		       quoted ? "t" : "nil",
		       interpreted ? "t" : "nil");
	}
}

void
find_func_name(int loc)
{
	w36 v;
	int args, ptr, len, i, j;
	char name[256];

	read_mem_36(loc, &v);
	args = v.word & 0xff;

	read_mem_36(loc - args - 2, &v);
	ptr = v.word;
//	printf("ptr %x\n", ptr);

	read_mem_36(ptr, &v);
	ptr = v.word;
//	printf("ptr2 %x\n", v.word);

	read_mem_36(ptr, &v);
	ptr = v.word;
//	printf("ptr3 %x\n", v.word);

	read_mem_36(ptr, &v);
//	printf("string %x\n", v.word);

	len = v.word & 0x0fff;
	j = 0;
	for (i = 0; i < (len/4)+1; i++) {
		read_mem_36(++ptr, &v);
		name[j++] = v.word;
		name[j++] = v.word >> 8;
		name[j++] = v.word >> 16;
		name[j++] = v.word >> 24;
	}
	name[len] = 0;
	printf("function: %s\n", name);
}

void
local_operand(int operand, w36 *v)
{
	int addr;

	if (operand & 0x80) {
		addr = stack_pointer - sign_extend_7(operand);
		printf("local-operand: sp - %d\n", sign_extend_7(operand));
	} else {
		addr = frame_pointer + unsigned_7(operand);
		printf("local-operand: fp + %d\n", unsigned_7(operand));
	}
	read_amem_36(addr, v);
	printf("local-operand: addr %o -> %x:%o\n", addr, v->tag, v->word);
}

void
write_local_operand(int operand, w36 *v)
{
	int addr;

	if (operand & 0x80) {
		addr = stack_pointer - sign_extend_7(operand);
		printf("local-operand: sp - %d\n", sign_extend_7(operand));
	} else {
		addr = frame_pointer + unsigned_7(operand);
		printf("local-operand: fp + %d\n", unsigned_7(operand));
	}
	write_amem_36(addr, v);
	printf("local-operand: addr %o <- %x:%o\n", addr, v->tag, v->word);
}

int
equal_typed_pointer(w36 *p1, w36 *p2)
{
	if (p1->word == p2->word &&
	    (p1->tag & 0x3f) == (p2->tag & 0x3f))
		return 1;
	return 0;
}

int
equal_fixnum(w36 *f1, w36 *f2)
{
	printf("equal_fixnum: %x == %x?\n", f1->word, f2->word);
	if (f1->word == f2->word)
		return 1;
	return 0;
}

int
zero_fixnum(w36 *f)
{
	if (f->word == 0)
		return 1;
	return 0;
}

int
plus_fixnum(w36 *f)
{
	if ((f->word & 0x80000000) == 0)
		return 1;
	return 0;
}

int
minus_fixnum(w36 *f)
{
	if (f->word & 0x80000000)
		return 1;
	return 0;
}

void
pc_add(int offset)
{
#if 0
	int hw;
	hw = (sim_pc & 1) ^ (offset & 1) ^ ((offset < 0) ? 1 : 0);
	sim_pc += offset;

	if (hw & 1)
		sim_pc |= 1;
	else
		sim_pc &= ~1;
#else
	sim_pc += offset >= 0 ? offset : (offset + 1);
#endif
}

int
exec_isn(u_long spc, int inst)
{
	int uop, operand, cdr_code, addr, nargs, vdisp;
	int next_instruction;
	int alen, atype, adisp;
	int pos, size, i;
	unsigned long mask;
	w36 v, v1;

	printf("exec_isn(pc=%o, isn=%o 0x%x)\n", spc>>1, inst, inst);

	disassemble36(inst);

	if ((inst & 0xff) == 0xff)
		uop = 01000 + ((inst >> 8) & 0x1ff);
//		uop = 01000 + ((inst >> 8) & 0x0ff);
	else
//		uop = (inst & 0xff) | ((inst & 0x10000) ? 0x100 : 0);
		uop = (inst & 0xff);

	operand = (inst >> 8) & 0xff;
	//printf("uop %o\n", uop);

	next_instruction = 1;

	switch (uop) {
	case 3:		/* %make-pointer-immed-offset */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		v.word += top_of_stack.word;
		set_type(&v, DTP_FIX);
		pop2push(&v);

		set_tag(&top_of_stack,
			(top_of_stack.tag & 0x0f) | (operand & 0x0f));

		newtop(&top_of_stack);
		break;
	case 032:	/* %io-read */
		break;
	case 0100:	/* push-indirect */
		vma = frame_function() - operand - 1;
		pushmemind();
		break;
	case 0101:	/* push-constant */
		vma = frame_function() - operand - 1;
		pushmem();
		break;
	case 0102:	/* push-local */
		local_operand(operand, &v);
#if 0
		if (operand & 0x80) {
			addr = stack_pointer - sign_extend_7(operand);
			printf("push-local: sp - %d\n", sign_extend_7(operand));
		} else {
			addr = frame_pointer + unsigned_7(operand);
			printf("push-local: fp + %d\n", unsigned_7(operand));
		}
		printf("push-local: addr %o\n", addr);
		read_amem_36(addr, &v);
#endif
		pushval(&v);
		break;
	case 0103:	/* push-immed */
		v.tag = DTP_FIX;
		v.word = sign_extend_7(operand);
		pushval(&v);
		break;
	case 0104:	/* push-address-local */
		if (operand & 0x80)
			v.word = stack_pointer - sign_extend_7(operand) + 1;
		else
			v.word = frame_pointer + sign_extend_7(operand);
		set_type(&v, DTP_LOCATIVE);
		pushval(&v);
		break;
	case 0105:	/* push-from-beyond-multiple */
		/* stack-pointer - a-memory[stack-pointer] + operand + 1 */
		read_amem_36(stack_pointer, &v);
		addr = stack_pointer - v.word + operand + 1;

		read_amem_36(addr, &v);
		set_type(&v, DTP_FIX);
		pushval(&v);
		break;
	case 0106:	/* movem-local */
		write_local_operand(operand, &top_of_stack);
		break;
	case 0107:	/* movem-indirect */
		pushval(&top_of_stack);
		vma = frame_function() - operand - 1;
		printf("movem-indirect: vma 0x%x 0%o\n", vma, vma);
		popmemind();
		break;
	case 0110:	/* pop-local */
		popval(&v);
		write_local_operand(operand, &v);
		break;
	case 0111:	/* pop-indirect */
		vma = frame_function() - operand - 1;
		printf("pop-indirect: vma 0x%x 0%o\n", vma, vma);
		popmemind();
		break;
	case 0120:	/* push-n-nils */
		for (i = 0; i < operand; i++) {
			pushval(&quote_nil);
		}
		break;
	case 0121:	/* pop-n */
		stack_pointer -= unsigned_7(operand);
		read_amem_36(stack_pointer, &top_of_stack);
		break;
	case 0122:	/* pop-n-save-1 */
		stack_pointer -= unsigned_7(operand);
		write_amem_36(stack_pointer, &top_of_stack);
		break;
	case 0160:	/* branch */
		pc_add(sign_extend_7(operand));
		next_instruction = 0;
		break;
	case 0161:	/* branch-true */
		read_amem_36(stack_pointer, &v);
		printf("branch-true: %x:%o\n", v.tag, v.word);
		if (v.tag != DTP_NIL) {
			printf("branch-true: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;
	case 0162:	/* branch-false */
		read_amem_36(stack_pointer, &v);
		printf("branch-false: %x:%o\n", v.tag, v.word);
		if (v.tag == DTP_NIL) {
			printf("branch-false: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;
	case 0163:	/* branch-true-else-pop */
		read_amem_36(stack_pointer, &v);
		if (v.tag != DTP_NIL) {
			printf("branch-true-else-pop: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		} else {
			popval(&v);
		}
		break;

	case 0176:	/* branch-eq */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		if (equal_typed_pointer(&v, &top_of_stack)) {
			printf("branch-eq: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;
	case 0177:	/* branch-not-eq */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		if (!equal_typed_pointer(&v, &top_of_stack)) {
			printf("branch-not-eq: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;
	case 0244:	/* multiply-immed */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			v.word *= operand;
			break;
		}
		newtop(&v);
		break;

	case 0251:	/* decrement-local */
		local_operand(operand, &v);
#if 0
		if (top_of_stack.tag != DTP_FIX ||
		    v.tag != DTP_FIX) {
			take_arithmetic_trap('decrement 'local);
		}
#endif
		v.word--;
		v.tag = DTP_FIX;
		write_local_operand(operand, &v);
//		newtop(&v);
		break;

	case 0260:	/* ldb-immed */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		/* form 10-bit-operand */
		operand |= (inst & 0x3) << 8;

		pos = operand >> 6;
		size = operand & 0x3f;

		printf("ldb-immed: operand 0%o, pos %d, size %d\n",
		       operand, pos, size);

		printf("ldb-immed: top-of-stack-a %x:%x\n", v.tag, v.word);

		if (pos + size <= 32) {
			mask = 0xffffffff >> (32 - size);
			v.word = (v.word >> pos) & mask;
			printf("ldb-immed: mask %x, result %x\n", mask, v.word);
		} else {
			unsigned long long val, lmask;
			lmask = 0x00fffffffffLL >> (36 - size);
			val = ((unsigned long long)v.tag << 28) | v.word;
			val = (val >> pos) & lmask;
			v.word = (unsigned long)val;
			printf("ldb-immed: ll mask %llx result %llx\n", lmask, val);
		}

		set_type(&v, DTP_FIX);
		newtop(&v);
		break;

	case 0300:	/* call-0-ignore */
		vdisp = 0;
		nargs = 0;
	call_common:
		/* prev previous-frame base pointer */
//		v.word = frame_function();
		v.word = frame_pointer;
		v.tag = DTP_LOCATIVE;
		pushval_with_cdr(&v);

		/* push previous-frame top pointer */
		/* cdr code is value disposition */
		v.word = stack_pointer - nargs - 1;
		v.tag = DTP_LOCATIVE;
		pushval_with_cdr(&v);
	  
		/* return pc */
		v.word = (sim_pc + 1) >> 1;
		v.tag = (sim_pc + 1) & 1 ? DTP_ODD_PC : DTP_EVEN_PC;
		pushval_with_cdr(&v);

		/* misc data */
		v.word = frame_funccalled();
		v.tag = 0;
		pushval_with_cdr(&v);

		/* function */
		vma = frame_function() - operand - 1;
		read_mem_36(vma, &v);
		vma = v.word;
		read_mem_36(vma, &v);
		sim_pc = (v.word << 1) | 1;

		set_cdr(&v, 0);
		pushval_with_cdr(&v);

		/* */
		frame_pointer = stack_pointer + 1;
		a_pclsr_top_of_stack = stack_pointer + 1;

		next_instruction = 0;

		if (1) dump_stack_frame("call-0-ignore");

//		find_func_name(sim_pc >> 1);
		show_compiled_function(sim_pc >> 1);

		break;
	case 0301:	/* call-0-stack */
		vdisp = 1;
		nargs = 2;
		goto call_common;
		break;
	case 0303:	/* call-0-multiple */
		vdisp = 3;
		nargs = 0;
		goto call_common;
		break;

	case 0310:	/* call-2-ignore */
		vdisp = 0;
		nargs = 2;
		goto call_common;
		break;

	case 0336:	/* lexpr-funcall-return */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		a_pclsr_top_of_stack = v.word;

		top_of_stack.tag = 0;
		top_of_stack.word = operand + 1;

		v.word = stack_pointer - top_of_stack.word - 1;
		printf("lexpr-funcall-return: %x\n", v.word);

		set_cdr(&v, 2); /* return */
		set_type(&v, DTP_LOCATIVE);
		write_amem_36(stack_pointer + 2, &v);

		stack_pointer++;

		/* check if rest arg is nil */
		read_amem_36(stack_pointer -1, &v);
		if (v.tag == DTP_NIL) {
			printf("lexpr-funcall-return: rest arg nil\n");
			read_amem_36(stack_pointer + 1, &v);
			write_amem_36(stack_pointer, &v);
			top_of_stack.word--;
			stack_pointer--;
			/* funcal-stack-n */
		}

		/* push previous-frame base pointer */
		v.word = frame_pointer;
		v.tag = DTP_LOCATIVE;
		pushval_with_cdr(&v);

		/* return pc */
		v.word = (sim_pc + 1) >> 1;
		v.tag = (sim_pc + 1) & 1 ? DTP_ODD_PC : DTP_EVEN_PC;
		pushval_with_cdr(&v);

		/* misc data */
		v.word = frame_funccalled() + frame_lexpr_called() + top_of_stack.word;
		v.tag = DTP_FIX;
		pushval_with_cdr(&v);

		/* function */
		vma = frame_function() - operand - 1;
		read_mem_36(vma, &v);
		vma = v.word;
		read_mem_36(vma, &v);
		sim_pc = (v.word << 1) | 1;

		set_cdr(&v, 0);
		pushval_with_cdr(&v);


		/* */
		frame_pointer = stack_pointer + 1;
		a_pclsr_top_of_stack = stack_pointer + 1;

		next_instruction = 0;

		if (1) dump_stack_frame("lexpr-funcall-return");

		show_compiled_function(sim_pc >> 1);

		break;

	case 0375:	/* %dispatch-elt */
		if (1) dump_stack_frame("%dispatch-elt");
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		v.word += top_of_stack.word;
		read_mem_36(v.word, &v);
		newtop(&v);
		break;

	case 01000:	/* %halt */
		sim_running = 0;
		break;

	case 01014:	/* %microsecond-clock */
		v.word = 0;
		v.tag = DTP_FIX;
		pushval(&v);
		break;
	case 01120:	/* push-nil */
		pushval(&quote_nil);
		break;

	case 01203:	/* zerop */
		printf("zerop: tag %x\n", top_of_stack.tag);
		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			if (zero_fixnum(&top_of_stack))
				newtop(&quote_t);
			else
				newtop(&quote_nil);
			break;
		}
		break;

	case 01204:	/* plusp */
		printf("plusp: tag %x\n", top_of_stack.tag);
		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			if (plus_fixnum(&top_of_stack))
				newtop(&quote_t);
			else
				newtop(&quote_nil);
			break;
		}
		break;

	case 01205:	/* minusp */
		printf("minusp: tag %x\n", top_of_stack.tag);
		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			if (minus_fixnum(&top_of_stack))
				newtop(&quote_t);
			else
				newtop(&quote_nil);
			break;
		}
		break;
	case 01210:	/* equal-number */
		if (1) dump_stack_frame("equal-number");
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		if (equal_fixnum(&v, &top_of_stack))
			newtop(&quote_t);
		else
			newtop(&quote_nil);
		break;

	case 01230:	/* push-2-nils */
		pushval(&quote_nil);
		pushval(&quote_nil);
		break;

	case 01240:	/* add-stack */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		v.word += top_of_stack.word;
		set_type(&v, DTP_FIX);
		pop2push(&v);
		break;

	case 01241:	/* sub-stack */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		v.word -= top_of_stack.word;
		set_type(&v, DTP_FIX);
		pop2push(&v);
		break;

	case 01265:	/* rational-quotient-stack */
		/* next-on-stack - dividend */
		read_amem_36(stack_pointer-1, &v);
		printf("rational-quotient-stack: dividend %d, divisor %d\n",
		       v.word, top_of_stack.word);
		if (top_of_stack.word != 0)
			v.word /= top_of_stack.word;
		else
			v.word = 0;
		stack_pointer--;
		newtop(&v);
		break;

	case 01270:	/* ar-1 */
#define ARRAY_NORMAL_LENGTH_FIELD(w)	((w) & 0x0003ffff)
#define ARRAY_DISPATCH_FIELD(w)		(((w) >> 22) & 0xf)
#define ARRAY_TYPE_FIELD(w)		(((w) >> 18) & 0xf)

#define ARRAY_LONG_PREFIX_LENGTH_FIELD(w)	((w) & 0x1f)
#define ARRAY_LONG_LEADER_LENGTH_FIELD(w)	(((w) >> 5) & 0x3ff)
#define ARRAY_DIMENSIONS_FIELD(w)		(((w) >> 15) & 0x3)

		if (1) dump_stack_frame("ar-1");

		read_amem_36(stack_pointer - 1, &v);

//		printf("tag %x, word %x\n", v.tag, v.word);
		show_taged("ptr from stack", &v);
		show_array(v.word);

		read_mem_36(v.word+4, &v);
//		printf("ar-1: array %x:%x %o\n", v.tag, v.word, v.word);
		show_taged("array", &v);
		if (v.tag != DTP_ARRAY) {
		}

		addr = v.word;

		read_mem_36(addr, &v);
//		printf("ar-1: array %x:%x %o\n", v.tag, v.word, v.word);
		show_taged("hdr", &v);

		read_amem_36(stack_pointer, &v1);
		read_mem_36(v1.word, &v1);
		read_mem_36(v1.word, &v1);
		printf("ar-1: subscript %x:%o\n", v1.tag, v1.word);
		if (v1.tag != DTP_FIX) {
		}

		alen = ARRAY_NORMAL_LENGTH_FIELD(v.word);
		adisp = ARRAY_DISPATCH_FIELD(v.word);
		atype = ARRAY_TYPE_FIELD(v.word);

		printf("ARRAY_NORMAL_LENGTH_FIELD %d 0x%x, ARRAY_DISPATCH_FIELD %d, ARRAY_TYPE_FIELD %d\n",
		       alen, alen, adisp, atype);
		if (atype >= 0 && atype <= 16)
			printf("%s %s\n", array_dispatch_names[adisp], array_type_names[atype]);

		switch (adisp) {
		case ARRAY_DISPATCH_LONG:
			printf("offset to leader %d, # elems in leader %d, # dims %d\n",
			       ARRAY_LONG_PREFIX_LENGTH_FIELD(v.word),
			       ARRAY_LONG_LEADER_LENGTH_FIELD(v.word),
			       ARRAY_DIMENSIONS_FIELD(v.word));

			/* read locative */
			read_mem_36(addr+1, &v);
			if (v.tag != DTP_LOCATIVE) {
			}
			show_taged("locative", &v);

			addr = v.word;
			read_mem_36(addr+0, &v);
			show_taged("+0", &v);
			read_mem_36(addr+1, &v);
			show_taged("+1", &v);

			v.tag = 0;
			v.word = 0x8;

			pop2push(&v);
			break;
		default:
			break;
		}

		break;

	case 01300:	/* funcall-0-ignore */
		vdisp = 0;
		nargs = 0;
//		goto call_common;

//		popval(&v);
		v = top_of_stack;
		addr = v.word;
		printf("funcall-0-ignore: 0x%x %o\n", addr, addr);
		
		/* prev previous-frame base pointer */
//		v.word = frame_function();
		v.word = frame_pointer;
		v.tag = DTP_LOCATIVE;
		pushval_with_cdr(&v);

		/* push previous-frame top pointer */
		/* cdr code is value disposition */
		v.word = stack_pointer - nargs - 1;
		v.tag = DTP_LOCATIVE;
		pushval_with_cdr(&v);
	  
		/* return pc */
		v.word = (sim_pc + 1) >> 1;
		v.tag = (sim_pc + 1) & 1 ? DTP_ODD_PC : DTP_EVEN_PC;
		pushval_with_cdr(&v);

		/* misc data */
		v.word = frame_funccalled();
		v.tag = 0;
		pushval_with_cdr(&v);

		/* function */
//		vma = frame_function() - operand - 1;
		v.word = addr;
		v.tag = DTP_ODD_PC;
		printf("funcall-0-ignore: function from stack %x\n", vma);
//		read_mem_36(vma, &v);
//		vma = v.word;
//		read_mem_36(vma, &v);
		sim_pc = (v.word << 1) | 1;

		set_cdr(&v, 0);
		pushval_with_cdr(&v);

		/* */
		frame_pointer = stack_pointer + 1;
		a_pclsr_top_of_stack = stack_pointer + 1;

		next_instruction = 0;

		if (1) dump_stack_frame("funcall-0-ignore");

		show_compiled_function(sim_pc >> 1);

		break;

	case 01370:	/* return-stack */
	return_stack:
		if (1) dump_stack_frame("return-pc");

		read_amem_36(frame_pointer - 3, &v);

	  	sim_pc = v.word << 1;
		if (v.tag == DTP_ODD_PC)
			sim_pc++;

		printf("return-stack: return pc %x:%o\n", v.tag, v.word);

		read_amem_36(frame_pointer - 4, &v);
		stack_pointer = v.word;

		cdr_code = v.tag >> 6;

		read_amem_36(frame_pointer - 5, &v);
		frame_pointer = v.word;

		printf("return-stack: cdr_code %d\n", cdr_code);

		switch (cdr_code) {
		case CDR_NEXT:    /* effect */
		  read_amem_36(stack_pointer, &top_of_stack);
		  break;
		case CDR_NIL:    /* value */
		  pushval(&top_of_stack);
		  break;
		case CDR_NORMAL: /* return */
		  pushval(&top_of_stack);
		  goto return_stack;
		  break;
		case CDR_SPARE:  /* multiple-values */
		  pushval(&top_of_stack);
		  v.word = 1;
		  v.tag = DTP_FIX;
		  pushval(&v);
		  break;
		}

		next_instruction = 0;
		break;

	case 01372:	/* popj */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		printf("popj: %x:%o\n", v.tag, v.word);

		if ((v.tag != DTP_EVEN_PC &&
		     v.tag != DTP_ODD_PC) ||
		    (top_of_stack.tag != DTP_EVEN_PC &&
		     top_of_stack.tag != DTP_ODD_PC))
		{
		}

	  	sim_pc = v.word << 1;
		if (v.tag == DTP_ODD_PC)
			sim_pc++;

		popval(&v);
		next_instruction = 0;
		break;

	case 01374:	/* return-nil */
		goto return_stack;
		break;

	case 01375:	/* location-boundp */
		read_amem_36(stack_pointer, &v);
		printf("location-boundp: [0x%x] -> %x:%o\n",
		       stack_pointer, v.tag, v.word);
		addr = v.word;
		read_amem_36(addr, &v);
		printf("location-boundp: %o -> %x:%o\n",
		       addr, v.tag, v.word);

		if (v.tag == DTP_NULL) {
			newtop(&quote_nil);
		} else {
			newtop(&quote_t);
		}
		break;

	case 01440:	/* %32-bit-plus */
		/* a guess */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		v.word += top_of_stack.word;
		newtop(&v);
		break;

	case 01441:	/* %32-bit-difference */
		/* a guess */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		v.word -= top_of_stack.word;
		newtop(&v);
		break;

	default:
		printf("UNIMPLEMENTED INSTRUCTION!\n");
		break;
	}

	if (next_instruction) {
		sim_pc++;
	}

	return 0;
}

int
start_sim(void *stream)
{
	int isn;
	w36 v;

	printf("l-machine macroinstruction simulator\n\n");

	quote_nil.tag = DTP_NIL;
	quote_nil.word = 0;

	quote_t.tag = DTP_SYMBOL;
	quote_t.word = 0525252;

	sim_pc = get_amem_sym_value("FUNCTION-SYSTEM-STARTUP") << 1;
	sim_pc |= 1;

	/* should be running on aux stack */
//	stack_pointer = 04000;
	stack_pointer = 07000;
	stack_low = stack_pointer;
	stack_limit = 0;

	v.word = 0x0ffff800;
	v.tag = 0;
	write_amem_36(0x402, &v);


//	v.word = sim_pc;
//	pushval(&v);

/* */
//	set_type(&v, DTP_LOCATIVE);
	v.word = 0;
	pushval(&v); /* prev frame */

	v.word = stack_pointer - 2;
	pushval(&v); /* prev top */

	v.word = 0;
	pushval(&v); /* return pc */

	v.word = 0;
	pushval(&v); /* frame-misc-data */

	v.word = sim_pc >> 1;
	pushval(&v); /* dtp-compiled-function */
	
	frame_pointer = stack_pointer + 1;
/* */

	sim_running = 1;
	world_stream = stream;

//	find_func_name(sim_pc >> 1);
	show_compiled_function(sim_pc >> 1);

	while (sim_running) {
		sim_cycles++;
		printf("\ntop; pc %o, sp %o, fp %o\n",
		       sim_pc >> 1, stack_pointer, frame_pointer);

		fetch_isn(sim_pc, &isn);
		exec_isn(sim_pc, isn);

		if (sim_cycles > 1000)
			break;
	}

	printf("simulation halted\n");
}
