/*
 * lsim.c
 * l-machine macroinstruction simulator
 *
 * this version is not designed to be fast...  the goal is to make
 * it work first!
 *
 * Copyright (C) 2004 Brad Parker <brad@heeltoe.com>
 * $Id: lsim.c,v 1.5 2004/07/07 09:52:20 brad Exp $
 */

#include <stdio.h>
#include <stdlib.h>

#include "lsim.h"

int sim_running;
u_long sim_pc;
u_long sim_cycles;

int trace_mem;
int trace_vm;

int trace_top;
int trace_stack_small;
int trace_isn_disasm;
int trace_push_pop;
int trace_args;
int trace_comp;
int trace_bits;
int trace_flow;
int trace_array;
int trace_call;
int trace_call_stack;
int trace_enter_exit;
int trace_hw;

int trace_func_header;

u_long perc_current_phtc;

extern int flag_enter_debugger;
extern int flag_max_cycles;
extern int flag_trace;

/* tags */
enum {
  DTP_NULL = 0,
  DTP_NIL = 1,
  DTP_SYMBOL = 2,
  DTP_EXTENDED_NUMBER = 3,
  DTP_LIST = 4,
  DTP_LOCATIVE = 5,
  DTP_COMPILED_FUNCTION = 6,
  DTP_ARRAY = 7,
  DTP_CLOSURE,
  DTP_CHARACTER,
  DTP_INSTANCE,
  DTP_LEXICAL_CLOSURE,
  DTP_LOGIC_VARIABLE,
  DTP_GENERIC_FUNCTION,
  DTP_16,
  DTP_17,

  DTP_FIX = 16,
  DTP_FLOAT = 32,

  DTP_EVEN_PC = 48,
  DTP_EXTERNAL_VALUE_CELL_POINTER,
  DTP_ONE_Q_FORWARD,
  DTP_HEADER_FORWARD,
  DTP_64,
  DTP_GC_FORWARD,
  DTP_HEADER_P,
  DTP_HEADER_I,
  DTP_ODD_PC,

  DTP_MONITOR_FORWARD,
  DTP_ELEMENT_FORWARD,

  DTP_73,
  DTP_74,
  DTP_75,
  DTP_76,
  DTP_77
};

#define dtp_fix_p(tag)	((tag & 0x30) == DTP_FIX)
#define tag(tag)	(tag & 0x3f)

char *tag_names[] = {
	"DTP-NULL",
	"DTP-NIL",
	"DTP-SYMBOL",
	"DTP-EXTENDED-NUMBER",
	"DTP-LIST",
	"DTP-LOCATIVE",
	"DTP-COMPILED-FUNCTION",
	"DTP-ARRAY",
	"DTP-CLOSURE",
	"DTP-CHARACTER",
	"DTP-INSTANCE",
	"DTP-LEXICAL-CLOSURE",
	"DTP-LOGIC-VARIABLE",
	"DTP-GENERIC-FUNCTION",
	"DTP-16",
	"DTP-17",

	"DTP-FIX", "DTP-FIX", "DTP-FIX", "DTP-FIX",
	"DTP-FIX", "DTP-FIX", "DTP-FIX", "DTP-FIX",
	"DTP-FIX", "DTP-FIX", "DTP-FIX", "DTP-FIX",
	"DTP-FIX", "DTP-FIX", "DTP-FIX", "DTP-FIX",

	"DTP-FLOAT", "DTP-FLOAT", "DTP-FLOAT", "DTP-FLOAT",
	"DTP-FLOAT", "DTP-FLOAT", "DTP-FLOAT", "DTP-FLOAT",
	"DTP-FLOAT", "DTP-FLOAT", "DTP-FLOAT", "DTP-FLOAT",
	"DTP-FLOAT", "DTP-FLOAT", "DTP-FLOAT", "DTP-FLOAT",

	"DTP-EVEN-PC",
	"DTP-EXTERNAL-VALUE-CELL-POINTER",
	"DTP-ONE-Q-FORWARD",
	"DTP-HEADER-FORWARD",
	"DTP-64",
	"DTP-GC-FORWARD",
	"DTP-HEADER-P",
	"DTP-HEADER-I",
	"DTP-ODD-PC",

	"DTP-MONITOR-FORWARD",
	"DTP-ELEMENT-FORWARD",

	"DTP-73",
	"DTP-74",
	"DTP-75",
	"DTP-76",
	"DTP-77"
};

char *array_dispatch_names[] = {
	"%ARRAY-DISPATCH-1-BIT",
	"%ARRAY-DISPATCH-2-BIT",
	"%ARRAY-DISPATCH-4-BIT",
	"%ARRAY-DISPATCH-8-BIT",
	"%ARRAY-DISPATCH-16-BIT",
	"%ARRAY-DISPATCH-WORD",
	"%ARRAY-DISPATCH-SHORT-INDIRECT",
	"%ARRAY-DISPATCH-FIXNUM",
	"%ARRAY-DISPATCH-BOOLEAN",
	"%ARRAY-DISPATCH-LEADER",
	"%ARRAY-DISPATCH-SHORT-2D",
	"%ARRAY-DISPATCH-CHARACTER",
	"%ARRAY-DISPATCH-14",
	"%ARRAY-DISPATCH-LONG",
	"%ARRAY-DISPATCH-LONG-MULTIDIMENSIONAL",
	"%ARRAY-DISPATCH-FAT-CHARACTER"
};

enum {
	ARRAY_DISPATCH_1_BIT = 0,
	ARRAY_DISPATCH_2_BIT,
	ARRAY_DISPATCH_4_BIT,
	ARRAY_DISPATCH_8_BIT,
	ARRAY_DISPATCH_16_BIT,
	ARRAY_DISPATCH_WORD,
	ARRAY_DISPATCH_SHORT_INDIRECT,
	ARRAY_DISPATCH_FIXNUM,
	ARRAY_DISPATCH_BOOLEAN,
	ARRAY_DISPATCH_LEADER,
	ARRAY_DISPATCH_SHORT_2D,
	ARRAY_DISPATCH_CHARACTER,
	ARRAY_DISPATCH_14,
	ARRAY_DISPATCH_LONG,
	ARRAY_DISPATCH_LONG_MULTIDIMENSIONAL,
	ARRAY_DISPATCH_FAT_CHARACTER
};

enum {
	ART_1B = 0,
	ART_2B,
	ART_4B,
	ART_8B,
	ART_16B,
	ART_Q,
	ART_Q_LIST,
	PERC_ARRAY_TYPE_7,
	ART_BOOLEAN,
	PERC_ARRAY_TYPE_11,
	PERC_ARRAY_TYPE_12,
	PERC_ARRAY_TYPE_13,
	PERC_ARRAY_TYPE_14,
	ART_STRING,
	ART_FAT_STRING,
	ART_FIXNUM
};

char *array_type_names[] = {
	"ART-1B",
	"ART-2B",
	"ART-4B",
	"ART-8B",
	"ART-16B",
	"ART-Q",
	"ART-Q-LIST",
	"%ARRAY-TYPE-7",
	"ART-BOOLEAN",
	"%ARRAY-TYPE-11",
	"%ARRAY-TYPE-12",
	"%ARRAY-TYPE-13",
	"%ARRAY-TYPE-14",
	"ART-STRING",
	"ART-FAT-STRING",
	"ART-FIXNUM"
};

#define ARRAY_NORMAL_LENGTH_FIELD(w)	((w) & 0x0003ffff)
#define ARRAY_DISPATCH_FIELD(w)		(((w) >> 22) & 0xf)
#define ARRAY_TYPE_FIELD(w)		(((w) >> 18) & 0xf)

#define ARRAY_LONG_PREFIX_LENGTH_FIELD(w)	((w) & 0x1f)
#define ARRAY_LONG_LEADER_LENGTH_FIELD(w)	(((w) >> 5) & 0x3ff)
#define ARRAY_DIMENSIONS_FIELD(w)		(((w) >> 15) & 0x3)

/* low memory (virtual), starting at 0 */
char *fep_communication_area_names[] = {
	"FEP-VERSION-NUMBER",
	"KLUDGE-INPUT-CHARACTER",
	"KLUDGE-OUTPUT-CHARACTER",
	"KLUDGE-MINI-BUFFER-NUMBER",
	"KLUDGE-MINI-BUFFER-FULL",
	"MACHINE-CONFIGURATION-TABLE",
	"LOAD-MAP-SIZE",
	"LOAD-MAP-ADDRESS",
	"LOAD-MAP-DPN-ADDRESS",
	"SWAP-MAP-SIZE",
	"SWAP-MAP-ADDRESS",
	"SWAP-MAP-DPN-ADDRESS",
	"BAD-MEMORY-PAGES-SIZE",
	"BAD-MEMORY-PAGES-ADDRESS",
	"NET-ADDRESS-1",
	"NET-ADDRESS-2",
	"CHAOS-ADDRESS",
	"%UNWIRED-VIRTUAL-ADDRESS-LOW",
	"%UNWIRED-VIRTUAL-ADDRESS-HIGH",
	"%UNWIRED-PHYSICAL-ADDRESS-LOW",
	"COLD-LOAD-STREAM-OUTPUT-CHARACTER",
	"COLD-LOAD-STREAM-INPUT-CHARACTER",
	"CART-STATUS",
	"CART-COMMAND",
	"CART-EXT-STATUS-1",
	"CART-EXT-STATUS-2",
	"CART-MAP-TABLE-1",
	"CART-MAP-COUNT-1",
	"CART-MAP-TABLE-2",
	"CART-MAP-COUNT-2",
	"CART-LISP-COUNTER",
	"CART-FEP-COUNTER",
	"CART-LAST-BUF-OFFSET",
	"FEP-CHANNELS",
	"NEW-FEP-CHANNEL",
	"TASK-3-REQUESTS",
	"TASK-3-ENABLES",
	"SCREEN-DATA-OFFSET",
	"SCREEN-LOCATIONS-PER-LINE",
	"SCREEN-RASTER-HEIGHT",
	"SCREEN-TOP-FAKE-LINES",
	"SCREEN-BOTTOM-FAKE-LINES",
	"SCREEN-TOP-MARGIN-LINES",
	"SCREEN-BOTTOM-MARGIN-LINES",
	"SCREEN-RASTER-WIDTH",
	"SCREEN-LEFT-MARGIN-PIXELS",
	"SCREEN-RIGHT-MARGIN-PIXELS",
	"FEP-PRELOADED-ADDRESS",
	"FEP-PRELOADED-SIZE",
	"MACHINE-ID-PROMS",
	"CALENDAR-CLOCK-COMMAND",
	"CALENDAR-CLOCK-DATA-1",
	"CALENDAR-CLOCK-DATA-2",
	"REQUESTING-LISP-TO-STOP",
	"FEP-COMMAND-STRING",
	"FEP-BUS-COMMAND",
	"FEP-BUS-ADDRESS",
	"FEP-BUS-DATA",
	"FEP-BUS-ADDRESS-2",
	"FEP-BUS-CONDITIONS",
	"CONSOLE-BRIGHTNESS",
	"CONSOLE-VOLUME",
	"CONSOLE-COMMAND",
	"%HARDWARE-CONFIGURATION",
	"%SOFTWARE-CONFIGURATION",
	"*CRASH-DATA-REQUEST*",
	"*CONSOLE-SWITCHES*",
	"*CONSOLE-SLOT*",
	(char *)0
};

/* starting at 400 */
char *system_communication_area_names[] = {
	"SYSCOM-MAJOR-VERSION-NUMBER",
	"SYSCOM-MINOR-VERSION-NUMBER",
	"SYSCOM-DESIRED-MICROCODE-VERSION",
	"SYSCOM-VALID-ADDRESS-SPACE-SIZE",
	"*ADDRESS-SPACE-MAP*",
	"*AREA-NAME*",
	"*AREA-MAXIMUM-QUANTUM-SIZE*",
	"*AREA-REGION-QUANTUM-SIZE*",
	"*AREA-REGION-LIST*",
	"*AREA-REGION-BITS*",
	"*REGION-QUANTUM-ORIGIN*",
	"*REGION-QUANTUM-LENGTH*",
	"*REGION-FREE-POINTER*",
	"*REGION-GC-POINTER*",
	"*REGION-BITS*",
	"*REGION-LIST-THREAD*",
	"%REGION-CONS-ALARM",
	"%PAGE-CONS-ALARM",
	"*PACKAGE*",
	"OBARRAY",
	"CHAOS-RESET",
	"CHAOS-RESET-ACK",
	"CHAOS-INT-PKTS-TO-NET-OUTPUT-POINTER",
	"CHAOS-INT-PKTS-TO-NET-INPUT-POINTER",
	"CHAOS-INT-PKTS-TO-NET-LAST-POINTER",
	"CHAOS-INT-PKTS-TO-NET-FIRST-POINTER",
	"CHAOS-INT-PKTS-FROM-NET-OUTPUT-POINTER",
	"CHAOS-INT-PKTS-FROM-NET-INPUT-POINTER",
	"CHAOS-INT-PKTS-FROM-NET-LAST-POINTER",
	"CHAOS-INT-PKTS-FROM-NET-FIRST-POINTER",
	"KBD-BUFFER-START",
	"KBD-BUFFER-END",
	"KBD-BUFFER-IN-PTR",
	"KBD-BUFFER-OUT-PTR",
	"FEP-MOUSE-X",
	"FEP-MOUSE-Y",
	"MOUSE-BUTTONS",
	"MOUSE-WAKEUP",
	"%MICROCODE-VERSION",
	"*PHTC*",
	"*PHT*",
	"*PHT-SIZE*",
	"*MMPT-Y*",
	"*MMPT*",
	"*SMPT*",
	"*LOAD-BITMAPS*",
	"*LOAD-MAP*",
	"*LOAD-MAP-DPN*",
	"*SWAP-MAP*",
	"*SWAP-MAP-DPN*",
	"*SYSOUT-BITMAPS*",
	"LOADED-BAND-NAME",
	"(QUOTE NIL)",
	"(QUOTE T)",
	"WIRED-FERROR-ARGS",
	"(QUOTE IGNORE)",
	"*DISK-UNIT-TABLE*",
	"*LOAD-PAGES-TO-SWAP-AREA-P*",
	"*ENABLE-SYSOUT-AT-COLD-BOOT*",
	"*SYSOUT-GENERATION-NUMBER*",
	"*SYSOUT-TIMESTAMP-1*",
	"*SYSOUT-TIMESTAMP-2*",
	"*SYSOUT-PARENT-TIMESTAMP-1*",
	"*SYSOUT-PARENT-TIMESTAMP-2*",
	"*PACKAGE-NAME-TABLE*",
	"PRIMARY-NETWORK-ADDRESS",
	"FEP-TABLET-X",
	"FEP-TABLET-Y",
	"FEP-TABLET-B",
	"TABLET-WAKEUP",
	"*CURRENT-FEP-OVERLAYS*",
	"*REGION-AREA*",
	"*REGION-CREATED-PAGES*",
	"*LISP-STOPPED-CLEANLY*",
	"*NETBOOT-CONTROL-STRING*",
	"*ESDI-DISK-INFO-TABLE*",
	"*REGION-FREE-POINTER-BEFORE-FLIP*",
	"*LISP-RELEASE-STRING*",
	(char *)0
};

/* starting at 02000 */
char *fep_a_memory_variable_names[] = {
	"%STACK-BUFFER-LOW",
	"%STACK-BUFFER-LIMIT",
	"%CURRENT-STACK-BUFFER",
	"%WIRED-VIRTUAL-ADDRESS-HIGH",
	"%WIRED-PHYSICAL-ADDRESS-LOW",
	"%WIRED-PHYSICAL-ADDRESS-HIGH",
	"%CURRENT-PHTC",
	"%DISK-SECTOR-MAX-TRIES",
	"%DISK-COMMAND-ADDRESS",
	"%DISK-STATUS-ADDRESS",
	"%DISK-DCW-ADDRESS",
	"%DISK-MICRO-STATUS",
	"%DISK-WAKEUP",
	"%DISK-MEMORY-ADDRESS",
	"%CURRENT-STACK-GROUP",
	"%SEQUENCE-BREAK-PENDING",
	"ALPHABETIC-CASE-AFFECTS-STRING-COMPARISON",
	"ZUNDERFLOW",
	"%AUDIO-COMMAND-POINTER",
	"ARITHMETIC-BINARY-OPERATION-DISPATCH",
	"ARITHMETIC-UNARY-OPERATION-DISPATCH",
	"%COUNT-MAP-RELOADS",
	"%NET-MICRO-STATUS",
	"%NET-FREE-LIST",
	"%NET-RECEIVED-LIST",
	"%NET-TRANSMIT-LIST",
	"%NET-COLLISIONS",
	"%NET-TRANSMIT-ABORTS",
	"%NET-CRC-ERRORS",
	"%NET-ALIGNMENT-ERRORS",
	"%NET-DATA-OVERFLOWS",
	"%NET-BUFFER-OVERFLOWS",
	"%NET-IGNORED",
	"%DISK-HISTORY-BUFFER",
	"%DISK-HISTORY-BUFFER-BEGIN",
	"%DISK-HISTORY-BUFFER-END",
	"%NET-CONTROL-ADDRESS",
	"%STACK-GROUP-LOCK",
	"INHIBIT-SCHEDULING-FLAG",
	"%TRANSPORT-IN-PROGRESS",
	"%PC-TABLE-ADDRESS",
	"%PC-TABLE-LENGTH",
	"%PC-TABLE-OFFSET",
	"%PC-TABLE-PC-ROTATION",
	"%PC-TABLE-MISSES",
	"%LIST-CACHE-AREA",
	"%LIST-CACHE-VIRTUAL-ADDRESS",
	"%LIST-CACHE-LENGTH",
	"%LIST-CACHE-FREE-POINTER-ADDRESS",
	"%LIST-TRANSPORT-CACHE-AREA",
	"%LIST-TRANSPORT-CACHE-VIRTUAL-ADDRESS",
	"%LIST-TRANSPORT-CACHE-LENGTH",
	"%LIST-TRANSPORT-CACHE-FREE-POINTER-ADDRESS",
	"%STRUCTURE-CACHE-AREA",
	"%STRUCTURE-CACHE-VIRTUAL-ADDRESS",
	"%STRUCTURE-CACHE-LENGTH",
	"%STRUCTURE-CACHE-FREE-POINTER-ADDRESS",
	"%STRUCTURE-TRANSPORT-CACHE-AREA",
	"%STRUCTURE-TRANSPORT-CACHE-VIRTUAL-ADDRESS",
	"%STRUCTURE-TRANSPORT-CACHE-LENGTH",
	"%STRUCTURE-TRANSPORT-CACHE-FREE-POINTER-ADDRESS",
	"%NETWORK-MODE",
	"%FORTRAN-GLOBAL-ARRAY",
	"*DEFAULT-CONS-AREA*",
	"%TAPE-COMMAND-ADDRESS",
	"%TAPE-STATUS-ADDRESS",
	"%TAPE-DCW-ADDRESS",
	"%TAPE-MICRO-STATUS",
	"%TAPE-WAKEUP",
	"%TAPE-MEMORY-ADDRESS",
	"%FORTRAN-GLOBAL-EVENT-COUNT",
	"%FORTRAN-GLOBAL-BASE-POINTER",
	"%FORTRAN-GLOBAL-LAST-LOCATION",
	"%DISK-INTERFACE",
	"%DISK-ERROR-MASK",
	"GC-FLIP-INHIBIT",
	"BITBLT-UNPACKING-CONSTANT",
	"BITBLT-UNPACKING-BITS-PER-ELT",
	"BITBLT-UNPACKING-MASK",
	"*SLB-CONSOLE-MASK*",
	"%CHOICE-POINTER",
	"%NEXT-GOAL-POINTER",
	"%STRUCTURE-STACK-BASE",
	"%STRUCTURE-STACK-LIMIT",
	"%STRUCTURE-STACK-POINTER",
	"%STRUCTURE-STACK-BACKTRACK-POINTER",
	"%TRAIL-STACK-BASE",
	"%TRAIL-STACK-LIMIT",
	"%TRAIL-STACK-POINTER",
	"*SLB-CONSOLE-BUFFER-START*",
	"*SLB-CONSOLE-BUFFER-END*",
	"*SLB-CONSOLE-BUFFER-POINTER*",
	"*IO-BOARD-BASE*",
	"*NBS-CART-BUFFER-POINTER*",
	"*NBS-CART-BUFFER-COUNT*",
	"*NBS-CART-DATA-REGISTER*",
	"*NBS-CART-DATA-POINTER*",
	"*NBS-CART-WORD*",
	"*NBS-CART-BYTE-COUNT*",
	"*NBS-CART-WORD-COUNT*",
	"*SERIAL-DATA-1*",
	"*SERIAL-DATA-2*",
	"*SERIAL-COMMAND-1*",
	"*SERIAL-COMMAND-2*",
	"*SERIAL-INPUT-STATE-1*",
	"*SERIAL-INPUT-HEADER-POINTER-1*",
	"*SERIAL-INPUT-DATA-POINTER-1*",
	"*SERIAL-INPUT-BYTE-COUNT-1*",
	"*SERIAL-INPUT-BUFFER-LENGTH-1*",
	"*SERIAL-INPUT-WORD-1*",
	"*SERIAL-INPUT-BYTE-WITHIN-WORD-COUNT-1*",
	"*SERIAL-OUTPUT-STATE-1*",
	"*SERIAL-OUTPUT-HEADER-POINTER-1*",
	"*SERIAL-OUTPUT-DATA-POINTER-1*",
	"*SERIAL-OUTPUT-END-BYTE-OFFSET-1*",
	"*SERIAL-OUTPUT-BUFFER-LENGTH-1*",
	"*SERIAL-OUTPUT-WORD-1*",
	"*SERIAL-OUTPUT-DRAIN-BYTE-OFFSET-1*",
	"*SERIAL-INPUT-STATE-2*",
	"*SERIAL-INPUT-HEADER-POINTER-2*",
	"*SERIAL-INPUT-DATA-POINTER-2*",
	"*SERIAL-INPUT-BYTE-COUNT-2*",
	"*SERIAL-INPUT-BUFFER-LENGTH-2*",
	"*SERIAL-INPUT-WORD-2*",
	"*SERIAL-INPUT-BYTE-WITHIN-WORD-COUNT-2*",
	"*SERIAL-OUTPUT-STATE-2*",
	"*SERIAL-OUTPUT-HEADER-POINTER-2*",
	"*SERIAL-OUTPUT-DATA-POINTER-2*",
	"*SERIAL-OUTPUT-END-BYTE-OFFSET-2*",
	"*SERIAL-OUTPUT-BUFFER-LENGTH-2*",
	"*SERIAL-OUTPUT-WORD-2*",
	"*SERIAL-OUTPUT-DRAIN-BYTE-OFFSET-2*",
	"*CART-TAPE-RDY-EXC*",
	"*SERIAL-TX-UNDERRUNS-1*",
	"*SERIAL-TX-UNDERRUNS-2*",
	"*SERIAL-RX-ABORTS-1*",
	"*SERIAL-RX-ABORTS-2*",
	"*SERIAL-CLOCK-CONTROL*",
	"*AUDIO-STATE*",
	"*SLB-CONSOLE-SCC-DATA-ADDRESS*",
	(char *)0
};

/* at 02400 */
char *current_stack_group_data_names[] = {
	"%CONTROL-STACK-LOW",
	"%CONTROL-STACK-LIMIT",
	"%BINDING-STACK-LOW",
	"%BINDING-STACK-LIMIT",
	"%BINDING-STACK-POINTER",
	"%CATCH-BLOCK-LIST",
	"%CURRENT-STACK-GROUP-STATUS-BITS",
	"FLOAT-OPERATING-MODE",
	"FLOAT-OPERATION-STATUS",
	(char *)0
};

/* at 02440 */
#define PERC_OTHER_PC			02440	/* %OTHER-PC */
#define PERC_OTHER_FRAME_POINTER	02441	/* %OTHER-FRAME-POINTER */
#define PERC_OTHER_STACK_POINTER	02442	/* %OTHER-STACK-POINTER */

char *other_stack_group_data_names[] = {
	"%OTHER-PC",
	"%OTHER-FRAME-POINTER",
	"%OTHER-STACK-POINTER",
	"%OTHER-CONTROL-STACK-LOW",
	"%OTHER-CONTROL-STACK-LIMIT",
	"%OTHER-BINDING-STACK-LOW",
	"%OTHER-BINDING-STACK-LIMIT",
	"%OTHER-BINDING-STACK-POINTER",
	"%OTHER-CATCH-BLOCK-LIST",
	"%OTHER-STACK-GROUP-STATUS-BITS",
	"%OTHER-FLOAT-OPERATING-MODE",
	"%OTHER-FLOAT-OPERATION-STATUS",
	"%OTHER-STACK-LIMIT",
	(char *)0
};




int fep_communication_area_entries = sizeof(fep_communication_area_names) / sizeof(char *);
int system_communication_area_entries = sizeof(system_communication_area_names) / sizeof(char *);


void write_amem_36(int addr, w36 *v);
void read_amem_36(int addr, w36 *v);

/* cdr codes (top 2 bits of tag) */
enum {
	CDR_NEXT = 0,	/* ignore */
	CDR_NIL,	/* stack */
	CDR_NORMAL,	/* return */
	CDR_SPARE	/* multiple-values */
};

struct {
	int min, max;
} arg_table[]= {
	{0 , 0777}, {0, 0}, {0, 1}, {1, 1},
	{0, 2}, {1, 2}, {2, 2},
	{0, 3}, {1, 3}, {2, 3}, {3, 3},
	{0, 4}, {1, 4}, {2, 4}, {3, 4}, {4, 4}
};

/*
 * simple mmu
 * implements basic 3-level page table
 */

/* single page entry */
struct page_s {
	int fpn;
	int paged_in;
	u_long *mem_addr;
	u_char *tag_addr;
};

/* directory of pages */
struct page_dir_s {
	struct page_s pages[256];
};

static struct page_dir_s *page_root;
static void *world_stream;

struct page_dir_s *
paging_new_dir(void)
{
	struct page_dir_s *d;
	int i;

	d = (struct page_dir_s *)malloc(sizeof(struct page_dir_s));
	if (d == 0)
		return 0;

	memset((char *)d, 0, sizeof(struct page_dir_s));

	for (i = 0; i < 256; i++)
	  d->pages[i].fpn = -1;

	return d;
}

int
paging_init(void)
{
	/* create root directory */
	page_root = paging_new_dir();
	if (page_root == 0)
		return -1;

	return 0;
}

/*
 *
 * 28 bit virtual address mapping
 *
 *  x1 22 33 44
 *                                       
 *         top 4 bits (x1)  8 bits (22) 8 bits (33)  8 bits (44)
 * root -> root dir[0]
 *         root dir[1]   -> dir[0]
 *         ...              dir[1]   -> dir[0]
 *         root dir[255]    ...         dir[1]   ->  mem/tag [44]
 *                          dir[255]    ...          
 *                                      dir[255]     
 */

struct page_s *
paging_find_entry(u_long vma)
{
	int p1, p2, p3;
	struct page_dir_s *dir = page_root;
	struct page_s *p;

	p1 = vma >> 24;
	p = &page_root->pages[p1];

	if (!p->paged_in) {
		page_root->pages[p1].paged_in = 1;
		page_root->pages[p1].mem_addr = (u_long *)paging_new_dir();

		if (0) printf("new p1 %d\n", p1);
	}

	p2 = (vma >> 16) & 0xff;
	dir = (struct page_dir_s *)p->mem_addr;
	p = &dir->pages[p2];

	if (!p->paged_in) {
		dir->pages[p2].paged_in = 1;
		dir->pages[p2].mem_addr = (u_long *)paging_new_dir();

		if (0) printf("new p2 %d\n", p2);
	}

	p3 = (vma >> 8) & 0xff;
	dir = (struct page_dir_s *)p->mem_addr;
	p = &dir->pages[p3];

	return p;
}

int
paging_map_to_pma(u_long vma, u_long **pma, u_char **pta)
{
	int p3;
	struct page_s *p;

	p = paging_find_entry(vma);
	if (p == 0)
		return -1;

	if (!p->paged_in) {
		int offset;

		/* xxx use a heap */
		p->mem_addr = malloc(256*4);
		p->tag_addr = malloc(256);
		memset((char *)p->mem_addr, 0, 256*4);
		memset((char *)p->tag_addr, 0, 256);

		if (p->fpn >= 0) {
		  offset = p->fpn * 1152;  /* (256/2)*9 */

		  if (trace_vm) {
			  printf("paging_map_to_pma() new page; vma=0%o, pfn=0x%x\n",
				 vma, p->fpn);
		  }

		  read36_page(world_stream, offset, p->mem_addr, p->tag_addr);
		} else {
			if (trace_vm) {
				printf("paging_map_to_pma() new page; vma=0%o (zeros)\n",
				       vma);
			}

		  memset(p->mem_addr, 0, 256*4);
		  memset(p->tag_addr, 0, 256);
		}

		p->paged_in = 1;
	}

	*pma = p->mem_addr;
	*pta = p->tag_addr;

	return 0;
}

int
paging_add_vma_entry(u_long vma, int fpn)
{
	struct page_s *p;

	p = paging_find_entry(vma);
	if (p == 0)
		return -1;

	if (0) printf("vma 0x%x, page entry %p, fpn %x\n", vma, p, fpn);

	p->fpn = fpn;
	return 0;
}

int
fetch_isn(u_long spc, int *pisn)
{
	u_long pc, isn;
	u_long *pma;
	u_char *pta;
	int offset;

	pc = spc >> 1;
	if (paging_map_to_pma(a28(pc), &pma, &pta))
		return -1;

	offset = pc % 256;

	if (0) printf("fetch_isn() %08x tag %x ", pma[offset], pta[offset]);

	if (spc & 1) {
		isn = pma[offset] >> 16;
		if (pta[offset] & 0x80)
			isn |= 0x10000;
	} else {
		isn = pma[offset] & 0xffff;
		if (pta[offset] & 0x40)
			isn |= 0x10000;
	}

	*pisn = isn;

	return 0;
}

void
read_mem_36(u_long vma, w36 *v)
{
	u_long *pma;
	u_char *pta;
	int offset;

	if ((vma & 0x0ffff000) == 0x0ffff000)
	{
		read_amem_36(vma & 0x0fff, v);

		if (trace_mem) {
			char *sym;
			extern char *find_sym_by_val(int, int);

			sym = find_sym_by_val(1, vma & 0x0fff);
		
			printf("read_mem_36(vma=0%o) -> amem %02x:%08x %o",
			       a28(vma), v->tag, v->word, v->word);
			if (sym) {
				printf(" %s", sym);
			}
			printf("\n");
		}

		return;
	}

	if (paging_map_to_pma(a28(vma), &pma, &pta))
		;

	offset = vma % 256;


	if (trace_mem) {
		char *sym = 0;

		if (a28(vma) < sizeof(fep_communication_area_names))
			sym = fep_communication_area_names[a28(vma)];

		printf("read_mem_36(vma=0%o) -> %02x:%08x %o",
		       a28(vma), pta[offset], pma[offset], pma[offset]);
		if (sym) {
			printf(" %s", sym);
		}
		printf("\n");
	}

	v->word = pma[offset];
	v->tag = pta[offset];
}

void
write_mem_36(u_long vma, w36 *v)
{
	u_long *pma;
	u_char *pta;
	int offset;

	if ((vma & 0x0f000000) == 0x0f00000)
		printf("write %x\n", vma);

	if ((vma & 0x0ffff000) == 0x0ffff000)
	{
		write_amem_36(vma & 0x0fff, v);

		if (trace_mem) {
			printf("write_mem_36(vma=0%o) <- amem %02x:%08x %o\n",
			       a28(vma), v->tag, v->word, v->word);
		}

		return;
	}

	if (paging_map_to_pma(a28(vma), &pma, &pta))
		;

	offset = vma % 256;

	if (trace_mem) {
		printf("write_mem_36(vma=0%o) <- %02x:%08x\n",
		       vma, v->tag, v->word);
	}

	pma[offset] = v->word;
	pta[offset] = v->tag;
}


w36 top_of_stack;
int stack_pointer;
int stack_low;
int stack_limit;
int frame_pointer;
int a_pclsr_top_of_stack;

w36 quote_nil;
w36 quote_t;

extern u_long amem[];
extern u_char amem_tag[];

void
write_amem_36(int addr, w36 *v)
{
	amem[addr & 0x0fff] = v->word;
	amem_tag[addr & 0x0fff] = v->tag;
}

void
read_amem_36(int addr, w36 *v)
{
	v->word = amem[addr & 0x0fff];
	v->tag = amem_tag[addr & 0x0fff];
}

void
set_cdr(w36 *v, int cdr_code)
{
	v->tag = (v->tag & 0x3f) | (cdr_code << 6);
}

void
set_tag(w36 *v, int tag)
{
	v->tag = tag;
}

void
set_type(w36 *v, int tag)
{
	v->tag = tag;
	v->word = (v->word & 0x0fffffff) | ((v->tag & 0xf) << 28);
}

void
pushval(w36 *v)
{
	set_cdr(v, CDR_NEXT);
	stack_pointer++;
	top_of_stack = *v;
	if (trace_push_pop) {
		printf("pushval: (%o) %3o:%011o (%x:%08x)\n",
		       stack_pointer, v->tag, v->word, v->tag, v->word);
	}
	write_amem_36(stack_pointer, v);
}

void
pushval_with_cdr(w36 *v)
{
	stack_pointer++;
	top_of_stack = *v;
	if (trace_push_pop) {
		printf("pushval: (%o) %3o:%011o (%x:%08x)\n", stack_pointer,
		       v->tag, v->word,
		       v->tag, v->word);
	}
	write_amem_36(stack_pointer, v);
}

void
popval(w36 *v)
{
	*v = top_of_stack;
	if (trace_push_pop) {
		printf("popval: (%o) %3o:%011o (%x:%08x)\n",
		       stack_pointer,
		       top_of_stack.tag, top_of_stack.word,
		       top_of_stack.tag, top_of_stack.word);
	}
	stack_pointer--;
	read_amem_36(stack_pointer, &top_of_stack);
}

void
pop2push(w36 *v)
{
	/* like doing two popval's and then pushval */
	set_cdr(v, CDR_NEXT);
	write_amem_36(stack_pointer - 1, v);
	top_of_stack = *v;
	stack_pointer--;
	if (trace_push_pop) {
		printf("pop2push: (%o) %3o:%011o (%x:%08x)\n", stack_pointer,
		       top_of_stack.tag, top_of_stack.word,
		       top_of_stack.tag, top_of_stack.word);
	}
}

void
newtop(w36 *v)
{
	set_cdr(v, CDR_NEXT);
	write_amem_36(stack_pointer, v);
	top_of_stack = *v;
	if (trace_push_pop) {
		printf("newtop: (%o) %3o:%011o (%x:%08x)\n", stack_pointer,
		       top_of_stack.tag, top_of_stack.word,
		       top_of_stack.tag, top_of_stack.word);
	}
}

int vma;

void
popmem(void)
{
	/* read mem to get tag; pop stack, write data but merge tag */
	w36 v, v1;
	read_mem_36(vma, &v);
	popval(&v1);
	v.word = v1.word;
	v.tag = (v.tag & 0x3f) | (v1.tag & 0xc0);
	write_mem_36(vma, &v);
	if (trace_push_pop) printf("popmem: vma %011o (0x%x) <- %o:%011o\n",
				   vma, vma, v.tag, v.word);
}

void
pushmem(void)
{
	/* read mem to get tag; popstack, write data but merge tag */
	w36 v;
	if (trace_push_pop) printf("pushmem: vma %011o (0x%x)\n", vma, vma);
	read_mem_36(vma, &v);
	pushval(&v);
}

void
popmemind(void)
{
	w36 v;
	if (trace_push_pop) printf("popmemind: vma %011o (0x%x)\n", vma, vma);
	/* indirect version of popmem */
	read_mem_36(vma, &v);
	vma = v.word;
	popmem();
}

void
pushmemind(void)
{
	w36 v;
	/* indirect version of popmem */
	if (trace_push_pop) printf("pushmemind: vma %011o (0x%x)\n", vma, vma);
	read_mem_36(vma, &v);
	vma = v.word;
	pushmem();
}

u_long
frame_function(void)
{
	if (trace_push_pop) {
		printf("frame_function() fp-1=%o; [fp-1]=%08x %11o\n",
		       frame_pointer-1,
		       amem[frame_pointer - 1], amem[frame_pointer - 1]);
	}

	return amem[frame_pointer - 1];
}

void
pop_indirect(int operand)
{
	vma = frame_function() - operand - 1;
	popmemind();
}

#define FRAME_NUMBER_OF_ARGS	(0x00ff)
#define FRAME_CLEANUP_BITS	(0xff00)
#define FRAME_LEXPR_CALLED	(1<<25)
#define FRAME_INSTANCE_CALLED	(1<<26)
#define FRAME_FUNCALLED		(1<<27)
#define FRAME_PART_DONE		(1<<28)
#define FRAME_CLEANUP_IN_PROGRESS (1<<29)
#define FRAME_THROWN_THROUGH	(1<<30)
#define FRAME_ARGUMENT_FORMAT	(3<<24)

/* *FRAME-ARGUMENT-FORMATS* */
enum {
	PERC_FRAME_ARGUMENTS_NORMAL = 0,
	PERC_FRAME_ARGUMENTS_LEXICAL,
	PERC_FRAME_ARGUMENTS_LEXPR,
	PERC_FRAME_ARGUMENTS_LEXPR_LEXICAL,
	PERC_FRAME_ARGUMENTS_INSTANCE,
	PERC_FRAME_ARGUMENTS_5,
	PERC_FRAME_ARGUMENTS_LEXPR_INSTANCE,
};

u_long
frame_misc_data(void)
{
	return amem[frame_pointer - 2];
}

u_long
frame_previous_top(void)
{
	return amem[frame_pointer - 4];
}

u_long
frame_previous_frame(void)
{
	return amem[frame_pointer - 5];
}

int
frame_funccalled(void)
{
  return frame_misc_data() & FRAME_FUNCALLED;
}

int
frame_lexpr_called(void)
{
  return frame_misc_data() & FRAME_LEXPR_CALLED;
}

int
frame_argument_format(void)
{
	return (frame_misc_data() & FRAME_ARGUMENT_FORMAT) >> 24;
}

int
frame_number_of_args(void)
{
	return (frame_misc_data() & FRAME_NUMBER_OF_ARGS);
}

void
dump_stack(void)
{
	int i;
	w36 v;
	char marker;

	printf("stack:\n");
	printf("stack-pointer %o, frame-pointer %o\n",
	       stack_pointer, frame_pointer);

	for (i = stack_low; i < stack_pointer + 10; i++) {
		marker = ' ';
		if (i == stack_pointer) marker = '>';
		if (i == frame_pointer) marker = '-';

		read_amem_36(i, &v);
		printf("%4o:%c%3o:%011o (%x:%x)\n",
		       i, marker,
		       v.tag, v.word, v.tag, v.word);
	}
}

void
dump_stack_frame(char *who)
{
	int i, from, to;
	w36 v;
	char marker;

	printf("%s: stack:\n", who);
	printf("stack-pointer %o, frame-pointer %o\n",
	       stack_pointer, frame_pointer);

	if (stack_pointer > frame_pointer) {
		from = frame_pointer - 10;
		to = stack_pointer + 10;
	} else {
		from = stack_pointer - 10;
		to = stack_pointer + 10;
	}

	for (i = from; i < to; i++) {
		marker = ' ';
		if (i == stack_pointer) marker = '>';
		if (i == frame_pointer) marker = '-';

		read_amem_36(i, &v);
		printf("%4o:%c%03o:%011o (%x:%x)\n",
		       i, marker,
		       v.tag, v.word, v.tag, v.word);
	}
}

void
dump_stack_small(void)
{
  int i, from, to;
  char marker;
  w36 v;

  if (stack_pointer > frame_pointer) {
	  from = frame_pointer;
	  to = stack_pointer + 2;
  } else {
	  from = stack_pointer - 2;
	  to = stack_pointer + 2;
  }

  for (i = from; i < to; i++) {
	  marker = ' ';
	  if (i == stack_pointer) marker = '>';
	  if (i == frame_pointer) marker = '-';

	  read_amem_36(i, &v);
	  printf("%4o:%c%03o:%011o (0x%x)\n",
		 i, marker,
		 v.tag, v.word, v.word);
  }
}

int
sign_extend_7(int operand)
{
	/* assumes 32 bit machine */
	if (operand & 0x80)
		return operand | 0xffffff80;
	return operand & 0x7f;
}

int
unsigned_7(int operand)
{
	return operand & 0x7f;
}

char *arg_dispatch_names[] = {
	"slow case",
	"no arguments",
	"no required, 1 optional",
	"1 required",
	"no required, 2 optional",
	"1 required, 1 optional",
	"2 required",
	"no required, 3 optional",
	"1 required, 2 optional",
	"2 required, 1 optional",
	"3 required",
	"no required, 4 optional",
	"1 required, 3 optional",
	"2 required, 2 optional",
	"3 required, 1 optional",
	"4 required"
};

/*
 * show string
 * (usually a dtp-header-i %header-type-array art-string array)
 */
void
show_pname(int loc)
{
	w36 v;
	int i, j, len;
	char str[256];

	read_mem_36(loc, &v);
	len = v.word & 0x0fff;

	j = 0;
	for (i = 0; i < (len/4)+1; i++) {
		read_mem_36(++loc, &v);
		str[j++] = v.word;
		str[j++] = v.word >> 8;
		str[j++] = v.word >> 16;
		str[j++] = v.word >> 24;
	}
	str[len] = 0;

	printf("%s", str);
}

static int
_ldb(int v, int size, int pos)
{
	int mask;

	mask = 0xffffffff >> (32 - size);
	return (v >> pos) & mask;
}

void
_show_tagged_loc(unsigned long loc, char *str, int flags)
{
	w36 v, p;
	int immed, ptr, cdr_code, i;

	read_mem_36(loc, &v);

	printf("%s %s %3o:%011o ",
	       str, tag_names[v.tag & 0x3f],
	       v.tag, a28(v.word));

	immed = ptr = v.word;
	cdr_code = v.tag >> 6;

	switch (v.tag & 0x3f) {
	case DTP_SYMBOL:
		printf("\n");
		ptr = v.word;
		for (i = 0; i < 5; i++) {
			char str[32];
			sprintf(str, " [%d]:", i);
			_show_tagged_loc(ptr+i, str, 0);
		}
		break;

	case DTP_ARRAY:
		printf("\n");
		ptr = v.word;
		_show_tagged_loc(ptr, "[] ", 0);
		break;

	case DTP_HEADER_I:
		switch (cdr_code) {
			int atype, adisp;
		case 0: printf("%%header-type-compiled-function "); break;
		case 1:
			printf("%%header-type-array\n");
			printf(" array-named-structure-bit %d\n",
			       _ldb(immed, 1, 27));
			printf(" array-discontiguous-bit %d\n",
			       _ldb(immed, 1, 26));

			adisp = _ldb(immed, 4, 22);
			printf(" array-dispatch-field %d ", adisp);
			printf("%s\n", adisp >= 0 && adisp <= 16 ?
			       array_dispatch_names[adisp] : "");

			atype = _ldb(immed, 4, 18);
			printf(" array-type-field %d ", atype);
			printf("%s\n", atype >= 0 && atype <= 16 ?
			       array_type_names[atype] : "");

			switch (adisp) {
			default:
				printf(" array-normal-length-field %d\n",
				       _ldb(immed, 18, 0));
				break;
			case ARRAY_DISPATCH_LEADER:
				printf(" array-short-length-field %d\n",
				       _ldb(immed, 12, 0));
				printf(" array-leader-length-field %d\n",
				       _ldb(immed, 6, 12));
				break;
			case ARRAY_DISPATCH_SHORT_2D:
				printf(" array-rows-field %d\n",
				       _ldb(immed, 9, 0));
				printf(" array-columns-field %d\n",
				       _ldb(immed, 9, 9));
				break;
			case ARRAY_DISPATCH_SHORT_INDIRECT:
				printf(" array-short-indirect-length-field %d\n",
				       _ldb(immed, 9, 0));
				printf(" array-short-indirect-offset-field %d\n",
				       _ldb(immed, 9, 9));
				/* 2nd word - array-indirect-field */
				break;
			case ARRAY_DISPATCH_LONG:
			case ARRAY_DISPATCH_LONG_MULTIDIMENSIONAL:
				printf(" array-long-prefix-length-field %d\n",
				       _ldb(immed, 5, 0));
				printf(" array-long-leader-length-field %d\n",
				       _ldb(immed, 10, 5));
				printf(" array-dimensions-field %d\n",
				       _ldb(immed, 3, 15));

				/* 2nd word - array-indirect-field */
				_show_tagged_loc(loc+1,
						 " array-indirect-field", 0);

				/* 3rd word - array-long-length-field */
				_show_tagged_loc(loc+2,
						 " array-long-length-field", 0);

				/* 4th word - array-index-offset-field */
				_show_tagged_loc(loc+3,
						 " array-index-offset-field", 0);
				break;
			}
			break;
		case 2:
			printf("%%header-type-number ");
			break;
		default:
			printf("cdr_code %d\n", cdr_code);
		}
		break;
	case DTP_HEADER_P:
		switch (cdr_code) {
		case 0:
			printf("%%header-type-symbol ");
			show_pname(v.word);
			break;
		case 1:
			printf("%%header-type-instance ");
			break;
		}

		break;
	}

	printf("\n");
}

void
show_tagged_loc(unsigned long loc, char *str, int flags)
{
	_show_tagged_loc(loc, str, flags);
}

void
_show_taged(char *str, w36 *v, int follow)
{
	w36 p;
	int immed, ptr, cdr_code, i;

	printf("%s %s %3o:%011o ",
	       str, tag_names[v->tag & 0x3f],
	       v->tag, a28(v->word));

	immed = ptr = v->word;
	cdr_code = v->tag >> 6;

	switch (v->tag & 0x3f) {
	case DTP_SYMBOL:
	case DTP_ARRAY:
		printf("\n");
		break;

	case DTP_HEADER_I:
		switch (cdr_code) {
			int atype, adisp;
		case 0: printf("%%header-type-compiled-function "); break;
		case 1:
			printf("%%header-type-array ");
			printf("array-named-structure-bit %d\n", _ldb(immed, 1, 27));
			printf("array-discontiguous-bit %d\n", _ldb(immed, 1, 26));

			adisp = _ldb(immed, 4, 22);
			printf("array-dispatch-field %d ", adisp);
			printf("%s\n", adisp >= 0 && adisp <= 16 ?
			       array_dispatch_names[adisp] : "");

			atype = _ldb(immed, 4, 18);
			printf("array-type-field %d ", atype);
			printf("%s\n", atype >= 0 && atype <= 16 ?
			       array_type_names[atype] : "");

			switch (adisp) {
			default:
				printf("array-normal-length-field %d\n",
				       _ldb(immed, 18, 0));
				break;
			case ARRAY_DISPATCH_LEADER:
				printf("array-short-length-field %d\n",
				       _ldb(immed, 12, 0));
				printf("array-leader-length-field %d\n",
				       _ldb(immed, 6, 12));
				break;
			case ARRAY_DISPATCH_SHORT_2D:
				printf("array-rows-field %d\n",
				       _ldb(immed, 9, 0));
				printf("array-columns-field %d\n",
				       _ldb(immed, 9, 9));
				break;
			case ARRAY_DISPATCH_SHORT_INDIRECT:
				printf("array-short-indirect-length-field %d\n",
				       _ldb(immed, 9, 0));
				printf("array-short-indirect-offset-field %d\n",
				       _ldb(immed, 9, 9));
				/* 2nd word - array-indirect-field */
				break;
			case ARRAY_DISPATCH_LONG:
			case ARRAY_DISPATCH_LONG_MULTIDIMENSIONAL:
				printf("array-long-prefix-length-field %d\n",
				       _ldb(immed, 5, 0));
				printf("array-long-leader-length-field %d\n",
				       _ldb(immed, 10, 5));
				printf("array-dimensions-field %d\n",
				       _ldb(immed, 3, 15));
				/* 2nd word - array-indirect-field */
				/* 3rd word - array-long-length-field */
				/* 4th word - array-index-offset-field */
				break;
			}
			break;
		case 2:
			printf("%%header-type-number ");
			break;
		default: printf("cdr_code %d\n", cdr_code);
		}
		break;
	case DTP_HEADER_P:
		switch (cdr_code) {
		case 0:
			printf("%%header-type-symbol ");
			show_pname(v->word);
			break;
		case 1:
			printf("%%header-type-instance ");
			break;
		}

		break;
	}

	printf("\n");
}

void
show_tagged(char *str, w36 *v)
{
	_show_taged(str, v, 1);
}

void
show_array(int loc)
{
	printf("array: %011o\n", loc);
	show_tagged_loc(loc+0, "+0:", 0);
	show_tagged_loc(loc+1, "+1:", 0);
	show_tagged_loc(loc+2, "+2:", 0);
	show_tagged_loc(loc+3, "+3:", 0);
	show_tagged_loc(loc+4, "+4:", 0);
}

/*
 *  interpret the header before a DTP-COMPILED-FUNCTION
 */
void
show_compiled_function(int loc)
{
	w36 v;
	int ptr, len, i, j;
	char name[256];
	int table_size, arg_dispatch;
	int compiled_function_table_size,
		compiled_function_total_size,
		compiled_function_args_info,
		cca_current_definition_p,
		compiled_function_language_index;

	read_mem_36(loc, &v);
	table_size = v.word & 0xff;
	arg_dispatch = (v.word >> 8) & 0xf;

	if (trace_func_header > 1) {
		show_loc(loc, 0);
		show_loc(loc - table_size - 4, 0);
		show_loc(loc - table_size - 3, 0);
		show_loc(loc - table_size - 2, 0);
		show_loc(loc - table_size - 1, 0);
	}

	/* DTP-HEADER-I, %HEADER-TYPE-COMPILED-FUNCTION */
	read_mem_36(loc - table_size - 4, &v);
	if (v.tag != DTP_HEADER_I) {
		printf("DTP-HEADER-I tag %x\n", v.tag);
		printf("word %x, table_size %d, arg_dispatch %d\n",
		       v.word, table_size, arg_dispatch);
		return;
	}

	compiled_function_table_size = v.word & 0xff; /* # of ext refs and constants */
	compiled_function_total_size = (v.word >> 8) & 0xffff; /* # of words total */

	/* COMPILED-FUNCTION-HEADER-WORD-2 */
	read_mem_36(loc - table_size - 3, &v);
	if (!dtp_fix_p(v.tag)) {
		printf("COMPILED-FUNCTION-HEADER-WORD-2 tag %x\n", v.tag);
		return;
	}

	compiled_function_args_info = v.word & 0x3ffff; /* %ARGS-INFO data */
	cca_current_definition_p = v.word & 0x40000;
	compiled_function_language_index = (v.word >> 28) & 0xf;

	/* COMPILED-FUNCTION-EXTRA-INFO */
	read_mem_36(loc - table_size - 2, &v);
	ptr = v.word;
	if (trace_func_header > 1)
		printf("ptr %x:%x\n", v.tag, v.word);
	if (tag(v.tag) != DTP_LIST)
		return;

	/* extra-info is a cons of the function name and debug-info a-list */

	read_mem_36(ptr, &v);
	ptr = v.word;
	if (trace_func_header > 1)
		printf("ptr2 %x:%x\n", v.tag, v.word);

	read_mem_36(ptr, &v);
	ptr = v.word;
	if (trace_func_header > 1)
		printf("ptr3 %x:%x\n", v.tag, v.word);

	/* pname string */

	read_mem_36(ptr, &v);
	if (trace_func_header > 1)
		printf("string %x:%x\n", v.tag, v.word);
	len = v.word & 0x0fff;

	if (v.tag == DTP_HEADER_P) {
		ptr = v.word;
		read_mem_36(ptr, &v);
		if (trace_func_header > 1)
			printf("string %x:%x\n", v.tag, v.word);
		len = v.word & 0x0fff;
	}

	j = 0;
	for (i = 0; i < (len/4)+1; i++) {
		read_mem_36(++ptr, &v);
		name[j++] = v.word;
		name[j++] = v.word >> 8;
		name[j++] = v.word >> 16;
		name[j++] = v.word >> 24;
	}
	name[len] = 0;

	printf("function: '%s'\n", name);
	if (trace_enter_exit)
		return;

	printf("table-size %d, arg-dispatch %d, '%s'\n",
	       table_size, arg_dispatch,
	       arg_dispatch_names[arg_dispatch]);

	{
		int max_args, min_args, rest_arg, quoted, interpreted;

		max_args = compiled_function_args_info & 0xff;
		min_args = (compiled_function_args_info >> 8) & 0xff;
		rest_arg = compiled_function_args_info & 0x1000 ? 1 : 0;
		quoted = compiled_function_args_info & 0x2000 ? 1 : 0;
		interpreted = compiled_function_args_info & 0x4000 ? 1 : 0;

		printf("max-args %d, min-args %d, rest-arg %s, quoted %s, interpreted %s\n", 
		       max_args, min_args,
		       rest_arg ? "t" : "nil",
		       quoted ? "t" : "nil",
		       interpreted ? "t" : "nil");
	}
}

void
find_func_name(int loc)
{
	w36 v;
	int args, ptr, len, i, j;
	char name[256];

	read_mem_36(loc, &v);
	args = v.word & 0xff;

	read_mem_36(loc - args - 2, &v);
	ptr = v.word;
	if (0) printf("ptr %x\n", ptr);

	read_mem_36(ptr, &v);
	ptr = v.word;
	if (0) printf("ptr2 %x\n", v.word);

	read_mem_36(ptr, &v);
	ptr = v.word;
	if (0) printf("ptr3 %x\n", v.word);

	read_mem_36(ptr, &v);
	if (0) printf("string %x\n", v.word);

	len = v.word & 0x0fff;
	j = 0;
	for (i = 0; i < (len/4)+1; i++) {
		read_mem_36(++ptr, &v);
		name[j++] = v.word;
		name[j++] = v.word >> 8;
		name[j++] = v.word >> 16;
		name[j++] = v.word >> 24;
	}
	name[len] = 0;
	printf("function: %s\n", name);

//if (strcmp("DISABLE-CONSOLES", name) == 0)
if (strcmp("INITIALIZE-CONSOLE", name) == 0)
{
set_trace(2);
flag_enter_debugger = 1;
}

}

u_long
local_operand_addr(int operand)
{
	int addr;

	if (operand & 0x80) {
		addr = stack_pointer - sign_extend_7(operand);

		if (trace_args)
			printf("local-operand: sp - %d\n",
			       sign_extend_7(operand));
	} else {
		addr = frame_pointer + unsigned_7(operand);
		if (trace_args)
			printf("local-operand: fp + %d\n",
			       unsigned_7(operand));
	}
	return addr;
}

void
local_operand(int operand, w36 *v)
{
	int addr;

	if (operand & 0x80) {
		addr = stack_pointer - sign_extend_7(operand);

		if (trace_args)
			printf("local-operand: sp - %d\n",
			       sign_extend_7(operand));
	} else {
		addr = frame_pointer + unsigned_7(operand);
		if (trace_args)
			printf("local-operand: fp + %d\n",
			       unsigned_7(operand));
	}
	read_amem_36(addr, v);
	if (trace_args)
		printf("local-operand: addr %o -> %x:%o\n",
		       addr, v->tag, v->word);
}

void
write_local_operand(int operand, w36 *v)
{
	int addr;

	if (operand & 0x80) {
		addr = stack_pointer - sign_extend_7(operand);
		if (trace_args)
			printf("local-operand: sp - %d\n",
			       sign_extend_7(operand));
	} else {
		addr = frame_pointer + unsigned_7(operand);
		if (trace_args)
			printf("local-operand: fp + %d\n",
			       unsigned_7(operand));
	}
	write_amem_36(addr, v);
	if (trace_args)
		printf("local-operand: addr %o <- %x:%o\n",
		       addr, v->tag, v->word);
}

int
equal_typed_pointer(w36 *p1, w36 *p2)
{
	if (p1->word == p2->word &&
	    (p1->tag & 0x3f) == (p2->tag & 0x3f))
		return 1;
	return 0;
}

int
equal_fixnum(w36 *f1, w36 *f2)
{
	if (trace_comp) printf("equal_fixnum: %x == %x?\n", f1->word, f2->word);
	if (f1->word == f2->word)
		return 1;
	return 0;
}

int
lesser_fixnum(w36 *f1, w36 *f2)
{
	if (trace_comp) printf("lesser_fixnum: %x < %x?\n", f1->word, f2->word);
	if (f1->word < f2->word)
		return 1;
	return 0;
}

int
greater_fixnum(w36 *f1, w36 *f2)
{
	if (trace_comp) printf("greater_fixnum: %x > %x?\n", f1->word, f2->word);
	if (f1->word > f2->word)
		return 1;
	return 0;
}

int
zero_fixnum(w36 *f)
{
	if (f->word == 0)
		return 1;
	return 0;
}

int
plus_fixnum(w36 *f)
{
	if ((f->word & 0x80000000) == 0)
		return 1;
	return 0;
}

int
minus_fixnum(w36 *f)
{
	if (f->word & 0x80000000)
		return 1;
	return 0;
}

void
pc_add(int offset)
{
#if 1
	int pc, word_offset, halfword_offset;

	pc = sim_pc >> 1;
	word_offset = offset >> 1;
	halfword_offset = (offset & 1) ^ (offset < 0 ? 1 : 0);
	sim_pc = ((pc + word_offset) << 1) + ((sim_pc & 1) ^ halfword_offset);
#endif
#if 0
	int hw;
	hw = (sim_pc & 1) ^ (offset & 1) ^ ((offset < 0) ? 1 : 0);
	sim_pc = (sim_pc & ~1) + (offset << 1);

	if (hw & 1)
		sim_pc |= 1;
	else
		sim_pc &= ~1;
#endif
#if 0
	sim_pc += offset >= 0 ? offset : (offset + 1);
#endif
}

int
exec_isn(u_long spc, int inst)
{
	int uop, operand, cdr_code, addr, nargs, vdisp;
	int next_instruction, blt_pointer, old_stack_pointer;
	int alen, atype, adisp;
	int a_temp, b_temp;
	int pos, size, i, n;
	unsigned long mask;
	w36 v, v1, v2;

	if (0) printf("exec_isn(pc=%o, isn=%o 0x%x)\n", spc>>1, inst, inst);

	if (trace_stack_small)
		dump_stack_small();

	if (trace_isn_disasm)
		disassemble36(sim_pc >> 1, sim_pc & 1, inst);

	if ((inst & 0xff) == 0xff)
		uop = 01000 + ((inst >> 8) & 0x1ff);
//		uop = 01000 + ((inst >> 8) & 0x0ff);
	else
//		uop = (inst & 0xff) | ((inst & 0x10000) ? 0x100 : 0);
		uop = (inst & 0xff);

	operand = (inst >> 8) & 0xff;
	//printf("uop %o\n", uop);

	next_instruction = 1;

	switch (uop) {
	case 2:		/* %make-pointer-immed */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		set_tag(&top_of_stack,
			(top_of_stack.tag & 0x0f) | (operand & 0x0f));

		newtop(&top_of_stack);
		break;
	case 3:		/* %make-pointer-immed-offset */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		v.word += top_of_stack.word;
		set_type(&v, DTP_FIX);
		pop2push(&v);

		set_tag(&top_of_stack,
			(top_of_stack.tag & 0x0f) | (operand & 0x0f));

		newtop(&top_of_stack);
		break;

	case 014:	/* p-dpb-immed */
		vma = top_of_stack.word;
		read_mem_36(vma, &v);

		if (trace_bits) {
		  printf("p-dpb-immed: vma 0%o 0x%x\n", vma, vma);
		  printf("p-dpb-immed: [vma] 0%o 0x%x\n", v.word, v.word);
		}

		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v1);
		stack_pointer--;

		popval(&v2);

		/* form 10-bit-operand */
		operand |= (inst & 0x3) << 8;

		size = (operand >> 5) + 1;
		pos = operand & 0x1f;

		if (trace_bits) {
		  printf("p-dpb-immed: operand 0%o 0x%x, pos %d, size %d\n",
			 operand, operand, pos, size);

		  printf("p-dpb-immed: next-on-stack %x:%x\n", v1.tag, v1.word);
		  printf("p-dpb-immed: [top-of-stack] %x:%x\n", v.tag, v.word);
		}

		/* memory-data <- dpb b-temp macro macro memory-data */

		mask = 0xffffffff >> (32 - size);
		v.word = (v.word & ~(mask << pos)) |
			((v1.word & mask) << pos);

		if (trace_bits) {
		printf("p-dpb-immed: mask %x, mask shifted %x\n",
		       mask, mask << pos);
		printf("p-dpb-immed: result %x\n", v.word);
		}

		set_type(&v, DTP_FIX);

		write_mem_36(vma, &v);
		break;

	case 032:	/* %io-read */
		break;
	case 0100:	/* push-indirect */
		vma = frame_function() - operand - 1;
		pushmemind();
		break;
	case 0101:	/* push-constant */
		vma = frame_function() - operand - 1;
		pushmem();
		break;
	case 0102:	/* push-local */
		local_operand(operand, &v);
		pushval(&v);
		break;
	case 0103:	/* push-immed */
		v.tag = DTP_FIX;
		v.word = sign_extend_7(operand);
		pushval(&v);
		break;
	case 0104:	/* push-address-local */
		if (operand & 0x80)
			v.word = stack_pointer - sign_extend_7(operand) + 1;
		else
			v.word = frame_pointer + sign_extend_7(operand);
		v.word += 0x0ffff000;
		set_type(&v, DTP_LOCATIVE);
		pushval(&v);
		break;
	case 0105:	/* push-from-beyond-multiple */
		/* stack-pointer - a-memory[stack-pointer] + operand + 1 */
		read_amem_36(stack_pointer, &v);
		addr = stack_pointer - v.word + operand + 1;

		read_amem_36(addr, &v);
		set_type(&v, DTP_FIX);
		pushval(&v);
		break;
	case 0106:	/* movem-local */
		write_local_operand(operand, &top_of_stack);
		break;
	case 0107:	/* movem-indirect */
		pushval(&top_of_stack);
		vma = frame_function() - operand - 1;
		if (trace_bits)
			printf("movem-indirect: vma 0%o (0x%x)\n", vma, vma);
		popmemind();
		break;
	case 0110:	/* pop-local */
//xxx - calc w/sp before pop?
		popval(&v);
		write_local_operand(operand, &v);
		break;
	case 0111:	/* pop-indirect */
		vma = frame_function() - operand - 1;
		if (trace_bits)
		printf("pop-indirect: vma 0x%x 0%o\n", vma, vma);
		popmemind();
		break;
	case 0120:	/* push-n-nils */
		for (i = 0; i < operand; i++) {
			pushval(&quote_nil);
		}
		break;
	case 0121:	/* pop-n */
		stack_pointer -= unsigned_7(operand);
		read_amem_36(stack_pointer, &top_of_stack);
		break;
	case 0122:	/* pop-n-save-1 */
		stack_pointer -= unsigned_7(operand);
		write_amem_36(stack_pointer, &top_of_stack);
		break;
	case 0145:	/* take-values */
//		if (1) dump_stack_frame("take-values before");
#if 0
		if (top_of_stack.tag != DTP_FIX) {
		}
#endif
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		if (trace_bits)
		printf("top-of-stack-a %o, operand %o\n", v.word, operand);

		if (v.word == operand) {
			if (trace_bits)
			printf("take-values: correct # of args\n");
			popval(&v);
			break;
		}

		n = v.word - operand;
		stack_pointer--;

		if (trace_bits)
		printf("n %d\n", n);

		if (n >= 0) {
			stack_pointer -= n;
			read_amem_36(stack_pointer, &top_of_stack);
			break;
		}

	push_missing_values:
		pushval(&quote_nil);
		while (++n < 0) {
			pushval(&quote_nil);
		}

//		if (1) dump_stack_frame("take-values after");

		break;

	case 0152:	/* take-argument-rest */
		printf("take-argument-rest: "
		       "frame_argument_format %d, frame_number_of_args %d\n",
		       frame_argument_format(),
		       frame_number_of_args());

		a_temp = frame_pointer - 5;

// this is fixing something else which is broken...
stack_pointer = frame_pointer - 1;

		switch (frame_argument_format()) {
		case PERC_FRAME_ARGUMENTS_NORMAL:
			nargs = frame_number_of_args();
			goto take_rest_arg_1;
			break;
		case PERC_FRAME_ARGUMENTS_LEXPR:
			a_temp--;
			nargs = frame_number_of_args() - 1;
			goto take_rest_arg_lexpr_1;
			break;
		case PERC_FRAME_ARGUMENTS_INSTANCE:
			nargs = frame_number_of_args() + 2;
			//error-if unsigned-immediate < 2 function-is-not-a-method
			goto take_rest_arg_lexpr_1;
			break;
		case PERC_FRAME_ARGUMENTS_LEXPR_INSTANCE:
			a_temp--;
			nargs = frame_number_of_args() + 1;
			//error-if unsigned-immediate < 2 function-is-not-a-method
			goto take_rest_arg_lexpr_1;
			break;
		}

	take_rest_arg_1:
		/* # of args that go into the rest arg */
		b_temp = nargs - operand - 1;
		/* enough args to be embedded? */
		if (nargs > operand) {

			read_amem_36(frame_pointer - 6, &v);
			set_cdr(&v, CDR_NIL);
			write_amem_36(frame_pointer - 6, &v);

			v.word = a_temp - b_temp - 1;
			v.tag = DTP_LIST;
			pushval(&v);
		} else {
			pushval(&quote_nil);
		}

		break;

	take_rest_arg_lexpr_1:
		/* # of args that go into the rest arg */
		b_temp = nargs - operand - 1;
		/* enough args to be embedded? */
		if (nargs > operand) {

			/* return pointer into caller's copy of args */
			read_amem_36(frame_pointer - 7, &v);
			set_cdr(&v, CDR_NORMAL);
			write_amem_36(frame_pointer - 7, &v);

			v.word = a_temp - b_temp - 1;
			v.tag = DTP_LIST;
			pushval(&v);
		} else {
			/* exactly the desired number of spread args */
			read_amem_36(frame_pointer - 6, &v);
			pushval(&v);
		}

		break;

	case 0157:	/* long-branch-immed */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (trace_flow)
		printf("long-branch-immed: operand 0%o\n", operand);
		pc_add(v.word + operand /*+ 1*/);
		next_instruction = 0;
		break;

	case 0160:	/* branch */
		pc_add(sign_extend_7(operand));
		next_instruction = 0;
		break;
	case 0161:	/* branch-true */
		read_amem_36(stack_pointer, &v);
		if (trace_flow)
		printf("branch-true: %x:%o\n", v.tag, v.word);
		if (v.tag != DTP_NIL) {
			if (trace_flow)
			printf("branch-true: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;
	case 0162:	/* branch-false */
		read_amem_36(stack_pointer, &v);
		if (trace_flow)
		printf("branch-false: %x:%o\n", v.tag, v.word);
		if (v.tag == DTP_NIL) {
			if (trace_flow)
			printf("branch-false: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;
	case 0163:	/* branch-true-else-pop */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag != DTP_NIL) {
			if (trace_flow)
			printf("branch-true-else-pop: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		} else {
			popval(&v);
		}
		break;
	case 0164:	/* branch-false-else-pop */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag == DTP_NIL) {
			if (trace_flow)
			printf("branch-false-else-pop: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		} else {
			popval(&v);
		}
		break;
	case 0165:	/* branch-true-and-pop */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag != DTP_NIL) {
			if (trace_flow)
			printf("branch-true-and-pop: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
			popval(&v);
		}
		break;
	case 0166:	/* branch-false-and-pop */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag == DTP_NIL) {
			if (trace_flow)
			printf("branch-false-and-pop: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
			popval(&v);
		}
		break;

	case 0176:	/* branch-eq */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		if (equal_typed_pointer(&v, &top_of_stack)) {
			if (trace_flow)
			printf("branch-eq: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;
	case 0177:	/* branch-not-eq */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		if (!equal_typed_pointer(&v, &top_of_stack)) {
			if (trace_flow)
			printf("branch-not-eq: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;
	case 0201:	/* branch-not-atom */
		if (tag(v.tag) == DTP_LIST) {
			if (trace_flow)
				printf("branch-not-atom: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;

	case 0202:	/* branch-endp */
		v = top_of_stack;

		if (tag(v.tag) != DTP_LIST && tag(v.tag) != DTP_NIL) {
		}

		if (tag(v.tag) == DTP_LIST) {
			vma = top_of_stack.word;
			read_mem_36(vma, &v1);

			cdr_code = v1.tag >> 6;
			switch (cdr_code) {
			case CDR_NIL:
				break;
			case CDR_NORMAL:
				break;
			}
		}

		if (tag(v.tag) == DTP_NIL) {
			if (trace_flow)
				printf("branch-endp: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;

	case 0203:	/* branch-not-endp */
		v = top_of_stack;

		if (tag(v.tag) != DTP_LIST && tag(v.tag) != DTP_NIL) {
		}

		if (tag(v.tag) == DTP_LIST) {
			vma = top_of_stack.word;
			read_mem_36(vma, &v1);

			cdr_code = v1.tag >> 6;
			switch (cdr_code) {
			case CDR_NIL:
				break;
			case CDR_NORMAL:
				break;
			}
		}

		if (tag(v.tag) != DTP_NIL) {
			if (trace_flow)
				printf("branch-not-endp: take branch\n");
			pc_add(sign_extend_7(operand));
			next_instruction = 0;
		}
		popval(&v);
		break;

	case 0240:	/* add-local */
		//xxx float, overflow bignum creat
		local_operand(operand, &v);
		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			v.word += top_of_stack.word;
			break;
		}
		newtop(&v);
		break;
	case 0241:	/* add-immed */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			v.word += sign_extend_7(operand);
			break;
		}
		newtop(&v);
		break;
	case 0242:	/* sub-local */
		//xxx float, overflow bignum creat
		local_operand(operand, &v);
		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			v.word = top_of_stack.word - v.word;
			break;
		}
		newtop(&v);
		break;
	case 0244:	/* multiply-immed */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			v.word *= sign_extend_7(operand);
			break;
		}
		newtop(&v);
		break;

	case 0250:	/* increment-local */
		local_operand(operand, &v);
#if 0
		if (top_of_stack.tag != DTP_FIX ||
		    v.tag != DTP_FIX) {
			take_arithmetic_trap('increment 'local);
		}
#endif
		v.word++;
		v.tag = DTP_FIX;
		write_local_operand(operand, &v);
		break;

	case 0251:	/* decrement-local */
		local_operand(operand, &v);
#if 0
		if (top_of_stack.tag != DTP_FIX ||
		    v.tag != DTP_FIX) {
			take_arithmetic_trap('decrement 'local);
		}
#endif
		v.word--;
		v.tag = DTP_FIX;
		write_local_operand(operand, &v);
		break;

	case 0252:	/* set-cdr-local */
		//xxx this is all hooey
		local_operand(operand, &v);

		read_mem_36(vma, &v1);
		cdr_code = v1.tag >> 6;
		if (cdr_code == CDR_NIL) {
			v = quote_nil;
		} else {
			v.word++;
		}
		write_local_operand(operand, &v);
		break;

	case 0254:	/* %set-cdr-code-2 */
		//xxx this is all hooey
		popval(&v);
		popval(&v);
		break;

	case 0255:	/* push-car-local */
		local_operand(operand, &v);
		if (v.tag != DTP_LIST) {
			printf("push-car-local: not list!\n");
//			flag_enter_debugger = 1;
		}
		vma = a28(v.word);
		printf("push-car-local: vma %011o\n", vma);
		read_mem_36(vma, &v);
		pushval(&v);
		break;

	case 0256:	/* push-cdr-local */
		vma = local_operand_addr(operand);
		if (v.tag != DTP_LIST) {
			printf("push-cdr-local: not list!\n");
//			flag_enter_debugger = 1;
		}
		read_amem_36(vma+1, &v);
	vma = a28(v.word);
	read_mem_36(vma, &v);
		pushval(&v);
		break;

	case 0260:	/* ldb-immed */
	case 0261:	/* ldb-immed */
	case 0262:	/* ldb-immed */
	case 0263:	/* ldb-immed */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		/* form 10-bit-operand */
		operand |= (inst & 0x3) << 8;

		size = (operand >> 5) + 1;
		pos = 32 - (operand & 0x1f);
		if (pos == 32)
			pos = 0;

		if (trace_bits) {
			printf("ldb-immed: operand 0%o 0x%x, pos %d, size %d\n",
			       operand, operand, pos, size);

			printf("ldb-immed: top-of-stack-a %x:%x\n",
			       v.tag, v.word);
		}

		if (pos + size <= 32) {
			mask = 0xffffffff >> (32 - size);
			v.word = (v.word >> pos) & mask;
			if (trace_bits)
			printf("ldb-immed: mask %x, result %x\n", mask, v.word);
		} else {
			unsigned long long val, lmask;
			lmask = 0x00fffffffffLL >> (36 - size);
			val = ((unsigned long long)v.tag << 28) | v.word;
			val = (val >> pos) & lmask;
			v.word = (unsigned long)val;
			if (trace_bits)
			printf("ldb-immed: ll mask %llx result %llx\n", lmask, val);
		}

		set_type(&v, DTP_FIX);
		newtop(&v);
		break;

	case 0264:	/* dpb immed */
	case 0265:	/* dpb immed */
	case 0266:	/* dpb immed */
	case 0267:	/* dpb immed */
		/* next-on-stack */
		read_amem_36(stack_pointer - 1, &v);

		/* form 10-bit-operand */
		operand |= (inst & 0x3) << 8;

		size = (operand >> 5) + 1;
		pos = operand & 0x1f;

		if (trace_bits) {
			printf("dpb-immed: operand 0%o 0x%x, pos %d, size %d\n",
			       operand, operand, pos, size);

			printf("dpb-immed: next-on-stack %x:%x\n", v.tag, v.word);
			printf("dpb-immed: top-of-stack  %x:%x\n",
			       top_of_stack.tag, top_of_stack.word);
		}

		mask = 0xffffffff >> (32 - size);
		v.word = (top_of_stack.word & ~(mask << pos)) |
			((v.word & mask) << pos);

		if (trace_bits)
		printf("dpb-immed: mask %x, result %x\n", mask, v.word);

		set_type(&v, DTP_FIX);
		pop2push(&v);
		break;

	case 0271:	/* ar-1-local */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		/* check tag - dtp-array */
		if (trace_array)
		printf("tag %x %s\n", v.tag, tag_names[v.tag & 0x3f]);

		vma = v.word;
		read_mem_36(vma, &v);
		/* check tag - dtp-fix */
		if (trace_array)
		printf("tag %x %s\n", v.tag, tag_names[v.tag & 0x3f]);

		if (operand & 0x80)
			v.word = stack_pointer - sign_extend_7(operand) + 1;
		else
			v.word = frame_pointer + sign_extend_7(operand);

		v.word += 0x0ffff000;
		pushval(&v);

		goto ar_1_common;
//		if (1) dump_stack_frame("ar-1-local");
		break;

	case 0273:	/* as-1-local */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		vma = v.word;
		read_mem_36(vma, &v);

		if (operand & 0x80)
			v.word = stack_pointer - sign_extend_7(operand) + 1;
		else
			v.word = frame_pointer + sign_extend_7(operand);

		v.word += 0x0ffff000;
		pushval(&v);

		goto as_1_common;

	as_1_common:
		popval(&v);
		popval(&v);
		popval(&v);

		break;

	case 0274:	/* array-leader-immed */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		/* check tag - dtp-array */
		if (trace_array)
		printf("tag %x %s; 0%o 0x%x\n",
		       v.tag, tag_names[v.tag & 0x3f], v.word, v.word);

		vma = v.word;
		read_mem_36(vma, &v);
		/* check tag - dtp-fix */
		if (trace_array)
		printf("tag %x %s; 0%o 0x%x\n",
		       v.tag, tag_names[v.tag & 0x3f], v.word, v.word);

#if 0
		/* array-register-dispatch-field */
		read_amem_36(stack_pointer+1, &v);
		printf("array-register-dispatch-field %d\n", v.word);
		atype = v.word;

		/* init length to zero; assume no leader */
		v.word = 0;
		v.tag = DTP_FIX;
		write_amem_36(stack_pointer+3, &v);

		switch (atype) {
		default:
			break;
		case ARRAY_DISPATCH_LEADER:
			break;
		case ARRAY_DISPATCH_SHORT_INDIRECT:
			break;
		}
#else
		read_mem_36(vma+1, &v);
		newtop(&v);
#endif
		break;

	case 0300:	/* call-0-ignore */
		vdisp = 0;
		nargs = 0;
	call_common:
		/* prev previous-frame base pointer */
//		v.word = frame_function();
		v.word = frame_pointer;
		v.tag = DTP_LOCATIVE;
		pushval_with_cdr(&v);

		/* push previous-frame top pointer */
		/* cdr code is value disposition */
		v.word = stack_pointer - nargs - 1;
		v.tag = DTP_LOCATIVE;
		set_cdr(&v, vdisp);
		if (trace_call)
			printf("call_common vdisp %d\n", vdisp);
		pushval_with_cdr(&v);
	  
		/* return pc */
		v.word = (sim_pc + 1) >> 1;
		v.tag = (sim_pc + 1) & 1 ? DTP_ODD_PC : DTP_EVEN_PC;
		pushval_with_cdr(&v);

		/* misc data */
		v.word = nargs;
		v.tag = 0;
		pushval_with_cdr(&v);

		/* function */
		vma = frame_function() - operand - 1;
		if (trace_call)
			printf("vma 0%o 0x%x\n", vma, vma);
		read_mem_36(vma, &v);
		vma = v.word;
		if (trace_call)
			printf("[vma] 0%o 0x%x\n", vma, vma);
		read_mem_36(vma, &v);
		sim_pc = (a28(v.word) << 1) | 1;
		if (trace_call)
			printf("[[vma]] 0%o 0x%x\n", v.word, v.word);

		set_cdr(&v, 0);
		pushval_with_cdr(&v);

		{
			int table_size, arg_dispatch;

			read_mem_36(sim_pc >> 1, &v);
			table_size = v.word & 0xff;
			arg_dispatch = (v.word >> 8) & 0xf;

			if (nargs < arg_table[arg_dispatch].min ||
			    nargs > arg_table[arg_dispatch].max)
				printf("call-common: wrong number of args!");

			/* skip optional arg init code */
			if (arg_table[arg_dispatch].min &&
			    arg_table[arg_dispatch].max &&
			    nargs > arg_table[arg_dispatch].min)
			{
				printf("call-common: adjusting pc!");
				sim_pc += nargs - arg_table[arg_dispatch].min;
			}
		}

		/* */
		frame_pointer = stack_pointer + 1;
		a_pclsr_top_of_stack = stack_pointer + 1;

#if 1
		/* copy the arguments */
		for (i = 0; i < nargs; i++) {
			if (trace_call)
			printf("arg #%d, fp + %d: %o\n",
			       i+1, i - (nargs+5),
			       frame_pointer + i - (nargs + 5));
			read_amem_36(frame_pointer + i - (nargs + 5), &v);
			pushval(&v);
		}
#endif

		next_instruction = 0;

		if (trace_call_stack)
			dump_stack_frame("call-common");

		if (trace_call)
			show_compiled_function(sim_pc >> 1);
		else
			if (trace_enter_exit)
				find_func_name(sim_pc >> 1);

		break;
	case 0301:	/* call-0-stack */
		vdisp = 1;
		nargs = 0;
		goto call_common;
		break;
	case 0302:	/* call-0-return */
		vdisp = 2;
		nargs = 0;
		goto call_common;
		break;
	case 0303:	/* call-0-multiple */
		vdisp = 3;
		nargs = 0;
		goto call_common;
		break;
	case 0304:	/* call-1-ignore */
		vdisp = 0;
		nargs = 1;
		goto call_common;
		break;
	case 0305:	/* call-1-stack */
		vdisp = 1;
		nargs = 1;
		goto call_common;
		break;
	case 0306:	/* call-1-return */
		vdisp = 2;
		nargs = 1;
		goto call_common;
		break;

	case 0310:	/* call-2-ignore */
		vdisp = 0;
		nargs = 2;
		goto call_common;
		break;

	case 0311:	/* call-2-stack */
		vdisp = 1;
		nargs = 2;
		goto call_common;
		break;

	case 0314:	/* call-3-ignore */
		vdisp = 0;
		nargs = 3;
		goto call_common;
		break;

	case 0315:	/* call-3-stack */
		vdisp = 1;
		nargs = 3;
		goto call_common;
		break;

	case 0320:	/* call-4-ignore */
		vdisp = 0;
		nargs = 4;
		goto call_common;
		break;

	case 0326:	/* call-n-return */
		popval(&v);
		vdisp = 2;
		nargs = v.word;
		goto call_common;
		break;

	case 0336:	/* lexpr-funcall-return */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		a_pclsr_top_of_stack = v.word;

		top_of_stack.tag = 0;
		top_of_stack.word = operand + 1;

		v.word = stack_pointer - top_of_stack.word - 1;
		if (trace_call)
		printf("lexpr-funcall-return: %x\n", v.word);

		set_cdr(&v, 2); /* return */
		set_type(&v, DTP_LOCATIVE);
		write_amem_36(stack_pointer + 2, &v);

		stack_pointer++;

		/* check if rest arg is nil */
		read_amem_36(stack_pointer -1, &v);
		if (v.tag == DTP_NIL) {
			if (trace_call)
			printf("lexpr-funcall-return: rest arg nil\n");
			read_amem_36(stack_pointer + 1, &v);
			write_amem_36(stack_pointer, &v);
			top_of_stack.word--;
			stack_pointer--;
			/* funcal-stack-n */
		}

		/* push previous-frame base pointer */
		v.word = frame_pointer;
		v.tag = DTP_LOCATIVE;
		pushval_with_cdr(&v);

		/* return pc */
		v.word = (sim_pc + 1) >> 1;
		v.tag = (sim_pc + 1) & 1 ? DTP_ODD_PC : DTP_EVEN_PC;
		pushval_with_cdr(&v);

		/* misc data */
		v.word = frame_funccalled() + frame_lexpr_called() + top_of_stack.word;
		v.tag = DTP_FIX;
		pushval_with_cdr(&v);

		/* function */
		vma = frame_function() - operand - 1;
		read_mem_36(vma, &v);
		vma = v.word;
		read_mem_36(vma, &v);
		sim_pc = (a28(v.word) << 1) | 1;

		set_cdr(&v, 0);
		pushval_with_cdr(&v);


		/* */
		frame_pointer = stack_pointer + 1;
		a_pclsr_top_of_stack = stack_pointer + 1;

		next_instruction = 0;

		if (trace_call) {
			dump_stack_frame("lexpr-funcall-return");
		}

		if (trace_call || trace_enter_exit)
			show_compiled_function(sim_pc >> 1);

		break;

	case 0371:	/* return-n */
		nargs = operand;

		if (frame_misc_data() &
		    FRAME_CLEANUP_BITS)
//		    (FRAME_BUFFER_OVERFLOW | FRAME_CLEANUP_BITS))
		{
		}

		if (trace_call)
			printf("frame-misc-data %x\n", frame_misc_data());

		/* frame-previous-pc */
		read_amem_36(frame_pointer - 3, &v);

		/* check arg-type return-pc */

	  	sim_pc = a28(v.word) << 1;
		if (v.tag == DTP_ODD_PC)
			sim_pc++;

		if (trace_call || trace_enter_exit)
		printf("return-n: return pc %x:%o\n", v.tag, v.word);

		/* frame-previous-top */
		read_amem_36(frame_pointer - 4, &v);
		cdr_code = v.tag >> 6;

		if (trace_call)
		printf("return-n: cdr_code %d\n", cdr_code);

		switch (cdr_code) {
		case CDR_NEXT:    /* effect (ignore) */
			/* frame-previous-top */
			read_amem_36(frame_pointer - 4, &v);
			stack_pointer = v.word;

			/* frame-previous-frame */
			read_amem_36(frame_pointer - 5, &v);
			frame_pointer = v.word;

			read_amem_36(stack_pointer, &top_of_stack);
			break;
		case CDR_NIL:    /* value (stack) */
			if (nargs == 0)
				top_of_stack = quote_nil;
			else {
				stack_pointer -= nargs;
				read_amem_36(stack_pointer+1, &top_of_stack);
			}

			/* frame-previous-top */
			read_amem_36(frame_pointer - 4, &v);
			stack_pointer = v.word;

			/* frame-previous-frame */
			read_amem_36(frame_pointer - 5, &v);
			frame_pointer = v.word;

			pushval(&top_of_stack);
			break;

		case CDR_NORMAL: /* return */
//			pushval(&top_of_stack);
//			goto return_stack;
			/* blt-values-down */
			blt_pointer = stack_pointer - nargs;
			old_stack_pointer = stack_pointer;

			/* frame-previous-top */
			read_amem_36(frame_pointer - 4, &v);
			stack_pointer = v.word;
			
			/* frame-previous-frame */
			read_amem_36(frame_pointer - 5, &v);
			frame_pointer = v.word;

			if (trace_call) {
			printf("return: stack_pointer %o\n",
			       stack_pointer);
			printf("return: frame_pointer %o\n",
			       frame_pointer);
			printf("return: blt_pointer %o\n",
			       blt_pointer);
			}

			/* blt-stack */
			blt_pointer++;
			while (blt_pointer <= old_stack_pointer) {
				read_amem_36(blt_pointer, &v);
				pushval_with_cdr(&v);
				blt_pointer++;
			}

			break;

		case CDR_SPARE:  /* multiple-values */
			/* blt-values-down */
			blt_pointer = stack_pointer - nargs;
			old_stack_pointer = stack_pointer;

			/* frame-previous-top */
			read_amem_36(frame_pointer - 4, &v);
			stack_pointer = v.word;
			
			/* frame-previous-frame */
			read_amem_36(frame_pointer - 5, &v);
			frame_pointer = v.word;

			if (trace_call) {
			printf("multiple-values: stack_pointer %o\n",
			       stack_pointer);
			printf("multiple-values: frame_pointer %o\n",
			       frame_pointer);
			printf("multiple-values: blt_pointer %o\n",
			       blt_pointer);
			}

			/* blt-stack */
			blt_pointer++;
			while (blt_pointer <= old_stack_pointer) {
				read_amem_36(blt_pointer, &v);
				pushval_with_cdr(&v);
				blt_pointer++;
			}

			v.word = nargs;
			v.tag = DTP_FIX;
			pushval(&v);
			break;
		}

		next_instruction = 0;
		break;

	case 0375:	/* %dispatch-elt */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		if (trace_flow)
		printf("%%dispatch-elt: offset from stack 0%o, base 0%o\n",
		       v.word, top_of_stack.word);
		v.word += top_of_stack.word;
		read_mem_36(v.word, &v);
		newtop(&v);
		break;

	case 01000:	/* %halt */
		sim_running = 0;
		break;

	case 01003:	/* %pointer */
		top_of_stack.tag = DTP_FIX;
		top_of_stack.word = a28(top_of_stack.word);

		newtop(&top_of_stack);
		break;

	case 01010:	/* %p-store-contents */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);

		popval(&v1);

		vma = v.word;
		read_mem_36(vma, &v);

		v.tag = (v.tag & 0x3f) | (top_of_stack.tag & 0xc0);
		v.word = v1.word;

		stack_pointer--;
		read_amem_36(stack_pointer, &top_of_stack);

		if (trace_bits)
		printf("%%p-store-contents: vma 0%011o (%x) <- %3o:%011o\n",
		       vma, vma, v.tag, v.word);
		write_mem_36(vma, &v);
		break;

	case 01013:	/* %p-structure-offset */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		vma = a28(v.word);

		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		v.word += vma;
		v.tag = DTP_LOCATIVE;

		pop2push(&v);
		break;

	case 01014:	/* %microsecond-clock */
		v.word = 0;
		v.tag = DTP_FIX;
		pushval(&v);
		break;

	case 01022:	/* %block-store-cdr-and-contents */
//		if (1) dump_stack_frame("%block-store-cdr-and-contents");
	{
		int a, b, c, addr;

		read_amem_36(stack_pointer-2, &v);
		a = v.word;

		read_amem_36(stack_pointer-1, &v);
		b = v.word;

		read_amem_36(stack_pointer-3, &v);
		c = v.word;

		read_amem_36(stack_pointer-4, &v);
		addr = v.word;

//		if (trace_bits)
		printf("%%block-store-cdr-and-contents: "
		       "a %x, b %x, c %x, addr %x\n",a, b, c, addr);
	}

		break;

	case 01030:	/* %map-cache-write */
		/* contents */
		popval(&v);
		/* virtual address */
		popval(&v1);
		if (trace_hw)
		printf("%%map-cache-write: %011o <- %011o\n", v1.word, v.word);
		break;

	case 01032:	/* %phtc-write */
		/* contents */
		popval(&v);
		/* virtual address */
		popval(&v1);
		if (trace_hw)
		printf("%%phtc-write: %011o <- %011o\n", v1.word, v.word);
		break;

	case 01033:	/* %phtc-setup */
		perc_current_phtc = top_of_stack.word;
		stack_pointer--;
		break;

	case 01042:	/* %gc-map-write */
		/* contents */
		popval(&v);
		/* virtual address */
		popval(&v1);
		if (trace_hw)
		printf("%%gc-map-write: %011o <- %011o\n", v1.word, v.word);
		break;

	case 01100:	/* car */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag == DTP_NIL) {
			newtop(&quote_nil);
			break;
		}

		vma = a28(v.word);
		read_mem_36(vma, &v);
		newtop(&v);
		break;

	case 01101:	/* cdr */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		if (v.tag == DTP_NIL) {
			newtop(&quote_nil);
			break;
		}

		vma = v.word;
		read_amem_36(vma, &v1);
		if (v.tag == DTP_LOCATIVE) {
			newtop(&v1);
			break;
		}

		cdr_code = v1.tag >> 6;
//		if (trace_bits)
			printf("cdr: vma %o, cdr_code %d\n", vma, cdr_code);

		vma++;
		switch (cdr_code) {
		case CDR_NIL:
			newtop(&quote_nil);
			break;
		case CDR_NEXT:
			v.word++;
			newtop(&v);
			break;
		case CDR_NORMAL:
			read_mem_36(vma, &v1);
			newtop(&v1);
			break;
		default:
			//signal-error bad-cdr-code
			break;
		}
		break;

	case 01102:	/* rplaca */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		vma = a28(v.word);

	rplaca1:
		read_mem_36(vma, &v1);
		popval(&v);
		popval(&v2);

		/* merge new and old cdr code */
		v.tag = (v.tag & 0x3f) | (v1.tag & 0xc0);

		if (trace_bits)
		printf("rplaca: [%011o] <- %3o:%011o\n", vma, v.tag, v.word);

		write_mem_36(vma, &v);
		break;

	case 01103:	/* rplacd */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		vma = a28(v.word);

		if (v.tag == DTP_LOCATIVE) {
			goto rplaca1;
		}

		popval(&v);

		/* next-on-stack */
		popval(&v1);

		if (trace_bits)
		printf("rplacd: [%011o] <- %3o:%011o\n", vma, v.tag, v.word);

		cdr_code = v1.tag >> 6;
		if (trace_bits)
		printf("rplacd: cdr code %d\n", cdr_code);

		if (cdr_code == CDR_NORMAL) {
			if (trace_bits)
			printf("rplacd: [%011o]\n", vma+1);
			write_mem_36(vma+1, &v);
			break;
		}

		if (v.tag != DTP_NIL) {
			//take-post-trap rplacd-escape restore-stack
		}

		read_mem_36(vma, &v);
		set_cdr(&v, DTP_NIL);
		write_mem_36(vma, &v);
		break;

	case 01120:	/* push-nil */
		pushval(&quote_nil);
		break;

	case 01202:	/* not */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag == DTP_NIL)
			newtop(&quote_t);
		else
			newtop(&quote_nil);
		break;

	case 01203:	/* zerop */
		if (trace_comp)
		printf("zerop: tag %x\n", top_of_stack.tag);
		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			if (zero_fixnum(&top_of_stack))
				newtop(&quote_t);
			else
				newtop(&quote_nil);
			break;
		}
		break;

	case 01204:	/* plusp */
		if (trace_comp)
		printf("plusp: tag %x\n", top_of_stack.tag);
		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			if (plus_fixnum(&top_of_stack))
				newtop(&quote_t);
			else
				newtop(&quote_nil);
			break;
		}
		break;

	case 01205:	/* minusp */
		if (trace_comp)
		printf("minusp: tag %x\n", top_of_stack.tag);
		switch (top_of_stack.tag) {
		default:
		case DTP_FIX:
			if (minus_fixnum(&top_of_stack))
				newtop(&quote_t);
			else
				newtop(&quote_nil);
			break;
		}
		break;
	case 01206:	/* lessp */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		if (lesser_fixnum(&v, &top_of_stack))
			newtop(&quote_t);
		else
			newtop(&quote_nil);
		break;
	case 01207:	/* greaterp */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		if (greater_fixnum(&v, &top_of_stack))
			newtop(&quote_t);
		else
			newtop(&quote_nil);
		break;
	case 01210:	/* equal-number */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		if (equal_fixnum(&v, &top_of_stack))
			newtop(&quote_t);
		else
			newtop(&quote_nil);
		break;

	case 01211:	/* atom */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag == DTP_LIST)
			newtop(&quote_nil);
		else
			newtop(&quote_t);
		break;

	case 01214:	/* numberp */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag == DTP_FIX ||
		    v.tag == DTP_FLOAT ||
		    v.tag == DTP_EXTENDED_NUMBER)
			newtop(&quote_t);
		else
			newtop(&quote_nil);
		break;

	case 01215:	/* symbolp */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag == DTP_SYMBOL || v.tag == DTP_NIL)
			newtop(&quote_t);
		else
			newtop(&quote_nil);
		break;

	case 01216:	/* arrayp */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag == DTP_ARRAY)
			newtop(&quote_t);
		else
			newtop(&quote_nil);
		break;

	case 01225:	/* array-length */
		addr = top_of_stack.word;
		read_mem_36(addr, &v);

		switch (ARRAY_DISPATCH_FIELD(v.word)) {
		default:
			v.word = ARRAY_NORMAL_LENGTH_FIELD(v.word);
			v.tag = DTP_FIX;
			newtop(&v);
			break;
		case ARRAY_DISPATCH_LONG:
			break;
		}
		break;

	case 01227:	/* stringp */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);
		if (v.tag == DTP_HEADER_I && _ldb(v.word, 4, 18) == ART_STRING)
			newtop(&quote_t);
		else
			newtop(&quote_nil);
//xxx
set_trace(2);
flag_enter_debugger = 1;
		break;

	case 01230:	/* push-2-nils */
		pushval(&quote_nil);
		pushval(&quote_nil);
		break;

	case 01231:	/* push-t */
		pushval(&quote_t);
		break;

	case 01236:	/* member-fast */
		//xxx this is all hooey
		popval(&v);
		popval(&v1);
		pushval(&quote_t);
		break;

	case 01240:	/* add-stack */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		v.word += top_of_stack.word;
		set_type(&v, DTP_FIX);
		pop2push(&v);
		break;

	case 01241:	/* sub-stack */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		v.word -= top_of_stack.word;
		set_type(&v, DTP_FIX);
		pop2push(&v);
		break;

	case 01243:	/* logand-stack */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		v.word &= top_of_stack.word;
		set_type(&v, DTP_FIX);
		pop2push(&v);
		break;

	case 01244:	/* logior-stack */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		v.word |= top_of_stack.word;
		set_type(&v, DTP_FIX);
		pop2push(&v);
		break;

	case 01245:	/* logxor-stack */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		v.word ^= top_of_stack.word;
		set_type(&v, DTP_FIX);
		pop2push(&v);
		break;

	case 01246:	/* multiply-stack */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		v.word *= top_of_stack.word;
		set_type(&v, DTP_FIX);
		pop2push(&v);
		// overflow?
		break;

	case 01265:	/* rational-quotient-stack */
		/* next-on-stack - dividend */
		read_amem_36(stack_pointer-1, &v);
		if (trace_bits)
		printf("rational-quotient-stack: dividend %d, divisor %d\n",
		       v.word, top_of_stack.word);
		if (top_of_stack.word != 0)
			v.word /= top_of_stack.word;
		else
			v.word = 0;
		stack_pointer--;
		newtop(&v);
		break;

	case 01270:	/* ar-1 */
//		if (1) dump_stack_frame("ar-1");

	ar_1_common:
		read_amem_36(stack_pointer - 1, &v);

		if (trace_array) {
//		printf("tag %x, word %x\n", v.tag, v.word);
			show_tagged("ptr from stack", &v);
			show_array(v.word);
		}

		/* hack */
		if (v.tag != DTP_ARRAY) {
			if (trace_array)
			printf("ar-1: hack!\n");
			v.tag = 0;
			v.word = 0;
			{ static int times = 0;
			if (++times == 30) {
				if (trace_array)
				printf("ar-1: hack! returns 2!\n");
				v.word = 0x2;
				times = 0;
			}
			}
			pop2push(&v);
			break;
		}

//		read_mem_36(v.word+4, &v);
////		printf("ar-1: array %x:%x %o\n", v.tag, v.word, v.word);
//		show_tagged("array", &v);
//		if (v.tag != DTP_ARRAY) {
//		}

		addr = v.word;

		read_mem_36(addr, &v);
//		printf("ar-1: array %x:%x %o\n", v.tag, v.word, v.word);
		if (trace_array)
		show_tagged("hdr", &v);

		read_amem_36(stack_pointer, &v1);
		if (trace_array)
		printf("sub addr %o\n", v1.word);

		read_mem_36(v1.word, &v1);
//		read_mem_36(v1.word, &v1);
		if (trace_array)
		printf("ar-1: subscript %x:%o\n", v1.tag, v1.word);
		if (v1.tag != DTP_FIX) {
		}

		alen = ARRAY_NORMAL_LENGTH_FIELD(v.word);
		adisp = ARRAY_DISPATCH_FIELD(v.word);
		atype = ARRAY_TYPE_FIELD(v.word);

		if (trace_array) {
			printf("ARRAY_NORMAL_LENGTH_FIELD %d 0x%x\n",
			       alen, alen);
			printf("ARRAY_DISPATCH_FIELD %d\n", adisp);
			printf("ARRAY_TYPE_FIELD %d\n", atype);

			if (atype >= 0 && atype <= 16)
				printf("%s %s\n",
				       array_dispatch_names[adisp],
				       array_type_names[atype]);
		}

		switch (adisp) {
		case ARRAY_DISPATCH_WORD:
			if (trace_array) {
				printf("ARRAY_DISPATCH_word:\n");	
				show_tagged_loc(addr + 1 + v1.word,
						"[addr+sub]", 0);
			}
			read_mem_36(addr + 1 + v1.word, &v);
			pop2push(&v);
			break;
		case ARRAY_DISPATCH_LONG:
			if (trace_array) {
				printf("ARRAY_DISPATCH_LONG:\n");
				printf("offset to leader %d, "
				       "# elems in leader %d, # dims %d\n",
			       ARRAY_LONG_PREFIX_LENGTH_FIELD(v.word),
			       ARRAY_LONG_LEADER_LENGTH_FIELD(v.word),
			       ARRAY_DIMENSIONS_FIELD(v.word));
			}

			/* read locative */
			read_mem_36(addr+1, &v);
			if (v.tag != DTP_LOCATIVE) {
			}

			if (trace_array)
				show_tagged("locative", &v);

			addr = v.word;

			if (trace_array) {
				show_tagged_loc(addr+0, "+0", 0);
				show_tagged_loc(addr+1, "+1", 0);
			}

			pop2push(&v);
			break;
		default:
			break;
		}

		break;

	case 01300:	/* funcall-0-ignore */
		vdisp = 0;
		nargs = 0;
	funcall_common:
//		popval(&v);
//		v = top_of_stack;
		read_amem_36(stack_pointer - nargs, &v);
		addr = v.word;
		if (trace_call)
			printf("funcall-0-ignore: 0x%x %o\n", addr, addr);
		
		/* prev previous-frame base pointer */
//		v.word = frame_function();
		v.word = frame_pointer;
		v.tag = DTP_LOCATIVE;
		pushval_with_cdr(&v);

		/* push previous-frame top pointer */
		/* cdr code is value disposition */
		v.word = stack_pointer - nargs - 2/*1*/;
		v.tag = DTP_LOCATIVE;
		set_cdr(&v, vdisp);
		pushval_with_cdr(&v);
	  
		/* return pc */
		v.word = (sim_pc + 1) >> 1;
		v.tag = (sim_pc + 1) & 1 ? DTP_ODD_PC : DTP_EVEN_PC;
		pushval_with_cdr(&v);

		/* misc data */
		v.word = nargs | FRAME_FUNCALLED;
		v.tag = DTP_FIX;
		pushval_with_cdr(&v);

		/* function */
//		vma = frame_function() - operand - 1;
		v.word = addr;
		v.tag = DTP_ODD_PC;
		if (trace_call)
			printf("funcall-0-ignore: function from stack %x\n",
			       addr);
//		read_mem_36(vma, &v);
//		vma = v.word;
//		read_mem_36(vma, &v);
		sim_pc = (a28(v.word) << 1) | 1;

		set_cdr(&v, 0);
		pushval_with_cdr(&v);

		/* */
		frame_pointer = stack_pointer + 1;
		a_pclsr_top_of_stack = stack_pointer + 1;

#if 1
		/* copy the arguments */
		for (i = 0; i < nargs; i++) {
			if (trace_call)
			printf("arg #%d, fp + %d: %o\n",
			       i+1, i - (nargs+5),
			       frame_pointer + i - (nargs + 5));
			read_amem_36(frame_pointer + i - (nargs + 5), &v);
			pushval(&v);
		}
#endif

		next_instruction = 0;

		if (trace_call) {
			dump_stack_frame("funcall-0-ignore");
		}

		if (trace_call || trace_enter_exit)
			show_compiled_function(sim_pc >> 1);

		break;

	case 01304:	/* funcall-1-ignore */
		vdisp = 0;
		nargs = 1;
		goto funcall_common;
		break;
	case 01311:	/*funcall-2-stack */
		vdisp = 1;
		nargs = 2;
		goto funcall_common;
		break;

	case 01360:	/* restart-trapped-call */
		popval(&v);
		vma = a28(v.word);
		read_mem_36(vma, &v);

		if (v.tag == DTP_HEADER_P) {
			/* SYMBOL-FUNCTION-CELL */
			show_tagged_loc(vma+2, "SYMBOL-FUNCTION-CELL", 0);

			read_mem_36(vma + 2, &v);

			if (v.tag == DTP_ONE_Q_FORWARD)
				;
			vma = a28(v.word);
			read_mem_36(vma, &v);

			show_tagged_loc(vma, "[SYMBOL-FUNCTION-CELL]", 0);

			if (v.tag == DTP_COMPILED_FUNCTION)
				;

			pushval(&v);
		}

		sim_pc = (a28(v.word) << 1) | 1;
		
		printf("restart-trapped-call: new pc %o\n", sim_pc >> 1);

		show_compiled_function(sim_pc >> 1);

		next_instruction = 0;
		break;

	case 01364:	/* %resume-main-stack-buffer */
		flag_enter_debugger = 1;

		read_amem_36(PERC_OTHER_PC, &v);
		sim_pc = a28(v.word) << 1;
		if (v.tag == DTP_ODD_PC)
			sim_pc++;

stack_low = 0;

		read_amem_36(PERC_OTHER_FRAME_POINTER, &v);
		frame_pointer = v.word & 0x0fff;
frame_pointer = v.word & 0x00ff;

		read_amem_36(PERC_OTHER_STACK_POINTER, &v);
		stack_pointer = v.word & 0x0fff;
stack_pointer = v.word & 0x00ff;

		read_amem_36(stack_pointer, &top_of_stack);
		next_instruction = 0;

		printf("%%resume-main-stack-buffer: sp %o, fp %o, pc %o\n",
		       stack_pointer, frame_pointer, sim_pc >> 1);

		break;

	case 01370:	/* return-stack */
	return_stack:
//		if (1) dump_stack_frame("return-pc");

		/* frame-previous-pc */
		read_amem_36(frame_pointer - 3, &v);

	  	sim_pc = a28(v.word) << 1;
		if (v.tag == DTP_ODD_PC)
			sim_pc++;

		if (trace_call || trace_enter_exit)
		printf("return-stack: return pc %x:%o\n", v.tag, v.word);

		/* frame-previous-top */
		read_amem_36(frame_pointer - 4, &v);
		stack_pointer = v.word;

		cdr_code = v.tag >> 6;

		/* frame-previous-frame */
		read_amem_36(frame_pointer - 5, &v);
		frame_pointer = v.word;

		if (trace_call)
		printf("return-stack: cdr_code %d\n", cdr_code);

		switch (cdr_code) {
		case CDR_NEXT:    /* effect */
		  read_amem_36(stack_pointer, &top_of_stack);
		  break;
		case CDR_NIL:    /* value */
		  pushval(&top_of_stack);
		  break;
		case CDR_NORMAL: /* return */
		  pushval(&top_of_stack);
		  goto return_stack;
		  break;
		case CDR_SPARE:  /* multiple-values */
		  pushval(&top_of_stack);
		  v.word = 1;
		  v.tag = DTP_FIX;
		  pushval(&v);
		  break;
		}

		next_instruction = 0;
		break;

	case 01372:	/* popj */
		/* top-of-stack-a */
		read_amem_36(stack_pointer, &v);

		if (trace_flow)
		printf("popj: %x:%o\n", v.tag, v.word);

		if ((v.tag != DTP_EVEN_PC &&
		     v.tag != DTP_ODD_PC) ||
		    (top_of_stack.tag != DTP_EVEN_PC &&
		     top_of_stack.tag != DTP_ODD_PC))
		{
		}

	  	sim_pc = a28(v.word) << 1;
		if (v.tag == DTP_ODD_PC)
			sim_pc++;

		popval(&v);
		next_instruction = 0;
		break;

	case 01374:	/* return-nil */
		goto return_stack;
		break;

	case 01375:	/* location-boundp */
		read_amem_36(stack_pointer, &v);

		if (trace_bits)
		printf("location-boundp: [0x%x] -> %x:%o\n",
		       stack_pointer, v.tag, v.word);

		addr = v.word;
		read_amem_36(addr, &v);

		if (trace_bits)
		printf("location-boundp: %o -> %x:%o\n",
		       addr, v.tag, v.word);

		if (v.tag == DTP_NULL) {
			newtop(&quote_nil);
		} else {
			newtop(&quote_t);
		}
		break;

	case 01440:	/* %32-bit-plus */
		/* a guess */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		v.word += top_of_stack.word;
		newtop(&v);
		break;

	case 01441:	/* %32-bit-difference */
		/* a guess */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		v.word -= top_of_stack.word;
		newtop(&v);
		break;

	case 01451:	/* floor-stack */
		/* a guess */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		v.word /= top_of_stack.word;
		newtop(&v);
		break;

	case 01453:	/* ceiling-stack */
		/* a guess */
		/* next-on-stack */
		read_amem_36(stack_pointer-1, &v);
		stack_pointer--;
		v.word = (v.word + (v.word-1)) / top_of_stack.word;
		newtop(&v);
		break;

	case 01454:	/* round-stack */
		/* a guess */
		break;

	default:
		printf("UNIMPLEMENTED INSTRUCTION!\n");
		disassemble36(sim_pc >> 1, sim_pc & 1, inst);
		break;
	}

	if (next_instruction) {
		sim_pc++;
	}

	return 0;
}

int
set_fep_variable(char *name, w36 *v)
{
	int i;

	for (i = 0; fep_communication_area_names[i]; i++) {
		if (strcasecmp(name, fep_communication_area_names[i]) == 0) {
			printf("set_fep_variable: %s [0x%x] 0%o\n",
			       fep_communication_area_names[i],
			       i,
			       v->word);
			write_mem_36(i, v);
			return 0;
		}
	}
	return -1;
}

int
setup_fep_lowmem(void)
{
	w36 v;

//	v.tag = DTP_FIX;
//	v.word = 1;
//	set_fep_variable("*console-slot*", &v);

	v.tag = DTP_FIX;
	v.word = 0;
	set_fep_variable("swap-map-size", &v);

	v.tag = DTP_FIX;
	v.word = 0;
	set_fep_variable("screen-data-offset", &v);

	v.tag = DTP_FIX;
	v.word = 1024;
	set_fep_variable("screen-locations-per-line", &v);

	v.tag = DTP_FIX;
	v.word = 1024;
	set_fep_variable("screen-raster-height", &v);

	v.tag = DTP_FIX;
	v.word = 0;
	set_fep_variable("screen-top-fake-lines", &v);

	v.tag = DTP_FIX;
	v.word = 0;
	set_fep_variable("screen-bottom-fake-lines", &v);

	v.tag = DTP_FIX;
	v.word = 0;
	set_fep_variable("screen-top-margin-lines", &v);

	v.tag = DTP_FIX;
	v.word = 0;
	set_fep_variable("screen-bottom-margin-lines", &v);

	v.tag = DTP_FIX;
	v.word = 0;
	set_fep_variable("screen-raster-width", &v);

	v.tag = DTP_FIX;
	v.word = 0;
	set_fep_variable("screen-left-margin-pixels", &v);

	v.tag = DTP_FIX;
	v.word = 0;
	set_fep_variable("screen-right-margin-pixels", &v);
}

void
set_trace(int mode) {
	switch (mode) {
	case 0:
		trace_top = 0;
		trace_stack_small = 0;
		trace_isn_disasm = 0;
		trace_push_pop = 0;
		trace_args = 0;
		trace_comp = 0;
		trace_bits = 0;
		trace_flow = 0;
		trace_array = 0;
		trace_call = 0;
		trace_call_stack = 0;
		trace_enter_exit = 0;
		trace_hw = 0;
		trace_func_header = 0;
		break;
	case 1:
		trace_top = 0;
		trace_stack_small = 0;
		trace_isn_disasm = 0;
		trace_push_pop = 0;
		trace_args = 0;
		trace_comp = 0;
		trace_bits = 0;
		trace_flow = 0;
		trace_array = 0;
		trace_call = 0;
		trace_call_stack = 0;
		trace_enter_exit = 1;
		trace_hw = 0;
		break;
	case 2:
	default:
		trace_top = 1;
		trace_stack_small = 1;
		trace_isn_disasm = 1;
		trace_push_pop = 1;
		trace_args = 1;
		trace_comp = 1;
		trace_bits = 1;
		trace_flow = 1;
		trace_array = 1;
		trace_call = 1;
		trace_call_stack = 1;
		trace_enter_exit = 0;
		trace_hw = 1;
		trace_func_header = 1;
//trace_mem = 1;
		break;
	}
}

int
start_sim(void *stream)
{
	int isn;
	w36 v;

	printf("l-machine macroinstruction simulator\n\n");

	quote_nil.tag = DTP_NIL;
	quote_nil.word = 0;

	quote_t.tag = DTP_SYMBOL;
	quote_t.word = 0525252;

	sim_pc = get_amem_sym_value("FUNCTION-SYSTEM-STARTUP") << 1;
	sim_pc |= 1;

	printf("FUNCTION-SYSTEM-STARTUP %o\n", a28(sim_pc >> 1));

	/* should be running on aux stack */
//	stack_pointer = 04000;
	stack_pointer = 07000;
	stack_low = stack_pointer;
	stack_limit = 0;

	v.word = 0x0ffff800;
	v.tag = 0;
	write_amem_36(0x402, &v);


/* */
//	set_type(&v, DTP_LOCATIVE);
	v.word = 0;
	pushval(&v); /* prev frame */

	v.word = stack_pointer - 2;
	pushval(&v); /* prev top */

	v.word = 0;
	pushval(&v); /* return pc */

	v.word = 0;
	pushval(&v); /* frame-misc-data */

	v.word = sim_pc >> 1;
	pushval(&v); /* dtp-compiled-function */
	
	frame_pointer = stack_pointer + 1;
/* */

	sim_running = 1;
	world_stream = stream;

	setup_fep_lowmem();

//	find_func_name(sim_pc >> 1);
	show_compiled_function(sim_pc >> 1);

	if (flag_trace == 0)
		set_trace(1);

	isn = 0;

	while (sim_running) {
		sim_cycles++;

		fetch_isn(sim_pc, &isn);

		if (trace_top && !flag_enter_debugger) {
			printf("\ntop; pc %o, isn %o %x, sp %o, fp %o\n",
			       sim_pc >> 1, isn, isn,
			       stack_pointer, frame_pointer);
		}

		if (flag_enter_debugger) {
			if (!trace_stack_small)
				dump_stack_small();

			if (!trace_isn_disasm)
				disassemble36(sim_pc >> 1, sim_pc & 1, isn);

			enter_debugger(stream);
		}
		
#if 0
//#define BP 057166
#define BP 0500215623
		if (a28(sim_pc >> 1) == BP) {
			set_trace(2);
			flag_enter_debugger = 1;
		}
#endif

		exec_isn(sim_pc, isn);

		if (flag_max_cycles && sim_cycles > flag_max_cycles)
			break;
	}

	printf("simulation halted\n");
}
