; -*- Mode:Lisp; Package:Cold; Lowercase:T; Base:8 -*-
;	** (c) Copyright 1980 Massachusetts Institute of Technology **

;Loader of QFASL files into cold-loads

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; To compile this:			       ;;;
;;;   (1) Load COLDUT QFASL		       ;;;
;;;   (2) Run (LOAD-PARAMETERS)		       ;;;
;;;   (3) Now you may compile it	       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(declare (special evals-to-be-sent-over
		  last-fasl-eval  ;the element of evals-to-be-sent-over created by the last 
				  ; fasl-op-eval.
		  cold-list-area
		  fef-debugging-info-alist
		  current-function))		;debugging aid

(declare (special fasl-table fasl-table-fill-pointer fasl-return-flag 
		  fasl-group-bits fasl-group-type fasl-group-length fasl-group-flag
		  q-fasl-group-dispatch m-fasl-group-dispatch fasl-group-dispatch-size
		  file-property-list cold-loaded-file-property-lists))

(declare (special qfasl-binary-file fdefine-file-pathname))

;Q-FASL-xxxx refers to functions which load into the cold load, and
; return a "Q", i.e. a list of data-type and address-expression.
;M-FASL-xxxx refers to functions which load into the current Lisp environment
; (M used to be for Maclisp), and return a Lisp object.
; However M-objects still have all their symbols in the SYM package.
;In the FASL-TABLE, each entry in both the prefix and the main part
; is a list whose car is the M (lisp) value and whose cadr is either
; NIL or the Q-value.  If it needs a Q-value and one hasn't been
; computed yet, it will compute one, but this may put it in the wrong area.

;These functions are used to refer to the FASL-TABLE

;Get a Q object from FASL table
(defun q-arft (x)
  (cond ((atom (setq x (aref fasl-table x)))
	 (ferror nil "not a q - q-arft"))
	((cadr x))
	(t (rplaca (cdr x) (make-q-list 'sym:init-list-area (car x)))
	   (cadr x))))

;Get an M object
(defun m-arft (x)
  (cond ((atom (setq x (aref fasl-table x)))
	 (ferror nil "not a q - m-arft"))
	(t (car x))))

;Store an M object
(defsubst m-asft (d x)
  (aset (list d nil) fasl-table x))

;Store both M and Q objects
(defsubst m-q-asft (d q x)
  (aset (list d q) fasl-table x))

;(DEFPROP USER/:SOURCE-FILE-NAME (USER SOURCE-FILE-NAME) PACKAGE-PATH)

(defun cold-fasload (filespec &aux qfasl-binary-file fdefine-file-pathname file-property-list)
  (or (boundp 'q-fasl-group-dispatch) (initialize-fasl-environment))
  (setq filespec (fs:merge-and-set-pathname-defaults filespec))
  (format t "~&Cold-fasload ~A" filespec)
  (with-open-file (qfasl-binary-file filespec '(:in :fixnum))
    (set-file-loaded-id filespec (funcall qfasl-binary-file ':info))
    (setq fdefine-file-pathname (store-string 'sym:p-n-string
					      (funcall (funcall filespec ':generic-pathname)
						       ':string-for-printing)))
    (or (and (= (qfasl-nibble) 143150)
	     (= (qfasl-nibble) 71660))
	(ferror nil "~A is not a QFASL file" filespec))
    (do () ((eq (qfasl-whack) 'eof)))))

;This remembers where the file that we are building comes from
;The value of *cold-loaded-file-property-lists* is a list of elements:
;	(filenamestring specific-file-plist generic-file-plist)
(defun set-file-loaded-id (pathname id)
  (cond ((not (boundp 'cold-loaded-file-property-lists))
	 (setq cold-loaded-file-property-lists
	       (qintern 'sym:*cold-loaded-file-property-lists*))
	 (vwrite (+ cold-loaded-file-property-lists 1) qnil)))
  (setq file-property-list
	(vmake-pointer sym:dtp-list
		       (store-cdr-q 'sym:property-list-area sym:cdr-next
				    (store-string 'sym:p-n-string
						  (funcall pathname ':string-for-printing)))))
  (store-cdr-q 'sym:property-list-area sym:cdr-next qnil)	;will get filled in
  (store-cdr-q 'sym:property-list-area sym:cdr-nil qnil)	;as will this
  (let ((newp (vmake-pointer sym:dtp-list
			     (store-cdr-q 'sym:property-list-area sym:cdr-normal
					  file-property-list))))
    (store-cdr-q 'sym:property-list-area sym:cdr-error
		 (vread (+ cold-loaded-file-property-lists 1)))
    (vstore-contents (+ cold-loaded-file-property-lists 1) newp))
  ;; (nil fileversionid)
  (let ((id-prop (vmake-pointer sym:dtp-list
				(store-cdr-q 'sym:property-list-area sym:cdr-next qnil))))
    (store-cdr-q 'sym:property-list-area sym:cdr-nil
		 (store-string 'sym:p-n-string id))
    ;; ((nil fileversionid))
    (setq id-prop (vmake-pointer sym:dtp-list (store-cdr-q 'sym:property-list-area
							   sym:cdr-nil id-prop)))
    (let ((plist (vmake-pointer sym:dtp-list
				(store-cdr-q 'sym:property-list-area sym:cdr-next
					     (qintern 'sym:file-id-package-alist)))))
      (store-cdr-q 'sym:property-list-area sym:cdr-nil id-prop)
      ;; plist = (file-id-package-alist ((nil fileversionid)))
      ;; The nil will get replaced with the SI package later
      (vstore-contents (+ file-property-list 1) plist))))

;This is the function which gets a 16-bit "nibble" from the fasl file.
(defun qfasl-nibble ()
  (funcall qfasl-binary-file ':tyi))

;This function processes one "whack" (independent section) of a fasl file.
(defun qfasl-whack ()
  (let ((fasl-table-fill-pointer sym:fasl-table-working-offset)
	(fasl-return-flag nil))
    (or (boundp 'fasl-table)
	(setq fasl-table (make-array nil 'art-q sym:length-of-fasl-table)))
    (fillarray fasl-table '(nil))
    (initialize-qfasl-table)
    (do () (fasl-return-flag)
      (qfasl-group nil))
    fasl-return-flag)) 

;Initialize FASL-TABLE prefix
(defun initialize-qfasl-table ()
  (aset '(sym:nr-sym nil) fasl-table sym:fasl-symbol-head-area)
  (aset '(sym:p-n-string nil) fasl-table sym:fasl-symbol-string-area)
  (aset '(sym:control-tables nil) fasl-table sym:fasl-array-area) ;I GUESS
  (aset '(sym:macro-compiled-program nil) fasl-table sym:fasl-frame-area)
  (aset '(sym:init-list-area nil) fasl-table sym:fasl-list-area) ;Not FASL-CONSTANTS-AREA!!
  (aset '(sym:fasl-temp-area nil) fasl-table sym:fasl-temp-list-area))

(defun initialize-fasl-environment ()
  (setq fef-debugging-info-alist nil)
  (setq fasl-group-dispatch-size (length sym:fasl-ops))
  (setq q-fasl-group-dispatch (make-array nil 'art-q fasl-group-dispatch-size))
  (setq m-fasl-group-dispatch (make-array nil 'art-q fasl-group-dispatch-size))
  (do ((i 0 (1+ i))
       (l sym:fasl-ops (cdr l))
       (package (pkg-find-package "cold"))
       (m-op) (q-op))
      ((= i fasl-group-dispatch-size))
    (setq m-op (intern (format nil "M-~A" (car l)))
	  q-op (intern (format nil "Q-~A" (car l))))
    (aset m-op m-fasl-group-dispatch i)
    (aset q-op q-fasl-group-dispatch i)))

;Process one "group" (a single operation)
;Argument is NIL for Q-FASL, T for M-FASL.
(defun qfasl-group (m-p &aux fasl-group-flag fasl-group-bits
			     fasl-group-type fasl-group-length)
  (setq fasl-group-bits (qfasl-nibble))
  (or (bit-test sym:%fasl-group-check fasl-group-bits)
      (ferror nil "fasl-group-nibble-without-check-bit"))
  (setq fasl-group-flag (bit-test sym:%fasl-group-flag fasl-group-bits)
	fasl-group-length (ldb sym:%%fasl-group-length fasl-group-bits))
  (and (= fasl-group-length 377)
       (setq fasl-group-length (qfasl-nibble)))
  (setq fasl-group-type (logand sym:%fasl-group-type fasl-group-bits))
  (or (< fasl-group-type fasl-group-dispatch-size)
      (ferror nil "~O erroneous fasl group type" fasl-group-type))
  (funcall (aref (if m-p m-fasl-group-dispatch q-fasl-group-dispatch) fasl-group-type)))

;Get next nibble out of current group
(defun qfasl-next-nibble ()
  (cond ((zerop fasl-group-length) (ferror nil "fasl-group-overflow"))
	(t (setq fasl-group-length (1- fasl-group-length))
	   (qfasl-nibble))))

;Get next value for current group.  Works by recursively evaluating a group.
;This one gets a Q value
(defun q-fasl-next-value ()
  (q-arft (qfasl-group nil)))

;This one gets an M value
(defun m-fasl-next-value ()
  (m-arft (qfasl-group t)))

;This one gets both
(defun m-q-fasl-next-value ()
  (let ((idx (qfasl-group nil)))
    (values (m-arft idx) (q-arft idx))))

;FASL-OP's that create a value end up by calling this.  The value is saved
;away in the FASL-TABLE for later use, and the index is returned (as the 
;result of QFASL-GROUP).
;This one enters an M object and a Q
(defun m-q-enter-fasl-table (m q)
  (cond ((not (< fasl-table-fill-pointer sym:length-of-fasl-table))
	 (ferror nil "fasl table overflow"))
	(t
	 (m-q-asft m q fasl-table-fill-pointer)
	 (prog1 fasl-table-fill-pointer
		(setq fasl-table-fill-pointer (1+ fasl-table-fill-pointer))))))

;This one enters an M value
(defun m-enter-fasl-table (v)
  (cond ((not (< fasl-table-fill-pointer sym:length-of-fasl-table))
	 (ferror nil "fasl table overflow"))
	(t
	 (m-asft v fasl-table-fill-pointer)
	 (prog1 fasl-table-fill-pointer
		(setq fasl-table-fill-pointer (1+ fasl-table-fill-pointer))))))

(defun m-q-store-evaled-value (m q)
  (m-q-asft m q sym:fasl-evaled-value)
  sym:fasl-evaled-value)


;--M-FASL ops

(defun m-fasl-op-noop () 0)

(defun m-fasl-op-index () (qfasl-next-nibble))

(defun m-fasl-op-string ()
  (m-enter-fasl-table (m-fasl-pname)))

(defun m-fasl-op-symbol ()
  (m-enter-fasl-table (cond (fasl-group-flag (make-symbol (m-fasl-pname)))
			    (t (intern (m-fasl-pname) sym-package)))))

(defun m-fasl-op-package-symbol ()
  (do ((i 0 (1+ i))
       (path nil)
       (sym)
       (len (qfasl-next-nibble)))
      ((= i len) 
       (setq path (nreverse path))
       (cond ((> (length path) 2)
	      (ferror nil "Package path ~S has more than one prefix, I can't handle this"
			  path))
	     ((or (eq (car path) 'sym:si)	;don't get faked out into splitting one
		  (eq (car path) 'sym:system-internals)	;symbol into two non-eq symbols
		  (eq (car path) 'sym:global))
	      (m-enter-fasl-table (cadr path)))
	     (t
	      (setq sym (intern (format nil "~{~A~^:~}" path) sym-package))
	      (putprop sym path 'package-path)
	      (m-enter-fasl-table sym))))
    (push (intern (m-fasl-next-value) sym-package) path)))  ;fasl-value is string

(defun m-fasl-pname ()	;Return a string
  (let ((str (make-array nil 'art-string (* fasl-group-length 2)))
	tem)
    (dotimes (i fasl-group-length)
      (setq tem (qfasl-next-nibble))
      (aset tem str (+ i i))
      (aset (setq tem (lsh tem -8)) str (+ i i 1)))
    (and (eq tem 200) (adjust-array-size str (1- (array-length str)))) ;padded
    str))

;Generate a FIXNUM (or BIGNUM) value.
(defun m-fasl-op-fixed ()
  (do ((pos (* (1- fasl-group-length) 20) (- pos 20))
       (c fasl-group-length (1- c))
       (ans 0))
      ((zerop c) (cond (fasl-group-flag (setq ans (minus ans))))
		 (m-enter-fasl-table ans))
    (setq ans (dpb (qfasl-next-nibble) (+ (lsh pos 6) 20) ans))))

(defun m-fasl-op-float ()
  (q-fasl-op-float))

(defun m-fasl-op-list () (q-fasl-op-list))

(defun m-fasl-op-temp-list () (m-fasl-op-list1))

(defun m-fasl-op-list-component () (q-fasl-op-list t))

(defun m-fasl-op-list1 ()
  (do ((list-length (qfasl-next-nibble) (1- list-length))
       (lst nil) (adr) (tem))
      ((zerop list-length)
       (m-q-enter-fasl-table lst '**screw**))
    (cond ((and fasl-group-flag (= list-length 1)) ;dotted
	   (rplacd adr (m-fasl-next-value)))
	  (t (setq tem (ncons (m-fasl-next-value)))
	     (and adr (rplacd adr tem))
	     (or lst (setq lst tem))
	     (setq adr tem)))))

;--Q-FASL ops

(defun q-fasl-op-noop () 0)

(defun q-fasl-op-index () (qfasl-next-nibble))

(defun q-fasl-op-string ()
  (let ((str (m-fasl-pname)))
    (m-q-enter-fasl-table str (store-string 'sym:p-n-string str))))

(defun q-fasl-op-package-symbol ()
  (let ((x (m-fasl-op-package-symbol)))
    (q-arft x)
    x))

(defun q-fasl-op-symbol ()
  (let ((sym (m-fasl-pname)))
    (m-q-enter-fasl-table (if fasl-group-flag (make-symbol sym)
			      (setq sym (intern sym sym-package)))
			  (if fasl-group-flag ;uninterned
			      (store-symbol-vector sym 'sym:nr-sym)
			      (qintern sym)))))

(defun q-fasl-op-fixed ()
  (let ((x (m-fasl-op-fixed)))
    (q-arft x)
    x))

(defun q-fasl-op-float ()
  (or fasl-group-flag (ferror nil "large flonums not supported"))
  (let ((num (%make-pointer dtp-small-flonum
			    (%logdpb (qfasl-next-nibble) 2010 (qfasl-next-nibble)))))
    (m-q-enter-fasl-table num (make-small-flonum num))))

;;; Total kludgery.  FASL-OP-TEMP-LIST makes an M list, assumed to be
;;; going to get fed to something like FASL-OP-ARRAY or FASL-OP-EVAL.
;;; FASL-OP-LIST, on the other hand, makes a Q list, assumed to
;;; be going to be used for something like a macro.  In either case the
;;; area specification in the FASL table is ignored.
;;; Hopefully this kludgery stands some chance of working.

(defun q-fasl-op-temp-list ()
  (m-fasl-op-list))
       
(defun q-fasl-op-list-component ()
  (q-fasl-op-list t))

(defun q-fasl-op-list (&optional component-flag)
  (let ((area cold-list-area)
	(list-length (qfasl-next-nibble))
	lst c-code maclisp-list fasl-idx)
    (or (memq area sym:list-structured-areas)
	(ferror nil "q-fasl-op-list in non-list-structured area"))
    (setq lst (allocate-block area list-length))
    (do ((adr lst (1+ adr))
	 (len list-length (1- len)))
	((zerop len))
      (setq c-code (cond ((and fasl-group-flag (= len 2)) sym:cdr-normal)
			 ((and fasl-group-flag (= len 1)) sym:cdr-error)
			 ((= len 1) sym:cdr-nil)
			 (t sym:cdr-next)))
      (setq fasl-idx (qfasl-group nil))
      (vwrite-cdr adr c-code (q-arft fasl-idx))
      (setq maclisp-list (nconc maclisp-list
				(if (and fasl-group-flag (= len 1)) (m-arft fasl-idx)
				    (ncons (m-arft fasl-idx))))))
    (if (null component-flag)
	(m-q-enter-fasl-table maclisp-list (vmake-pointer sym:dtp-list lst))
	(m-q-store-evaled-value maclisp-list (vmake-pointer sym:dtp-list lst)))))

;Array stuff

;FASL-OP-ARRAY arguments are
; <value>  Area 
; <value>  Type symbol
; <value>  The dimension or dimension list (use temp-list)
; <value>  Displace pointer (NIL if none)
; <value>  Leader (NIL, number, or list) (use temp-list)
; <value>  Index offset (NIL if none)
; <value>  Named-structure (only present if flag bit set)
(defun q-fasl-op-array ()
  (let ((flag fasl-group-flag)
	(area (m-fasl-next-value))
	(type-sym (m-fasl-next-value))
	(dims (m-fasl-next-value))
	(displaced-p (m-fasl-next-value))  ;if non-nil, will it work?
	(leader (m-fasl-next-value))
	(index-offset (m-fasl-next-value)) ;if non-nil, will it work?
	(named-structure nil)
	(array nil) (data-length nil) (adr nil))
     (setq area 'sym:control-tables) ;kludge, may not be needed any more
     (cond (flag
	    (setq named-structure (m-fasl-next-value))))
     (and (not (atom leader))
	  (setq leader (mapcar (function (lambda (x) (make-q-list 'sym:init-list-area x)))
			       leader)))
     (setq array (init-q-array-named-str area
					 nil  ;return list of address and data-length
					 index-offset
					 type-sym
					 dims
					 displaced-p
					 leader
					 named-structure))
     (setq data-length (cadr array)
	   array (vmake-pointer sym:dtp-array-pointer (car array)))
     ;now store the data area
     (and displaced-p (ferror nil "displaced array not handled"))
     (setq adr (allocate-block area data-length))
     (cond ((cdr (assq type-sym sym:array-bits-per-element)) ;numeric
	    (dotimes (i data-length)
	      (vwrite (+ adr i) 0)))
	   (t
	    (cond ((and named-structure (not leader))
		   (vwrite adr (qintern named-structure))
		   (setq adr (1+ adr)
			 data-length (1- data-length))))
	    (dotimes (i data-length)
	      (vwrite (+ adr i) qnil))))
     (m-q-enter-fasl-table
       "note - you have been screwed to the wall by an array"
       array)))

;Get values and store them into an array.
(defun q-fasl-op-initialize-array ()
  (prog (array num hack ptr header long-flag ndims)
     (setq hack (qfasl-group nil))
     (setq array (q-arft hack))
     (or (= (vdata-type array) sym:dtp-array-pointer)
	 (ferror nil "fasl-op-initialize-array of non-array"))
     (setq num (m-fasl-next-value))	;number of values to initialize with
     ;; Take header apart to find address of data
     (setq ptr (logand q-pointer-mask array))
     (setq header (vread ptr))
     (setq long-flag (bit-test sym:array-long-length-flag header)
	   ndims (logand (// header sym:array-dim-mult) 7))
     (and (bit-test sym:array-displaced-bit header)
	  (ferror nil "attempt to initialize displaced array, give it up"))
     (setq ptr (+ ptr (if long-flag 1 0) ndims))	;To data
     (dotimes (n num)				;Initialize specified num of vals
       (vwrite ptr (q-fasl-next-value))
       (setq ptr (1+ ptr)))
     (return hack)))

;Get 16-bit nibbles and store them into an array.
(defun q-fasl-op-initialize-numeric-array ()
  (prog (array num hack ptr header long-flag ndims)
     (setq hack (qfasl-group nil))
     (setq array (q-arft hack))
     (or (= (vdata-type array) sym:dtp-array-pointer)
	 (ferror nil "fasl-op-initialize-array of non-array"))
     (setq num (m-fasl-next-value))	;number of values to initialize with
     ;; Take header apart to find address of data
     (setq ptr (logand q-pointer-mask array))
     (setq header (vread ptr))
     (setq long-flag (bit-test sym:array-long-length-flag header)
	   ndims (logand (// header sym:array-dim-mult) 7))
     (and (bit-test sym:array-displaced-bit header)
	  (ferror nil "attempt to initialize displaced array, give it up"))
     (setq ptr (+ ptr (if long-flag 1 0) ndims))	;To data
     (dotimes (n (// num 2))			;Initialize specified num of vals
       (vwrite ptr (+ (qfasl-nibble) (ash (qfasl-nibble) 16.)))
       (setq ptr (1+ ptr)))
     (cond ((oddp num)				;odd, catch last nibble
	    (vwrite ptr (qfasl-nibble))))
     (return hack)))

(defun q-fasl-op-eval ()
  (ferror nil "**This fasl op has been decommitted")
  (let ((exp (m-arft (qfasl-next-nibble))))
    (cond ((and (not (atom exp))
		(eq (car exp) 'sym:record-source-file-name)
		(not (atom (cadr exp)))
		(eq (caadr exp) 'sym:quote)
		(symbolp (cadadr exp)))
	   (store-source-file-name-property (qintern (cadadr exp))))
	  (t (setq evals-to-be-sent-over
		   (setq last-fasl-eval (cons exp evals-to-be-sent-over))))))
  (m-q-store-evaled-value 'value-only-available-in-the-future
			  'value-only-available-in-the-future))

(defun q-fasl-op-move ()
  (let ((from (qfasl-next-nibble))
	(to (qfasl-next-nibble)))
    (cond ((= to 177777) (m-q-enter-fasl-table (car (aref fasl-table from))
					       (cadr (aref fasl-table from))))
	  (t (aset (aref fasl-table from) fasl-table to)
	     to))))

;Macrocompiled code

(defun q-fasl-op-frame ()
  (let ((q-count (qfasl-next-nibble))		;number of boxed qs
	(unboxed-count (qfasl-next-nibble))	;number of unboxed qs (half num instructions)
	(fef)					;the fef being created
	(obj)
	(m-obj)
	(fname)
	(tem)
	(offset 0)
	(area 'sym:macro-compiled-program))	;(m-arft sym:fasl-frame-area)
     (setq fasl-group-length (qfasl-next-nibble))	;amount of stuff that follows
     (setq fef (vmake-pointer sym:dtp-fef-pointer	;Store header
			      (storeq area (vmake-pointer sym:dtp-header
							  (m-fasl-next-value)))))
     (qfasl-next-nibble)			;skip modifier nibble for header q
     (do i 1 (1+ i) (>= i q-count)		;fill in boxed qs
       (multiple-value (m-obj obj) (m-q-fasl-next-value))	;get object to be stored
       (setq tem (qfasl-next-nibble))		;get ultra-kludgey modifier
       (or (zerop (setq offset (logand 17 tem)))	;add offset if necessary
	   (setq obj (+ obj offset)))
       (and (bit-test 420 tem)			;try not to get shafted totally
	    (or (= (vdata-type obj) sym:dtp-symbol)
		(ferror nil "about to get shafted totally - q-fasl-op-frame")))
       (and (bit-test 20 tem)			;make into external value cell pointer
	    (setq obj (vmake-pointer sym:dtp-external-value-cell-pointer obj)))
       (and (bit-test 400 tem)			;make into locative
	    (setq obj (vmake-pointer sym:dtp-locative obj)))
       (setq obj (dpb (lsh tem -6) sym:%%q-cdr-code obj))
       (and (bit-test 40 tem)			;flag bit
	    (setq obj (dpb 1 sym:%%q-flag-bit obj)))
       (storeq area obj)
     @QS@`PzAQgsZs%MKMQ[MGbF[]CQUR@Qb5ibA@fC[J@[_EQ!RR @@@@ akgQ QG_QcAM]@JAZQWETR@KL[@EEkO@q]N[QM^[@SghQ@@@ @QE@WS\[b4_eJQPCYMbweIf@eKB@]E_sH[GQu]hR]_n@i_e@PiQJ@]E_sHAcb5@@@ @QIQtS[Kb0Q\@QAk]@'qKHQS_k]b@dRRR@@@@ @@Qb4_eJQPCYMbweH@QMCgQE]KqbE]SE@$JRRQ@@@ @QKQd[giQrJ[Q@Mo_b$fR @@@@ Z[bQU]iKb%MCgQEiCEQE@@ @@@@ E]_bE@ZAsjAQ@JAE@U\Ag@2KoK@@i^Ab@JAo@XAEsBAM@VDKLRQ IKMbVAb[@agX[Qp[MkQciS_QeQKC@Ed@PQ@@Q e_N@ L[gs RgKibGkeb%]h[@e]GiQ\@QQUMCgQE]KqbEmCYbUR@@@@ L[gs @QZQVCgXQVKqhQVCYk@QReKibR\@`QRRQIK@e\AbQVCgXQW`[MbVGiSQv[K]@@PR@b IKMbVAb[@agX[b4_eKQ[gsQR_X[@5YX@QAakbEg_kb#J[MQJ[]@J[ab'aKebAR@ QaeQw@Q]@W`AC@BAICbAAgsQP]SDQ@@@ @Qg@TbA]Q@Qc@agX[Qeqh[QaEEY@QR@ @@@QKib@sZ@Q[MCb4[]Ks[mCQEJRRR@@@@ QG_Qd@PPsP]SD@sZu@agX[@VCYK@EmCYbUR@w"5iiSQgAgsQR_XAbGAeKb5YhAQvAKm@kCiQ\@@@QKeeQrA]SQ@E
"4[ QU-@YkgQHXAQaKMbTYrAQemKd@KhAQeJDQ@@ @QGQvH@PQi_Z@sZR@@@@ @@@@_IQrAiQAK]bBrASQ`-C[)QR[' V(["e$@@@Q_]H@ Q]kQDAYCb4[MCb4[Km@R@@QKeeQrA]SQ@E}&@]mCQAHAgbGeKSQegs[@'XDAb1ZRRQ@ @QebCGB@CghQVCgXQUmCXR@ A@PQ@gKY@SibAQePb@@sKhRRd@O@cKhRR_iQ@RoSg@PQMKb"_dAQaX@@@ E%Kb5YhAQvAKm@kCiQ\A[bShAE@Pgi_b%HASQ`mCYbUA_d@k]GbA_\A@5YXDQR@@@ @QgsuckQtJ@Yb1ZR@Q@GCd@CghQVCgXQUmCXQR@@@QKikb&@`RQw"3S`Ab@JAe@ShA_@`iQSb0Mk]@4S_\R@@ h@Q@eee_b ]SX@kgb@EJA@gsZ@mCY@T[mCQEJDRQRR @@@@ gKibICi@Qb[@MhAQaDRRR@@@@ QG_Qd@PQ@_ZAb1ZR!@@@@ gKibgsZ@ cS]bEe\Ab1ZRRR@@@ QmgbGeJ[@7]iKQdf@PQ0gsZ@RAI@BR!@@@@ G_]@@QakbEg_kb#J[MQJ[]@J[ab'aKebA@@@Qi_e@Ug_kb#J[MQJ[]@J[ab'aKebAAgsQQRRRR@@@ QKb@ GCd@sZR@gsZs'uS]bEe]CQA@ @@QQr@Pz@@dR@ MKeb'dA]Q@E}"0_]YsCYYQwKHA@gdAMbVGiSQvAGKQDDAgsRR!@@@@ YKhQ PQa@K]h@ GCIb gsZQ@@QQIKp@ GCI@BAgsQQR@@@QCEY@PQGIb QCgb1@Ogst^uQiKeQaX[M@V[_M@cKifR@@@@ GId@ CggQsAaCb%]hA@eL[I@RkOOQN[SQf^[CQAghRQRR!@@@ MKL@ cMI@VS]SbA_\AbeK]bARR!@@@@ @Q_b iCEQE@QM@Re_d@SX@@#C]]QtAY_@1iJAQiKeQaX[M@V[_M@cKif@_dAscDAgsRR!@@@@ @Q_b Pz@QIDAb1ZtJ@Q[ICbA[isbAMK@aAgsQRIi`QVKL[bS]i@RR@@Q@eee_b ]SX@}&AQghAM@VACf@k]GbA_\A@EMS]QS_\@LA}"2AMK@`aCe@VhRRR@@@ @@Qbci_e@UG_]bE]if@ VAM@V@Q]b@AS]@EpAi@YJRQICi@RR!@@@s3A]@v@Q	 V+@QA" R@ 0A2Q	3@ @vv@AiQQWfAQOfAb4_eSQgAMkQciS_Q`GKYQ@Ekh@KCYQAA!+"@% R@@@ Q]_b@QC]@@PzAQ`dR@Qb@Qb`AgsR@Ob1Zt^s e_a@RirRQ@ @@Q@eee_b ]SX@}&AQghAB@s[EQtA_d@ e_a@RirAb0KFD@sZRQ@@ Qh@QKib@Id@QS]i@R\@Q@1dAgsRRRR@@@ @@Qb5ibAQeo`@Q[CW@Ua_SQdKdAb1ZuIb@[YSb4Qb4_eJQSId[bOgsQRae_beirQTSghQQeKB@sZu@4d[]@Ph@@@@ cS]bEe\@QCId@sZRQRR!@@@@ @QgbGeJ[@4d[b@gsZs e_a@Rir[QAgh[@KBAb1ZuG@B[]_b%CXA@AiBRR@@@ @@Qb4_eJQSId[bOgsQRae_beirQTSghQQeKB@sZu@4d[Kb"_d@QeKC@@PVA@d@fQR@@@@ QmgbGeJ[@7]iKQdf@PQ0CId@RA]@W`RRQ@@@ @Qe@Tke\@ RRRRQ@EMk\@MIK@a]SiQ\@Qb1ZR @QGQvH@PQs[EQt`AgsR@QbbKCH@ V@Qb]iKb&AgsQQ@dRQQ Kb@QCdAb1ZR@@ssZtQrS]i@R]CXQ@QKhT@ QaCb%]h@QCId@sZRQQ]IKs QGC@DdAgsRR!Qi@YJ@QId@Qggb@gsZs'uS]bEe]CQEMKLQWMMg@Tf@@ GId@ CggQsAaCb%]hA@eL[I@RkOOQN[SQf^[CQAghRQRR!QM@V@Qc@dKMSQaiS_Q`aCe@VhRRQ@@ Q_d@CEY@PQMKb"_dAQaX@E 1]]_b@Y_G@JASQdKe]@[MK@e_MMb5ifA@gdA}"2AgsQQR@@QbbKCH@ VAM@V@Q]b@AS]@EpAi@YJRQRR!Qh@QKeeQrA]SQ@E}&@CeE@JAMbVGiSQvAga@SDAgsRRRQQKMkQ`gi_b%[g_bRGJ[@aYJ[Qa[J[b_aKb$r@Qb1ZR @QY@T@PQQeo`@Q[CW@Ua_SQdKdAb1ZuIb@[YSb4@@@@ QgiQrJ[G@B[b@@ssZub_aKb$r[YQh[Cb%BAgsuGIb%]KqbE@@QS]i@R\@Ob1ZugQueGJQVSYJQVC[JQRRRRwo@A+' R^u'u%QVQVR@@@@ gi_b%[GIb%b@Ob1Zuab'aKebA[YSb4[Ce@QAgsQRGIdQV_e[@AMI@VS]JQVSYJQPCiQQa[JRR@@@@ gi_b%[GIb%b@Ob1Zuab'aKebA[YSb4[Ce@QAgsQRGIdQUee_b Qme@QH@PQ0gsZ@RRRR@@@@ mgiQrJ[GQviK]bC@PV@sZ@b1A]KbpRRRRQ@EMk\@[MCb4[_`QSi_e@Q\[gsE_XQVCYk@PPR @QbQVCgXQSi_e@Q\[gsE_XQSKYX@A]SQARRQIK@e\AbQVCgXQW`[gbGeKSQeMk]@4S_\QSKYX@ R@ Qb[@agX[b4_eKQ[gsQR_X[@5YX@b hRRRQ@EMk\@[MCb4[_`QSi_e@Q\[ab'aKebA[GKQD@PRR@@QbMCgQEgi_b%S\[b1[E_QEGKYQ@fA]QRR%QI@Vk\AbMCgQE_`[b4_eKQ[Ceb!r[Y@QIKd@ R@ QYKb@PQCb"Cr@Q[Ce@d@Qc@agX[Qeqh[QaEEY@QRR!Qgk@#Gd@Q[Ce@d@Qc@agX[Qeqh[QaEEY@QRR!QmCQEJ@QbCeMb@QcM@X[]@Ph[]QEYJQRR @@@s3KeeQrAGQ@SWS]@p[SOQAEJ@SGJR@@@@QgiQrJ[Cb"Cr[QECIKb mCYbUACeb!rAgbRgGdQ@@@ Qmob!iJ@QACeb!r@PQ0dAgbRgGdQAmCQEJR @@@bRRQIK@e\AbQVCgXQVKiGQgs[@'X[G@TX@QQa@@ YKh@ QmCQ@QmGQviK]bC@PV@ b[M@X[]@Ph[m@kJR@RRRQ@@@ QSL@ z@QbdCiBQTsaJ@CXR@sZu@D`[]bTXR!QMKb"_dAQaX@E@eiGP@LAkQb_k]@@gs[@'XAG@TXDRQ@@@ QZ[bK]i@R[MCb4[iC@$J@Eb1[E_Q@G_[b]K]bBAmCQARR%QI@Vk\AbMCgQE_`[@eiGPQSs[EQt[mCQEJ@PQ@@Q[MCb4[MKbCP[gsE_XQSKYX@RR%QI@Vk\AbMCgQE_`[@eiGPQVk]GbA_\[@5YX@Q@@ b[M@X[M@TGP[b1[E_QEGKYQ@dRRRQ@EMk\@[MCb4[_`QVKiGQae_beirQSKYX@ R@ Qb[@agX[@eiGPQSs[EQt[GKQD@fRQQKMkQ`b[M@X[_bK]HQWL[oQGV@Q@@ gKibMCgQEeKibR\[MQAN@O@VH[_@eoQC@3R@ `R%QI@Vk\AbMCgQE_`[@VH[_@eMSY@PPR @Qg@TbAM@X[e@Tke\QVYCN@K_LQ@@b IKMbVAb[@agX[Qp[g_@@PRR@@Q@GiS[@S@QGQu]h@QMCgQE]KqbE]SE@$JRRR@@@@ Z[M@X[]@Ph[m@kJRQ@@QMCgQEOe_bPAhRQQKMkQ`b[M@X[_bgKhQPCeCQUiKd@ R@ QYKb@PQiQpQZ[@agX[Qeqh[baYkJQQe_Z@ cMCb4[OeQu`AhQR@ @@Q@Kh@QeKL@CgXQTCEY@PMe_QQAMCb4[iC@$JAiQq@@ @`RQQKMkQ`b[M@X[_bMSY@Uae_beirQTSgh@ R@ QmgbGeJ[@7]iKQdf@PQ0MSY@Uae_beirQTSgh@R@QbMCgQE]KqbEmCYbURR @`RRvs3A!Cb@]C[@SACe@PIk[bHA_bTAg^@fAiQpikeQ`S]iQpeKCQ@_]Kb0oQKQ`MCgQGCIK@Dvs3AMCQ5Ak`@Agib!]NAQgiK@\@A@cuGCQg]SG@SuJQS_YHQT_CHQPCiQ@KfAbqYXA@apASb@ECGQ6Q@EMk\@[MCb4[_`QUmCXbPMCbPAM_b%AaCb@]C[@Q@@ gKibM_eQPQZ[@agX[Qeqh[baYkJQ@@ _d@Q]H@QSgib M_eQQ@Q[@Ub@Q@1dAMQrZR@@pgsZs&f^uQQWJ[biQ]@J[SQdKe]@@@ gsZs&f^uQQWJ[@agY_@[aCb@]C[@QR@@Qb5ibAbiQ]@J@Q@G@PQ@dSgh@ GId@_eZQQGIb MYSb4RR!@@ @@@@ aYSb4A]SQA@@@@ @@Q@TKZRQ@@@@ PQ]bTXAMQAghRR@ @@@@ QCabr@F@vfu[@J[a@Q]CQU[S]bEe]CQE@@@@ @Qg@TbAaQAgh@QeKm@RgJAbSghQRR!@@ @Qg@TbAKQEZ@Q@1dAMQAghRQ@@@Qd@Q@H@QQAgi`@YKZQ@@@@ QKb@ GCd@YKZQOgsQRck_bERR!QKikb&A]SQAR@@@ QgKbAAKY@U@QG@dAKQEZRRR@ @@Q@7]H@Q [K[bKYKQPNQgsuk]b0KGS@aFAgst^ubVgaK@1MSFQ@@@ gKibKYKQPNukQcaKGQSFRQ@@PQd@QQeYXA@TKZR@ gieQO`A@TKZR@ ]k[@%e`A@TKZRQ@@Q@Qe@Tke\@SXRQ@@@@ akgQ KYKQPaYSb4RRRQ@@@ @@Q@eee_b ]SX@C]QghAKbaXAiQfAMQrZXAscXAsQuAoSQDAY_b5DAMQrZRRR@@Qb5ibAbiQ]@J@Q@e]GCQDAaCb@]C[@PNugbBS]NQV_d[bS]iQNRRR@@QQUb[KQdKd[@agX[bAEYJ@ CiQQa[J@Qi_e@UgieQN@Ob1Zu`QV[gib!]NAbiQ]@JRRQ