;;; -*- Mode: Lisp; Package: System-Internals; Base: 8 -*-

;;; This file contains random hacks to help recompile and build new lispm
;;; worlds.  Feel free to add to it.

;;; DLA's hack to find the creation dates of all installed QFASL files.

(DEFVAR QFASL-SOURCE-FILE-PLISTS-ALIST NIL)
(DEFUN QFASL-SOURCE-FILE-PLISTS (&OPTIONAL (SYSTEM "System"))
  (AND (OR (NOT (ASSOC SYSTEM QFASL-SOURCE-FILE-PLISTS-ALIST))
	   (Y-OR-N-P "Recompute QFASL source files? "))
       (LET ((FILES (MAPCAR #'(LAMBDA (X)
				(FUNCALL X ':NEW-PATHNAME ':TYPE "QFASL" ':VERSION ':NEWEST))
			    (SYSTEM-SOURCE-FILES SYSTEM ':ALL))))
	 (PUSH (CONS SYSTEM
		     (SORT (DEL #'(LAMBDA (IGNORE X) (NULL (GET X ':CREATION-DATE-AND-TIME)))
				NIL
				(FS:MULTIPLE-FILE-PROPERTY-LISTS T FILES))
			   #'(LAMBDA (X Y)
			       (< (GET X ':CREATION-DATE-AND-TIME)
				  (GET Y ':CREATION-DATE-AND-TIME)))))
	       QFASL-SOURCE-FILE-PLISTS-ALIST)))
  (CDR (ASSOC SYSTEM QFASL-SOURCE-FILE-PLISTS-ALIST)))

(DEFUN LIST-QFASL-SOURCE-FILES (FILENAME &OPTIONAL (SYSTEM "System"))
  (LET ((FILES (QFASL-SOURCE-FILE-PLISTS SYSTEM)))
    (WITH-OPEN-FILE (STREAM FILENAME '(OUT))
      (DOLIST (P FILES)
	(FORMAT STREAM "~30A" (CAR P))
	(TIME:PRINT-UNIVERSAL-TIME (GET P ':CREATION-DATE-AND-TIME) STREAM)
	(FUNCALL STREAM ':TYO #\CR)))))

(DEFUN RECOMPILE-FILES-AFTER-DATE (AFTER-DATE &OPTIONAL (SYSTEM "System")
				   &AUX RECOM-FILES TEM)
  (AND (STRINGP (SETQ AFTER-DATE (TIME:PARSE-UNIVERSAL-TIME AFTER-DATE 0 NIL NIL)))
       (FERROR NIL AFTER-DATE))
  (DOLIST (P (QFASL-SOURCE-FILE-PLISTS SYSTEM))
    (COND ((AND (> (GET P ':CREATION-DATE-AND-TIME) AFTER-DATE)
		(SETQ TEM (FUNCALL (FUNCALL (CAR P) ':GENERIC-PATHNAME)
				   ':GET ':QFASL-SOURCE-FILE-UNIQUE-ID)))
	   (PUSH (FUNCALL (FS:MERGE-PATHNAME-DEFAULTS TEM) ':NEW-VERSION ':NEWEST)
		 RECOM-FILES))))
  (DOLIST (F (NREVERSE RECOM-FILES))
    (COND ((PROBEF F)
	   (FORMAT T "~%Recompiling ~A ..." F)
	   (QC-FILE F))
	  (T (FORMAT T "~%~A does not exist.")))))

(DEFUN RECOMPILE-FILES-BEFORE-DATE (BEFORE-DATE &OPTIONAL (SYSTEM "System")
				    &AUX RECOM-FILES TEM)
  (AND (STRINGP (SETQ BEFORE-DATE (TIME:PARSE-UNIVERSAL-TIME BEFORE-DATE 0 NIL NIL)))
       (FERROR NIL BEFORE-DATE))
  (DOLIST (P (QFASL-SOURCE-FILE-PLISTS SYSTEM))
    (COND ((AND (> BEFORE-DATE (GET P ':CREATION-DATE-AND-TIME))
		(SETQ TEM (FUNCALL (FUNCALL (CAR P) ':GENERIC-PATHNAME)
				   ':GET ':QFASL-SOURCE-FILE-UNIQUE-ID)))
	   (PUSH (FUNCALL (FS:MERGE-PATHNAME-DEFAULTS TEM) ':NEW-VERSION ':NEWEST)
		 RECOM-FILES))))
  (DOLIST (F (NREVERSE RECOM-FILES))
    (COND ((PROBEF F)
	   (FORMAT T "~%Recompiling ~A ..." F)
	   (QC-FILE F))
	  (T (FORMAT T "~%~A does not exist.")))))


;;; DLA's hack to search through the sources of QFASL files compiled after a certain
;;; date for bad strings.  Useful when a macro has expanded wrong, or the compiler
;;; broke in certain cases.

(DEFUN SOURCE-SCAN (SUBSTRING AFTER-DATE &OPTIONAL (SYSTEM "System") &AUX BAD-FILES FILE)
  (SETQ AFTER-DATE (TIME:PARSE-UNIVERSAL-TIME AFTER-DATE))
  (DOLIST (F (QFASL-SOURCE-FILE-PLISTS SYSTEM))
    (COND ((> (GET F ':CREATION-DATE-AND-TIME) AFTER-DATE)
	   (SETQ FILE (FUNCALL (CAR F) ':NEW-PATHNAME ':TYPE "LISP" ':VERSION ':NEWEST))
	   (FUNCALL STANDARD-OUTPUT ':CLEAR-SCREEN)
	   (FORMAT T "File ~A ..." FILE)
	   (AND (SCAN-SOURCE-FILE FILE SUBSTRING)
		(PUSH FILE BAD-FILES)))))
  BAD-FILES)

(DEFUN SCAN-SOURCE-FILE (FILE SUBSTRING &AUX LINE EOF)
  (LET* ((BUFFER (CIRCULAR-LIST NIL NIL NIL NIL NIL NIL NIL))
	 (LINE-IN-POINT (CDDDDR BUFFER)))
    (WITH-OPEN-FILE (STREAM FILE)
      (DO () (NIL)
	(MULTIPLE-VALUE (LINE EOF)
	  (FUNCALL STREAM ':LINE-IN T))
	(AND EOF (OR (NULL LINE)
		     (EQUAL LINE ""))
	     (SETQ LINE ':EOF))
	(SETF (CAR LINE-IN-POINT) LINE)
	(SETQ BUFFER (CDR BUFFER)
	      LINE-IN-POINT (CDR LINE-IN-POINT))
	(COND ((NULL (CAR BUFFER)))		;Beginning of file
	      ((EQ (CAR BUFFER) ':EOF)
	       (RETURN NIL))
	      ((STRING-SEARCH SUBSTRING (CAR BUFFER))
	       (FUNCALL STANDARD-OUTPUT ':SET-CURSORPOS 0 3 ':CHARACTER)
	       (FUNCALL STANDARD-OUTPUT ':CLEAR-EOF)
	       (DO ((L (CDR LINE-IN-POINT) (CDR L)))
		   ((EQ L LINE-IN-POINT))
		 (FUNCALL STANDARD-OUTPUT ':LINE-OUT
			  (COND ((NULL (CAR L)) "[Beginning of file]")
				((SYMBOLP (CAR L)) "[End of file]")
				(T (CAR L)))))
	       (FORMAT T "~% Is this OK? ")
	       (OR (Y-OR-N-P)
		   (RETURN T))))))))

;;; RG's Source compare hack.

;start dribble file if you want to save results.
(DEFUN SOURCE-COMPARE-SYSTEM-UPDATES (&OPTIONAL SYSTEMS)
  (COND ((NULL SYSTEMS) (SETQ SYSTEMS *SYSTEMS-LIST*))
	((NLISTP SYSTEMS) (SETQ SYSTEMS (LIST SYSTEMS))))
  (LET (FILES)
    (DOLIST (SYSTEM SYSTEMS)
      (SETQ FILES (APPEND FILES (SYSTEM-SOURCE-FILES SYSTEM))))
    (SETQ FILES (ELIMINATE-DUPLICATES FILES))
    (DOLIST (FILE FILES)
      (SOURCE-COMPARE-FILE-UPDATES FILE))))

(DEFUN SOURCE-COMPARE-FILE-UPDATES (FILE)
  (PROG (CURRENT-VERSION INSTALLED-VERSION FILE-TO-COMPARE)
	(SETQ CURRENT-VERSION (PROBEF FILE))
	(COND ((NULL CURRENT-VERSION)
	       (FORMAT T "~% No source for ~S" FILE)
	       (RETURN NIL)))
	(SETQ INSTALLED-VERSION (FUNCALL (FUNCALL CURRENT-VERSION ':GENERIC-PATHNAME)
					 ':GET
					 ':QFASL-SOURCE-FILE-UNIQUE-ID))
	(IF (STRINGP INSTALLED-VERSION)
	    (SETQ INSTALLED-VERSION (FS:PARSE-PATHNAME INSTALLED-VERSION)))
	(COND ((NULL INSTALLED-VERSION)
	       (FORMAT T "~%Installed version of ~S unrecorded, trying oldest" FILE)
	       (GO OLD))
	      ((PROBEF INSTALLED-VERSION)
	       (SETQ FILE-TO-COMPARE INSTALLED-VERSION)
	       (GO COMP)))
  OLD	(COND ((SETQ FILE-TO-COMPARE
		     (PROBEF (FUNCALL CURRENT-VERSION ':NEW-VERSION ':OLDEST)))
	       (FORMAT T "~%Installed version of ~s unavailable, using oldest"
		       CURRENT-VERSION)
	       (GO COMP))
	      (T (FERROR NIL "")))
  COMP  (COND ((SAME-FILE-P FILE-TO-COMPARE CURRENT-VERSION)
	       (COND ((AND INSTALLED-VERSION
			   (SAME-FILE-P INSTALLED-VERSION CURRENT-VERSION))
		      (FORMAT T "~%No change to file ~s" FILE))
		     (T (FORMAT T "~%Only one version to compare ~s" FILE))))
	      (T (FORMAT T "~%Comparing ~s and ~s" FILE-TO-COMPARE CURRENT-VERSION)
		 (SRCCOM:SOURCE-COMPARE FILE-TO-COMPARE CURRENT-VERSION)))))

(DEFUN SAME-FILE-P (F1 F2)
  (OR (EQ F1 F2)
      (AND (EQUAL (FUNCALL F1 ':DIRECTORY) (FUNCALL F2 ':DIRECTORY))
	   (EQUAL (FUNCALL F1 ':NAME) (FUNCALL F2 ':NAME))
	   (EQUAL (FUNCALL F1 ':VERSION) (FUNCALL F2 ':VERSION))
	   (OR (EQUAL (FUNCALL F1 ':TYPE) (FUNCALL F2 ':TYPE))
	       (AND (MEMQ (FUNCALL F1 ':TYPE) '(:UNSPECIFIC NIL))
		    (MEMQ (FUNCALL F2 ':TYPE) '(:UNSPECIFIC NIL)))))))

;;; DLA's hack to purge source files of a system.

(DEFUN UNIQUE-IDS-OF-SYSTEM (&OPTIONAL (SYSTEM "System"))
  (LET ((FILES (SYSTEM-SOURCE-FILES SYSTEM)))
    (DO ((F FILES (CDR F)))
	((NULL F))
      (SETF (CAR F)
	    (FUNCALL (FUNCALL (CAR F) ':GENERIC-PATHNAME)
		     ':GET ':QFASL-SOURCE-FILE-UNIQUE-ID))
      (AND (STRINGP (CAR F))
	   (SETF (CAR F) (FS:MERGE-PATHNAME-DEFAULTS (CAR F)))))
    (DELQ NIL FILES)))

;; This function maps through the source files of a system, and deletes all files
;; which are not either installed or newest.
(DEFUN SYSTEM-SOURCE-FILE-PURGE (&OPTIONAL (SYSTEM "System") (QUERY-P T) (KEEP 1)
				 &AUX DIR DIR-LIST)
  (LET ((FILES (UNIQUE-IDS-OF-SYSTEM SYSTEM)))
    (DO ((DIR-DELETE-P NIL NIL)) ((NULL FILES))
      (SETQ DIR (FS:PATHNAME-DIRECTORY (CAR FILES))
	    DIR-LIST (FS:DIRECTORY-LIST (FUNCALL (CAR FILES) ':NEW-PATHNAME
					  ':NAME ':WILD ':TYPE ':WILD ':VERSION ':WILD)))
      ;; This puts the directory in descending order.
      (SETQ DIR-LIST (SORT (CDR DIR-LIST)
			   #'(LAMBDA (X Y) (NOT (< (GET X ':CREATION-DATE)
						   (GET Y ':CREATION-DATE))))))
      (DOLIST (FILE FILES)
	(COND ((EQUAL (FS:PATHNAME-DIRECTORY FILE) DIR)
	       (DO ((D DIR-LIST (CDR D))
		    (NEWEST KEEP)
		    (NAME (FS:PATHNAME-NAME FILE))
		    (TYPE (FS:PATHNAME-TYPE FILE))
		    (VERSION (FS:PATHNAME-VERSION FILE))
		    (RELEVANT-FILES NIL)
		    (DELETE-P NIL))
		   ((NULL D)
		    (COND (DELETE-P
			   (FORMAT T "~%~%")
			   (COND ((NOT DIR-DELETE-P)
				  (FORMAT T "~A:~%" DIR)
				  (SETQ DIR-DELETE-P T)))
			   (MAPC 'ZWEI:DEFAULT-LIST-ONE-FILE RELEVANT-FILES)
			   (COND ((OR (NOT QUERY-P)
				      (Y-OR-N-P "Delete them? "))
				  (DOLIST (F RELEVANT-FILES)
				    (AND (GET F ':DELETED)
					 (DELETEF (CAR F)))))))))
		 (AND (EQUAL NAME (FS:PATHNAME-NAME (CAAR D)))
		      (EQUAL TYPE (FS:PATHNAME-TYPE (CAAR D)))
		      (COND ((OR (PLUSP NEWEST)
				 (EQUAL VERSION (FS:PATHNAME-VERSION (CAAR D))))
			     (PUSH (CAR D) RELEVANT-FILES)
			     (SETQ NEWEST (1- NEWEST)))
			    (T (PUTPROP (CAR D) T ':DELETED)
			       (SETQ DELETE-P T)
			       (PUSH (CAR D) RELEVANT-FILES))))))))
      (SETQ FILES (DEL #'(LAMBDA (DIR FILE) (EQUAL DIR (FS:PATHNAME-DIRECTORY FILE)))
		       DIR FILES)))))

(DEFUN LIST-MISSING-INSTALLED-VERSIONS (&OPTIONAL (SYSTEMS *SYSTEMS-LIST*))
  (OR (LISTP SYSTEMS) (SETQ SYSTEMS (LIST SYSTEMS)))
  (FUNCALL STANDARD-OUTPUT ':FRESH-LINE)
  (LOOP FOR (FILE . PLIST)
	 IN (FS:MULTIPLE-FILE-PROPERTY-LISTS NIL
	      (LOOP FOR FILE IN (ELIMINATE-DUPLICATES
				  (LOOP FOR SYSTEM IN SYSTEMS
					APPEND (SYSTEM-SOURCE-FILES SYSTEM)))
		    AS SOURCE = (FUNCALL (FUNCALL FILE ':GENERIC-PATHNAME)
					 ':GET ':QFASL-SOURCE-FILE-UNIQUE-ID)
		    AS VERSION = (AND SOURCE (FUNCALL (FS:PARSE-PATHNAME SOURCE) ':VERSION))
		    UNLESS (NULL VERSION)
		      COLLECT (FUNCALL (FUNCALL FILE ':NEW-VERSION VERSION)
				       ':TRANSLATED-PATHNAME)))
       WHEN (NULL PLIST)
         DO (PRINC FILE) (TERPRI)))