/* 8MB Memory Board implementation
 
 This file is subject to the terms and conditions of the GNU General Public
 License.  See the file COPYING in the main directory of this archive for
 more details.

*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "meroko.h"
#include "nubus.h"

#define RAM_TOP 0x800000
#define PTY_TOP 0x100000

// PARITY STORAGE: 8 bytes per byte.
// Rotate address right 3 to get offset, 7-0 is rotation count.

unsigned char MEM_ROM[2048];
unsigned char MEM_RAM[RAM_TOP];
unsigned char MEM_PTY[PTY_TOP];

char msg[256];
int mem_tracerq=0;

// Registers
unsigned long mem_test_reg = 0;
unsigned char mem_base_reg=0xF4;
unsigned char mem_failure_reg=0;
unsigned long mem_term_reg=0; 
unsigned char mem_ptyf_bits=0; // Parity Failed bits
unsigned char mem_ptyu_bits=0; // Parity Used bits

#define PROM_FNAME "proms/2243924-2_8MB"

// Externals
extern int cpu_die_rq;
extern unsigned int ldb(unsigned long long value, int size, int position);
extern void logmsg(char *msg);
extern int breakpoint_armed; 
extern int NUbus_error;

// Registers
unsigned char mem8_config_register = 0x0; 

void mem8_init(){
  FILE *romfile;
  int x=0;

  bzero(MEM_ROM,2048);
  bzero(MEM_RAM,RAM_TOP);

  romfile = fopen(PROM_FNAME,"r");
  if(romfile == NULL){
    perror("mem8-fopen");
    exit(-1);
  }
  x=0;
  while(x < 2048){
    fread(&MEM_ROM[0x400^x],1,1,romfile);
    x++;
  }

  fclose(romfile);
}

inline unsigned int genparity(unsigned char data){
  unsigned int x=0x1;
  unsigned int y=0x0;
  while(x < 0x100){
    if((data&x)==x){
      y++;
    } 
    x = x << 1; 
  }  
  y &= 1;
  return(y); // Return EVEN parity
}

inline unsigned int storepty(unsigned long address,unsigned char data){
  unsigned long ptyadr;
  unsigned int bitcnt;
  unsigned char stgbit;
  unsigned char ptybit;

  ptyadr = (address>>3);
  bitcnt = (address&0x07);

  stgbit = (0x01 << bitcnt);

  // Generate parity to store
  if((mem_test_reg&0x1000) == 0x1000){
    ptybit = (((mem_test_reg&0xF00)>>(8+(address&0x3)))&0x01);
    // The parity-test register bits are inverted.
    ptybit ^= 0x01; // and flip
    /*    if(mem_tracerq){ 
      sprintf(msg,"PTY-STOR: %X\n",ptybit); logmsg(msg);
      } */
  }else{
    ptybit = genparity(data);
  }

  if(ptybit==0){
    MEM_PTY[ptyadr] &= (~stgbit);
  }else{
    MEM_PTY[ptyadr] |= stgbit;
  }
  mem_ptyu_bits |= (ptybit << (address&0x03));
  return(ptybit);
}

inline unsigned char fetchpty(unsigned long address){
  unsigned long ptyadr;
  unsigned int bitcnt;
  unsigned char stgbit;

  ptyadr = (address>>3);
  bitcnt = (address&0x07);
  stgbit = MEM_PTY[ptyadr];
  stgbit >>= bitcnt;
  return(stgbit&1);  
}

inline unsigned char chkpty(unsigned long address){
  // Parity Test
  int x,y;
  x = fetchpty(address); 
  y = genparity(MEM_RAM[address]);

  // Replace fetched parity if testing?
  if((mem_test_reg&0x1000) == 0x1000){
    x = ((mem_test_reg&0xF00)>>(8+(address&0x3)));
    // The parity-test register bits are inverted.
    x = ((x&0x1)^0x1); // and flip
  }
  

  if(x != y){
    NUbus_error = 1;
    mem_ptyf_bits |= (0x01 << (address&0x03));
    if((mem_test_reg&0x1000) != 0x1000){
      sprintf(msg,"PTY-FAIL OUTSIDE TEST @ 0x%lX, pty %d %d\n",address,x,y); logmsg(msg);
    }
  }else{
    /*    if(mem_tracerq != 0){
      sprintf(msg,"PTY-PASS: x = %d, y = %d\n",x,y); logmsg(msg);
      } */
    mem_ptyu_bits |= (1 << (address&0x03));
  }
  return(x);
}

void mem8_nubus_io(){
  unsigned long NUbus_Addr = ldb(NUbus_Address,24,0);  

  if(mem_tracerq){
    sprintf(msg,"MEM8: NUbus IO-(%ld): Addr %lX, Data %lX\n",NUbus_Request,NUbus_Addr,NUbus_Data); logmsg(msg); 
  } 

  // Handle RAM here
  if(((NUbus_Request&0xF) == VM_READ || (NUbus_Request&0xF) == VM_BYTE_READ) && NUbus_Addr < RAM_TOP){
    unsigned long RAM_Addr;

    // Initialize
    NUbus_Data = 0;
    RAM_Addr = NUbus_Addr;
    mem_failure_reg = 0; // Reset
    mem_ptyu_bits = 0;
    mem_ptyf_bits = 0;

    /*
    if(NUbus_Addr == 0x00001B){
      mem_tracerq=1;
      logmsg("2510'd!\n"); 
      sprintf(msg,"DEBUG: Nubus-Write-IO-(%ld): %lX @ %lX\n",NUbus_Request,NUbus_Data,RAM_Addr); logmsg(msg);         
      cpu_die_rq=1; }
    */

    if((NUbus_Request&0xF) == VM_READ){ // Read four bytes
      if((RAM_Addr&3) != 0){
	//	mem_tracerq=1;
	//	logmsg("HALFWORD READ\n");
	switch(RAM_Addr&0x03){
	case 1: // Read Low Half
	  NUbus_Data =  MEM_RAM[RAM_Addr];
	  chkpty(RAM_Addr);
	  NUbus_Data = NUbus_Data << 8;
	  
	  NUbus_Data |= MEM_RAM[RAM_Addr-1]; 
	  chkpty(RAM_Addr-1);
	  //	  NUbus_Data = NUbus_Data << 16;
	  break;

	case 2: // Block Transfer (ILLEGAL)
	  logmsg("MEM8: BLOCK READ REQUESTED\n");
	  cpu_die_rq=1;
	  break;

	case 3: // Read High Half
	  NUbus_Data = MEM_RAM[RAM_Addr];
	  chkpty(RAM_Addr);
	  NUbus_Data = NUbus_Data << 8;
	  
	  NUbus_Data |= MEM_RAM[RAM_Addr-1]; 
	  chkpty(RAM_Addr-1);
	  NUbus_Data = NUbus_Data << 16;
	  break;
	}
      }else{
	// Full word read
	NUbus_Data =  MEM_RAM[RAM_Addr+3];
	chkpty(RAM_Addr+3);
	NUbus_Data = NUbus_Data << 8;
	
	NUbus_Data |= MEM_RAM[RAM_Addr+2]; 
	chkpty(RAM_Addr+2);
	NUbus_Data = NUbus_Data << 8;
	
	NUbus_Data |= MEM_RAM[RAM_Addr+1]; 
	chkpty(RAM_Addr+1);
	NUbus_Data = NUbus_Data << 8;
	
	NUbus_Data |= MEM_RAM[RAM_Addr+0];
	chkpty(RAM_Addr+0);
      }
    }else{
      unsigned int ptydata;
      // BYTE READ
      NUbus_Data = MEM_RAM[RAM_Addr+0];
      ptydata = chkpty(RAM_Addr+0);
      
      // ptyu is not PTY USED, it's PTY TEST OK?
      /*
      if(ptydata != 0){
	mem_ptyu_bits |= 0x0F;
      }else{
	mem_ptyu_bits &= 0xF0;
      }
      */
    }

    if(mem_ptyf_bits != 0){
      mem_term_reg |= 0x8000;
      mem_failure_reg = (RAM_Addr&0x03);
      mem_failure_reg <<= 5;
      mem_failure_reg |= 0x10; // Was 0x40
      // mem_failure_reg |= (mem_ptyf_bits^0xF);
      // 0 means failure?
    }else{
      mem_term_reg &= 0x7FFF;
      mem_failure_reg = (RAM_Addr&0x03);
      mem_failure_reg <<= 5;
      mem_failure_reg |= 0x0F; // was ptyu
    }

    // Rotation for byte reads
    if((NUbus_Request&0xF) == VM_BYTE_READ){
      NUbus_Data = NUbus_Data << (8 * (NUbus_Addr&0x3));
    }

#ifdef TRACELOG
    sprintf(msg,"DEBUG: Nubus-Read-IO- (%ld): %lX @ %lX\n",NUbus_Request,NUbus_Data,RAM_Addr); logmsg(msg);     
#endif

    NUbus_acknowledge=1;
    return;
  }

  if(((NUbus_Request&0xF) == VM_WRITE || (NUbus_Request&0xF) == VM_BYTE_WRITE) && NUbus_Addr < RAM_TOP){
    unsigned long RAM_Data = NUbus_Data;
    unsigned long RAM_Addr = NUbus_Addr;

    mem_failure_reg = 0; // Reset
    mem_ptyu_bits = 0;
    mem_ptyf_bits = 0;

    // Store
    if((NUbus_Request&0xF) == VM_BYTE_WRITE){ // && (RAM_Addr&0x3) != 0){
      // BYTE WRITE
      unsigned int ptydata;

      // Derotation for byte IO
      RAM_Data = RAM_Data >> (8 * (RAM_Addr&0x3));

      MEM_RAM[RAM_Addr+0] = RAM_Data&0xFF;     // Store data
      ptydata = storepty(RAM_Addr,RAM_Data&0xFF);        // And parity
      if(ptydata != 0){
	mem_ptyu_bits |= 0x0F;
      }else{
	mem_ptyu_bits &= 0xF0;
      }
    }else{
      // WORD WRITE
      if((RAM_Addr&3) != 0){
	//	cpu_die_rq=1;
	switch(RAM_Addr&0x03){
	case 1: // Write low half
	  //	  sprintf(msg,"MEM8: LOW HALF WRITE: Data = 0x%lX\n",RAM_Data); logmsg(msg);
	  MEM_RAM[RAM_Addr-1] = RAM_Data&0xFF;     
	  storepty(RAM_Addr-1,RAM_Data&0xFF);
	  RAM_Data = RAM_Data >> 8; // Next!

	  MEM_RAM[RAM_Addr] = RAM_Data&0xFF; 
	  storepty(RAM_Addr,RAM_Data&0xFF);
	  break;

	case 2: // BLOCK TRANSFER (ILLEGAL)
	  logmsg("MEM8: BLOCK TRANSFER REQUESTED\n");
	  cpu_die_rq=1;
	  break;

	case 3: // Write high half
	  // sprintf(msg,"MEM8: HIGH HALF WRITE: Data = 0x%lX\n",RAM_Data); logmsg(msg);
	  RAM_Data = RAM_Data >> 16; // Eliminate low half
	  MEM_RAM[RAM_Addr-1] = RAM_Data&0xFF;     
	  storepty(RAM_Addr-1,RAM_Data&0xFF);
	  RAM_Data = RAM_Data >> 8; // Next!

	  MEM_RAM[RAM_Addr] = RAM_Data&0xFF; 
	  storepty(RAM_Addr,RAM_Data&0xFF);
	  break;
	}
      }else{
	MEM_RAM[RAM_Addr] = RAM_Data&0xFF;     
	storepty(RAM_Addr,RAM_Data&0xFF);
	RAM_Data = RAM_Data >> 8; // Next!
	
	MEM_RAM[RAM_Addr+1] = RAM_Data&0xFF; 
	storepty(RAM_Addr+1,RAM_Data&0xFF);
	RAM_Data = RAM_Data >> 8;
	
	MEM_RAM[RAM_Addr+2] = RAM_Data&0xFF; 
	storepty(RAM_Addr+2,RAM_Data&0xFF);
	RAM_Data = RAM_Data >> 8;
	
	MEM_RAM[RAM_Addr+3] = RAM_Data&0xFF; 
	storepty(RAM_Addr+3,RAM_Data&0xFF);
      }
    }

    mem_term_reg &= 0x7FFF;
    mem_failure_reg = (RAM_Addr&0x03);
    mem_failure_reg <<= 5;
    mem_failure_reg |= mem_ptyu_bits;

#ifdef TRACELOG
    sprintf(msg,"DEBUG: Nubus-Write-IO-(%ld): %lX @ %lX\n",NUbus_Request,NUbus_Data,RAM_Addr); logmsg(msg);         
#endif

    NUbus_acknowledge=1;
    return;
  }

  // Handle ROM here
  if(((NUbus_Request&0xF) == VM_READ || (NUbus_Request&0xF) == VM_BYTE_READ) && NUbus_Addr >= 0xFFE000){
    unsigned int ROMaddr = ((NUbus_Addr >> 2)&0x7FF);

    // ROMaddr ^= 0x400; // HO HO HO, VERY FUNNY
    if(NUbus_Addr&0x3){logmsg("MEM8: ODD ROM ADDR\n"); cpu_die_rq=1; }
   
    NUbus_Data = MEM_ROM[ROMaddr]&0xFF;

    //    sprintf(msg,"MEM8: Read %lX for address %lX (ROM address %X)\n",NUbus_Data,NUbus_Addr,ROMaddr); logmsg(msg);

    NUbus_acknowledge=1;
    return;
  }  

  // Registers
  if((NUbus_Request&0xF) == VM_BYTE_WRITE){
    switch(NUbus_Addr){

    case 0xFFC000: // Configuration Register
      /* BITS ARE:
	 0x1 = BOARD RESET (WRITE ONLY)
	 0x4 = BOARD LED (RW?)
      */
      if((NUbus_Data&0x01)==0x01){ 
	// mem8_config_register = 0; 
	// logmsg("MEM8: RESET\n");
	mem_test_reg = 0;
	mem_base_reg = 0xF4;
      }
      if((NUbus_Data&0x04)==0x04){ 
	mem8_config_register |= 0x04; 
      }else{ 
	mem8_config_register &= (~0x04); 
      }
      NUbus_acknowledge=1;
      // cpu_die_rq=1;
      return;

    case 0xFFC008: // Base Register
      mem_base_reg = NUbus_Data;
      //      sprintf(msg,"MEM8: TEST-REGISTER-WRITE 0x%lX\n",mem_test_reg); logmsg(msg);
      NUbus_acknowledge=1;
      return;

    case 0xFFC010: // Failure Location Latch
      // mem_failure_reg = NUbus_Data; -- THIS IS A READ-ONLY REGISTER!
      //      sprintf(msg,"MEM8: Failure Latch write, 0x%lX\n",NUbus_Data); logmsg(msg);
      // Reset errors, maybe?
      //      mem_ptyf_bits = 0;
      // mem_term_reg &= 0x7FFF;
      // Then
      mem_test_reg = NUbus_Data; // LOAD THIS
      NUbus_acknowledge=1;
      // It looks like the diagnostic expects it to fall into FFC011.      
      //      NUbus_Data >>= 8;
      // cpu_die_rq=1;
      // sprintf(msg,"MEM8: FLL-WRITE 0x%lX\n",NUbus_Data); logmsg(msg);
      return;
    
    case 0xFFC011: // Test Register
      /* BITS ARE:
	 100 - Test Parity 0 (0x000000FF)
	 200 - Test Parity 1 (0x0000FF00)
	 400 - Test Parity 2 (0x00FF0000)
	 800 - Test Parity 3 (0xFF000000)
	1000 - Test Enable
      */
      mem_test_reg = NUbus_Data&0xFF00; // breakpoint_armed = 1; // cpu_die_rq=1;
      // sprintf(msg,"MEM8: MTR-WRITE 0x%lX\n",NUbus_Data); logmsg(msg);
      NUbus_acknowledge=1;
      return;

    }
  }
  if((NUbus_Request&0xF) == VM_BYTE_READ){
    switch(NUbus_Addr){
    case 0xFFC000: // Configuration Register
      /* BITS ARE:
	 0x1 = BOARD RESET (WRITE ONLY)
	 0x4 = BOARD LED (RW?)
      */
      NUbus_Data = mem8_config_register;
      NUbus_acknowledge=1;
      // cpu_die_rq=1;
      return;

    case 0xFFC008: // Base Register
      NUbus_Data = mem_base_reg;
      //      sprintf(msg,"MEM8: TEST-REGISTER-WRITE 0x%lX\n",mem_test_reg); logmsg(msg);
      NUbus_acknowledge=1;
      return;

    case 0xFFC010: // Failure Location Latch
      NUbus_Data = mem_failure_reg&0xFF;
      // sprintf(msg,"MEM8: Failure Latch Read, 0x%lX\n",NUbus_Data); logmsg(msg);
      NUbus_acknowledge=1;
      // cpu_die_rq=1;
      // mem_tracerq=1;
      return;      

    case 0xFFC011: // Test Register
      NUbus_Data = mem_test_reg&0xFF00;
      // sprintf(msg,"MEM8: TEST-REGISTER-READ 0x%lX\n",mem_test_reg); logmsg(msg);
      NUbus_acknowledge=1;
      return;
    }
  }

  sprintf(msg,"MEM8: Unknown NUbus IO-(%ld): %lX\n",NUbus_Request,NUbus_Addr); logmsg(msg); 
  cpu_die_rq=1;
}
