/* NUbus Peripheral Interface - Implementation

 This file is subject to the terms and conditions of the GNU General Public
 License.  See the file COPYING in the main directory of this archive for
 more details.

*/


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "meroko.h"
#include "nubus.h"

char nupi_msg[256];
unsigned char NUPI_ROM[0x4000]; // 16K ROM
unsigned char NUPI_RAM[0x1000]; // 4K MPU RAM

unsigned long NUPI_XFER_BUFFER[0x1000]; // 1KW transfer area
unsigned int nupi_xfer_pointer=0; 
unsigned int nupi_unit_pointer=0;

// The NuPI has a 68000 MPU on board that runs the ROM code.
// The MPU is clocked at 10MHz.

// Externals
extern int cpu_die_rq,Memory_Busy;
extern unsigned int ldb(unsigned long long value, int size, int position);
extern void logmsg(char *nupi_msg);
extern unsigned long nubus_io_request(int access, unsigned long address, unsigned long data);

// Registers
unsigned long nupi_test_addr=0;
unsigned long nupi_test_indx=0;
unsigned int  nupi_test_loop=0;
unsigned long nupi_test_prog=0;
unsigned long nupi_master_event_addr=0;

unsigned long nupi_cmd_addr = 0;
unsigned long nupi_cmd_prog = 0;
unsigned long nupi_cmd_dadr = 0;
unsigned long nupi_cmd_word = 0;
unsigned int nupi_go = 0;

unsigned int nupi_retry_count = 0;
unsigned int nupi_dead_cmd_disable = 0;
unsigned int nupi_periodic_poll_disable = 0;
unsigned int nupi_retry_disable = 0;
unsigned int nupi_ecc_disable = 0;

// Fields in command block
// Word 0
unsigned char nupi_dst_bit,nupi_formatter_bit,nupi_command;
unsigned char nupi_event,nupi_scatter,nupi_swap_completion;
unsigned char nupi_formatterselect,nupi_devselect;
// Word 1
unsigned long nupi_status;
// Word 2
unsigned long nupi_buffer_pointer;
// Word 3
unsigned long nupi_transfer_count;
// Word 4
unsigned long nupi_device_block_addr;
// Word 5
unsigned long nupi_event_gen_addr;
// Word 6 and 7 are reserved.

// A device
struct nupi_device_entry{
  int fd;                           // Disk file descriptor
  unsigned int last_cmd_option;     // Option field of last command completed
  unsigned int last_cmd;            // Last command completed
  unsigned int cmd_option;          // Option of currently executing command
  unsigned int cmd;                 // Currently executing command
  unsigned int removable;           // Removable device
  unsigned int busy;                // Device is busy
  unsigned int attention;           // Unit requires service
  unsigned int indt_status;         // Unit has indeterminate status
  unsigned int writeprotect;        // Unit is write-protected
  unsigned int overtemp;            // Unit has overtemperature alarm
  unsigned int selftest;            // Unit self-test status
  unsigned int error;               // Unit has error condition
  unsigned int offline;             // Unit is offline
  unsigned int devtype;             // Device type
};

struct nupi_device_entry nupi_device[20];

// NUPI devices:
// 00 = Formatter 0
// 01 = Formatter 1
// 02 = Formatter 2
// 03 = Formatter 3
// 04 = Formatter 4
// 05 = Formatter 6
// 06 = Formatter 7
// 07 = Formatter 0,Device 0
// 08 = Formatter 0,Device 1
// 09 = Formatter 1,Device 0
// 10 = Formatter 1,Device 1
// 11 = Formatter 2,Device 0
// 12 = Formatter 2,Device 1
// 13 = Formatter 3,Device 0
// 14 = Formatter 3,Device 1
// 15 = Formatter 4,Device 0
// 16 = Formatter 4,Device 1
// 17 = Formatter 6,Device 0
// 18 = Formatter 6,Device 1
// 19 = Formatter 7,Device 0
// 20 = Formatter 7,Device 1

#define PROM1_FNAME "proms/2238056-5_NUPI"
#define PROM2_FNAME "proms/2238057-5_NUPI"

void nupi_init(){
  FILE *romfile1,*romfile2;
  int x=0;
  unsigned long tmp;
  romfile1 = fopen(PROM1_FNAME,"r");
  romfile2 = fopen(PROM2_FNAME,"r");
  if(romfile1 == NULL){
    perror("nupi-fopen-1");
    exit(-1);
  }
  if(romfile2 == NULL){
    perror("nupi-fopen-2");
    exit(-1);
  }
  while(x < 0x4000){
    fread(&tmp,1,1,romfile1);
    NUPI_ROM[x+3] = tmp;
    x++;
    fread(&tmp,1,1,romfile2);
    NUPI_ROM[x+1] = tmp; 
    x++;
    fread(&tmp,1,1,romfile1);
    NUPI_ROM[x-1] = tmp; 
    x++;
    fread(&tmp,1,1,romfile2);
    NUPI_ROM[x-3] = tmp; 
    x++;
  }
  fclose(romfile1);
  fclose(romfile2);
  /*
  printf("NUPI ROM BYTES = %X %X %X %X - %X %X %X %X - %X %X %X %X - %X %X %X %X\n",
	 NUPI_ROM[0x3B60],NUPI_ROM[0x3B61],NUPI_ROM[0x3B62],NUPI_ROM[0x3B63],
	 NUPI_ROM[0x3B64],NUPI_ROM[0x3B65],NUPI_ROM[0x3B66],NUPI_ROM[0x3B67],
	 NUPI_ROM[0x3B68],NUPI_ROM[0x3B69],NUPI_ROM[0x3B6A],NUPI_ROM[0x3B6B],
	 NUPI_ROM[0x3B6C],NUPI_ROM[0x3B6D],NUPI_ROM[0x3B6E],NUPI_ROM[0x3B6F]);
  exit(-1);
  */  

  // Initialize devices to disconnected-offline
  {
    int x=0;
    while(x<7){
      nupi_device[x].offline=1;
      nupi_device[x].last_cmd_option=0;
      nupi_device[x].last_cmd=0;
      nupi_device[x].cmd_option=0;
      nupi_device[x].cmd=0;       
      nupi_device[x].removable=0; 
      nupi_device[x].busy=0;      
      nupi_device[x].attention=0; 
      nupi_device[x].indt_status=0;
      nupi_device[x].writeprotect=0;
      nupi_device[x].overtemp=0;    
      nupi_device[x].selftest=0;    
      nupi_device[x].error=0;       
      nupi_device[x].devtype=0;     
      x++;
    }
  }
  // Set formatter 0,1,3 to online
  nupi_device[0].offline = 0;
  nupi_device[1].offline = 0;
  nupi_device[3].offline = 0;
  // Attach drives 0,1,4
  nupi_device[7].fd = open("/X1-DISKS/c0-d0.dsk",O_RDWR);
  nupi_device[9].fd = open("/X1-DISKS/c0-d1.dsk",O_RDWR); 
  // Was 15 in Nevermore
  nupi_device[13].fd = open("/X1-DISKS/c2-d0.dsk",O_RDWR);
  
  if(nupi_device[7].fd < 0){
    perror("Disk-0:open");
    exit(-1);
  }
  if(nupi_device[9].fd < 0){
    perror("Disk-1:open");
    exit(-1);
  }
  if(nupi_device[13].fd < 0){
    perror("Disk-2:open");
    exit(-1);
  }
  nupi_device[7].offline=0;
  nupi_device[8].offline=1;
  nupi_device[9].offline=0;
  nupi_device[10].offline=1;
  nupi_device[13].offline=0;
  nupi_device[14].offline=1; 

  nupi_device[7].devtype = 0x02; // HARD DISK DRIVE
  nupi_device[9].devtype = 0x02; // HARD DISK DRIVE 
  nupi_device[13].devtype = 0x02; // HARD DISK DRIVE
}

int nupi_trace_rq = 0;

void nupi_nubus_io(){
  unsigned long NUbus_Addr = ldb(NUbus_Address,24,0);  

  //  cpu_die_rq=1;

  // Exclude ROM here
  if((NUbus_Request == VM_READ || NUbus_Request == VM_BYTE_READ) && NUbus_Addr >= 0xFFC000){
    unsigned int ROMaddr = (NUbus_Addr&0x3FFF);

    if(NUbus_Request == VM_READ){
      // Read other 3 bytes (This is probably wrong)
      NUbus_Data =  NUPI_ROM[ROMaddr+3]; NUbus_Data <<= 8;
      NUbus_Data |= NUPI_ROM[ROMaddr+2]; NUbus_Data <<= 8;
      NUbus_Data |= NUPI_ROM[ROMaddr+1]; NUbus_Data <<= 8; 
      NUbus_Data |= NUPI_ROM[ROMaddr+0]; // NUbus_Data &= 0xFF;
    }else{
      NUbus_Data = NUPI_ROM[ROMaddr+0];
      NUbus_Data <<= (8*(NUbus_Addr&0x3));
      //      NUbus_Data &= 0xFF;
    }
    NUbus_acknowledge=1;
    // cpu_die_rq=1;

    /*
    if(nupi_trace_rq){
      sprintf(nupi_msg,"NuPI: IO-(%ld) data %lX for address %lX\n",NUbus_Request,NUbus_Data,NUbus_Addr); logmsg(nupi_msg);    
    }
    */
   
    return;
  }

  // Registers
  if(NUbus_Request == VM_BYTE_READ){
    switch(NUbus_Addr){
    case 0xD40002: // FLAG REGISTER
      // BITS ARE:
      // 0x10000 = 0-Self-Test Completed
      // 0x20000 = 0-Self-Test Passed
      // 0x40000 = 0-SCSI Test Passed
      // One indicates failures
      // We will cheat and return all passes.
      NUbus_Data = 0x0;
      NUbus_acknowledge = 1;
      return;

    case 0xE0000B: // Configuration Register
      /* BITS ARE:
	 01 = RESET
	 02 = BUS-MASTER ENABLE
	 04 = FAULT-LED
	 08 = SYSTEM-BUS-TEST
	 30 = MUST-BE-ZERO
	 40 = FAILURE-OVERRIDE
      */

      NUbus_Data = NUPI_RAM[0xB];
      NUbus_Data <<= 24; 
      NUbus_Data = 0;
      NUbus_acknowledge=1;
      return;	
    }
  }

  if(NUbus_Request == VM_WRITE){
    switch(NUbus_Addr){
    case 0xE00004: // Command Address Register
      {
	nupi_cmd_addr = NUbus_Data;
	nupi_go = 1;
	NUbus_acknowledge=1;
      }
      return;
      
    default:
      sprintf(nupi_msg,"NuPI: Unknown NUbus word-write - 0x%lX @ %lX\n",NUbus_Data,NUbus_Addr);
      logmsg(nupi_msg);      
      cpu_die_rq=1;      
    }
  }

  if(NUbus_Request == VM_BYTE_WRITE){
    unsigned char NUPI_DATA;

    // Derotate
    NUPI_DATA = ((NUbus_Data >> (8*(NUbus_Addr&0x3)))&0xFF);

    switch(NUbus_Addr){
    case 0xE0000B: // Configuration Register
      // Handle bits that must be handled
      if(NUPI_DATA&0x1){ 
	logmsg("NUPI: RESET\n"); 
	NUPI_RAM[0xB] = 0; // Clobber status
      }
      if(NUPI_DATA&0x2){
	logmsg("NUPI: BUS-MASTER-ENABLE\n"); 
	//	cpu_die_rq=1;
      }
      if(NUPI_DATA&0x8){
	logmsg("NUPI: SYSTEM-BUS-TEST STARTED\n");
	// This causes the NuPI to become a master, copy the configuration ROM to the memory location indicated in
	// E0000F, read it back, and repeat this three times.
	nupi_test_prog=1;
	//	cpu_die_rq=1;
      }
      NUPI_DATA &= 0x0A; // Don't write reset or LED bit into RAM
      break;

    case 0xE0000F: // DMA-Test-Register
      // Safe to load without intervention
      // Contains memory test address for bus test.
      // 0xF0 = PA:27-24
      // 0x0F = PA:15-12
      // So for 0xAB, PA = 0x0A00B000
      nupi_test_addr = 0xF0000;                // F0000
      nupi_test_addr |= (NUPI_DATA&0xF0) << 8; // FA000
      nupi_test_addr |= (NUPI_DATA&0xF);       // FA00B
      nupi_test_addr <<= 12;                   // FA00B000
      break;

    default:
      sprintf(nupi_msg,"NuPI: Unknown NUbus byte-write - 0x%lX @ %lX\n",NUbus_Data,NUbus_Addr); logmsg(nupi_msg);      
      cpu_die_rq=1;
    }
    
    // handle debug window
    if(NUbus_Addr >= 0xE00000 && NUbus_Addr <= 0xE01000){
      // Microprocessor RAM area
      unsigned int NUPI_ADDR = (NUbus_Addr - 0xE00000);
      // Store
      NUPI_RAM[NUPI_ADDR] = NUPI_DATA;
      // Return
      NUbus_acknowledge=1;
      return;
    }
  }

  sprintf(nupi_msg,"NuPI: Unknown NUbus IO-(%ld): %lX\n",NUbus_Request,NUbus_Addr); logmsg(nupi_msg); 
  cpu_die_rq=1;
}

void nupi_clock_pulse(){

  // Check things
  if((NUPI_RAM[0xB]&0x2)==0x2 && nupi_test_prog > 0){
    switch(nupi_test_prog){
    case 0: // STOP (Shouldn't ever get here!)
      break;
    case 1: // START WRITE PHASE
      if(nupi_test_indx < 0x40){
	if(Memory_Busy == 0){
	  nubus_io_request(NB_WRITE,nupi_test_addr+(4*nupi_test_indx),NUPI_ROM[0xFC0+nupi_test_indx]);
	  nupi_test_indx++;
	}
      }else{
	nupi_test_prog++;
	nupi_test_indx=0;
      }
      return;
    case 2: // START READ PHASE
      if(nupi_test_indx < 0x40){
	// Don't actually check the result, for now.
	if(Memory_Busy == 0){
	  nubus_io_request(NB_READ,nupi_test_addr+(4*nupi_test_indx),0);
	  nupi_test_indx++;
	}
      }else{ 
	nupi_test_prog++;
	nupi_test_indx=0;
      }
      return;
    case 3: // LOOP
      if(nupi_test_loop < 3){
	nupi_test_indx=0;
	nupi_test_prog=1;
	nupi_test_loop++;
      }else{
	nupi_test_prog=0;
	NUPI_RAM[0xB] &= 0xF7; // Turn off test bit
	logmsg("NUPI: System Bus Test Completed\n");
      }      
      return;
    }
  }
  // Test loop finished or not running, check for commands.
  if((NUPI_RAM[0xB]&0x2)==0x2 && nupi_go != 0){
    switch(nupi_cmd_prog){
    case 0: // FETCH
      nubus_io_request(NB_READ,nupi_cmd_addr,0);
      nupi_cmd_prog++;
      break;

    case 1: // LOAD COMMAND WORD
      if(Memory_Busy > 0){
	break;
      }else{
	nupi_cmd_word = NUbus_Data;
	nubus_io_request(NB_READ,nupi_cmd_addr+0x04,0);
	nupi_cmd_prog++;
      }
      break;

    case 2: // LOAD STATUS WORD
      if(Memory_Busy > 0){
	break;
      }else{
	nupi_status = NUbus_Data;
	nubus_io_request(NB_READ,nupi_cmd_addr+0x08,0);
	nupi_cmd_prog++;
      }
      break;

    case 3: // LOAD BUFFER POINTER
      if(Memory_Busy > 0){
	break;
      }else{
	nupi_buffer_pointer = NUbus_Data;
	sprintf(nupi_msg,"NUPI: Buffer Pointer = 0x%lX\n",nupi_buffer_pointer); logmsg(nupi_msg);
	nubus_io_request(NB_READ,nupi_cmd_addr+0x0C,0);
	nupi_cmd_prog++;
      }
      break;

    case 4: // LOAD TRANSFER COUNT
      if(Memory_Busy > 0){
	break;
      }else{
	nupi_transfer_count = NUbus_Data;
	nubus_io_request(NB_READ,nupi_cmd_addr+0x10,0);
	nupi_cmd_prog++;
      }
      break;

    case 5: // LOAD DEVICE BLOCK ADDRESS COUNT
      if(Memory_Busy > 0){
	break;
      }else{
	nupi_device_block_addr = NUbus_Data;
	nubus_io_request(NB_READ,nupi_cmd_addr+0x14,0);
	nupi_cmd_prog++;
      }
      break;

    case 6: // LOAD EVENT GENERATION ADDRESS
      if(Memory_Busy > 0){
	break;
      }else{
	nupi_event_gen_addr = NUbus_Data;
	sprintf(nupi_msg,"NUPI: Event Address = 0x%lX\n",nupi_event_gen_addr); logmsg(nupi_msg);
	nupi_cmd_prog++;
      }
      // Fall into execute

    case 7: // EXECUTE
      nupi_dst_bit = (nupi_cmd_word&0x80000000)>>30;              // The NUPI is our destination
      nupi_formatter_bit = (nupi_cmd_word&0x40000000)>>29;        // A formatter is our destination
      nupi_command = ((nupi_cmd_word&0x3F000000)>>24);            // The command itself
      nupi_event = (nupi_cmd_word&0x800000)>>23;
      nupi_scatter = (nupi_cmd_word&0x400000)>>22;
      nupi_swap_completion = (nupi_cmd_word&0x100000)>>20;
      nupi_formatterselect = ((nupi_cmd_word&0x38)>>3);
      nupi_devselect = (nupi_cmd_word&0x1);

      sprintf(nupi_msg,"NUPI: Got command word %lX\n",nupi_cmd_word); logmsg(nupi_msg);
      if(nupi_scatter){ logmsg("SCATTER MODE DOESN'T WORK YET!\n"); cpu_die_rq=1; }

      // Tell CPU we are busy
      nupi_cmd_word = 0x80000000; // BUSY code
      nubus_io_request(NB_WRITE,nupi_cmd_addr+4,nupi_cmd_word); // Write it back
      
      switch(nupi_command){

      case 0x01: // SETUP COMMAND
	// Setting nupi status
	if(nupi_dst_bit){
	  sprintf(nupi_msg,"NUPI: SETUP COMMAND with 0x%lX words of output\n",(nupi_transfer_count>>2)); logmsg(nupi_msg);
	  // TRIGGER BUFFER LOAD
	  nupi_cmd_prog=20; // IO BUFFER BUSY-LOOP
	  nupi_xfer_pointer = 0;
	  return;
	}
	logmsg("NUPI-Setup: Unknown destination\n");
	cpu_die_rq=1; nupi_cmd_prog=0; nupi_go=0;
	return;
	break;

      case 0x02: // REQUEST STATUS
	// Requesting nupi status?
	if(nupi_dst_bit){
	  int x,y;
	  x=0;
	  sprintf(nupi_msg,"NUPI: status requested with 0x%lX words of output\n",(nupi_transfer_count>>2)); logmsg(nupi_msg);
	  // Write out NUPI status
	  NUPI_XFER_BUFFER[0] = 0x0; // No error conditions set
	  if(nupi_transfer_count > x){ // More bytes to write?
	    NUPI_XFER_BUFFER[1] = 0x0; // Selftest information
	    x += 4;
	  }
	  y=0;
	  while(nupi_transfer_count > x && y < 20){ // More bytes to write?
	    NUPI_XFER_BUFFER[2+y] = nupi_device[y].devtype;
	    NUPI_XFER_BUFFER[2+y] <<= 1;
	    NUPI_XFER_BUFFER[2+y] |= nupi_device[y].offline; 
	    NUPI_XFER_BUFFER[2+y] <<= 1;
	    NUPI_XFER_BUFFER[2+y] |= nupi_device[y].selftest; 
	    NUPI_XFER_BUFFER[2+y] <<= 1;
	    NUPI_XFER_BUFFER[2+y] |= nupi_device[y].overtemp; 
	    NUPI_XFER_BUFFER[2+y] <<= 1;
	    NUPI_XFER_BUFFER[2+y] |= nupi_device[y].writeprotect; 
	    NUPI_XFER_BUFFER[2+y] <<= 1;
	    NUPI_XFER_BUFFER[2+y] |= nupi_device[y].indt_status; 
	    NUPI_XFER_BUFFER[2+y] <<= 1;
	    NUPI_XFER_BUFFER[2+y] |= nupi_device[y].attention; 
	    NUPI_XFER_BUFFER[2+y] <<= 1;
	    NUPI_XFER_BUFFER[2+y] |= nupi_device[y].busy; 
	    NUPI_XFER_BUFFER[2+y] <<= 1;
	    NUPI_XFER_BUFFER[2+y] |= nupi_device[y].removable; 
	    NUPI_XFER_BUFFER[2+y] <<= 1;
	    NUPI_XFER_BUFFER[2+y] <<= 4; // Reserved bits
	    NUPI_XFER_BUFFER[2+y] <<= 8;	    
	    NUPI_XFER_BUFFER[2+y] |= nupi_device[y].last_cmd;
	    NUPI_XFER_BUFFER[2+y] <<= 8;
	    NUPI_XFER_BUFFER[2+y] |= nupi_device[y].last_cmd_option;
	    x += 4; y++;
	  }
	  // Next is ... wierd stuff.
	  if(nupi_transfer_count > x){ // More bytes to write?
	    sprintf(nupi_msg,"NuPI: More status? Geez... It wants %ld more words of data.\n",((nupi_transfer_count-x)>>2));
	    logmsg(nupi_msg);
	  }
	  // Done
	  nupi_cmd_prog=8; // WRITE RESULT
	  nupi_xfer_pointer = 0;
	  return;
	}
	if(nupi_formatter_bit){	
	  logmsg("NUPI-Request-Status: Unknown destination\n");
	  cpu_die_rq=1; nupi_cmd_prog=0; nupi_go=0;
	  return;
	}
	// Target is a device.
	{
	  int x=0,y=0;
	  sprintf(nupi_msg,"NUPI: DRIVE STATUS RQ: Target is ID %d, LUN %d\n",nupi_formatterselect,nupi_devselect); logmsg(nupi_msg);
	  sprintf(nupi_msg,"NUPI: status requested with 0x%lX words of output\n",(nupi_transfer_count>>2)); logmsg(nupi_msg);
	  x = 4;
	  
	  y = 7 + (nupi_formatterselect<<1);
	  y |= nupi_devselect;
	  sprintf(nupi_msg,"NUPI: Internal device # is %d\n",y); logmsg(nupi_msg);
	  NUPI_XFER_BUFFER[0] = nupi_device[y].devtype;
	  NUPI_XFER_BUFFER[0] <<= 1;
	  NUPI_XFER_BUFFER[0] |= nupi_device[y].offline; 
	  NUPI_XFER_BUFFER[0] <<= 1;
	  NUPI_XFER_BUFFER[0] |= nupi_device[y].selftest; 
	  NUPI_XFER_BUFFER[0] <<= 1;
	  NUPI_XFER_BUFFER[0] |= nupi_device[y].overtemp; 
	  NUPI_XFER_BUFFER[0] <<= 1;
	  NUPI_XFER_BUFFER[0] |= nupi_device[y].writeprotect; 
	  NUPI_XFER_BUFFER[0] <<= 1;
	  NUPI_XFER_BUFFER[0] |= nupi_device[y].indt_status; 
	  NUPI_XFER_BUFFER[0] <<= 1;
	  NUPI_XFER_BUFFER[0] |= nupi_device[y].attention; 
	  NUPI_XFER_BUFFER[0] <<= 1;
	  NUPI_XFER_BUFFER[0] |= nupi_device[y].busy; 
	  NUPI_XFER_BUFFER[0] <<= 1;
	  NUPI_XFER_BUFFER[0] |= nupi_device[y].removable; 
	  NUPI_XFER_BUFFER[0] <<= 1;
	  NUPI_XFER_BUFFER[0] <<= 4; // Reserved bits
	  NUPI_XFER_BUFFER[0] <<= 8;	    
	  NUPI_XFER_BUFFER[0] |= nupi_device[y].last_cmd;
	  NUPI_XFER_BUFFER[0] <<= 8;
	  NUPI_XFER_BUFFER[0] |= nupi_device[y].last_cmd_option;
	  if(nupi_transfer_count > x){ // More bytes to write?
	    sprintf(nupi_msg,"NuPI: More status? Geez... It wants %ld more words of data.\n",((nupi_transfer_count-x)>>2));
	    logmsg(nupi_msg);
	  }
	  nupi_cmd_prog=8; // WRITE RESULT
	  nupi_xfer_pointer = 0;
	  return;
	}
	break;

      case 0x10: // RESTORE DEVICE
	{
	  // For disks - Reposition to track 0, clear all faults.
	  // For tapes - rewind to BOT, clear all faults
	  // A bus reset does this to every device.
	  // We don't have to do anything yet, as no errors exist.
	  sprintf(nupi_msg,"NUPI: Restore Device requested with 0x%lX words of output, ignored.\n",(nupi_transfer_count>>2)); logmsg(nupi_msg);
	  nupi_cmd_prog=8; // DONE	  
	  nupi_xfer_pointer = 0;
	  return;
	}
      	break;

      case 0x12: // READ DATA
	{
	  int y=0;
	  sprintf(nupi_msg,"NUPI: DRIVE READ RQ: Target is ID %d, LUN %d\n",nupi_formatterselect,nupi_devselect); logmsg(nupi_msg);
	  sprintf(nupi_msg,"NUPI: Data requested with 0x%lX words of output from block 0x%lX\n",
		  (nupi_transfer_count>>2),nupi_device_block_addr); logmsg(nupi_msg);	  
	  y = 7 + (nupi_formatterselect<<1);
	  y |= nupi_devselect;
	  sprintf(nupi_msg,"NUPI: Internal device # is %d\n",y); logmsg(nupi_msg);
	  // Reposition the file pointer.
	  lseek(nupi_device[y].fd,(nupi_device_block_addr*0x400),SEEK_SET);
	  nupi_unit_pointer = y;
	  nupi_xfer_pointer = 0;
	  // Dispatch to read-io-busy-loop
	  nupi_cmd_prog=10; // GO 
	  return;
	}

      default:
	sprintf(nupi_msg,"NuPI: Unknown Command 0x%X\n",nupi_command); logmsg(nupi_msg);
	cpu_die_rq=1;
	nupi_cmd_prog=0; nupi_go=0;
	return;
      }
      break;

    case 8: // WRITE RESULTS
      if(nupi_xfer_pointer < nupi_transfer_count){
	// Wait for the bus.
	if(Memory_Busy == 0){
	  nubus_io_request(NB_WRITE,nupi_buffer_pointer+nupi_xfer_pointer,NUPI_XFER_BUFFER[(nupi_xfer_pointer>>2)]);
	  NUPI_XFER_BUFFER[(nupi_xfer_pointer>>2)]=0; // Reset
	  nupi_xfer_pointer += 0x04;
	}	  
	return;
      }
      // Otherwise fall into

    case 9: // TRANSFER COMPLETED
      // Completed!
      // Notify the processor
      nupi_cmd_word = 0x40000000; // COMPLETED code
      nubus_io_request(NB_WRITE,nupi_cmd_addr+4,nupi_cmd_word); // Write it back
      logmsg("NUPI: Operation Completed!\n");
      if(nupi_event){
	logmsg("Posting event...\n");
	nubus_io_request(NB_WRITE,nupi_event_gen_addr,0xFF);
      }
      nupi_go=0; nupi_cmd_prog=0;
      break;	

    case 10: // READ BUSY-LOOP
      {
	unsigned char diskio[4];

	if(nupi_xfer_pointer < nupi_transfer_count){
	  // Is memory busy?
	  if(Memory_Busy == 0){
	    // Read a word in from the disk.
	    read(nupi_device[nupi_unit_pointer].fd,&diskio[0],1);
	    read(nupi_device[nupi_unit_pointer].fd,&diskio[1],1);
	    read(nupi_device[nupi_unit_pointer].fd,&diskio[2],1);
	    read(nupi_device[nupi_unit_pointer].fd,&diskio[3],1);
	    
	    NUPI_XFER_BUFFER[0]  = diskio[3]; NUPI_XFER_BUFFER[0] <<= 8;
	    NUPI_XFER_BUFFER[0] |= diskio[2]; NUPI_XFER_BUFFER[0] <<= 8;
	    NUPI_XFER_BUFFER[0] |= diskio[1]; NUPI_XFER_BUFFER[0] <<= 8;
	    NUPI_XFER_BUFFER[0] |= diskio[0];

	    nubus_io_request(NB_WRITE,nupi_buffer_pointer+nupi_xfer_pointer,NUPI_XFER_BUFFER[0]);
	    NUPI_XFER_BUFFER[0] = 0;

	    nupi_xfer_pointer += 0x04; // Repeat
	    return;
	  }	  
	}else{
	  sprintf(nupi_msg,"NUPI: Read %d words of data.\n",(nupi_xfer_pointer>>2)); logmsg(nupi_msg);
	  nupi_cmd_prog=9; // DONE TRANSFERRING	  
	  nupi_xfer_pointer = 0;
	  return;
	}
      }
      break;

    case 20: // READ INTO BUFFER BUSY-LOOP
      {
	if(nupi_xfer_pointer < nupi_transfer_count){
	  // Is memory busy?
	  if(Memory_Busy == 0){
	    // Read a word in from the memory
	    nubus_io_request(NB_READ,nupi_buffer_pointer+nupi_xfer_pointer,0);
	    return;
	  }else{
	    // Get the byte
	    NUPI_XFER_BUFFER[(nupi_xfer_pointer>>2)] = NUbus_Data;
	    nupi_xfer_pointer += 4;
	  }
	}else{
	  // FINISHED READING TO BUFFER
	  logmsg("NUPI: Memory Buffering Completed, Executing Command\n");
	  nupi_cmd_prog=30+nupi_command; // DONE TRANSFERRING	  
	  nupi_xfer_pointer = 0;
	}
      }
      break;

    case 31: // CONTINUATION OF SETUP COMMAND
      {
	// Setting nupi status
	if(nupi_dst_bit){
	  int x=0;
	  if(x<nupi_transfer_count){
	    nupi_master_event_addr = NUPI_XFER_BUFFER[0];
	    sprintf(nupi_msg,"NUPI: Master Event Address reset to 0x%lX\n",nupi_master_event_addr); logmsg(nupi_msg);
	    x += 4;
	  }else{
	    // ERROR!
	    logmsg("NUPI: SETUP command has no data?\n");
	    cpu_die_rq=1; nupi_cmd_prog=0; nupi_go=0;
	  }
	  if(x<nupi_transfer_count){
	    nupi_retry_count = NUPI_XFER_BUFFER[1]&0xFF;
	    nupi_dead_cmd_disable = (NUPI_XFER_BUFFER[1]>>28)&0x01;
	    nupi_periodic_poll_disable = (NUPI_XFER_BUFFER[1]>>29)&0x01;
	    nupi_retry_disable = (NUPI_XFER_BUFFER[1]>>30)&0x01;
	    nupi_ecc_disable = (NUPI_XFER_BUFFER[1]>>31)&0x01;
	    sprintf(nupi_msg,"NUPI: Setup Word 2 = 0x%lX\n",NUPI_XFER_BUFFER[1]); logmsg(nupi_msg);
	  }
	  nupi_cmd_prog=9; // DONE
	  return;
	}
	logmsg("NUPI-Setup-2: Unknown destination\n");
	cpu_die_rq=1; nupi_cmd_prog=0; nupi_go=0;
	return;
      }
      break;

    default:
      sprintf(nupi_msg,"NUPI: BAD INTERNAL PC %ld\n",nupi_cmd_prog); logmsg(nupi_msg);
      nupi_go=0; cpu_die_rq=1;
    } 
  }
}
