/* Raven CPU - Implementation

 This file is subject to the terms and conditions of the GNU General Public
 License.  See the file COPYING in the main directory of this archive for
 more details.

*/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ncurses.h>
#include <string.h>
#include "meroko.h"
#include "nubus.h"
#include "nupi.h"
#include "mem8.h"
#include "sib.h"
#include "raven_rom.h"

// Pick-up externals
extern void logmsg(char *msg);
extern void nupi_clock_pulse();
extern inline void sib_clock_pulse();

// NCurses
WINDOW *cpu_w;
// Stuff
char msg[256];

int breakpoint_armed=0;
unsigned long tracepoint_data=0;
unsigned int delay_tune=20;
unsigned int delay_indx=0;

/* Definitions */
#define CLK_HI 1
#define CLK_LO 0
/* Data pipeline cycles */
#define DPC_SRC 0
#define DPC_EXE 1
#define DPC_DED 2
#define DPC_DST 3
/* Control pipeline cycles */
#define CPC_PC 0
#define CPC_FETCH 1
#define CPC_DECEXE 2
#define CPC_PENDST 3

/* MCR bits */
#define MCR_Slot_ID                    0xF0000000 
#define MCR_Self_Test_Flag             0x08000000
#define MCR_Chaining_Enable            0x04000000
#define MCR_Misc_Op_Group_1            0x02000000
#define MCR_Misc_Op_Group_0            0x01000000
#define MCR_Loop_On_Test               0x00800000
#define MCR_Need_Fetch                 0x00400000
#define MCR_NU_Bus_Reset               0x00200000
#define MCR_Local_Reset                0x00100000
#define MCR_Int_Level_3                0x00080000
#define MCR_Int_Level_2                0x00040000
#define MCR_Int_Level_1                0x00020000
#define MCR_Int_Level_0                0x00010000
// All interrupts
#define MCR_Interrupts                 0x000F0000
#define MCR_Int_Enable                 0x00008000
#define MCR_Sequence_Break             0x00004000
// CPU reference calls this abort-on-bus-error?
#define MCR_Parity_Trap_Enable         0x00002000
#define MCR_Parity_Halt_Enable         0x00001000
#define MCR_PROM_Disable               0x00000800
#define MCR_Interlocked_Memory_Control 0x00000400
#define MCR_Forced_Access_Request      0x00000200
#define MCR_Memory_Cycle_Enable        0x00000100
#define MCR_Subsys_Flag                0x00000080
#define MCR_Test_Fail_Flag             0x00000040
// LEDs
#define MCR_LED_5_                     0x00000020
#define MCR_LED_4_                     0x00000010
#define MCR_LED_3_                     0x00000008
#define MCR_LED_2_                     0x00000004
#define MCR_LED_1_                     0x00000002
#define MCR_LED_0_                     0x00000001
// Mine
#define MCR_LED_CODE                   0x0000003F

// String enumeration
const char ctlphase[4][16] = { "GNPC","FTCH","DECX","PNDT" };
const char dtaphase[4][16] = { "FSRC","EXEC","DEAD","DEST" };

// Instruction fields
const char opcode[4][16]   = { "ALU ","BYTE","JUMP","DISP" };  

const char alu_op_str[040][12] = {
  "SETZ",       // 0
  "AND",        // 1
  "ANDCA",      // 2
  "SETM",       // 3
  "ANDCM",      // 4
  "SETA",       // 5
  "XOR",        // 6
  "IOR",        // 7
  "ANDCB",      // 10
  "EQV",        // 11   
  "SETCA",      // 12
  "ORCA",       // 13
  "SETCM",      // 14
  "ORCM",       // 15
  "ORCB",       // 16
  "SETO",       // 17
  "MUL",        // 20
  "MUL-Last",   // 21
  "DIV",        // 22
  "DIV-First",  // 23
  "DIV-Corr",   // 24
  "ALU-25",     // 25
  "ALU-26",     // 26
  "ALU-27",     // 27
  "ALU-30",     // 30
  "ADD",        // 31
  "ALU-32",     // 32
  "ALU-33",     // 33
  "ALU-M",      // 34
  "ALU-35",     // 35
  "SUB",        // 36
  "M+M"         // 37
};

const char obus_control_str[8][32] = {
  "OBus-A-Bus",
  "OBus-R-Bus",
  "OBus-A-Bus2",
  "OBus-Normal",
  "OBus-LeftShift-1",
  "OBus-RightShift-1",
  "OBus-Pointer-Extend",
  "OBus-Mirror" };

const char q_control_str[4][14] = {
  "NOP",
  "Shift-Left",
  "Shift-Right",
  "Load" };

const char byte_op_str[4][24] = {
  "NOP",
  "Load-Byte",
  "Selective-Deposit",
  "Deposit-Byte" };

const char jump_op_str[10][18] = {
  "Jump-XCT-Next",
  "JUMP",
  "Call-XCT-Next",
  "CALL",
  "Return-XCT-Next",
  "RETURN",
  "NOP",
  "SKIP" };

const char disp_src_str[4][10] = {
  "M-Source",
  "M-Tag",
  "MIR",
  "MIR2" };

const char disp_map_ena_str[4][18] = {
  "Normal",
  "GC-Volatile",
  "Old-Space",
  "GCV-Or-Oldspace" };
  
const char disp_op_str[4][10] = {
  "DISPATCH",
  "I-READ",
  "I-WRITE",
  "DISP-ERR" };

const char mbs_str[060][24] = {
  "VMA",            // 100
  "Q-R",
  "Macro-Argument",
  "Macro-Stack-Ptr",
  "MCR",
  "Location-Counter",
  "Map-lv2-addr",
  "Read-I-Arg",
  "Map-level-1",     // 110
  "Map-Lv2-Cntl",
  "Macro-IR",
  "Macro-Branch-Field",
  "114",
  "115",
  "116",
  "117",
  "uStack-Data",     // 120
  "uStack-Pop",
  "MD",
  "123",
  "124",
  "125",
  "126",
  "127",
  "130",
  "131",
  "132",
  "133",
  "134",
  "135",
  "136",
  "137",
  "C-PDL-Pointer",    // 140
  "C-PDL-Index",
  "142",
  "143",
  "C-PDL-Pointer-Pop",
  "C-PDL-Index-Dec",
  "146",
  "147",
  "PDL-Pointer",
  "PDL-Index",
  "PDL-Pointer-Pop",
  "PDL-Index-Dec"     // 155 
};
  
const char mbd_str[0100][56] = {
  "NOP",
  "Location-Counter",
  "MCR",
  "uStack-Pointer",
  "uStack-Data",
  "uStack-Data-Push",
  "IMod-Low",
  "IMod-Hi",
  "Macro-IR", // 010
  "11",
  "12",
  "13",
  "14",
  "15",
  "16",
  "TEST-SYNC",
  "VMA",        // 020
  "VMA-Write-Map-LV1",
  "VMA-Write-Map-LV2-C",
  "VMA-Write-Map-LV2-A",
  "VMA-Start-Read",
  "VMA-Start-Write",
  "VMA-Start-Unmapped-Read",
  "VMA-Start-Unmapped-Write",
  "MD",         // 030
  "MD-Write-Map-LV1",
  "MD-Write-Map-LV2-C",
  "MD-Write-Map-LV2-A",
  "MD-Start-Read",
  "MD-Start-Write",
  "MD-Start-Unmapped-Read",
  "MD-Start-Unmapped-Write",
  "C-PDL-Pointer", // 040
  "C-PDL-Index",
  "42",
  "43",
  "C-PDL-Pointer-Push",
  "C-PDL-Pointer-Inc",
  "46",
  "47",
  "PDL-Pointer", // 050
  "PDL-Index",
  "52",
  "53",
  "54",
  "55",
  "56",
  "57",
  "60",
  "61",
  "62",
  "63",
  "64",
  "65",
  "VMA-Start-Unmapped-Byte-Read",
  "VMA-Start-Unmapped-Byte-Write",
  "70",
  "71",
  "72",
  "73",
  "74",
  "75",
  "MD-Start-Unmapped-Byte-Read",
  "MD-Start-Unmapped-Byte-Write" };  

const char abj_str[010][24] = {
  "A-NOP",
  "A-SKIP",
  "A-CALL-ILLOP",
  "A-CALL-TRAP",
  "A-CALL-BUSERR",
  "A-UNUSED",
  "A-RETURN",
  "A-RETURN-XCT-NEXT" };

const char cond_str[021][24] = {
  "BIT-SET",
  "LESS",
  "LESS-OR-EQ",
  "NOT-EQ",
  "PAGE-FAULT",
  "PAGE-FAULT-OR-INT",
  "PAGE-FAULT-OR-INT-OR-SQB",
  "ALWAYS",
  "TAG-NOT-EQ",   // 010
  "NOT-MEM-BUSY",
  "Q-ZERO",
  "NUBUS-ERROR",
  "NOT-FIXNUM-OVERFLOW",
  "BOXED-SIGN-BIT",
  "NO-INTERRUPT",
  "17",
  "TAG-CLASSIFY" };

/* ***** Onboard stuffs ***** */
/* Pipeline control */
int data_cycle;
int ctl_cycle;
/* Buses */
unsigned long Abus; // A-bus
unsigned long Mbus; // M-bus
unsigned long Rbus; // R-bus
unsigned long long Ibus; // I-bus
unsigned long Obus; // O-bus

/* Local Bus Interface */
int Memory_Busy;      // Local Bus Busy

unsigned long PI_Sources;
int Highest_PI=0;
int Active_PI=0;
int Current_PI=0;

int Page_Fault;
int cached_gcv=0;

/* Clocks */
int master_clock;     // 28MHz
int minorcycle_clock; // 14MHz
int microcycle_clock; //  7MHz

/* Halt control */
int cpu_die = 0;
int cpu_die_rq = 0;
int cpu_stop_en = 0;
/* Debug */
int tracerq = 0;

/* Memories */
unsigned char Irom[7][2048];      // Control Store PROMs
unsigned long long PCS[2048];     // PROM Control Store
unsigned long long WCS[16384];    // Writable Control Store
unsigned long Amemory[1024];      // A-memory
int AMem_Addr_Sel,AMem_Addr_Go;   // AMem address to store results in & trigger
unsigned long Mmemory[64];        // M-memory
int MMem_Addr_Sel,MMem_Addr_Go;   // MMem address to store results in & trigger
unsigned long PDLmemory[1024];    // PDL-memory
unsigned long Dmemory[4096];      // Dispatch Memory
unsigned long long Tmemory[16];   // Tag-Classifier RAM 

const unsigned int shift_left_mask[040] = {
  0x00000001,0x00000003,0x00000007,0x0000000F,0x0000001F,0x0000003F,0x0000007F,0x000000FF,
  0x000001FF,0x000003FF,0x000007FF,0x00000FFF,0x00001FFF,0x00003FFF,0x00007FFF,0x0000FFFF,
  0x0001FFFF,0x0003FFFF,0x0007FFFF,0x000FFFFF,0x001FFFFF,0x003FFFFF,0x007FFFFF,0x00FFFFFF,
  0x01FFFFFF,0x03FFFFFF,0x07FFFFFF,0x0FFFFFFF,0x1FFFFFFF,0x3FFFFFFF,0x7FFFFFFF,0xFFFFFFFF };

const unsigned int shift_right_mask[040] = {
  0xFFFFFFFF,0xFFFFFFFE,0xFFFFFFFC,0xFFFFFFF8,0xFFFFFFF0,0xFFFFFFE0,0xFFFFFFC0,0xFFFFFF80,
  0xFFFFFF00,0xFFFFFE00,0xFFFFFC00,0xFFFFF800,0xFFFFF000,0xFFFFE000,0xFFFFC000,0xFFFF8000,
  0xFFFF0000,0xFFFE0000,0xFFFC0000,0xFFF80000,0xFFF00000,0xFFE00000,0xFFC00000,0xFF800000,
  0xFF000000,0xFE000000,0xFC000000,0xF8000000,0xF0000000,0xE0000000,0xC0000000,0x80000000 };
    
/* MF-bus sources */
unsigned long disp_constant_reg;  // Dispatch Constant
unsigned long pdl_index_reg;      // PDL Index Register
unsigned long pdl_ptr_reg;        // PDL Pointer Register
unsigned long uPCS_ptr_reg=0;     // Microcode counter stack pointer 
unsigned long loc_ctr_reg=0;      // (Micro)Location Counter Register
unsigned long Qshift_reg;         // Q Shift Register
unsigned long vm_lv1_map[4096];   // VM Map, level 1
unsigned long vm_lv2_ctl[4096];   // VM Map, level 2, Control Storage
unsigned long vm_lv2_adr[4096];   // VM Map, level 2, Address Storage

/* L registers */
unsigned long last_loc_ctr;       // Lpc

/* Others */
unsigned long BARREL_SHIFTER;     // Barrel Shifter
unsigned long Obus_mux;           // O-bus multiplexer
unsigned long Obus_mask;          // O-bus mask
unsigned long uPCS_stack[64];     // Microcode counter stack memory
unsigned long pc_source_select;   // Select uPC source
unsigned long pcss_pc1;           //   PC+1 register
unsigned long pcss_jdst;          //   Jump DeSTination register
unsigned long pcss_ddst;          //   Delayed Destination Register
unsigned int  pcss_skip;          //   Skip after source selection
unsigned long dispatch_ram[4096]; // Dispatch RAM
unsigned long long Iregister;     // 56b Instruction register
unsigned long LCregister;         // (Macro)Location Counter Register
unsigned int  MIbuffer;           // Macroinstruction Buffer
unsigned long MDregister;         // Memory Data register for local bus
unsigned long VMAregister;        // Virtual Memory Address
unsigned long Qregister;          // Q register
unsigned long MCregister;         // Machine Control Register
unsigned long long ALU_Result;    // Result of ALU operation
unsigned long long Obus_Input;    // Obus Multiplexer Input
bool ALU_Carry_Out;               // Carry-Out
unsigned int ALU_Fixnum_Oflow;    // FIXNUM Overflow
unsigned int imod_en;             // Enable IMOD
unsigned int imod_lo,imod_hi;     // IMOD storage
unsigned char raven_config_reg=0; // NUbus Configuration Register


// Instruction Data that gets shared around
unsigned int opword;
unsigned int MInst_Parity;
unsigned int MInst_Abbrv_Jump;
unsigned int MInst_M_Source_Addr;
unsigned int MInst_A_Source_Addr;
unsigned int MInst_Halt;
unsigned int MInst_M_Source_Addr;
unsigned int MInst_A_Source_Addr;
unsigned int MInst_Dest_Functnl;
unsigned int MInst_Cond_ClRAM_Sel;
unsigned int MInst_Cond_Invrt_Sns;
unsigned int MInst_Cond_Select;
unsigned int MInst_ALU_Opcode;
unsigned int MInst_ALU_Carry_In;
unsigned int MInst_Output_Bus_Ctl;
unsigned int MInst_Write_ClRAM;   //    = (bool)(Iregister&0x200);      // ((Iregister>>9)&0x01);   // ldb(Iregister,1,9); 
unsigned int MInst_Mask_Rotate;
unsigned int MInst_Source_Rotate;
unsigned int MInst_Rotation_Directn;
unsigned int MInst_Rotation_Count;
unsigned int MInst_Rotation_Len;

// Software flags (Nonexistent on real hardware)
int power_on_reset=1;             // Power-On Reset (LOAD PROM CONTENTS)
int data_pipe_inhibit=1;          // Data Pipeline Inhibit
int test_true;                    // Condition Test True
unsigned long test_src;           // Test source data

// Performance monitoring
unsigned long long ccount;

// Utility functions
void raven_halt(){
  cpu_die_rq=1;
  cpu_stop_en=1; // Enable stop condition
}

void raven_step(){
  cpu_die_rq=1;
  cpu_die=0;
}

void raven_cont(){
  cpu_die_rq=0;
  cpu_die=0;
  tracerq=0;
}

inline unsigned int ldb(unsigned long long value, int size, int position){
  unsigned long long mask=0,result=0;
  // Create mask
  mask = 0xFFFFFFFFFFFFFFFFLL; // Get all ones
  mask = mask >> (64 - size);  // Left-shift out the ones we don't need
  // sprintf(msg,"Generated mask 0x%LX for length %d",mask,size); logmsg(msg);
  // Shift value into position
  result = (value >> position);
  return(result&mask);
}

// ROTATION (THIS IS VERY INEFFICIENT FOR THIS! C NEEDS ROTATION OPERATORS! SOMEONE FIX THAT!)
inline unsigned int left_rotate(unsigned int data, int rot){
  unsigned int x=0xFFFFFFFF;
  unsigned int y=0;
  unsigned int result=0;
  // Grab a mask for the shifted-off part.
  x = x << (32 - rot);  // Left-shift out the ones we don't need
  // Save those bits and fix position
  y = (data&x)>>(32-rot);
  // Shift data
  result = data << rot;
  // Add saved bits
  result |= y;
  // Done
  return(result);
}

inline unsigned int right_rotate(unsigned int data, int rot){
  unsigned int x=0xFFFFFFFF;
  unsigned int y=0;
  unsigned int result=0;
  // Grab a mask for the shifted-off part.
  x = x >> (32 - rot);  // Left-shift out the ones we don't need
  // Save those bits and fix position
  y = (data&x)<<(32-rot);
  // Shift data
  result = data >> rot;
  // Add saved bits
  result |= y;
  // Done
  return(result);
}

inline unsigned int gen_i_parity(unsigned long long data){
  unsigned long long x=0x1;
  unsigned int y=0x0;

  while(x < 0x100000000000000LL){
    if(x == 0x004000000000000LL){
      x <<= 1; // Skip parity bit
    } 

    if(data&x){
      y++;
    } 
    x = x << 1; 
  }  
  y &= 1;
  /*
  sprintf(msg,"Parity for 0x%LX is %d\n",data,y);
  logmsg(msg); */
  return(y^1); // Return ODD parity
}

void disassemble_IR(){

  unsigned char disasm[128]; // Disassembled instruction
  unsigned char dtemp[64];   // Temporary
  unsigned int opcode = ldb(Iregister,2,54); 

  // Disassemble the instruction in IR.
  
  // First, get opcode and switch
  switch(opcode){
  case 0: // ALU
    {
      // Destination first.
      if(ldb(Iregister,1,31) != 0){
	// A-Memory-Select
	sprintf(dtemp,"(A-%d)",ldb(Iregister,10,19));
      }else{
	// M-Memory-Select
	if(ldb(Iregister,6,25) != 0){
	  // Include functional destination
	  sprintf(dtemp,"(M-%d MF-%s)",ldb(Iregister,6,19),mbd_str[ldb(Iregister,6,25)]);
	}else{
	  sprintf(dtemp,"(M-%d)",ldb(Iregister,6,19));
	}
      }
      sprintf(disasm,"[%ld] %s ",loc_ctr_reg,dtemp); // Create string
      // Now do function type
      if(ldb(Iregister,1,2) != 0){
	strcat(disasm,"CARRY-IN ");
      }
      if(ldb(Iregister,1,8) != 0){
	strcat(disasm,"TAGGED ");
      }
      sprintf(dtemp,"%s ",alu_op_str[ldb(Iregister,5,3)]);
      strcat(disasm,dtemp);
      // And source
      if(ldb(Iregister,7,42) > 077){
	sprintf(dtemp,"<A-%d,M-%s> ",ldb(Iregister,10,32),mbs_str[ldb(Iregister,7,42)-0100]);
      }else{
	sprintf(dtemp,"<A-%d,M-%d> ",ldb(Iregister,10,32),ldb(Iregister,7,42));
      }
      strcat(disasm,dtemp);
      // O-control
      sprintf(dtemp,"%s ",obus_control_str[ldb(Iregister,3,16)]);
      strcat(disasm,dtemp);
      // Conditional?
      if(ldb(Iregister,1,9) != 0){
	// T-Write
	sprintf(dtemp,"TM-Wt-%d ",ldb(Iregister,4,10));
	strcat(disasm,dtemp);
      }
      if(ldb(Iregister,1,14) != 0){
	// T-Read
	sprintf(dtemp,"TM-Rd-%d ",ldb(Iregister,4,10));
	strcat(disasm,dtemp);
      }else{
	if(ldb(Iregister,1,15) != 0){ strcat(dtemp,"NOT "); }
	sprintf(dtemp,"IF %s ",cond_str[ldb(Iregister,4,10)]);
	strcat(disasm,dtemp);
      }
      // ABJ
      if(ldb(Iregister,3,51) != 0){
	sprintf(dtemp,"%s ",abj_str[ldb(Iregister,3,51)]);
	strcat(disasm,dtemp);	
      }
      // Q-Control
      if(ldb(Iregister,2,0) != 0){
	sprintf(dtemp,"Q-%s ",q_control_str[ldb(Iregister,2,0)]);
	strcat(disasm,dtemp);
      }
      strcat(disasm,"\n"); // Terminate
    }
    break;
    
  case 1: // BYTE
    {
      // Destination first.
      if(ldb(Iregister,1,31) != 0){
	// A-Memory-Select
	sprintf(dtemp,"(A-%d)",ldb(Iregister,10,19));
      }else{
	// M-Memory-Select
	if(ldb(Iregister,6,25) != 0){
	  // Include functional destination
	  sprintf(dtemp,"(M-%d MF-%s)",ldb(Iregister,6,19),mbd_str[ldb(Iregister,6,25)]);
	}else{
	  sprintf(dtemp,"(M-%d)",ldb(Iregister,6,19));
	}
      }
      sprintf(disasm,"[%ld] %s ",loc_ctr_reg,dtemp); // Create string
      // Operation
      if(ldb(Iregister,1,16) != 0){
	strcat(disasm,"RIGHT ");
      }
      sprintf(dtemp,"%s (L=%d C=%d) ",byte_op_str[ldb(Iregister,2,17)],ldb(Iregister,5,5),ldb(Iregister,5,0));
      strcat(disasm,dtemp);
      // And source
      if(ldb(Iregister,7,42) > 077){
	sprintf(dtemp,"<A-%d,M-%s> ",ldb(Iregister,10,32),mbs_str[ldb(Iregister,7,42)-0100]);
      }else{
	sprintf(dtemp,"<A-%d,M-%d> ",ldb(Iregister,10,32),ldb(Iregister,7,42));
      }
      strcat(disasm,dtemp);
      // Conditional?
      if(ldb(Iregister,1,14) != 0){
	// T-Read
	sprintf(dtemp,"TM-Rd-%d ",ldb(Iregister,4,10));
	strcat(disasm,dtemp);
      }else{
	if(ldb(Iregister,1,15) != 0){ strcat(dtemp,"NOT "); }
	sprintf(dtemp,"IF %s ",cond_str[ldb(Iregister,4,10)]);
	strcat(disasm,dtemp);
      }
      // ABJ
      if(ldb(Iregister,3,51) != 0){
	sprintf(dtemp,"%s ",abj_str[ldb(Iregister,3,51)]);
	strcat(disasm,dtemp);	
      }
      strcat(disasm,"\n");
    }
    break;

  case 2: // JUMP
    {
      // Operation and new-micro-PC
      sprintf(disasm,"[%ld] %s %d ",loc_ctr_reg,jump_op_str[ldb(Iregister,3,5)],ldb(Iregister,14,18));

      // And source
      if(ldb(Iregister,7,42) > 077){
	sprintf(dtemp,"<A-%d,M-%s> ",ldb(Iregister,10,32),mbs_str[ldb(Iregister,7,42)-0100]);
      }else{
	sprintf(dtemp,"<A-%d,M-%d> ",ldb(Iregister,10,32),ldb(Iregister,7,42));
      }
      strcat(disasm,dtemp);
      // Conditional?
      if(ldb(Iregister,1,9) != 0){
	// WCS-Write
	strcat(disasm,"WCS-Write ");
      }
      if(ldb(Iregister,1,8) != 0){
	// WCS-Write
	strcat(disasm,"WCS-Read ");
      }

      if(ldb(Iregister,1,14) != 0){
	// T-Read
	sprintf(dtemp,"TM-Rd-%d ",ldb(Iregister,4,10));
	strcat(disasm,dtemp);
      }else{
	if(ldb(Iregister,1,15) != 0){ strcat(dtemp,"NOT "); }
	sprintf(dtemp,"IF %s ",cond_str[ldb(Iregister,4,10)]);
	strcat(disasm,dtemp);
      }
      if(ldb(Iregister,4,10) == 0){
	sprintf(dtemp,"(Bit %d ",ldb(Iregister,5,0));
	if(ldb(Iregister,1,16) != 0){
	  strcat(dtemp,"RIGHT ");
	}
	strcat(dtemp,") ");
	strcat(disasm,dtemp);
      }
      // ABJ
      if(ldb(Iregister,3,51) != 0){
	sprintf(dtemp,"%s ",abj_str[ldb(Iregister,3,51)]);
	strcat(disasm,dtemp);	
      }
      strcat(disasm,"\n");
    }
    break;

  case 3: // DISP
    sprintf(disasm,"DISASM-UNIMPLIMENTED\n");
    break;
  }
  logmsg(disasm);
}

inline void operate_shifter(){
  unsigned long x=0;
  unsigned long Mask = 0;
  unsigned int left_mask_index; 
  unsigned int right_mask_index;
  unsigned long R = Mbus;
 
  // Shifter run
  // Rotate R
  if(MInst_Source_Rotate != 0){
    if(MInst_Rotation_Directn == 0){
      R = left_rotate(R,MInst_Rotation_Count);
    }else{
      R = right_rotate(R,MInst_Rotation_Count);
    }
  }
  // Create mask
  // Get Right mask
  if(MInst_Mask_Rotate != 0){
    if(MInst_Rotation_Directn == 0){
      right_mask_index = MInst_Rotation_Count;
    }else{
      right_mask_index = (040-MInst_Rotation_Count);
    }
  }else{
    right_mask_index = 0;
  }
  right_mask_index &= 037;
  left_mask_index = (right_mask_index + (MInst_Rotation_Len-1)) % 32;
  
  Mask = shift_left_mask[left_mask_index]&shift_right_mask[right_mask_index];
  
  // Merge A with R, using bits from R if the mask bit is 1
  Obus = 0;
  // Bottom cycle
  if((Mask&0x01) == 0x01){
    // Bit set, pull this one from R
    Obus |= (R&0x01);
  }else{
    // Bit clear, pull this one from A
    Obus |= (Abus&0x01);
  }	    
  // Continue
  x=0x01;
  while(x != 0x80000000){
    x = x << 1;
    if((Mask&x) == x){
      // Bit set, pull this one from R
      Obus |= (R&x);
    }else{
      // Bit clear, pull this one from A
      Obus |= (Abus&x);
    }	    
  }
  // SAVE FOR LATER ABUSE
  Rbus = R;
}

// *** INTERRUPT CONTROL ***
void pi_system_check(){
  int IMask=0;
  int PI_Level=15;
  int PI_RQ = 0;
  
  Highest_PI = 0;
  Active_PI = 0;

  while(PI_Level >= 0){
    IMask = 0x1 << PI_Level;
    if((PI_Sources&IMask) != 0){
      Highest_PI = PI_Level;
      PI_RQ = 1;
    }
    PI_Level--;
  }
  if(PI_RQ != 0 && (MCregister&MCR_Int_Enable) != 0){
    Active_PI = 1; // Signal that PI needs handled
  }
  // Update MCR
  MCregister &= 0xFFF0FFFF;
  MCregister |= ((Highest_PI&0xF) << 16);
}

// *** RAVEN'S NUBUS SLAVE ***
void raven_nubus_io(){
  unsigned long NUbus_Addr = ldb(NUbus_Address,24,0);  

  // C00000 == Flag Register
  // D00000 == Configuration Register
  // E00000 == Post PI-0
  // E00004 == Post PI-1
  // E00008 == Post PI-2
  // E0000C == Post PI-3
  // E00010 == Post PI-4
  // E00014 == Post PI-5
  // and so on until...
  // E0003C == Post PI-15 (lowest priority)
  // FFFC00
  // thru   == Configuration PROM
  // FFFFFF 

  // Configuration Register
  if(NUbus_Addr == 0xD00000){
    if(NUbus_Request == VM_BYTE_WRITE){
      sprintf(msg,"RAVEN: Configuration Register Write, Data = 0x%lX\n",NUbus_Data); logmsg(msg);
      // Bit 01 = RESET
      // Bit 02 = Light Error LED
      // Bit 04 = ??
      if(NUbus_Data&0x01){
	// RESET
	logmsg("RESET NOT IMPLEMENTED\n");
	cpu_die_rq=1;
      }
      raven_config_reg = NUbus_Data&0xFF;
      NUbus_acknowledge=1;
      return;
    }
    if(NUbus_Request == VM_BYTE_READ){
      NUbus_Data = raven_config_reg;
      NUbus_acknowledge=1;
      return;
    }
  }

  // Flag Register
  if(NUbus_Addr == 0xC00000){
    // Flag register bits:
    // 0x4 = SUBSYSTEM TEST FAILED (Always 1 for CPU)
    // 0x2 = BOARD TEST FAILED     (1 = Passed Test, 0 = Test Failed or Still Running)
    // 0x1 = SELF-TEST RUNNING     (0 = Test Completed)
    if(NUbus_Request == VM_READ || NUbus_Request == VM_BYTE_READ){
      NUbus_Data = 0x0;
      if((MCregister&MCR_Subsys_Flag)==0){
	NUbus_Data |= 0x04;
      }
      if((MCregister&MCR_Test_Fail_Flag)==0){
	NUbus_Data |= 0x02;
      }
      if((MCregister&MCR_Self_Test_Flag)==0){
	NUbus_Data |= 0x01;
      }
    }
    NUbus_acknowledge=1;
    //    logmsg("Raven NUbus Response Sent\n");
    return;
  }

  if((NUbus_Addr&0xFFFFC3) == 0xE00000){
    // PI
    int PI_Nr = ldb(NUbus_Addr,4,2);
    unsigned long PI_Mask = (0x1 << PI_Nr);
    if((NUbus_Data&0x1) == 0){
      PI_Sources &= (~PI_Mask);
    }else{
      PI_Sources |= PI_Mask;
    }
    NUbus_acknowledge=1;
    pi_system_check();      
    return;
  }

  if((NUbus_Request == VM_READ || NUbus_Request == VM_BYTE_READ) && NUbus_Addr >= 0xFFFC00){
    // For Configuration ROM addresses
    // Use ROM instead!
    NUbus_Data = raven_rom[((NUbus_Addr&0x3FF)>>2)];
    NUbus_acknowledge=1;
    return;
    
    switch(NUbus_Addr){

    case 0xFFFF00: // IPL Resource Type
      NUbus_Data = 0x10; // CPU
      NUbus_acknowledge=1;
      return;

    case 0xFFFF04: // Configuration PROM IOD
      NUbus_Data = 0xC3; NUbus_acknowledge=1; return;

    case 0xFFFF08: // Self-Test Time (log2 seconds)
      NUbus_Data = 0x02; NUbus_acknowledge=1; return;

    case 0xFFFF0C: // ROM layout version
      NUbus_Data = 0x03; NUbus_acknowledge=1; return;

    case 0xFFFF10: // ROM Flags
      NUbus_Data = 0x25; NUbus_acknowledge=1; return;

    case 0xFFFF14: // Flag Register Offset
    case 0xFFFF18:
      NUbus_Data = 0x00; NUbus_acknowledge=1; return;
    case 0xFFFF1C:
      NUbus_Data = 0xC0; NUbus_acknowledge=1; return;

    case 0xFFFF20: // Diagnostic Offset
    case 0xFFFF24:
    case 0xFFFF28:
    case 0xFFFF2C: // Device Driver Offset
    case 0xFFFF30:
    case 0xFFFF34:
      NUbus_Data = 0xFF; NUbus_acknowledge=1; return;

    case 0xFFFF38: // Configuration Register Offset
      NUbus_Data = 0x00; NUbus_acknowledge=1; return;
    case 0xFFFF3C: // Configuration Register Offset
      NUbus_Data = 0x00; NUbus_acknowledge=1; return;
    case 0xFFFF40: // Configuration Register Offset
      NUbus_Data = 0xD0; NUbus_acknowledge=1; return;

    case 0xFFFF44: // Part Number
    case 0xFFFF48:
    case 0xFFFF4C:
    case 0xFFFF50:
    case 0xFFFF64:
    case 0xFFFF6C:
    case 0xFFFF74:
    case 0xFFFF78:
    case 0xFFFF7C:
      NUbus_Data = 0x30; NUbus_acknowledge=1; return;
    case 0xFFFF54:
    case 0xFFFF58:
      NUbus_Data = 0x32; NUbus_acknowledge=1; return;
    case 0xFFFF5C:
      NUbus_Data = 0x33; NUbus_acknowledge=1; return;
    case 0xFFFF60:
      NUbus_Data = 0x38; NUbus_acknowledge=1; return;
    case 0xFFFF68:
      NUbus_Data = 0x34; NUbus_acknowledge=1; return;
    case 0xFFFF70:
      NUbus_Data = 0x2D; NUbus_acknowledge=1; return;
    case 0xFFFF80:
      NUbus_Data = 0x31; NUbus_acknowledge=1; return;

    case 0xFFFF84: // Board Type
      NUbus_Data = 0x43; NUbus_acknowledge=1; return;

    case 0xFFFF8C:
      NUbus_Data = 0x50; NUbus_acknowledge=1; return;
    case 0xFFFF90:
      NUbus_Data = 0x55; NUbus_acknowledge=1; return;
    case 0xFFFF94:
    case 0xFFFF98:
    case 0xFFFF9C:
    case 0xFFFFA0:
      NUbus_Data = 0x00; NUbus_acknowledge=1; return;

    case 0xFFFFA4: // Vendor ID 'TIAU'
      NUbus_Data = 0x54; NUbus_acknowledge=1; return;
    case 0xFFFFA8:
      NUbus_Data = 0x49; NUbus_acknowledge=1; return;
    case 0xFFFFAC:
      NUbus_Data = 0x41; NUbus_acknowledge=1; return;
    case 0xFFFFB0:
      NUbus_Data = 0x55; NUbus_acknowledge=1; return;

    case 0xFFFFB4: // ROM size (Log2)
      NUbus_Data = 0x08; NUbus_acknowledge=1; return;
    
    case 0xFFFFB8: // CRC
    case 0xFFFFBC: 
      NUbus_Data = 0xFF; NUbus_acknowledge=1; return;

    case 0xFFFFC0: // Revision
    case 0xFFFFC4:
    case 0xFFFFC8:
    case 0xFFFFCC:
    case 0xFFFFD0:
    case 0xFFFFD4:
    case 0xFFFFD8:
    case 0xFFFFDC: // Serial Number
    case 0xFFFFF0:
    case 0xFFFFF4:
    case 0xFFFFF8:
    case 0xFFFFFC:
      NUbus_Data = 0x42; // SN 42...
      NUbus_acknowledge=1;
      return;

    default: // What the hell
//      sprintf(msg,"CPU: Unknown NUbus ROM IO-(%ld) 0x%lX for addr 0x%lX\n",NUbus_Request,NUbus_Data,NUbus_Addr); logmsg(msg);
      NUbus_Data = 0x00;
      //      NUbus_acknowledge=1;
      return;

    }
  }

  sprintf(msg,"RAVEN: Unknown NUbus IO-(%ld): %lX\n",NUbus_Request,NUbus_Addr); logmsg(msg); 
  cpu_die_rq=1;
}

// *** NUBUS INTERFACE ***
void nubus_io_pulse(){
  switch(Memory_Busy){

  case 1: // Start!
    // Having taken a short delay, handle the request.
    {
      int NUbus_Slot = ldb(NUbus_Address,8,24);
      switch(NUbus_Slot&0xFF){

	// UNIMPLEMENTED
	// case 1: // Ethernet

      case 0xF2: // NuPI
	nupi_nubus_io();
	break;

      case 0xF4: // Memory
	mem8_nubus_io();
	break;

      case 0xF5: // SIB	
	sib_nubus_io();
	break;

      case 0xF6: // CPU
	raven_nubus_io();
	break;

	
      default:
	{
	  // sprintf(msg,"Unknown NUbus slot %d\n",NUbus_Slot&0xF); logmsg(msg);
	  // cpu_die_rq=1;
	  extern unsigned char mem_base_reg;
	  if(NUbus_Slot == mem_base_reg){
	    mem8_nubus_io();
	  }
	}
      }
      // If this is a CPU request, load MD here.
      if((NUbus_Request&0x10)==0 && NUbus_acknowledge > 0){
	if(NUbus_Request == VM_BYTE_READ){
	  // Insanity Test
	  NUbus_Data &= (0xFF<<(8*(NUbus_Address&0x03)));
	}
	MDregister = NUbus_Data;      
	if(tracerq){
	  sprintf(msg,"CPU: NUBUS-IO COMPLETED: OP %ld PA %lX, Data %lX\n",
		  NUbus_Request,NUbus_Address,NUbus_Data); logmsg(msg);
	}
      }
    }
    break;
    // NUbus IO devices can pick up their data at clock 1.

  case 3:
    // If this is a CPU request, re-load MD here.
    if((NUbus_Request&0x10)==0 && NUbus_acknowledge > 0){
      if(NUbus_Request == VM_BYTE_READ){
	// Insanity Test
	NUbus_Data &= (0xFF<<(8*(NUbus_Address&0x03)));
      }
      MDregister = NUbus_Data;      
    }
    break;

  case 0: // Deadline!
    // Anyone take this request?
    // If this is a CPU request, re-load MD here.
    if((NUbus_Request&0x10)==0){
      if(NUbus_acknowledge > 0){
	if(NUbus_Request == VM_BYTE_READ){
	  // Insanity Test
	  NUbus_Data &= (0xFF<<(8*(NUbus_Address&0x03)));
	}
	MDregister = NUbus_Data;      
      }else{
	// It's ours, but wasn't accepted
	NUbus_error = 1;
#ifdef TRACELOG
	sprintf(msg,"DEBUG: NUbus Timeout for %lX\n",NUbus_Address); logmsg(msg);
#endif
      }
    }
    break;

  }
  // Done
}

unsigned long nubus_io_request(int access, unsigned long address, unsigned long data){

#ifdef TRACELOG
  int NUbus_Slot = ldb(address,8,24);
  unsigned long NUbus_Addr = ldb(address,24,0);
#endif

  if(Memory_Busy > 0){
    /*
    sprintf(msg,"CPU: Warning, memory bus conflict! (%d clocks remain) (Op %d @ 0x%lX for %lX)\n",
	    Memory_Busy,access,data,address);
	    logmsg(msg);
    */
    // Stall for completion?
    while(Memory_Busy){
      Memory_Busy--;
      nubus_io_pulse();
    }
  }

  Page_Fault = 0; // Reset flags
  NUbus_error = 0;
  NUbus_acknowledge = 0;

  if((MCregister&MCR_Memory_Cycle_Enable) == 0){
    Memory_Busy = 0;
    return(0);
  }

#ifdef TRACELOG
  sprintf(msg,"DEBUG: NUBUS-IO: OP %d PA %lX, Slot %d, Local Addr %lX, Data %lX\n",
    access,address,NUbus_Slot&0xF,NUbus_Addr,data); logmsg(msg);
#endif

  // Only 3 is a legal value. All others fail the CPU selftest, except 4 which breaks the
  // slot-empty testing.

  Memory_Busy = 3; // 2 INSTRUCTIONS PASS BEFORE COMPLETION (counting this one)
  NUbus_Address = address;
  NUbus_Request = access;
  NUbus_Data = data;

  //  cpu_die_rq=1;

  return(0);
}

// *** MMU ***
unsigned long VM_resolve_address(int access);

unsigned long VM_unmapped_io(int access){
  unsigned long result=0;
  // Unmapped accesses update the VM system.
  VM_resolve_address(access|0x100); // Access in unmapped mode
  // Inhibit page-faults in unmapped mode
  Page_Fault = 0;
  // Continue
  result = nubus_io_request(access,VMAregister,MDregister);
  return(result);
}

unsigned long VM_resolve_address(int access){
  unsigned int vpage_block = 0;
  unsigned int vpage_offset = 0;
  unsigned int page_offset = 0;
  unsigned long lvl_1_map_data = 0;
  unsigned int lv2_index = 0;
  unsigned long lv2_control = 0;
  unsigned long result = 0;

  vpage_block = ((VMAregister&0x01FFE000)>>13); // ldb(VMAregister,12,13);
  vpage_offset = ((VMAregister&0x1F00)>>8); //  ldb(VMAregister,5,8);
  page_offset = (VMAregister&0xFF); // ldb(VMAregister,8,0);

  lvl_1_map_data = vm_lv1_map[vpage_block];

  /*
  if(vpage_block == 0x01 && cpu_stop_en != 0){
    // logmsg("BLOCK 1 ACCESS\n");
    sprintf(msg,"B1 MMU: ACCESS 0x%X @ 0x%lX (L1=0x%X) FORCE=%lX\n",access,VMAregister,vpage_block,MCregister&MCR_Forced_Access_Request);
    logmsg(msg);
    //    cpu_die_rq=1;
  }
  */

  if(cpu_stop_en==2){
    sprintf(msg,"MMU: ACCESS 0x%X @ 0x%lX (L1=0x%X)\n",access,VMAregister,vpage_block);
    logmsg(msg);
  }

  lv2_index = ((lvl_1_map_data&0x7F)<<5);
  lv2_index |= vpage_offset;

  // Update cached-gcv
  cached_gcv = ((vm_lv2_ctl[lv2_index]&0x1800)>>11);

  // Set this
  vm_lv1_map[vpage_block] |= 0x4000;   // Map-Not-Forced-Access

  // Pagefault checks
  if((lvl_1_map_data&0x800)==0){ // Map-Entry-Valid
    vm_lv1_map[vpage_block] |= 0x1000; // Map-Access-Fault 
    // If force is allowed and active
    if(MCregister&MCR_Forced_Access_Request){
      //      logmsg("USE FORCE?\n");
      // cpu_die_rq=1;
      //      if(cpu_stop_en){ logmsg("FORCED LV1\n"); }
    }else{
      Page_Fault = 1;
      // Note that last access failed
      if(cpu_stop_en){
	// logmsg("PAGEFAULT 1\n");
      }
      // Log a write-fault if this was a write.
      if(access&0x01){
	vm_lv1_map[vpage_block] |= 0x2000; // Map-Access-Write-Fault    
      }
      return(0);
    }
  }else{
    vm_lv1_map[vpage_block] &= 0xEFFF; // Map-Not-Access-Fault
  }

  lv2_control = vm_lv2_ctl[lv2_index];

  // Level-2 Pagefault checks

  // Check access bit
  if((lv2_control&0x0200)==0){
    if(MCregister&MCR_Forced_Access_Request && (lv2_control&0x400) != 0){
      logmsg("FORCED LV2-ACCESS\n");
      vm_lv1_map[vpage_block] &= 0xBFFF; // Map-Forced-Access
      //      cpu_die_rq=1;
    }else{
      // ML2-Access-Bit == 0 
      vm_lv1_map[vpage_block] |= 0x1000; // Map-Access-Fault 
      Page_Fault = 2;
    }
  }

  if((lv2_control&0x0100)==0 && (access&0x01)){ // If this is a write and writes are disallowed
    // If force is allowed and active
    if(MCregister&MCR_Forced_Access_Request && (lv2_control&0x400) != 0){
      vm_lv1_map[vpage_block] &= 0xBFFF; // Map-Forced-Access
      vm_lv1_map[vpage_block] &= 0xDFFF; // No write-fault
      if(cpu_stop_en){ logmsg("FORCED LV2-WRITE\n"); }
    }else{
      // ML2-Read-Write == 0 ?
      vm_lv1_map[vpage_block] |= 0x2000; // Map-Access-Write-Fault    
      Page_Fault = 3;
    }
  }else{
    /*
    if(MCregister&MCR_Forced_Access_Request && access&0x01 && ((lv2_control&0x400) != 0)){
      vm_lv1_map[vpage_block] &= 0xBFFF; // Map-Forced-Access
      logmsg("FORCED FORCE FAILURE\n");
    }
    */
    vm_lv1_map[vpage_block] &= 0xDFFF;
  }

  result = (vm_lv2_adr[lv2_index]&0x3FFFFF);
  result = result << 8;
  result |= (page_offset&0xFF);
  result = result << 2;

  //  sprintf(msg,"VM-IO-(%d): VMA %lX VBK %X VBO %X PPO %X | LV1 %lX L2I %X PA=%lX\n",
  //	  access,VMAregister,vpage_block,vpage_offset,page_offset,lvl_1_map_data,lv2_index,result); logmsg(msg);

  // Update UNMAPPED for Level 1 map
  if(access&0x100){
    // Unmapped access
    vm_lv1_map[vpage_block] |= 0x8000;
  }else{
    // Mapped access
    vm_lv1_map[vpage_block] &= 0x7FFF;
  }

  // Update TM0,TM1,LOCK for level 2 map
  vm_lv2_ctl[lv2_index] &= 0x1FFF;
  if(access&0x01){ // If a write
    vm_lv2_ctl[lv2_index] |= 0x4000; // Set TM1
  }
  if(access&0x02){ // If byte-mode
    vm_lv2_ctl[lv2_index] |= 0x8000; // Set TM0
  }
  // INTERLOCK clears bit 13
  if(MCregister&MCR_Interlocked_Memory_Control){
    vm_lv2_ctl[lv2_index] |= 0x2000; // Last-Access-Locked
  }
  vm_lv2_ctl[lv2_index] ^= 0xE000; // Invert bits


  // Return 0 if there was a page fault.
  if(Page_Fault != 0){
    if(cpu_stop_en){
      // logmsg("PAGEFAULT 2\n");
    }
    return(0);
  }
  // Otherwise
  return(result);
}

// Update status display
void raven_dispupdate(){
  char LEDstring[10];
  // Draw clock information
  // Obsoleted
  //  mvwprintw(cpu_w,1,2,"CTL:%s | DTA:%s | CK/MS/MC %d/%d/%d",
  //    ctlphase[ctl_cycle],dtaphase[data_cycle],master_clock,minorcycle_clock,microcycle_clock);
  // Draw rate
  mvwprintw(cpu_w,1,2,"UCYCLE RATE = %Ld IPS (%d)\n",ccount,delay_tune);
  // Adjust delay
  if(ccount < 7000000){
   if(delay_tune > 0){
      delay_tune -= 1;
    }
  }else{
    delay_tune += 1;
  }
  ccount=0;

  // Draw LC and IR
  mvwprintw(cpu_w,2,2,"UPC %.5d | IR %.14LX",loc_ctr_reg,Iregister);
  // Parse instruction in IR if we can
  MInst_Parity     = (bool)(Iregister&0x4000000000000LL);  // ((Iregister >> 50)&0x1);  //     = ldb(Iregister,1,50);
  MInst_Halt       = (bool)(Iregister&0x2000000000000LL);  // ((Iregister >> 49)&0x01); //     = ldb(Iregister,1,49);

  // GLOBAL FIELDS AND REGISTERS
  mvwprintw(cpu_w,3,2,"INST:%s AJMP:%.2o MSRC:%.3o ASRC:%.4o",opcode[opword],MInst_Abbrv_Jump,MInst_M_Source_Addr,
	    MInst_A_Source_Addr);
  mvwprintw(cpu_w,4,2,"P:%s H:%s OBUS:%.8X uPC-SP:%.2d",
	    MInst_Parity ? "Y" : "N", MInst_Halt ? "Y" : "N",Obus,uPCS_ptr_reg);
  mvwprintw(cpu_w,5,2,"QR:0x%.8X MCR:0x%.8X",Qregister,MCregister);
  mvwprintw(cpu_w,6,2,"MD:0x%.8X VMA:0x%.8X",MDregister,VMAregister);

  mvwprintw(cpu_w, 7,2,"M-00:0x%.8X M-01:0x%.8X M-02:0x%.8X",Mmemory[0], Mmemory[1], Mmemory[2]);
  mvwprintw(cpu_w, 8,2,"M-03:0x%.8X M-04:0x%.8X M-05:0x%.8X",Mmemory[3], Mmemory[4], Mmemory[5]);
  mvwprintw(cpu_w, 9,2,"M-06:0x%.8X M-07:0x%.8X M-08:0x%.8X",Mmemory[6], Mmemory[7], Mmemory[8]);
  mvwprintw(cpu_w,10,2,"M-09:0x%.8X M-10:0x%.8X M-11:0x%.8X",Mmemory[9], Mmemory[10], Mmemory[11]);
  mvwprintw(cpu_w,11,2,"M-12:0x%.8X M-13:0x%.8X M-14:0x%.8X",Mmemory[12], Mmemory[13], Mmemory[14]);
  mvwprintw(cpu_w,12,2,"M-15:0x%.8X M-16:0x%.8X M-17:0x%.8X",Mmemory[15], Mmemory[16], Mmemory[17]);
  mvwprintw(cpu_w,13,2,"M-18:0x%.8X M-19:0x%.8X M-20:0x%.8X",Mmemory[18], Mmemory[19], Mmemory[20]);
  mvwprintw(cpu_w,14,2,"M-21:0x%.8X M-22:0x%.8X M-23:0x%.8X",Mmemory[21], Mmemory[22], Mmemory[23]);

  //  mvwprintw(cpu_w,15,2,"M-24:0x%.8X M-25:0x%.8X M-26:0x%.8X",Mmemory[24], Mmemory[25], Mmemory[26]);
  //  mvwprintw(cpu_w,16,2,"M-27:0x%.8X M-28:0x%.8X M-29:0x%.8X",Mmemory[27], Mmemory[28], Mmemory[29]);
  mvwprintw(cpu_w,16,2,"M-30:0x%.8X M-31:0x%.8X M-32:0x%.8X",Mmemory[30], Mmemory[31], Mmemory[32]);
  mvwprintw(cpu_w,17,2,"M-33:0x%.8X M-34:0x%.8X M-35:0x%.8X",Mmemory[33], Mmemory[34], Mmemory[35]);
  mvwprintw(cpu_w,18,2,"M-36:0x%.8X M-37:0x%.8X M-38:0x%.8X",Mmemory[36], Mmemory[37], Mmemory[38]);
  mvwprintw(cpu_w,19,2,"M-39:0x%.8X M-40:0x%.8X M-41:0x%.8X",Mmemory[39], Mmemory[40], Mmemory[41]);

  /*
  mvwprintw(cpu_w, 8,2,"M-02:0x%.8X M-03:0x%.8X",Mmemory[2], Mmemory[3]);
  mvwprintw(cpu_w, 9,2,"M-04:0x%.8X M-05:0x%.8X",Mmemory[4], Mmemory[5]);
  mvwprintw(cpu_w,10,2,"M-06:0x%.8X M-07:0x%.8X",Mmemory[6], Mmemory[7]);
  mvwprintw(cpu_w,11,2,"M-08:0x%.8X M-09:0x%.8X",Mmemory[8], Mmemory[9]);
  mvwprintw(cpu_w,12,2,"M-10:0x%.8X M-11:0x%.8X",Mmemory[10],Mmemory[11]);
  mvwprintw(cpu_w,13,2,"M-12:0x%.8X M-13:0x%.8X",Mmemory[12],Mmemory[13]);
  mvwprintw(cpu_w,14,2,"M-14:0x%.8X M-15:0x%.8X",Mmemory[14],Mmemory[15]);
  */

  mvwprintw(cpu_w,20,2,"MBUS:0x%.8X ABUS:0x%.8X",Mbus,Abus);

  /*
  else{
    mvwprintw(cpu_w,3,2,"INST:XXXX AJMP:XX MSRC:XXX ASRC:XXXX");
    mvwprintw(cpu_w,4,2,"P:X H:X OBUS:XXXXXXXX");
  }
  */

  sprintf(LEDstring,"%c%c%c%c%c%c",
	  MCregister&MCR_LED_5_ ? 'O' : '*',
	  MCregister&MCR_LED_4_ ? 'O' : '*',
	  MCregister&MCR_LED_3_ ? 'O' : '*',
	  MCregister&MCR_LED_2_ ? 'O' : '*',
	  MCregister&MCR_LED_1_ ? 'O' : '*',
	  MCregister&MCR_LED_0_ ? 'O' : '*');

  // Draw box and title
  box(cpu_w,0,0);
  wmove(cpu_w,0,(COLS/4)-6);          // Move cursor
  wprintw(cpu_w,"CPU [%s]",LEDstring);
  wrefresh(cpu_w);
}

// Reset
void raven_initialize(){

  if(power_on_reset){
    int x;
    FILE *in;

    power_on_reset=0; // Clear flag

    if(!(in=fopen("proms/2236480-03","r"))){ perror("in0"); exit(1); }
    x=2047;
    while(!feof(in)){
      fread(&Irom[0][x],1,1,in); /* IR48-55 */
      x--;
    }
    fclose(in);
    if(!(in=fopen("proms/2236481-03","r"))){ perror("in1"); exit(1); }
    x=2047;
    while(!feof(in)){
      fread(&Irom[1][x],1,1,in); /* IR40-47 */
      x--;
    }
    fclose(in);
    if(!(in=fopen("proms/2236482-03","r"))){ perror("in2"); exit(1); }
    x=2047;
    while(!feof(in)){
      fread(&Irom[2][x],1,1,in); /* IR32-39 */
      x--;
    }
    fclose(in);
    if(!(in=fopen("proms/2236483-03","r"))){ perror("in3"); exit(1); }
    x=2047;
    while(!feof(in)){
      fread(&Irom[3][x],1,1,in); /* IR24-31 */
      x--;
    }
    fclose(in);
    if(!(in=fopen("proms/2236484-03","r"))){ perror("in4"); exit(1); }
    x=2047;
    while(!feof(in)){
      fread(&Irom[4][x],1,1,in); /* IR16-23 */
      x--;
    }
    fclose(in);
    if(!(in=fopen("proms/2236485-03","r"))){ perror("in5"); exit(1); }
    x=2047;
    while(!feof(in)){
      fread(&Irom[5][x],1,1,in); /* IR08-15 */
      x--;
    }
    fclose(in);
    if(!(in=fopen("proms/2236486-03","r"))){ perror("in6"); exit(1); }
    x=2047;
    while(!feof(in)){
      fread(&Irom[6][x],1,1,in); /* IR00-07 */
      x--;
    }
    fclose(in);
    
    x=0;
    while(x<2048){
      // PROM overlay is as follows:
      // Start with rom 6, read backwards
      PCS[x]  = Irom[6][x]; PCS[x] = PCS[x] << 8;
      PCS[x] |= Irom[5][x]; PCS[x] = PCS[x] << 8;
      PCS[x] |= Irom[4][x]; PCS[x] = PCS[x] << 8;
      PCS[x] |= Irom[3][x]; PCS[x] = PCS[x] << 8;
      PCS[x] |= Irom[2][x]; PCS[x] = PCS[x] << 8;
      PCS[x] |= Irom[1][x]; PCS[x] = PCS[x] << 8;
      PCS[x] |= Irom[0][x];
      x++;
    }
    // Initialize other cards
    nupi_init();
    mem8_init();
    sib_init();
  }

  data_cycle=DPC_SRC; // Hold at SOURCE cycle until CPC_DECEXE
  ctl_cycle=CPC_PC;   // Ready at PC cycle
  // Force all clocks high
  master_clock=CLK_HI; 
  minorcycle_clock=CLK_HI; 
  microcycle_clock=CLK_HI;
  // Force start PC
  pc_source_select = 1;
  pcss_jdst = 0;
  pcss_ddst = 0;
  pcss_skip = 0;
  // Force low
  MCregister = 0x90800000; // CPU in slot 9 (Which is NUbus 6), loop-self-test
  Memory_Busy = 0;
  Page_Fault = 0;
  NUbus_error = 0;
  VMAregister = 0x0A0; // Is this correct?
  imod_lo = 0;
  imod_hi = 0;
  uPCS_ptr_reg = 0;
}

void raven_disp_init(){
  // Create CPU window
  cpu_w = newwin((LINES/2),(COLS/2),0,0);
  // Update it.
  raven_dispupdate();
}

inline void handle_m_fcn_src(){
  switch(MInst_M_Source_Addr){
  case 0100: // MBS-VMA
    Mbus = VMAregister;
    break;
    
  case 0101: // MBS-Q-R
    Mbus = Qregister;
    break;
    
  case 0102: // MBS-Macro-Instruction-Argument-Field
    if((LCregister&0x01) == 0){
      Mbus = ldb(MIbuffer,6,16);
    }else{
      Mbus = ldb(MIbuffer,6,0);
    }
    break;
    
  case 0103: // MBS-Micro-Stack-Pointer
    uPCS_ptr_reg &= 0x3F;
    Mbus = uPCS_ptr_reg;
    break;
    
  case 0104: // MBS-MCR
    Mbus = MCregister;
    break;
    
  case 0105: // MBS-Location-Counter
    LCregister &= 0x03FFFFFF;
    Mbus = LCregister;
    break;

  case 0106: // MBS-Memory-Map-Level-2-Address
    {
      unsigned long map1_addr = 0;
      unsigned long map1_data = 0;
      
      unsigned long map2_addr = 0;
      unsigned long map2_indx = 0; // INDEX bits from level 1
      unsigned long map2_mdrg = 0; // Bits from MD
      
      // Get Level-1 index into 2
      map1_addr = ldb(MDregister,12,13);
      map1_data = vm_lv1_map[map1_addr];
      // Get level-2 address
      map2_indx = (map1_data&0x7F);            // Get index bits
      map2_mdrg = ((MDregister&0x1F00)>>8);    // And MD register
      map2_addr = (map2_indx << 5);            // Load index bits
      map2_addr |= map2_mdrg;                  // and vpage-offset
      // Check
      if(map2_addr > 0x1000){ logmsg("L2MEMOVR\n"); cpu_die_rq=1; }
      // Read
      Mbus = vm_lv2_adr[map2_addr];
      
      // if(loc_ctr_reg > 334){ sprintf(msg,"L2AM: map1 = %lX map2_a = %lX M = %lX\n",map1_data,map2_addr,Mbus); logmsg(msg); }
    }
    break;

  case 0107: // MBS-Read-I-Arg (Dispatch constant?)
    disp_constant_reg &= 0x3FF;
    Mbus = disp_constant_reg;
    break;
	    
  case 0110: // MBS-Memory-Map-Level-1
    Mbus = vm_lv1_map[ldb(MDregister,12,13)];
    if(cpu_stop_en){
      cpu_die_rq=1;
      cpu_stop_en=2;
    }
    break;
    
  case 0111: // MBS-Memory-Map-Level-2-Control
    {
      unsigned long map1_addr = 0;
      unsigned long map1_data = 0;
      
      unsigned long map2_addr = 0;
      unsigned long map2_indx = 0; // INDEX bits from level 1
      unsigned long map2_mdrg = 0; // Bits from MD
      
      // Get Level-1 index into 2
      map1_addr = ((MDregister&0x01FFE000)>>13); // ldb(MDregister,12,13);
      map1_data = vm_lv1_map[map1_addr];
      // Get level-2 address
      map2_indx = (map1_data&0x7F);            // Get index bits
      map2_mdrg = ((MDregister&0x1F00)>>8);    // And MD register
      map2_addr = (map2_indx << 5);            // Load index bits
      map2_addr |= map2_mdrg;                  // and vpage-offset
      // Check
      if(map2_addr > 0x1000){ logmsg("L2MEMOVR\n"); cpu_die_rq=1; }
      // Read
      Mbus = vm_lv2_ctl[map2_addr];
    }
    break;

  case 0112: // MBS-Macro-Instruction-Buffer;
    if((LCregister&0x01) == 0x01){
      Mbus = MIbuffer;
    }else{
      // Swap halfwords
      unsigned int rtemp = MIbuffer&0xFFFF;
      Mbus = ((MIbuffer&0xFFFF0000)>>16);
      Mbus |= (rtemp<<16);
    }
    break;
    
  case 0113: // MBS-Macro-Instruction-Branch-Field
    if((LCregister&0x01) == 0){
      Mbus = ldb(MIbuffer,9,16);
    }else{
      Mbus = ldb(MIbuffer,9,0);
    }
    break;
    
  case 0120: // MBS-Micro-Stack-Data
    // Returns the microstack data at the pointer.
    // Push is pre-increment
    Mbus = (uPCS_stack[uPCS_ptr_reg]&0xFFFFF);
    break;
    
  case 0121: // MBS-Micro-Stack-Data-Pop
    Mbus = (uPCS_stack[uPCS_ptr_reg]&0xFFFFF);
    uPCS_ptr_reg--;  uPCS_ptr_reg &= 0x3F;
    break;
    
  case 0122: // MBS-MD
    Mbus = MDregister;
    break;
    
  case 0140: // MBS-C-PDL-Buffer-Pointer
    Mbus = PDLmemory[pdl_ptr_reg];
    // sprintf(msg,"C-PDL-Buffer-Pointer = 0x%lX\n",pdl_ptr_reg); logmsg(msg);
    break;

  case 0141: // MBS-C-PDL-Buffer-Index
    Mbus = PDLmemory[pdl_index_reg];
    break;
    
  case 0144: // MBS-C-PDL-Buffer-Pointer-Pop
    Mbus = PDLmemory[pdl_ptr_reg];
    pdl_ptr_reg--; pdl_ptr_reg &= 0x3FF;
    break;
    
  case 0145: // MBS-C-PDL-Buffer-Index-Decrement
    Mbus = PDLmemory[pdl_index_reg];
    pdl_index_reg--; pdl_index_reg &= 0x3FF;
    break;

  case 0150: // MBS-PDL-Buffer-Pointer
    Mbus = pdl_ptr_reg;
    break;
    
  case 0151: // MBS-PDL-Buffer-Index
    Mbus = pdl_index_reg;
    break;
    
  default:
    sprintf(msg,"UNKNOWN-M-FCN-SRC(0%o)\n",MInst_M_Source_Addr); logmsg(msg);
    cpu_die_rq=1;
  }
}

inline void handle_m_fcn_dst(){
  // Handle MF bus
  switch(MInst_Dest_Functnl){
  case 0: // NOP
    break;
    
  case 01: // MBD-Location-Counter
    LCregister = (Obus&0x03FFFFFF);
    break;
    
  case 02: // MBD-MCR
    // sprintf(msg,"MCR = 0x%lX\n",Obus); logmsg(msg);
    {
      // Test
      if((MCregister&MCR_PROM_Disable) == 0 && (NUbus_Data&MCR_PROM_Disable)!= 0){
	logmsg("CPU: PROM DISABLED\n");
      }
      if((MCregister&MCR_Chaining_Enable) == 0 && (NUbus_Data&MCR_Chaining_Enable) != 0){
	logmsg("MACROINSTRUCTION CHAINING ENABLED\n");
      }
      // AND off the bits we can set
      MCregister &= 0xF08F0000; // Not allowed to write slotid,loop-on-test,interrupts
      // OR in the bits we want set
      MCregister |= (Obus&0x0F70FFFF);
      // Test bits we haven't supported yet
      if(MCregister&MCR_Misc_Op_Group_1){
	logmsg("MISC-OP GROUP 1 ENABLED\n");
	cpu_die_rq=1;
      }
      if(MCregister&MCR_Misc_Op_Group_0){
	logmsg("MISC-OP GROUP 0 ENABLED\n");
	cpu_die_rq=1;
      }
      if(MCregister&MCR_Need_Fetch){
	logmsg("NEED-FETCH ENABLED\n");
	cpu_die_rq=1;
      }
      if(MCregister&MCR_NU_Bus_Reset){
	logmsg("CPU: NUBUS RESET\n");
      }
      /* This doesn't cause a state change, it changes the bits set in the level-2 map
	 when an access occurs. */
      /*
      if(MCregister&MCR_Interlocked_Memory_Control){
	logmsg("MEMORY-CONTROL INTERLOCK\n");
	cpu_die_rq=1;
      }
      */
      /* This doesn't cause a state change, it changes bits set in the level-1 map
	 when an access occurs.
      */
      /*
      if(MCregister&MCR_Forced_Access_Request){
	logmsg("FORCED ACCESS REQUEST\n");
	cpu_die_rq=1;
      }
      */

    }
    pi_system_check();      
    break;
    
  case 03: // MBD-Micro-Stack-Pointer
    uPCS_ptr_reg = Obus&0x3F;
    break;
    
  case 04: // MBD-Micro-Stack-Data
    // Returns the microstack data at the pointer.
    // Push is pre-increment
    uPCS_stack[uPCS_ptr_reg] = Obus;
    break;
    
  case 05: // MBD-Micro-Stack-Data-Push
    uPCS_ptr_reg++;  uPCS_ptr_reg &= 0x3F;
    uPCS_stack[uPCS_ptr_reg] = Obus;
    break;
    
  case 06: // MBD-OA-Reg-Low
    // (IMOD-LOW)
    // Data stored here will be deposited over the low half
    // of the next instruction.
    imod_lo = Obus;
    imod_en = 1;
    break;
    
  case 07: // MBD-OA-Reg-High
    // (IMOD-HI)
    // Data stored here will be deposited over the high half
    // of the next instruction.
    imod_hi = Obus&0xFFFFFF;
    imod_en = 1;
    break;

  case 010: // MBD-Macro-Instruction-Buffer
    MIbuffer = Obus;
    break;

  case 017: // MBD-Test-Synch
    sprintf(msg,"[CPU] TEST-SYNCH @ %ld\n",loc_ctr_reg); logmsg(msg);
    MDregister = 0;
    NUbus_error = 0;
    pi_system_check();      
    break;
    
  case 020: // MBD-VMA
    VMAregister = Obus;
    break;
    
  case 021: // MBD-VMA-Write-Map-Level-1
    VMAregister = Obus;
    vm_lv1_map[ldb(MDregister,12,13)] = (VMAregister&0xFFFF);
    break;
    
  case 022: // MBD-Write-Map-Level-2-Control
    {
      unsigned long map1_addr = 0;
      unsigned long map1_data = 0;
      
      unsigned long map2_addr = 0;
      unsigned long map2_indx = 0; // INDEX bits from level 1
      unsigned long map2_mdrg = 0; // Bits from MD
      
      // Load VMA
      VMAregister = Obus;
      // Get Level-1 index into 2
      map1_addr = ((MDregister&0x01FFE000)>>13); // ldb(MDregister,12,13);
      map1_data = vm_lv1_map[map1_addr];
      // Get level-2 address
      map2_indx = (map1_data&0x7F);            // Get index bits
      map2_mdrg = ((MDregister&0x1F00)>>8);    // And MD register
      map2_addr = (map2_indx << 5);            // Load index bits
      map2_addr |= map2_mdrg;                  // and vpage-offset
      // Check
      if(map2_addr > 0x1000){ logmsg("L2MEMOVR\n"); cpu_die_rq=1; }
      // Write
      vm_lv2_ctl[map2_addr] = VMAregister&0xFFFF;
    }
    break;
    
  case 023: // MBD-Write-Map-Level-2-Address
    {
      unsigned long map1_addr = 0;
      unsigned long map1_data = 0;
      
      unsigned long map2_addr = 0;
      unsigned long map2_indx = 0; // INDEX bits from level 1
      unsigned long map2_mdrg = 0; // Bits from MD
      
      // Load VMA
      VMAregister = Obus;
      // Get Level-1 index into 2
      map1_addr = ldb(MDregister,12,13);
      map1_data = vm_lv1_map[map1_addr];
      // Get level-2 address
      map2_indx = (map1_data&0x7F);            // Get index bits
      map2_mdrg = ((MDregister&0x1F00)>>8);    // And MD register
      map2_addr = (map2_indx << 5);            // Load index bits
      map2_addr |= map2_mdrg;                  // and vpage-offset
      // Checking
      if(map2_addr > 0x1000){ logmsg("L2MEMOVR\n"); cpu_die_rq=1; }
      // Write
      vm_lv2_adr[map2_addr] = VMAregister&0x3FFFFF;
      
      if(tracerq){ sprintf(msg,"L2AM:(W) map1 = %lX map2_a = %lX M = %lX\n",map1_data,map2_addr,VMAregister); logmsg(msg); }
    }
    break;
    
  case 024: // MBD-VMA-Start-Read
    {
      unsigned long physaddr=0;
      
      Page_Fault = 0;
      VMAregister = Obus; // Load VMA
      
      physaddr = VM_resolve_address(VM_READ);
      if(Page_Fault == 0){
	nubus_io_request(VM_READ,physaddr,0);
      }
    }
    break;
    
  case 025: // MBD-VMA-Start-Write
    {
      unsigned long physaddr=0;
      
      Page_Fault = 0;
      VMAregister = Obus; // Load VMA
      
      physaddr = VM_resolve_address(VM_WRITE);
      if(Page_Fault == 0){
	nubus_io_request(VM_WRITE,physaddr,MDregister);
      }
    }
    break;

  case 026: // MBD-VMA-Start-Read-Unmapped
    {
      VMAregister = Obus; // Load VMA
      VM_unmapped_io(VM_READ);
    }
    break;
    
  case 027: // MBD-VMA-Start-Write-Unmapped
    {
      VMAregister = Obus; // Load VMA
      VM_unmapped_io(VM_WRITE);
    }
    break;
    
  case 030: // MBD-MD
    // Don't overwrite if memory is busy?
    if(Memory_Busy > 1 && (NUbus_Request&0x1)==0){ 
      sprintf(msg,"CPU: MD CLOBBER PROTECTION STOP 0 (%d)\n",Memory_Busy);
      logmsg(msg); cpu_die_rq=1; }
    MDregister = Obus;
    break;

  case 031: // MBD-MD-Write-Map-Level-1
    if(Memory_Busy > 0 && (NUbus_Request&0x1)==0){ logmsg("CPU: MD CLOBBER PROTECTION STOP 1\n"); cpu_die_rq=1; }
    MDregister = Obus;
    vm_lv1_map[ldb(MDregister,12,13)] = VMAregister;
    break;
    
  case 032: // MBD-MD-Write-Map-Level-2-Control
    {
      unsigned long map1_addr = 0;
      unsigned long map1_data = 0;
      
      unsigned long map2_addr = 0;
      unsigned long map2_indx = 0; // INDEX bits from level 1
      unsigned long map2_mdrg = 0; // Bits from MD
      
      // Load MD
      if(Memory_Busy > 0 && (NUbus_Request&0x1)==0){ logmsg("CPU: MD CLOBBER PROTECTION STOP 2\n"); cpu_die_rq=1; }
      MDregister = Obus;
      // Get Level-1 index into 2
      map1_addr = ldb(MDregister,12,13);
      map1_data = vm_lv1_map[map1_addr];
      // Get level-2 address
      map2_indx = (map1_data&0x7F);            // Get index bits
      map2_mdrg = ((MDregister&0x1F00)>>8);    // And MD register
      map2_addr = (map2_indx << 5);            // Load index bits
      map2_addr |= map2_mdrg;                  // and vpage-offset
      // Check
      if(map2_addr > 0x1000){ logmsg("L2MEMOVR\n"); cpu_die_rq=1; }
      // Write
      vm_lv2_ctl[map2_addr] = VMAregister;
    }
    break;
    
  case 033: // MBD-MD-Write-Map-Level-2-Address
    {
      unsigned long map1_addr = 0;
      unsigned long map1_data = 0;
      
      unsigned long map2_addr = 0;
      unsigned long map2_indx = 0; // INDEX bits from level 1
      unsigned long map2_mdrg = 0; // Bits from MD
      
      // Load MD
      if(Memory_Busy > 0 && (NUbus_Request&0x1)==0){ logmsg("CPU: MD CLOBBER PROTECTION STOP 3\n"); cpu_die_rq=1; }
      MDregister = Obus;
      // Get Level-1 index into 2
      map1_addr = ldb(MDregister,12,13);
      map1_data = vm_lv1_map[map1_addr];
      // Get level-2 address
      map2_indx = (map1_data&0x7F);            // Get index bits
      map2_mdrg = ((MDregister&0x1F00)>>8);    // And MD register
      map2_addr = (map2_indx << 5);            // Load index bits
      map2_addr |= map2_mdrg;                  // and vpage-offset
      // Checking
      if(map2_addr > 0x1000){ logmsg("L2MEMOVR\n"); cpu_die_rq=1; }
      // Write
      vm_lv2_adr[map2_addr] = VMAregister;
      
      if(tracerq){ sprintf(msg,"L2AM:(W) map1 = %lX map2_a = %lX M = %lX\n",map1_data,map2_addr,VMAregister); logmsg(msg); }
    }
    break;
    
  case 034: // MBD-MD-Start-Read
    {
      unsigned long physaddr=0;
      
      Page_Fault = 0;
      if(Memory_Busy > 0 && (NUbus_Request&0x1)==0){ logmsg("CPU: MD CLOBBER PROTECTION STOP 4\n"); cpu_die_rq=1; }
      MDregister = Obus; // Load MD
      
      physaddr = VM_resolve_address(VM_READ);
      if(Page_Fault == 0){
	nubus_io_request(VM_READ,physaddr,0);
      }
    }
    break;
    
  case 035: // MBD-MD-Start-Write
    {
      unsigned long physaddr=0;

      Page_Fault = 0;
      if(Memory_Busy > 0 && (NUbus_Request&0x1)==0){ logmsg("CPU: MD CLOBBER PROTECTION STOP 5\n"); cpu_die_rq=1; }
      MDregister = Obus; // Load MD
      
      physaddr = VM_resolve_address(VM_WRITE);
      if(Page_Fault == 0){
	nubus_io_request(VM_WRITE,physaddr,MDregister);
      }
    }
    break;
    
  case 036: // MBD-MD-Start-Read-Unmapped
    {
      if(Memory_Busy > 0 && (NUbus_Request&0x1)==0){ logmsg("CPU: MD CLOBBER PROTECTION STOP 6\n"); cpu_die_rq=1; }
      MDregister = Obus; // Load MD
      VM_unmapped_io(VM_READ);
    }
    break;
    
  case 037: // MBD-MD-Start-Write-Unmapped
    {
      if(Memory_Busy > 0 && (NUbus_Request&0x1)==0){ logmsg("CPU: MD CLOBBER PROTECTION STOP 7\n"); cpu_die_rq=1; }
      MDregister = Obus; // Load VMA
      VM_unmapped_io(VM_WRITE);
    }
    break;

  case 040: // MBD-C-PDL-Buffer-Pointer
	    // Write to PDL buffer at pointer
    PDLmemory[pdl_ptr_reg] = Obus;
    // sprintf(msg,"C-PDL-Buffer-Pointer = 0x%lX\n",pdl_ptr_reg); logmsg(msg);
    break;
    
  case 041: // MBD-C-PDL-Buffer-Index
	    // Write to PDL buffer at index
    PDLmemory[pdl_index_reg] = Obus;
    break;
    
  case 044: // MBD-C-PDL-Buffer-Pointer-Push
    pdl_ptr_reg++; pdl_ptr_reg &= 0x3FF;
    PDLmemory[pdl_ptr_reg] = Obus;
    break;
    
  case 045: // MBD-C-PDL-Buffer-Index-Increment
	    // Write to PDL buffer at index++
    pdl_index_reg++; pdl_index_reg &= 0x3FF;
    PDLmemory[pdl_index_reg] = Obus;
    break;
    
  case 050: // MBD-PDL-Buffer-Pointer
    pdl_ptr_reg = Obus&0x3FF;
    break;
    
  case 051: // MBD-PDL-Buffer-Index
    pdl_index_reg = Obus&0x3FF;
    break;
    
  case 066: // MBD-VMA-Start-Read-Unmapped-NU (NU = BYTE)
    {
      VMAregister = Obus; // Load VMA
      VM_unmapped_io(VM_BYTE_READ);
    }
    break;
    
  case 067: // MBD-VMA-Start-Write-Unmapped-NU (NU = BYTE)
    {
      VMAregister = Obus; // Load VMA
      VM_unmapped_io(VM_BYTE_WRITE);
    }
    break;
    
  case 076: // MBD-MD-Start-Read-Unmapped-NU (NU = BYTE)
    {
      if(Memory_Busy > 0 && (NUbus_Request&0x1)==0){ logmsg("CPU: MD CLOBBER PROTECTION STOP 8\n"); cpu_die_rq=1; }
      MDregister = Obus; // Load MD
      VM_unmapped_io(VM_BYTE_READ);
    }
    break;

  case 077: // MBD-MD-Start-Write-Unmapped-NU (NU = BYTE)
    {
      if(Memory_Busy > 0 && (NUbus_Request&0x1)==0){ logmsg("CPU: MD CLOBBER PROTECTION STOP 9\n"); cpu_die_rq=1; }
      MDregister = Obus; // Load VMA
      VM_unmapped_io(VM_BYTE_WRITE);
    }
    break;    
    
  default:
    sprintf(msg,"UNKNOWN-FCN-DST(0%o)\n",MInst_Dest_Functnl); logmsg(msg);
    cpu_die_rq=1;
    break;
  }
}

inline void handle_condition_select(){
  // Initialize
  test_true = 0;

  if(MInst_Cond_ClRAM_Sel == 0){
    switch(MInst_Cond_Select){
    case 00: // Condition-Bit-Set
      {
	unsigned int MInst_Rotation_Directn  = (bool)(Iregister&0x10000); // ((Iregister>>16)&0x01); // ldb(Iregister,1,16);
	unsigned int MInst_Bit_Set_Pos    = (Iregister&0x1F);         // ldb(Iregister,5,0);
	
	// What bit where?
	// sprintf(msg,"MInst_Bit_Set_Pos = %d\n",MInst_Bit_Set_Pos); logmsg(msg);
	// sprintf(msg,"Expecting 0x%X",0x01 << MInst_Bit_Set_Pos); logmsg(msg);
	if(MInst_Rotation_Directn == 0){
	  // if(loc_ctr_reg > 100){ sprintf(msg,"LEFT-BIT-SET@%ld: %lX pos %d\n",loc_ctr_reg,test_src,MInst_Bit_Set_Pos); logmsg(msg); }
	  if((left_rotate(test_src,MInst_Bit_Set_Pos)&0x01) != 0){
	    test_true = 1;
	  }
	}else{
	// RIGHT ROTATION, rotate the mask from the opposite side
	// if(loc_ctr_reg > 100){ sprintf(msg,"RIGHT-BIT-SET@%ld: %lX pos %d\n",loc_ctr_reg,test_src,MInst_Bit_Set_Pos); logmsg(msg); }
	  if((right_rotate(test_src,MInst_Bit_Set_Pos)&0x01) != 0){
	    test_true = 1;
	  }
	}
      }
      break;
	
    case 01: // Condition-Less
      //	  sprintf(msg,"CONDITION-LESS %lX %lX\n",0x80000000^Mbus,0x80000000^Abus); logmsg(msg);
      if((0x80000000^Mbus) < (0x80000000^Abus)){
	test_true = 1;
      }
      break;

    case 02: // Condition-Less-Or-Equal
      if(!((0x80000000^Mbus) > (0x80000000^Abus))){
	test_true = 1;
      }
      break;
      
    case 03: // Condition-Not-Equal
      {
	if((MInst_ALU_Opcode&0x10) == 0x10){
	  // Arithmetic Mode
	  // logmsg("ALUHAX/ARITH\n");
	  if(Mbus&0x80000000){ Mbus = (~Mbus)&0xFFFFFFFF; }
	  if(Abus&0x80000000){ Abus = (~Abus)&0xFFFFFFFF; }
	}
	if(MInst_ALU_Carry_In != 0){ 
	  // logmsg("ALUHAX/CARRY-IN\n");
	  Abus += 1; 
	}
	// sprintf(msg,"ALUHAX - CI 0x%X, MBus = 0x%lX, ABus = 0x%lX\n",MInst_ALU_Carry_In,Mbus,Abus); logmsg(msg);
	if(Mbus != Abus){
	  test_true = 1;
	}
      }
      break;
      
    case 04: // Condition-Page-Fault
      if(Page_Fault != 0){
	test_true = 1;
      }
      break;
      
    case 05: // Condition-Page-Fault-Or-Interrupt
      if(Page_Fault != 0 || Active_PI != 0){
	test_true = 1;
      }
      break;
	    
    case 06: // Condition-Page-Fault-Or-Interrupt-Or-Sequence-Break
      if(Page_Fault != 0 || Active_PI != 0 || (MCregister&MCR_Sequence_Break) != 0){
	test_true = 1;
      }
      break;
      
    case 07: // Condition-True
      test_true=1;
      break;
      
    case 010: // Condition-Tag-Not-Equal
      if(ldb(Mbus,5,25) != ldb(Abus,5,25)){
	test_true=1;
      }
      break;
      
    case 011: // Condition-Not-Memory-Busy
      if(Memory_Busy == 0){
	test_true = 1;
      }
      break;
      
    case 012: // Condition-Q0
      test_true = (Qregister&0x01);
      break;
      
    case 013: // Condition-NU-Bus-Error
      if(NUbus_error != 0){
	test_true=1;
      }
      break;
      
    case 014: // Condition-Not-Fixnum-Overflow
      if(ALU_Fixnum_Oflow == 0){
	test_true=1;
      }
      break;
      
    case 015: // Condition-Boxed-Sign-Bit
      if((ALU_Result&0x01000000) == 0x01000000){
	test_true=1;
      }
      break;
      
    case 016: // Condition-No-Interrupt
      if(Active_PI==0){
	test_true=1;
      }
      break;

    case 017: // RESERVED-UNKNOWN-OPERATION
      // EXPT uses this one during the T-memory test.
      // I don't know what it does, and so I will fake it out.
      logmsg("CONDITION-17-USED\n");
      test_true=1;
      break;

    default:
      sprintf(msg,"UNKNOWN COND SELECT (0%o) - OP %d @ uPC %ld TMW %d TMS %d\n",MInst_Cond_Select,opword,loc_ctr_reg,
	      MInst_Write_ClRAM,MInst_Cond_ClRAM_Sel); logmsg(msg);
      cpu_die_rq=1;
    }
    
    /*
      if(MInst_ALU_Tagged != 0){
      logmsg("MInst-ALU-Tagged\n");
      cpu_die_rq=1;
      }
    */
  }else{
    // T-MEM READ: MInst_Cond_Select has address
    unsigned long long data;
    int Tpos,result;
    data = Tmemory[MInst_Cond_Select];
    Tpos = ldb(Mbus,5,25);     // Tag position
    // Mtag = ldb(Mbus,1,30);
    result = ldb(data,1,Tpos); // Isolate tag bit
    if(tracerq){ sprintf(msg,"TMEM-READ: Loc %X, Data %LX, want bit %d, got %d\n",MInst_Cond_Select,data,Tpos,result); logmsg(msg); }
    // cpu_die_rq=1;
    // ?
    // Compare to Mbus tag bit 
    // if(Mtag == 1){ logmsg("MTag\n"); cpu_die_rq = 1; }
    if(result == 1){ test_true = 1; }
  }

  if(MInst_Cond_Invrt_Sns != 0){
    if(test_true == 0){
      test_true = 1;
    }else{
      test_true = 0;
    }
  }
}

inline void handle_abj(){
  switch(MInst_Abbrv_Jump){
  case 0: // A-Jump-Field-Nop
    break;
    
  case 1: // A-Jump-Field-Skip
    // Skip next instruction
    pcss_skip = 1;
    break;
    
  case 2: // A-Jump-Call-Illop
    // PUSH ADDRESS
    uPCS_ptr_reg++;  uPCS_ptr_reg &= 0x3F;
    uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;
    // Jump
    pcss_jdst = 010;
    pc_source_select = 1; // JUMP
    break;
    
  case 3: // A-Jump-Call-Trap
    // PUSH ADDRESS
    uPCS_ptr_reg++;  uPCS_ptr_reg &= 0x3F;
    uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;
    // Jump
    pcss_jdst = 012;
    pc_source_select = 1; // JUMP
    break;
    
  case 4: // A-Jump-Call-Buserr
    // PUSH ADDRESS
    uPCS_ptr_reg++;  uPCS_ptr_reg &= 0x3F;
    uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;
    // Jump
    pcss_jdst = 014;
    pc_source_select = 1; // JUMP
    break;
    
  case 5: // A-Jump-Call-Unused
    // PUSH ADDRESS
    uPCS_ptr_reg++;  uPCS_ptr_reg &= 0x3F;
    uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;
    // Jump
    pcss_jdst = 016;
    pc_source_select = 1; // JUMP
    break;
    
  case 6: // A-Jump-Popj
    // POP ADDRESS
    pcss_jdst = uPCS_stack[uPCS_ptr_reg];
    uPCS_ptr_reg--;  uPCS_ptr_reg &= 0x3F;
    pc_source_select = 1; // JUMP
    break;
    
  case 7: // A-Jump-Popj-After-Next
    pc_source_select = 4; // ARM
    break;
    
  default:
    sprintf(msg,"AJMP UNKNOWN(0%o)\n",MInst_Abbrv_Jump); logmsg(msg);
    cpu_die_rq=1;
  }
}

inline void handle_output_select(){
  switch(MInst_Output_Bus_Ctl){
	    
  case 00: // Output-Bus-A-Bus
  case 02: // Output-Bus-A-Bus2
    Obus = Abus;
    break;
	    
  case 01: // Output-Bus-R-Bus
    {
      MInst_Rotation_Count = (Iregister&0x1F);             // ldb(Iregister,5,0);
      MInst_Rotation_Directn = (bool)(Iregister&0x10000);  // ((Iregister>>16)&0x01); // ldb(Iregister,1,16);
      MInst_Rotation_Len = 32;
      MInst_Mask_Rotate = 0;
      MInst_Source_Rotate = 1;
      /*
      Rbus = Mbus;
      if(MInst_Rotation_Directn == 0){
	Rbus = left_rotate(Rbus,MInst_Rotation_Count);
      }else{
	Rbus = right_rotate(Rbus,MInst_Rotation_Count);	      
      }
      */
      operate_shifter();
      Obus = Rbus;
    }
    break;
	  
  case 03: // Output-Bus-Normal
    Obus = Obus_Input;
    break;
	    
  case 04: // Output-Bus-LeftShift-1
    // Left shift gets SIGN from Qregister...
    Obus = (Obus_Input << 1);
    if(Qregister&0x80000000){ Obus |= 1; }
    break;
	    
  case 05: // Output-Bus-RightShift-1
    Obus = Obus_Input >> 1;
    // Different method
    if(ALU_Carry_Out != 0){
      Obus |= 0x80000000;
    }
    break;
	    
  case 06: // Output-Bus-Sign-Extend
    if((Obus_Input&0x01000000)==0){
      Obus = Obus_Input&0x00FFFFFF;
    }else{
      Obus = Obus_Input|0xFF000000;
    }
    break;
	    
  case 07: // Output-Bus-Mirror
    /* Reverse the bits in the ALU output
       Finally, a chance to use this famous example of confusing code!
       (Is it bad that I immediately recognised what it was doing when first reading it?) */
    Obus = Obus_Input;
    Obus = ((Obus >>  1) & 0x55555555) | ((Obus <<  1) & 0xaaaaaaaa);
    Obus = ((Obus >>  2) & 0x33333333) | ((Obus <<  2) & 0xcccccccc);
    Obus = ((Obus >>  4) & 0x0f0f0f0f) | ((Obus <<  4) & 0xf0f0f0f0);
    Obus = ((Obus >>  8) & 0x00ff00ff) | ((Obus <<  8) & 0xff00ff00);
    Obus = ((Obus >> 16) & 0x0000ffff) | ((Obus << 16) & 0xffff0000);
    /* I also just realized that Nevermore does the same thing in LISP.
       LISP somehow makes it look much more confusing.
       Maybe it's just me. */
    break;
	    
  default:
    sprintf(msg,"UNKNOWN-OBUS-CTL(%d)\n",MInst_Output_Bus_Ctl); logmsg(msg);
    cpu_die_rq=1;
    break;
  }
}

// Clock Pulse
void raven_clockpulse(){
  // Perform control cycle
  while(cpu_die == 0){    
    ccount++;
    delay_indx=0;
    while(delay_indx < delay_tune){
      delay_indx++;
    }

    // Run in one clock
    // case CPC_PC:
    //  if(minorcycle_clock == CLK_HI){
    //  if(master_clock == CLK_HI){
    // RBUS > DISPATCH ADDRESS SELECT
    // MF > DISPATCH ADDRESS SELECT
    // IBUF > DISPATCH ADDRESS SELECT
    // TRAP PC GEN > PC SOURCE SELECT
    // PC+1 > PC SOURCE SELECT
    // pcss_pc1 = loc_ctr_reg + 1;
    // IR JUMP ADDR > PC SOURCE SELECT
    // RETURN ADDRESS SELECT MX > uPCS
    //	uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;
    // then
    // MCLK-LO TIME    
    // uPCS > PC SOURCE SELECT
    
    // Otherwise
    //  if(master_clock == CLK_HI){
    // PC SOURCE SELECT > PC

    switch(pc_source_select){
    case 0: // Default PC+1
      loc_ctr_reg++; // = pcss_pc1;
      last_loc_ctr = loc_ctr_reg+1; // Expect next
      break;
    case 1: // JUMP
      loc_ctr_reg = pcss_jdst;
      last_loc_ctr = loc_ctr_reg+1; // Expect next
      // Switch back to defaults
      pc_source_select = 0;
      break;
    case 2: // PENDING JUMP
      loc_ctr_reg++; // Execute next
      last_loc_ctr = pcss_jdst; // Expect dest
      pcss_ddst = pcss_jdst;    // Save this
      pc_source_select = 8;     // Arm JUMP
      break;
    case 3: // PENDING CALL
      loc_ctr_reg++; // Continue
      // PUSH ADDRESS
      uPCS_ptr_reg++; uPCS_ptr_reg &= 0x3F;
      uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;
      // Arm JUMP
      last_loc_ctr = pcss_jdst; // Expect dest
      pcss_ddst = pcss_jdst;    // Protect destination
      pc_source_select = 8;     // Armed and dangerous
      break;
    case 4: // PENDING POPJ
      pc_source_select = 5;
      loc_ctr_reg++;
      break;
    case 5: // POPJ
      loc_ctr_reg = uPCS_stack[uPCS_ptr_reg];
      uPCS_ptr_reg--; uPCS_ptr_reg &= 0x3F;
      pc_source_select = 0; // CONTINUE
      break;
    case 6: // PENDING RETURN
      pc_source_select = 7; // ARM RETURN
      loc_ctr_reg++; // Continue
      break;
    case 7: // RETURN AFTER XCT
      // POP ADDRESS
      loc_ctr_reg = uPCS_stack[uPCS_ptr_reg];
      uPCS_ptr_reg--;  uPCS_ptr_reg &= 0x3F;
      pc_source_select = 0; // CONTINUE
      break;
    case 8: // PENDING JUMP COMPLETION
      loc_ctr_reg = pcss_ddst;
      last_loc_ctr = loc_ctr_reg+1; // Expect next
      // Switch back to defaults
      pc_source_select = 0;
      break;

    }	
    if(pcss_skip == 1){
      loc_ctr_reg++;
      // ??
      last_loc_ctr++; // Expect... uh, something maybe?
      pcss_skip = 0;
    }
   
    /*
    if(loc_ctr_reg == 1144){
      sprintf(msg,"[CPU] UCode breakpoint halt\n"); logmsg(msg);
      tracerq=1;
      cpu_die_rq=1;
    }        
    */

    // Die on errors
    /*
    if((MCregister&MCR_PROM_Disable) == 0 && loc_ctr_reg > 2048){
      sprintf(msg,"[CPU] loc_ctr_reg (from %lX) IS OUT OF BOUNDS FOR PROM!\n",last_loc_ctr); logmsg(msg);
      tracerq=1;
      cpu_die=1;
      break;
      } */
    if((MCregister&MCR_PROM_Disable) == 1 && loc_ctr_reg > 16384){
      sprintf(msg,"[CPU] loc_ctr_reg (from %lX) IS OUT OF BOUNDS FOR WCS!\n",last_loc_ctr); logmsg(msg);
      tracerq=1;
      cpu_die=1;
      break;
    }        

    // FETCH    
    if(MCregister&MCR_PROM_Disable || loc_ctr_reg > 2048){
      Iregister = WCS[loc_ctr_reg];
    }else{
      Iregister = PCS[loc_ctr_reg];
    }

    // IMOD LOGIC > IR
    if(imod_en != 0){
      unsigned long long Imod;
      Imod = imod_hi;
      Imod = Imod << 32;
      Imod |= imod_lo;
      Iregister |= Imod;
      imod_en = 0;
      imod_hi = 0;
      imod_lo = 0;
    }else{
      // IMod inhibits parity checking?
      
      if((MCregister&(MCR_Parity_Halt_Enable|MCR_Parity_Trap_Enable)) != 0){
	// PARITY TEST
	/*
	MInst_Parity = ((Iregister >> 50)&0x1);  //     = ldb(Iregister,1,50);
	if(gen_i_parity(Iregister) != MInst_Parity){
	  sprintf(msg,"PARITY ERROR STOP: %d\n",gen_i_parity(Iregister)); logmsg(msg);
	  cpu_die_rq=1;
	}
	*/
	if(MCregister&MCR_PROM_Disable || loc_ctr_reg > 2048){
	  if(((Iregister >> 50)&0x1) != ((Iregister >> 60)&0x1)){
	    logmsg("WCS PARITY STOP\n");
	    cpu_die_rq=1;
	  }
	}
      }
      
    }
    
    // Collect source addresses
    MInst_M_Source_Addr  = ((Iregister>>42)&0x7F);  // ldb(Iregister,7,42);
    MInst_A_Source_Addr  = ((Iregister>>32)&0x3FF); // ldb(Iregister,10,32);
    // Prepare source reads

    // Trigger read operations
    // if(AMem_Addr_Go == 2){
    Abus = Amemory[MInst_A_Source_Addr];
    //  AMem_Addr_Go = 0;
    // }
    // if(MMem_Addr_Go == 2){
    if(MInst_M_Source_Addr > 077){
      Mbus = 0;
    }else{
      Mbus = Mmemory[MInst_M_Source_Addr];
    }
    // MMem_Addr_Go = 0;
    //}
    
    // case CPC_DECEXE:
    // if(minorcycle_clock == CLK_HI){
    // if(master_clock == CLK_HI){
    // PC > LPC (This doesn't seem right...)
    // last_loc_ctr = loc_ctr_reg;
    // IR > IR DECODE LOGIC

    // GLOBAL FIELD DECODE
    opword           = ((Iregister >> 54)&0x3);  //     = ldb(Iregister,2,54); 
    MInst_Abbrv_Jump = ((Iregister >> 51)&0x07); //     = ldb(Iregister,3,51);
    
    // MInst_Halt
    if((Iregister&0x2000000000000LL) != 0){
      logmsg("**MInst-HALT**\n");
      cpu_die_rq=1;
    }

#ifdef TRACELOG
//    sprintf(msg,"DEBUG: [%ld]\n",loc_ctr_reg); logmsg(msg);
#endif

    switch(opword){
    case 0: // ALU-OP
      {
	// Fields
	unsigned int MInst_Dest_A_Select  = (bool)(Iregister&0x80000000); // ((Iregister>>31)&0x01);   // ldb(Iregister,1,31);
	unsigned int MInst_Dest_AMem_Addr = ((Iregister>>19)&0x3FF);      // ldb(Iregister,10,19);
	unsigned int MInst_Dest_MMem_Addr = ((Iregister>>19)&0x3F);       // ldb(Iregister,6,19);
	//	unsigned int MInst_Output_Bus_Ctl = ((Iregister>>16)&0x07);       // ldb(Iregister,3,16);

	MInst_Write_ClRAM    = (bool)(Iregister&0x200);      // ((Iregister>>9)&0x01);   // ldb(Iregister,1,9); 
	// unsigned int MInst_ALU_Tagged     = (bool)(Iregister&0x100);      // ((Iregister>>8)&0x01);   // ldb(Iregister,1,8); 
	unsigned int MInst_Q_Control      = (Iregister&0x03);             // ldb(Iregister,2,0);

	MInst_Output_Bus_Ctl = ((Iregister>>16)&0x07);       // ldb(Iregister,3,16);
	MInst_Dest_Functnl   = ((Iregister>>25)&0x3F);       // ldb(Iregister,6,25);
	MInst_Cond_Invrt_Sns = (bool)(Iregister&0x8000);     // ((Iregister>>15)&0x01);   // ldb(Iregister,1,15);
	MInst_Cond_ClRAM_Sel = (bool)(Iregister&0x4000);     // ((Iregister>>14)&0x01);  // ldb(Iregister,1,14); 
	MInst_Cond_Select    = ((Iregister>>10)&0x0F);       // ldb(Iregister,4,10);
	MInst_ALU_Opcode     = ((Iregister>>3)&0x1F);        // ldb(Iregister,5,3);	  
	MInst_ALU_Carry_In   = (bool)(Iregister&0x4);        // ((Iregister>>2)&0x01);   // ldb(Iregister,1,2);
      
	if(MInst_M_Source_Addr > 077){
	  handle_m_fcn_src();
	}
	
	// Reset flags
	ALU_Fixnum_Oflow=0;
	ALU_Carry_Out=0;

	switch(MInst_ALU_Opcode){
	case 0: // ALU-Opcode-SETZ
	  ALU_Result = 0;
	  break;	      

	case 1: // ALU-Opcode-AND
	  ALU_Result = Mbus&Abus;
	  // If bit sign is set, carry-out.
	  if(ALU_Result&0x80000000){
	    ALU_Result |= 0x100000000LL;
	  }
	  break;
	
	case 2: // ALU-Opcode-ANDCA
	  ALU_Result = Mbus&(~Abus);
	  break;

	case 3: // ALU-Opcode-SETM
	  ALU_Result = Mbus;
	  break;

	case 4: // ALU-Opcode-ANDCM
	  ALU_Result = (~Mbus)&Abus;
	  break;

	case 5: // ALU-Opcode-SETA
	  ALU_Result = Abus;
	  break;

	case 6: // ALU-Opcode-XOR
	  ALU_Result = Mbus^Abus;
	  break;

	case 7: // ALU-Opcode-IOR
	  ALU_Result = Mbus|Abus;
	  break;

	case 010: // ALU-Opcode-ANDCB
	  ALU_Result = (~Mbus)&(~Abus);
	  break;

	case 011: // ALU-Opcode-EQV
	  Abus = Mbus; // What for?
	  ALU_Result = Mbus;
	  break;

	case 012: // ALU-Opcode-SETCA
	  ALU_Result = ~Abus;
	  break;

	case 013: // ALU-Opcode-ORCA
	  ALU_Result = Mbus|(~Abus);
	  break;

	case 014: // ALU-Opcode-SETCM
	  ALU_Result = ~Mbus;
	  break;
	  
	case 015: // ALU-Opcode-ORCM
	  ALU_Result = (~Mbus)|Abus;
	  break;

	case 017: // ALU-Opcode-SETO
	  ALU_Result = 0xFFFFFFFF;
	  break;

	case 020: // ALU-Opcode-MUL
	  if((Qregister&0x01)==0x01){
	    ALU_Result = Mbus + Abus;
	  }else{
	    ALU_Result = Mbus;
	  }
	  // Carry out = sign.
	  if(ALU_Result&0x80000000){ ALU_Result |= 0x100000000LL; }
	  break;

	case 022: // ALU-Opcode-DIV
	  if((Qregister&0x01)==0){
	    ALU_Result = Mbus + Abus;
	  }else{
	    ALU_Result = Mbus - Abus;
	  }
	  break;

	case 023: // ALU-Opcode-DIV-First
	  ALU_Result = Mbus - Abus;
	  break;

	case 024: // ALU-Opcode-DIV-Corr
	  if((Qregister&0x01)==0x00){
	    ALU_Result = Mbus + Abus;
	  }else{
	    ALU_Result = Mbus;
	  }
	  break;

	case 031: // ALU-Opcode-ADD
	  ALU_Result = (unsigned long long)Mbus+(unsigned long long)Abus;
	  //	  sprintf(msg,"%ld: ADD: 0x%lX + 0x%lX = 0x%LX\n",loc_ctr_reg,Mbus,Abus,ALU_Result); logmsg(msg);
	  if(MInst_ALU_Carry_In != 0){
	    ALU_Result++;
	  }
	  // FIXNUM Overflow Check
	  if((((Mbus^ALU_Result)&(Abus^ALU_Result))&0x01000000)==0x01000000){
	    ALU_Fixnum_Oflow=1;
	  }
	  // HACKHACK is this correct?
	  if(ALU_Result < Mbus){
	    ALU_Result |= 0x100000000LL; // Arrange for carry-out
	  }
	  break;
	
	case 034: // ALU-Opcode-M
	  // Output = M or M+1
	  ALU_Result = Mbus;
	  if(MInst_ALU_Carry_In != 0){
	    ALU_Result++;
	  }
	  // HACKHACK is this correct?
	  if(ALU_Result < Mbus){
	    ALU_Result |= 0x100000000LL; // Arrange for carry-out
	  }
	  break;

	case 036: // ALU-Opcode-SUB
	  ALU_Result = (unsigned long long)Mbus - (unsigned long long)Abus;
	  if(MInst_ALU_Carry_In == 0){
	    ALU_Result--;
	  }
	  // FIXNUM Overflow Check
	  if((((Mbus^Abus)&(Mbus^ALU_Result))&0x01000000)==0x01000000){
	    ALU_Fixnum_Oflow=1;
	  }
	  // HACKHACK is this correct?
	  if(ALU_Result < Mbus){
	    ALU_Result |= 0x100000000LL; // Arrange for carry-out
	  }
	  break;

	case 037: // ALU-Opcode-M+M
	  // Output = M+M or M+M+1
	  ALU_Result = Mbus+Mbus;
	  if(MInst_ALU_Carry_In != 0){
	    ALU_Result++;
	  }
	  /*
	  // HACKHACK is this correct?
	  if(ALU_Result < Mbus){
	    ALU_Result |= 0x100000000LL; // Arrange for carry-out
	  }
	  */
	  break;

	default:
	  sprintf(msg,"UNKNOWN-ALU-OPCODE(0%o)\n",MInst_ALU_Opcode); logmsg(msg);
	  cpu_die_rq=1;
	  break;
	}
	// Reduce carry-out to a flag
	ALU_Carry_Out = (ALU_Result&0xFFFFFFFF00000000LL);
	// Clean Output (ALU is 32 bits wide)
	ALU_Result &= 0xFFFFFFFF;

	// Make the result
	Obus = ALU_Result;

	if(MInst_Write_ClRAM != 0){
	  unsigned long data = 0;
	  unsigned long long Tmask=0,Tdata=0;
	  int Trotn=0;

	  data = ALU_Result;

	  // Get position of tag bit to overwrite
	  Trotn = ldb(data,5,25); // ((data&0x3E000000)>>25);
	  // And make a mask
	  Tmask = (0x1 << Trotn);
	  
	  // Test
	  if(Trotn > 31){ sprintf(msg,"Trotn %d > 31!\n",Trotn); logmsg(msg); cpu_die_rq=1; }

	  Tdata = Tmemory[MInst_Cond_Select];
	  // Just...
	  if((data&0x40000000) == 0x40000000){
	    // Set it...
	    Tdata |= Tmask;
	    if(tracerq){ sprintf(msg,"T-MEM WRITE: Bit %d set for %X = %LX\n",Trotn,MInst_Cond_Select,Tdata); logmsg(msg); }
	  }else{
	    // Or forget it!
	    Tdata &= (~Tmask);
	    if(tracerq){ sprintf(msg,"T-MEM WRITE: Bit %d reset for %X = %LX\n",Trotn,MInst_Cond_Select,Tdata); logmsg(msg); }
	  }
	  // But wait, there's more!	
	  Tmemory[MInst_Cond_Select] = Tdata;
	  // Now how much would you pay?
	  test_true=1;
	}
	// Run conditionalizer
	handle_condition_select(); 
	
	if(test_true == 1){
	  // Don't handle ABJ if no test truth
	  handle_abj();
	}

	// Determine output location
	Obus_Input = ALU_Result;
	handle_output_select();
	// If in arithmetic mode...
	if((Iregister&0x10100) == 0x10100){

	  Obus_Input = (ALU_Result&0x00FFFFFF);
	  Obus_Input |= (Abus&0xFF000000);

	  switch(MInst_Output_Bus_Ctl){

	  case 0x01: // Rbus
	    {
	      MInst_Rotation_Count = (Iregister&0x1F);             // ldb(Iregister,5,0);
	      MInst_Rotation_Directn = (bool)(Iregister&0x10000);  // ((Iregister>>16)&0x01); // ldb(Iregister,1,16);
	      MInst_Rotation_Len = 32;
	      MInst_Mask_Rotate = 0;
	      MInst_Source_Rotate = 1;
	      /*
	      Rbus = Mbus;
	      if(MInst_Rotation_Directn == 0){
		Rbus = left_rotate(Rbus,MInst_Rotation_Count);
	      }else{
		Rbus = right_rotate(Rbus,MInst_Rotation_Count);	      
	      }
	      */
	      operate_shifter();
	      Obus  = Obus_Input&0xFE000000; // Preserve A bits
	      Obus |= (Rbus&0x1FFFFFF);      // Include R bus
	      // sprintf(msg,"CPU: RBus-AM Test: RBus = 0x%lX\n",Rbus); logmsg(msg);
	    }
	    break;
	    
	  case 0x03: // ALU
	    Obus = Obus_Input;
	    break;

	  case 05: // Output-Bus-RightShift-1
	    Obus = Obus_Input >> 1;                   // Rightshift...
	    Obus &= 0x1FFFFFF;                        // Then eliminate the high bits
	    // Different method for shorter math
	    {
	      // Funny carry arithmetic...
	      unsigned long atmp=0;
	      atmp =  ((Abus&0xFF000000)>>24);        // Insert A-bus
	      atmp += ((Obus_Input&0xFF000000)>>24);  // and the high output bits?
	      if(atmp&0xFFFFFF00){atmp |= 0x80; }     // Summation of overflow
	      Obus |= (atmp<<24);                     // Tack it on the O bus
	    }
	    if(ALU_Carry_Out){                        // If ALU carried-out
	      Obus ^= 0x80000000;                     // invert 32-bit sign.
	    }
	    break;
	    
	  case 0x07: // MIRROR
	    {
	      // Invert
	      //	      sprintf(msg,"CPU: OBus-AM Test-1: OBus_Input = 0x%LX,ALU-Output = 0x%LX\n",Obus_Input,ALU_Result); logmsg(msg);
	      Obus_Input = (Obus_Input>>8);
	      Obus = ALU_Result; // Obus_Input;
	      Obus = ((Obus >>  1) & 0x55555555) | ((Obus <<  1) & 0xaaaaaaaa);
	      Obus = ((Obus >>  2) & 0x33333333) | ((Obus <<  2) & 0xcccccccc);
	      Obus = ((Obus >>  4) & 0x0f0f0f0f) | ((Obus <<  4) & 0xf0f0f0f0);
	      Obus = ((Obus >>  8) & 0x00ff00ff) | ((Obus <<  8) & 0xff00ff00);
	      Obus = ((Obus >> 16) & 0x0000ffff) | ((Obus << 16) & 0xffff0000);
	      // sprintf(msg,"CPU: OBus-AM Test-2: OBus = 0x%lX\n",Obus); logmsg(msg);
	      // Sign-Extend
	      if((ALU_Result&0x01000000)==0){
		Obus = Obus&0x01FFFFFF;
	      }else{
		Obus = Obus|0xFE000000;
	      }
	      // sprintf(msg,"CPU: OBus-AM Test-3: OBus = 0x%lX\n",Obus); logmsg(msg);
	    }
	    break;
	    
	  default:
	    sprintf(msg,"CPU: Unknown Arithmetic-Mode OBus Control 0x%X\n",MInst_Output_Bus_Ctl); logmsg(msg);
	    cpu_die_rq=1;
	  }
	}	
      
	// Run Q
	if(MInst_Q_Control != 0){
	  switch(MInst_Q_Control){
	  case 0: // Q-Control-NOP
	    break;
	    
	  case 01: // Q-Control-Shift-Left
	    // Shift
	    Qregister = Qregister << 1;
	    // Carry-In inverse of ALU sign bit
	    if((ALU_Result&0x80000000) == 0){
	      Qregister |= 1;
	    }
	    break;
	    
	    // For later reference, a right-shift is not inverted from the ALU output.
	    
	  case 02: // Q-Control-Shift-Right
	    Qregister = Qregister >> 1;
	    // Carry-In ALU result
	    if((ALU_Result&01) == 0x01){
	      Qregister |= 0x80000000;
	    }
	    break;
	    
	  case 03: // Q-Control-Load
	    Qregister=ALU_Result;
	    break;
	    
	  default:
	    sprintf(msg,"UNKNOWN-Q-CONTROL(0%o)\n",MInst_Q_Control); logmsg(msg);
	    cpu_die_rq=1;
	  }
	}

	// Store result
	if(MInst_Dest_A_Select != 0){
	  // Store in A-Memory
	  Abus = Obus;
	  AMem_Addr_Sel = MInst_Dest_AMem_Addr;
	  AMem_Addr_Go = 1;
	}else{
	  // Store in M-Memory
	  Mbus = Obus;
	  MMem_Addr_Sel = MInst_Dest_MMem_Addr;
	  MMem_Addr_Go = 1;

	  handle_m_fcn_dst();
	}
	break;
      }
    case 1: // BYTE-OP
      {
	unsigned int MInst_Dest_A_Select  = (bool)(Iregister&0x80000000); // ((Iregister>>31)&0x01);   // ldb(Iregister,1,31);
	unsigned int MInst_Dest_MMem_Addr = ((Iregister>>19)&0x03F);      // ldb(Iregister,6,19);
	unsigned int MInst_Dest_AMem_Addr = ((Iregister>>19)&0x3FF);      // ldb(Iregister,10,19);
	// BYTE-specific
	MInst_Mask_Rotate      = (bool)(Iregister&0x40000);  // ldb(Iregister,1,18);
	MInst_Source_Rotate    = (bool)(Iregister&0x20000);  // ldb(Iregister,1,17);
	MInst_Rotation_Directn = (bool)(Iregister&0x10000);  // ((Iregister>>16)&0x01); // ldb(Iregister,1,16);
	// unsigned int MInst_Byte_Op           = ((Iregister>>17)&0x03);    // ldb(Iregister,2,17);
	// Shared
	// Byte-specific
	MInst_Rotation_Len   = ((Iregister>>5)&0x1F);        // ldb(Iregister,5,5);
	MInst_Rotation_Count = (Iregister&0x1F);             // ldb(Iregister,5,0);

	MInst_Dest_Functnl   = ((Iregister>>25)&0x03F);      // ldb(Iregister,6,25);
	MInst_Cond_Invrt_Sns = (bool)(Iregister&0x8000);     // ((Iregister>>15)&0x01);    // ldb(Iregister,1,15);
	MInst_Cond_ClRAM_Sel = (bool)(Iregister&0x4000);     // ((Iregister>>14)&0x01);    // ldb(Iregister,1,14); 
	MInst_Cond_Select    = ((Iregister>>10)&0x0F);       // ldb(Iregister,4,10);

	// Whack some fields
	MInst_ALU_Opcode = 0;
	MInst_ALU_Carry_In = 0;

	// Run conditionalizer
	handle_condition_select();
      
	if(test_true == 1){
	  // Only handle ABJ if test true
	  handle_abj();
	}

	if(MInst_M_Source_Addr > 077){
	  handle_m_fcn_src();
	}
      
	// ** DO WORK HERE **
	operate_shifter();

	// Select O-bus
      
	// Store result
	if(MInst_Dest_A_Select != 0){
	  // Store in A-Memory
	  Abus = Obus;
	  AMem_Addr_Sel = MInst_Dest_AMem_Addr;
	  AMem_Addr_Go = 1;
	}else{
	  // Store in M-Memory
	  Mbus = Obus;
	  MMem_Addr_Sel = MInst_Dest_MMem_Addr;
	  MMem_Addr_Go = 1;
	  
	  // Handle MF bus
	  handle_m_fcn_dst();
	}
      }
      break;

    case 2: // JUMP-OP
      {
	// Fields
	unsigned int MInst_New_Micro_PC   = ((Iregister>>18)&0x3FFF);  // ldb(Iregister,14,18);
	unsigned int MInst_M_Source_Sel   = (bool)(Iregister&0x20000); // ((Iregister>>17)&0x01);   // ldb(Iregister,1,17);
	/* More unused fields were here... */
	
	unsigned int MInst_Write_to_WCS   = (bool)(Iregister&0x200);   // ((Iregister>>9)&0x01);    // ldb(Iregister,1,9);
	unsigned int MInst_Read_from_WCS  = (bool)(Iregister&0x100);   // ((Iregister>>8)&0x01);    // ldb(Iregister,1,8);
	unsigned int MInst_Jump_Op        = ((Iregister>>5)&0x07);     // ldb(Iregister,3,5);
	// unsigned int MInst_Return_Bit     = ((Iregister>>7)&0x01);    // ldb(Iregister,1,7);
	// unsigned int MInst_Call_Bit       = ((Iregister>>6)&0x01);    // ldb(Iregister,1,6);
	// unsigned int MInst_Inhibit_Bit    = ((Iregister>>5)&0x01);    // ldb(Iregister,1,5);
	/* More unused fields were here */

	MInst_Cond_Invrt_Sns = (bool)(Iregister&0x8000);  // ((Iregister>>15)&0x01);   // ldb(Iregister,1,15);
	MInst_Cond_ClRAM_Sel = (bool)(Iregister&0x4000);     // ((Iregister>>14)&0x01);  // ldb(Iregister,1,14); 
	MInst_Cond_Select   = ((Iregister>>10)&0x0F);    // ldb(Iregister,4,10);

	// Whack some fields
	MInst_ALU_Opcode = 0;
	MInst_ALU_Carry_In = 0;

	if(MInst_Read_from_WCS != 0){
	  unsigned int MInst_Dest_A_Select  = (bool)(Iregister&0x80000000); 
	  unsigned int MInst_Dest_MMem_Addr = ((Iregister>>19)&0x03F); // WCS M bits (31:0) go here
	  unsigned int MInst_Dest_AMem_Addr = ((Iregister>>19)&0x3FF); // WCS A bits (56:32) go here

	  // See below for bad explanation of how this works
	  if((MCregister&0x800) == 0 && last_loc_ctr < 2048){
	    // Read PCS
	    if(MInst_Dest_A_Select == 0){
	      Mmemory[MInst_Dest_MMem_Addr] = ldb(PCS[last_loc_ctr],32,0);
	    }else{
	      Amemory[MInst_Dest_AMem_Addr] = ldb(PCS[last_loc_ctr],24,32);
	    }
	    if(tracerq){sprintf(msg,"R-CS %ld = %LX\n",last_loc_ctr,PCS[last_loc_ctr]); logmsg(msg); }
	  }else{
	    // Read WCS
	    if(MInst_Dest_A_Select == 0){
	      Mmemory[MInst_Dest_MMem_Addr] = ldb(WCS[last_loc_ctr],32,0);	      
	    }else{
	      Amemory[MInst_Dest_AMem_Addr] = ldb(WCS[last_loc_ctr],24,32);
	    }
	    /*
	    if((MCregister&(MCR_Parity_Halt_Enable|MCR_Parity_Trap_Enable)) != 0){
	      // PARITY TEST
	      MInst_Parity = ((WCS[last_loc_ctr] >> 50)&0x1);  //     = ldb(Iregister,1,50);
	      if(gen_i_parity(WCS[last_loc_ctr]) != MInst_Parity){
		sprintf(msg,"WCS-READ PARITY ERROR STOP: %d\n",gen_i_parity(Iregister)); logmsg(msg);
		cpu_die_rq=1;
	      }
	    }
	    */
	    if(tracerq){sprintf(msg,"R-CS %ld = %LX\n",last_loc_ctr,WCS[last_loc_ctr]); logmsg(msg); }
	  }

	  // Continue as nothing happened - No jump.
	  //	  cpu_die_rq=1;
	}

	if(MInst_Write_to_WCS != 0){
	  // last-micro-PC is the destination to write.
	  // Data comes from A and M.
	  // Write WCS
	  WCS[last_loc_ctr] = Abus;
	  WCS[last_loc_ctr] = WCS[last_loc_ctr] << 32;
	  WCS[last_loc_ctr] |= Mbus;
	  {
	    unsigned long long x = gen_i_parity(WCS[last_loc_ctr]);
	    x <<= 60;
	    WCS[last_loc_ctr] |= x;
	  }
	  /*
	  if((MCregister&(MCR_Parity_Halt_Enable|MCR_Parity_Trap_Enable)) != 0){
	    // PARITY TEST
	    MInst_Parity = ((WCS[last_loc_ctr] >> 50)&0x1);  //     = ldb(Iregister,1,50);
	    if(gen_i_parity(WCS[last_loc_ctr]) != MInst_Parity){
	      sprintf(msg,"WCS-WRITE PARITY ERROR STOP: %d\n",gen_i_parity(Iregister)); logmsg(msg);
	      cpu_die_rq=1;
	    }
	  }
	  */  
	}

	/*
	if(MInst_Inhibit_Bit != 0){
	printw("INH ");
	// This is a pipeline thing - Kill the next instruction fetch
	// if it's currently occuring. It's not.
	}
	*/

	// Initialize
	test_src = 0;
	
	if(MInst_M_Source_Addr > 077){
	  handle_m_fcn_src();
	}
	  
	if(MInst_M_Source_Sel == 0){
	  test_src = Mbus;
	}else{
	  logmsg("ASEL (THIS IS BAD?)\n");
	  test_src = Abus;
	  cpu_die_rq=1;
	}

	handle_condition_select();
      
	if(test_true == 0){
	  break;
	}

	handle_abj();

	/* CODES ARE:
	   R P I  (RETURN, PUSH, INHIBIT)
	   
	   0 0 0 = Branch-Xct-Next
	   0 0 1 = Branch
	   0 1 0 = Call-Xct-Next
	   0 1 1 = Call
	   1 0 0 = Return-Xct-Next
	   1 0 1 = Return
	   1 1 0 = NOP (JUMP2-XCT-NEXT)
	   1 1 1 = SKIP (JUMP2)

	   We ALWAYS load jdst!

	*/
	
	switch(MInst_Jump_Op){
	case 6: // RAVFMT says Jump-Nop, THIS IS WRONG! It's really JUMP2-XCT-NEXT
	  // PUSH ADDRESS
	  uPCS_ptr_reg++;  uPCS_ptr_reg &= 0x3F;
	  uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;
	  // POP ADDRESS
	  pcss_jdst = uPCS_stack[uPCS_ptr_reg];
	  uPCS_ptr_reg--;  uPCS_ptr_reg &= 0x3F;
	  // FALL THRU
	case 0: // Jump-Branch-Xct-Next
	  // Jump, but DO NOT inhibit the next instruction!
	  pcss_jdst = MInst_New_Micro_PC;
	  pc_source_select = 2; // PENDING JUMP
	  break;

	case 7: // RAVFMT says Jump-Skip with is WRONG. It's really JUMP2
	  // PUSH ADDRESS
	  uPCS_ptr_reg++;  uPCS_ptr_reg &= 0x3F;
	  uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;
	  // POP ADDRESS
	  pcss_jdst = uPCS_stack[uPCS_ptr_reg];
	  uPCS_ptr_reg--;  uPCS_ptr_reg &= 0x3F;
	  // FALL THRU
	case 1: // Jump-Branch
	  pcss_jdst = MInst_New_Micro_PC;
	  pc_source_select = 1; // JUMP
	  break;

	case 2: // Jump-Call-Xct-Next
	  // Call, but DO NOT inhibit the next instruction!
	  pcss_jdst = MInst_New_Micro_PC;
	  pc_source_select = 3; // PENDING CALL
	  break;	  

	case 3: // Jump-Call
	  // PUSH ADDRESS
	  uPCS_ptr_reg++;  uPCS_ptr_reg &= 0x3F;
	  uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;
	  // Jump
	  pcss_jdst = MInst_New_Micro_PC;
	  pc_source_select = 1; // JUMP
	  break;

	case 4: // Jump-Return-XCT-Next
	  pc_source_select = 6; // Arm RETURN
	  break;

	case 5: // Jump-Return
	  // POP ADDRESS
	  pcss_jdst = uPCS_stack[uPCS_ptr_reg];
	  uPCS_ptr_reg--;  uPCS_ptr_reg &= 0x3F;
	  pc_source_select = 1; // JUMP
	  break;

	default:
	  sprintf(msg,"UNKNOWN-JUMP-OP(%d)\n",MInst_Jump_Op); logmsg(msg);
	  cpu_die_rq=1;
	}
      }
      break;

    case 3: // DISP-OP
      {
	// Fields
	unsigned int MInst_Disp_Constant    = MInst_A_Source_Addr;
	unsigned int MInst_Disp_Address     = (Iregister>>20)&0xFFF;
	/* Unused fields were here... */
	// Alter-Retn-Addr means STACK-OWN-ADDRESS
	// If N is set, instead of putting the next instruction's address on the stack during a uPush, put ours.
	unsigned int MInst_Alter_Retn_Addr  = (bool)(Iregister&0x20000); 

	unsigned int MInst_Enable_IStream   = (bool)(Iregister&0x8000);  // ((Iregister>>15)&0x01);   // ldb(Iregister,1,15);
	unsigned int MInst_Disp_Source      = (Iregister>>12)&0x03;
	unsigned int MInst_Map_Oldspace     = (bool)(Iregister&0x800);
	unsigned int MInst_Map_GC_Volatile  = (bool)(Iregister&0x400);
	/* These are in DISP-OPCODE
	unsigned int MInst_Write_Disp_Mem   = (bool)(Iregister&0x200);
	unsigned int MInst_Read_Disp_Mem    = (bool)(Iregister&0x100);
	*/
	// MInst-Map-Enable is MAP-OLDSPACE and GC-VOLATILE together
	unsigned int MInst_Disp_Opcode      = (Iregister>>8)&0x03;
	// R bus controls
	MInst_Rotation_Directn = (bool)(Iregister&0x10000); // ((Iregister>>16)&0x01); // ldb(Iregister,1,16);
	MInst_Rotation_Len   = ((Iregister>>5)&0x7);         // ldb(Iregister,3,5);
	MInst_Rotation_Count = (Iregister&0x1F);             // ldb(Iregister,5,0);
	// Force operation code
	MInst_Mask_Rotate = 0;
	MInst_Source_Rotate = 1; 

	// UHO!
	// unsigned int MInst_Mask_Length      = ((Iregister>>5)&0x07);     // ldb(Iregister,3,5);

	unsigned long dispatch_source=0;

	// Read MF if we must
	if(MInst_M_Source_Addr > 077){
	  handle_m_fcn_src();
	}

	switch(MInst_Disp_Source){

	case 00: // Dispatch-Source-M-Source (Rbus)
	  {
	    // Mbus is already loaded, Abus already loaded, opcode set
	    operate_shifter();
	    dispatch_source = Rbus; // Get result
	  }
	  break;

	  /*	  
		 case 01: // Dispatch-Source-MF-Bus
		 {
		 handle_m_fcn_src();
		 
		 }
		 break;
	  */

	default:
	  sprintf(msg,"UNKNOWN-DISP-SOURCE 0x%X\n",MInst_Disp_Source); logmsg(msg);
	  cpu_die_rq=1;
	}

	if(MInst_Enable_IStream != 0){
	  logmsg("ENABLE-ISTREAM\n");
	  cpu_die_rq=1;
	}

	if(MInst_Map_Oldspace != 0){
	  logmsg("MAP-OLDSPACE\n");
	  cpu_die_rq=1;
	}

	// Load dispatch-constant register
	disp_constant_reg = MInst_Disp_Constant;

	switch(MInst_Disp_Opcode){
	  
	case 00: // DISPATCH
	  {
	    unsigned long disp_entry=0;
	    unsigned long disp_action=0;
	    unsigned long MInst_New_Micro_PC=0;
	    unsigned int  MInst_Jump_Op=0;
	    
	    //	    disp_entry = (MInst_Disp_Address&0x7F) | (dispatch_source&0x7F);

	    switch(MInst_Disp_Source){
	    case 0:
	      // High bits of dispatch source come from Dispatch-Address
	      disp_entry = (MInst_Disp_Address|(dispatch_source&0x7F)); 
	      break;

	    default:
	      logmsg("UNKNOWN DISPATCH SOURCE IN DISP-OP 2\n");
	      cpu_die_rq=1;
	    }
	    // sprintf(msg,"DISPATCH: DISP-ADDRESS 0x%X, DISP-SOURCE 0x%lX, DISP-ENTRY 0x%lX\n",MInst_Disp_Address,dispatch_source,disp_entry); logmsg(msg);

	    // disp_entry points to an index into D-memory
	    disp_action = Dmemory[disp_entry];

	    /* DISPATCH-ACTION FORMAT:
	       0x03FFF = NEW-MICRO-PC
	       0x04000 = R
	       0x08000 = P
	       0x10000 = N	       
	    */

	    MInst_New_Micro_PC = (disp_action&0x3FFF);
	    MInst_Jump_Op = ((disp_action&0x1C000)>>14);

	    //	    sprintf(msg,"DISPATCH: DISP-ACTION 0x%lX, JUMP-OP %X, NEW-PC %lX\n",disp_action,MInst_Jump_Op,MInst_New_Micro_PC); logmsg(msg);

	    if(MInst_Map_GC_Volatile != 0){
	      unsigned long l1_gcv = 0;
	      unsigned long map1_addr,map1_data;

	      map1_addr = ldb(MDregister,12,13);
	      map1_data = vm_lv1_map[map1_addr];
	      l1_gcv = (map1_data&0x0380)>>7;

	      // sprintf(msg,"MAP-GC-VOLATILE: cached-gcv = 0x%X, l1-gcv = 0x%lX\n",cached_gcv,l1_gcv);  logmsg(msg);
	      if((cached_gcv+4) > (l1_gcv^0x07)){
		//		logmsg("NONVOLATILE\n");
	      }else{
		// logmsg("VOLATILE\n");
		MInst_New_Micro_PC |= 0x01;
	      }
	      //	      cpu_die_rq=1;
	    }
	       
	    if(MInst_Alter_Retn_Addr != 0){
	      sprintf(msg,"DISPATCH-STACK-SELF: DW-OP %X DW-ADDR %lX (%ld) DW %lX (DMEM-ADDR %lX) ROT-Dn %X RTC %d RTL %d MBus 0x%lX\n",
		      MInst_Jump_Op,MInst_New_Micro_PC,MInst_New_Micro_PC,disp_action,
		      disp_entry,MInst_Rotation_Directn,MInst_Rotation_Count,MInst_Rotation_Len,Mbus);
	      logmsg(msg);
	      cpu_die_rq=1;
	    }

	    switch(MInst_Jump_Op){
	      
	      /*
	    case 6: // RAVFMT says Jump-Nop, THIS IS WRONG! It's really JUMP2-XCT-NEXT
	      // PUSH ADDRESS
	      uPCS_ptr_reg++; uPCS_ptr_reg &= 0x3F;
	      uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;	      
	      // POP ADDRESS
	      pcss_jdst = uPCS_stack[uPCS_ptr_reg];
	      uPCS_ptr_reg--;  uPCS_ptr_reg &= 0x3F;
	      // FALL THRU
	      */
	    case 0: // Jump-Branch-Xct-Next
	      // Jump, but DO NOT inhibit the next instruction!
	      pcss_jdst = MInst_New_Micro_PC;
	      pc_source_select = 2; // PENDING JUMP
	      break;

	    case 7: // RAVFMT says Jump-Skip.
	      // Quoth the CPU manual: With R and P both 1, the dispatch is ignored and the status of the
	      // next instruction depends on N.
	      // In this case, Nhibit is set, so we skip.
	      // PUSH ADDRESS
	      uPCS_ptr_reg++; uPCS_ptr_reg &= 0x3F;
	      if(MInst_Alter_Retn_Addr != 0 && (disp_action&0x4000)){
		uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg;
	      }else{
		uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg+1;
	      }
	      // POP ADDRESS
	      pcss_jdst = uPCS_stack[uPCS_ptr_reg];
	      uPCS_ptr_reg--;  uPCS_ptr_reg &= 0x3F;
	      pc_source_select = 1; // JUMP
	      // This is basically a skip. We don't fall through, like the JUMP version.
	      // Which makes no sense.
	      break;
	      
	      /*
	      // case 1: // Jump-Branch
	      pcss_jdst = MInst_New_Micro_PC;
	      pc_source_select = 1; // JUMP
	      break;
	      
	      
	      case 2: // Jump-Call-Xct-Next
	      // Call, but DO NOT inhibit the next instruction!
	      pcss_jdst = MInst_New_Micro_PC;
	      pc_source_select = 3; // PENDING CALL
	      break;	  
	      
	      case 3: // Jump-Call
	      // PUSH ADDRESS
	      uPCS_ptr_reg++;  uPCS_ptr_reg &= 0x3F;
	      uPCS_stack[uPCS_ptr_reg] = loc_ctr_reg + 1;
	      // Jump
	      pcss_jdst = MInst_New_Micro_PC;
	      pc_source_select = 1; // JUMP
	      break;
	      
	      case 4: // Jump-Return-XCT-Next
	      pc_source_select = 6; // Arm RETURN
	      break;
	      */

	    case 5: // Jump-Return
	      // POP ADDRESS
	      // Peculiarity for DISP-OP?
	      pcss_jdst = (uPCS_stack[uPCS_ptr_reg]&0x3FFF);
	      uPCS_ptr_reg--;  uPCS_ptr_reg &= 0x3F;
	      pc_source_select = 1; // JUMP
	      break;	      

	    default:
	      sprintf(msg,"UNKNOWN-DISPATCH-WORD-JUMP-OP(%d)\n",MInst_Jump_Op); logmsg(msg);
	      cpu_die_rq=1;
	    }	    
	    // Go!
	    //	    cpu_die_rq=1;
	  }
	  break;
	  
	case 01: // READ (AND FALL THROUGH)
	  Qregister = Dmemory[MInst_Disp_Address];
	  break;

	case 02: // WRITE (AND FALL THROUGH)
	  Dmemory[MInst_Disp_Address] = (Abus&0x1FFFF);
	  break;

	default:
	  sprintf(msg,"UNKNOWN-DISP-OPCODE 0%o\n",MInst_Disp_Opcode); logmsg(msg);
	  cpu_die_rq=1;
	}

	// Do ABJ
	handle_abj();
      }
      break;
    }
    // Done parsing

    // Otherwise
    // LPC > RETURN ADDRESS MX
    // PC+1 > RETURN ADDRESS MX
    // PC > RETURN ADDRESS MX	
  
    // DATA-PIPE EXEC
    // Collect results and place in transfer buffers
    // Select sources for SLOW DATA.
    // Trigger WRITES of fast data
    // Store Amemory
    if(AMem_Addr_Go == 1){
      Amemory[AMem_Addr_Sel] = Abus;
      if(AMem_Addr_Sel == 88 && cpu_stop_en != 0){
	logmsg("A88\n");
	cpu_die_rq=1;
      }
      /*
      if(AMem_Addr_Sel < 0100){
	Mmemory[AMem_Addr_Sel] = Abus;
      }
      */
      AMem_Addr_Go = 0;
    }
    // and MMemory
    if(MMem_Addr_Go == 1){
      Mmemory[MMem_Addr_Sel] = Mbus;
      Amemory[MMem_Addr_Sel] = Mbus;
      MMem_Addr_Go = 0;
    }  

    /*
    if(pcss_jdst == 18 && pc_source_select == 1){
      sprintf(msg,"[CPU] JUMP TO SELFTEST-ERROR FROM %ld\n",loc_ctr_reg); logmsg(msg);	
      cpu_die_rq=1;
    }
    */
    /*
    if(pcss_jdst == 1215 && pc_source_select == 1){
      sprintf(msg,"[CPU] JUMP TO MICROLOAD-ERROR FROM %ld\n",loc_ctr_reg); logmsg(msg);	
      cpu_die_rq=1;
    }
    */

    //    if(Mmemory[56] != tracepoint_data && loc_ctr_reg > 600){
    /*
    if(Mmemory[56] > 245 && tracepoint_data < 246 && loc_ctr_reg > 600){
      // TEST FAILED?
      tracepoint_data = Mmemory[56];
      sprintf(msg,"[CPU] U-Code checkpoint:(0x%lX) @ %ld\n",tracepoint_data,loc_ctr_reg); logmsg(msg);
      cpu_die_rq=1;
    }   
    */

    // Thwack bus devices
    nupi_clock_pulse();
//    sib_clock_pulse();

    // Do a memory cycle
    if(Memory_Busy > 0){
      Memory_Busy--;
      nubus_io_pulse();
    }

    // Die on error
    /*
    if((MCregister&MCR_PROM_Disable) == 0 && loc_ctr_reg > 2048){
      sprintf(msg,"[CPU] loc_ctr_reg OUT OF BOUNDS FOR PROM!\n"); logmsg(msg);
      tracerq=1;
      cpu_die_rq=1;
      } */       
    if((MCregister&MCR_PROM_Disable) == 1 && loc_ctr_reg > 16384){
      sprintf(msg,"[CPU] loc_ctr_reg OUT OF BOUNDS FOR WCS!\n"); logmsg(msg);
      tracerq=1;
      cpu_die_rq=1;
    }        

    // Die if requested
    if(cpu_die_rq){
      cpu_die=1;
      logmsg("[CPU] MASTER CLOCK STOPPED\n");
      raven_dispupdate();
      disassemble_IR();
    }
  }
  while(cpu_die != 0){
    sleep(99);
  }
}
