/* System Interface Board - Implementation

 This file is subject to the terms and conditions of the GNU General Public
 License.  See the file COPYING in the main directory of this archive for
 more details.

*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <linux/fb.h>
#include <linux/vt.h>
#include <linux/kd.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <strings.h>
#include <signal.h>
#include <termios.h>
#include <time.h>
#include <pthread.h>
#include <sched.h>
#include <sys/resource.h>

#include "meroko.h"
#include "nubus.h"

pthread_t clk_thread_id;

// REGISTERS
unsigned int keyboard_usart_status=0;
unsigned int keyboard_usart_command=0;
unsigned int keyboard_usart_control=0;
unsigned int keyboard_usart_mode=0;
unsigned char keyboard_usart_tx_data=0;
unsigned char keyboard_usart_rx_fifo[25];
unsigned char keyboard_usart_rx_bot=0;
unsigned char keyboard_usart_rx_top=0;

unsigned int sib_config_reg=0;
unsigned int sib_rtc_int_control=0;
unsigned int sib_rtc_int_status=0;
unsigned char sib_sound_control=0;
unsigned int sib_int_diag_reg=0;
unsigned int sib_int_diag_data=0;
unsigned int sib_vox_data_reg=0;

unsigned long sib_mask_reg=0;
unsigned long sib_opn_reg=0;

unsigned char sib_mouse_motion_reg=0;
unsigned int sib_mouse_x_pos=0;
unsigned int sib_mouse_y_pos=0;

unsigned long sib_itimer_ctr_0=0;
unsigned int  sib_itimer_ctl_0=0;
unsigned int  sib_itimer_ena_0=0;
unsigned int  sib_itimer_lod_0=0;

unsigned long sib_itimer_ctr_1=0;
unsigned long sib_itimer_ped_1=0;
unsigned int  sib_itimer_ctl_1=0;
unsigned int  sib_itimer_ena_1=0;
unsigned int  sib_itimer_lod_1=0;
unsigned int  sib_itimer_lvl_1=0;
unsigned int  sib_itimer_edg_1=0;

unsigned long sib_itimer_ctr_2=0;
unsigned int  sib_itimer_ctl_2=0;
unsigned int  sib_itimer_ena_2=0;
unsigned int  sib_itimer_lod_2=0;

// THESE ARE BCD NUMBERS!
unsigned long sib_rtc_100ns=0; // 2 digits, lower digit unused. This means we are actually counting 1000ns!
unsigned long sib_rtc_ticks=0; // NOT A REAL REGISTER - HERE FOR MY CONVENIENCE
unsigned long sib_rtc_10ms=0;  // 2 digits, 100ms and then 10ms
unsigned long sib_rtc_sec=0;   // 2 digits for all the rest
unsigned long sib_rtc_min=0;
unsigned long sib_rtc_hour=0;
unsigned long sib_rtc_dow=0;
unsigned long sib_rtc_date=0;
unsigned long sib_rtc_month=0;
// THESE ARE BCD NUMBERS!
unsigned long sib_ram_100ns=0;
unsigned long sib_ram_10ms=0;
unsigned long sib_ram_sec=0;
unsigned long sib_ram_min=0;
unsigned long sib_ram_hour=0;
unsigned long sib_ram_dow=0;
unsigned long sib_ram_date=0;
unsigned long sib_ram_month=0;

unsigned int end_of_month[12] = { 0x31,0x28,0x31,0x30,0x31,0x30,0x31,0x31,0x30,0x31,0x30,0x31 };

long previous_nsec;
struct timespec current_time;

unsigned long sib_rtc_int_vector=0;
unsigned long sib_sintv_expired_vector=0;
unsigned long sib_lintv_expired_vector=0;
unsigned long sib_serio_status_vector=0;
unsigned long sib_lpt_ack_vector=0;
unsigned long sib_gcmd_ack_vector=0;
unsigned long sib_kbd_rtt_vector=0;
unsigned long sib_psu_overtemp_vector=0;
unsigned long sib_kbd_bootchord_vector=0;
unsigned long sib_mouse_moved_vector=0;
unsigned long sib_mouse_button_vector=0;
unsigned long sib_voxdata_present_vector=0;
unsigned long sib_sound_parity_vector=0;
unsigned long sib_fiber_link_vector=0;
unsigned long sib_power_fail1_vector=0;
unsigned long sib_power_fail2_vector=0;

unsigned char sib_video_attr=0;
unsigned char sib_crt_r00=0x2a;
unsigned char sib_crt_r01=0x1f;
unsigned char sib_crt_r02=0x07;
unsigned char sib_crt_r03=0x0a;
unsigned char sib_crt_r04=0x24;
unsigned char sib_crt_r05=0x25;
unsigned char sib_crt_r06=0x80;
unsigned char sib_crt_r07=0x64;
unsigned char sib_crt_r08=0x67;
unsigned char sib_crt_r09=0x4a;
unsigned char sib_crt_r0a=0;
unsigned char sib_crt_r0b=0;
unsigned char sib_crt_r0c=0;
unsigned char sib_crt_r0d=0x40;
unsigned char sib_crt_r0e=0;
unsigned char sib_crt_r0f=0;
unsigned char sib_crt_r10=0x80;
unsigned char sib_crt_r11=0x80;
unsigned char sib_crt_r12=0x80;
unsigned char sib_crt_r13=0;
unsigned char sib_crt_r14=0;
unsigned char sib_crt_r15=0;
unsigned char sib_crt_r16=0;
unsigned char sib_crt_r17=0;
unsigned char sib_crt_r18=0;
unsigned char sib_crt_r19=0;
unsigned char sib_crt_r1a=0;

unsigned char sib_lpt_r00=0;

// Software
char msg[256];
unsigned char SIB_ROM[8192];
unsigned char NVRAM[2048];
unsigned char VRAM[0x20000]; /* 8K * 16, for 1024x1024 px */

int sib_log_arm = 0;

#define PROM_FNAME "proms/2236662_SIB"

// FRAMEBUFFER CONTROLS
int fbfd=0,consfd=0;
struct fb_var_screeninfo vinfo;
struct fb_fix_screeninfo finfo;
char *framebuffer = 0;
unsigned long int screensize=0;
long iodata=0;

void sib_clock_pulse();

// Externals
extern int cpu_die_rq;
extern unsigned int ldb(unsigned long long value, int size, int position);
extern unsigned long nubus_io_request(int access, unsigned long address, unsigned long data);
extern void logmsg(char *msg);

// RTC UPDATION
unsigned long nsec_passed = 0;
unsigned long msec_passed = 0;
long diff_ns=0;

inline void cascade_time(){
  /*
  // RTC 100NS
  if((sib_rtc_100ns&0xF) > 0x9){ // 900 ns
    sib_rtc_100ns += 0x10;
    sib_rtc_100ns &= 0xF0;
  }else{return;}
  if((sib_rtc_100ns&0xF0) > 0x90){ // 9000 ns
    sib_rtc_100ns &= 0xF;
    sib_rtc_ticks++; // 10000 nanoseconds have passed.
  }else{return;}
  */

  // INTERNAL TICKS
  if(sib_rtc_ticks > 10000000){
    sib_rtc_ticks = 0; // Reset
    //    sprintf(msg,"Over by %ld\n",sib_rtc_ticks); logmsg(msg);
    sib_rtc_10ms++; // Tick 10ms
  }else{return;}
  // RTC 10MS
  if((sib_rtc_10ms&0xF) > 9){
    sib_rtc_10ms += 0x10; // Carry over
    sib_rtc_10ms &= 0xF0; // Reset to 0
  }else{return;}
  // RTC 100MS
  if((sib_rtc_10ms&0xF0) > 0x90){
    sib_rtc_sec++;       // Tick 1sec
    sib_rtc_10ms &= 0x0F; // Reset to 0
  }else{return;}
  // RTC 1SEC

  if((sib_rtc_sec&0xF) > 9){
    sib_rtc_sec += 0x10;
    sib_rtc_sec &= 0xF0;
  }else{return;}
  // RTC 10SEC
  if((sib_rtc_sec&0xF0) > 0x50){
    sib_rtc_min++;
    sib_rtc_sec &= 0xF;
  }else{return;}
  // RTC 1MIN
  if((sib_rtc_min&0xF) > 9){
    sib_rtc_min += 0x10;
    sib_rtc_min &= 0xF0;
  }else{return;}
  // RTC 10MIN
  if((sib_rtc_min&0xF0) > 0x90){
    sib_rtc_hour++;
    sib_rtc_min &= 0x0F;
  }else{return;}
  // RTC 1HOUR
  if((sib_rtc_hour&0xF) > 9){
    sib_rtc_hour += 0x10;
    sib_rtc_hour &= 0xF0;
  }
  // RTC 24HOUR CHECK
  if(sib_rtc_hour > 0x24){
    sib_rtc_dow++;
    sib_rtc_date++;
    sib_rtc_hour=0;
  }else{return;}
  // RTC DOW
  if(sib_rtc_dow > 7){
    sib_rtc_dow = 0;
  }
  // RTC 1DAY
  if((sib_rtc_date&0xF) > 9){
    sib_rtc_date += 0x10;
    sib_rtc_date &= 0xF0;
  }
  // RTC END-OF-MONTH
  {
    int month = (sib_rtc_month&0xF)+(10*((sib_rtc_month&0xF0)>>4));
    if(sib_rtc_date > end_of_month[month]){
      sib_rtc_date = 0; // Clear
      sib_rtc_month++;
    }else{return;}
  }
  // RTC 1MONTH
  if((sib_rtc_month&0xF) > 9){
    sib_rtc_month += 0x10;
  }
  // RTC 12MONTH
  if(sib_rtc_month > 0x12){
    sib_rtc_month = 0;
  }
}

void updatetime(int arg){
  struct timespec delay;
  int timer2_ticks;
  unsigned long temp;

  while(1){
    delay.tv_sec = 0;
    delay.tv_nsec = 0;
    nanosleep(&delay,NULL);
  
    // Get time (This is very time-expensive...)
    clock_gettime(CLOCK_REALTIME,&current_time);

    // Figure difference
    if(previous_nsec > current_time.tv_nsec){
      // Time has rolled over
      // sprintf(msg,"TIMER ROLLED NEGATIVE! (%ld vs %ld)\n",current_time.tv_nsec,previous_nsec); logmsg(msg);
      previous_nsec = 0 - previous_nsec;
    }
    diff_ns = (current_time.tv_nsec - previous_nsec);
    previous_nsec = current_time.tv_nsec;

    // diff_ns 
    //    nsec_passed += diff_ns;
  
    // Update 100ns counter
    sib_rtc_100ns = ((diff_ns/100)&0xF)|(((diff_ns/100)&0xF0)<<4);
    // and the internal timer, which does the real work
    sib_rtc_ticks += diff_ns;
    msec_passed += diff_ns; 

    // sprintf(msg,"nsec_passed = %ld\n",sib_rtc_ticks); logmsg(msg);  
    
    // Let time cascade thru the clock registers
    cascade_time();

      //      sprintf(msg,"tick = %ld\n",temp); logmsg(msg);  
      //temp -= 100;
    
    // Test for RTC interrupt
    // Interrupt test.
    if(sib_rtc_int_control&0x01){ // COMPARE armed
      if(sib_rtc_month == sib_ram_month &&
	 sib_rtc_date == sib_ram_date &&
	 sib_rtc_dow == sib_ram_dow &&
	 sib_rtc_hour == sib_ram_hour &&
	 sib_rtc_min == sib_ram_min &&
	 sib_rtc_sec == sib_ram_sec &&
	 sib_rtc_10ms == sib_ram_10ms){
	
	logmsg("SIB: TIMER MATCH INTERRUPT\n");      
	sib_rtc_int_control ^= 0x01; // Disable future interrupt
	sib_rtc_int_status |= 0x01; // Tell CPU what happened
	sib_rtc_100ns = sib_ram_100ns; // Equalize
	sprintf(msg,"RTC Interrupt Vector = 0x%lX\n",sib_rtc_int_vector); logmsg(msg);
	// Write 0xFF to this location if interrupts are enabled
	if(sib_config_reg&0x02){
	  nubus_io_request(NB_BYTE_WRITE,sib_rtc_int_vector,0xFF);
	}
	// Die
	// cpu_die_rq=1;
      }
    }

    // Update msec
    if(msec_passed > 1000000){
      msec_passed /= 1000000; // Turn into microseconds

      // Interval timer 0 runs at 1MHz, or 1 microsecond.
      if(sib_itimer_ena_0){
	if(msec_passed > sib_itimer_ctr_0){
	  sib_itimer_ctr_0 = 0;
	  logmsg("SIB: Interval timer 0 expired\n");
	  sib_itimer_ena_0 = 0;
	}else{
	  sib_itimer_ctr_0 -= msec_passed;
	}
      }
  
      // Interval timer 1 runs at 1MHz as well, but in either mode 0 or 3.
      // In mode 0, it interrupts when it recaches 0.
      // In mode 3, it drives interval timer 2.
      // If loaded even, it is high for half it's period, and low for the other half.
      // If loaded odd, it is high for (N+1)/2 pulses and low for (N-1)/2 pulses.
      // When it expires, it returns to it's loaded value.
      // I guess that means that when it cycles low, timer 2 will decrement (if enabled)
  
      timer2_ticks=0;  
      if(sib_itimer_ena_1){
	if(msec_passed > sib_itimer_ctr_1){
	  //       sprintf(msg,"SIB: Interval timer 1 expired (%ld vs %ld)\n",msec_passed,sib_itimer_ctr_1); logmsg(msg);
	  if(sib_itimer_ctl_1&0xE){
	    timer2_ticks += (msec_passed/sib_itimer_ped_1); 
	    sib_itimer_ctr_1 = sib_itimer_ped_1; // Reset for square-wave mode
	    // Transitions happened, too.
	    sib_itimer_edg_1 = 1;
	    sib_itimer_lvl_1 = sib_itimer_edg_1;
	  }else{
	    sib_itimer_ctr_1 = 0; // Reset for terminal-count mode
	    sib_itimer_ena_1 = 0;
	    timer2_ticks++;
	  }
	}else{
	  sib_itimer_ctr_1 -= msec_passed;
	  if(sib_itimer_ctl_1&0xE){
	    // Determine signal level
	    if(sib_itimer_ctr_1 < (sib_itimer_ped_1/2)){
	      // Rise half - Logic 1
	      sib_itimer_edg_1 = 1;
	    }else{
	      // Fall half - Logic 0
	      sib_itimer_edg_1 = 0;
	    }
	    if(sib_itimer_edg_1 != sib_itimer_lvl_1){
	      // PULSE TIMER 2
	      timer2_ticks++;
	    }
	    sib_itimer_lvl_1 = sib_itimer_edg_1;
	  }
	}      
      }
  
      if(sib_itimer_ena_2){
	if(timer2_ticks > sib_itimer_ctr_2){
	  sib_itimer_ctr_2 = 0;
	  logmsg("SIB: Interval timer 2 expired\n");
	  sib_itimer_ena_2 = 0;
	  // Write 0xFF to this location if interrupts are enabled
	  if(sib_config_reg&0x2){
	    nubus_io_request(NB_BYTE_WRITE,sib_lintv_expired_vector,0xFF);
	  }
	}else{
	  sib_itimer_ctr_2 -= timer2_ticks;
	}
      }
    }
  }  
  // Done, return.
}

void usrhandler(int arg){
  signal(SIGUSR1,usrhandler);
}

void kbdhandler(int arg){
  unsigned char iochar[3];
  unsigned char outchar=0;
  int done=0;

  // Recapture signal
  signal(SIGIO,kbdhandler);
  // Get keycode
  while(done==0){
    outchar=0;
    if(read(consfd,&iochar[0],1)==1){
      // Remap and handle
      switch(iochar[0]){	
      case 0x02: // 1
      case 0x82:
      case 0x03: // 2
      case 0x83:
      case 0x04: // 3
      case 0x84:
      case 0x05: // 4
      case 0x85:
      case 0x06: // 5
      case 0x86:
      case 0x07: // 6
      case 0x87:
      case 0x08: // 7
      case 0x88:
      case 0x09: // 8
      case 0x89:
      case 0x0A: // 9
      case 0x8A:
      case 0x0B: // 0
      case 0x8B:
      case 0x0C: // -
      case 0x8C:
      case 0x0D: // +
      case 0x8D:
	outchar = iochar[0]+0x22; break;

      case 0x10: // Q
      case 0x90:
      case 0x11: // W
      case 0x91:
      case 0x12: // E
      case 0x92:
      case 0x13: // R
      case 0x93:
      case 0x14: // T
      case 0x94:
      case 0x15: // Y
      case 0x95:
      case 0x16: // U
      case 0x96:
      case 0x17: // I
      case 0x97:
      case 0x18: // O
      case 0x98:
      case 0x19: // P
      case 0x99:
	outchar = iochar[0]+0x29; break;
	
      case 0x1C: // ENTER
	outchar = 0x5B; break;
      case 0x9C:
	outchar = 0xDB; break;
	
      case 0x1E: // A
      case 0x9E:
      case 0x1F: // S
      case 0x9F:
      case 0x20: // D
      case 0xA0:
      case 0x21: // F
      case 0xA1:
      case 0x22: // G
      case 0xA2: 
      case 0x23: // H
      case 0xA3:
      case 0x24: // J
      case 0xA4:
      case 0x25: // K
      case 0xA5:
      case 0x26: // L
      case 0xA6:
	outchar = iochar[0]+0x32; break;
	
      case 0x2C: // Z
      case 0xAC:
      case 0x2D: // X
      case 0xAD:
      case 0x2E: // C
      case 0xAE:
      case 0x2F: // V
      case 0xAF:
      case 0x30: // B
      case 0xB0:
      case 0x31: // N
      case 0xB1:
      case 0x32: // M
      case 0xB2:
	outchar = iochar[0]+0x3C; break;
	
      default:
	sprintf(msg,"Got unmapped keycode 0x%X\n",iochar[0]); logmsg(msg);
      }
      if(outchar != 0 && (keyboard_usart_command&0x04) != 0){
	// If reciever's running, take the key
	keyboard_usart_rx_fifo[keyboard_usart_rx_top] = outchar;
	keyboard_usart_rx_top++;
	if(keyboard_usart_rx_top > 25){ keyboard_usart_rx_top = 0; }
      }else{
	logmsg("Keystroke lost - Unknown key or reciever not operating\n");
      }
      iochar[0]=0;
    }else{
      done=1;
    }
  }
}

void sib_init(){
  FILE *romfile;
  int x=0;
  romfile = fopen(PROM_FNAME,"r");
  if(romfile == NULL){
    perror("sib-fopen");
    exit(-1);
  }
  while(x < 8192){
    fread(&SIB_ROM[x],1,1,romfile);
    x++;
  }
  fclose(romfile);
  /* REGISTER RESETS */
  keyboard_usart_status = 0x80; // DSR set, transmit and recieve disabled
  keyboard_usart_tx_data = 0;  
  keyboard_usart_rx_fifo[0]=0;
  keyboard_usart_rx_bot=0;
  keyboard_usart_rx_top=0;
  /* CONSOLE SETUP */
  // Arrange for SIGIO on IO availability
  signal(SIGIO,kbdhandler);
  signal(SIGUSR1,usrhandler);

  // Become a realtime process (Well, as close as timesharing Linux can get...)
  if(setpriority(PRIO_PROCESS,getpid(),-20)<0){
    perror("setpriority");
    exit(-1);
  }

  // Check RTC resolution
  clock_getres(CLOCK_REALTIME,&current_time);
  printf("CLOCK_REALTIME has %ld second, %ld nsec resolution.\n",current_time.tv_sec,current_time.tv_nsec);
  previous_nsec = 0;

  // Kick RTC thread
  pthread_create(&clk_thread_id,NULL,(void *)updatetime,NULL);
  //  sleep(30);

  // Open console and keyboard FD
  consfd = open("/dev/tty0",O_RDWR|O_NONBLOCK);
  if(!consfd){
    perror("console-open");
    exit(-1);
  }
  
  // Disassociate from process group
  setpgrp();
  
  // Detach CTY
  {
    int fd=0;
    fd=open("/dev/tty",O_RDWR);
    if(ioctl(fd,TIOCNOTTY,0)){
      perror("ioctl(TIOCNOTTY)");
      exit(-1);
    }
    close(fd);
  }

  {
    struct vt_mode vtmode;
    vtmode.mode = VT_PROCESS;
    vtmode.relsig = SIGUSR1;
    vtmode.acqsig = SIGUSR1;

    if(ioctl(consfd,VT_SETMODE,&vtmode)){
      perror("ioctl(VT_SETMODE)");
      exit(-1);
    }     
  }

  // Make TTY7 the active console
  if(ioctl(consfd,VT_ACTIVATE,7)){
    perror("ioctl(VT_ACTIVATE)");
    exit(-1);
  } 

  // Wait for it...
  if(ioctl(consfd,VT_WAITACTIVE,7)){
    perror("ioctl(VT_WAITACTIVE)");
    exit(-1);
  } 


  if(ioctl(consfd,KDSETMODE,KD_GRAPHICS)){
    perror("ioctl(VT_ACTIVATE)");
    exit(-1);
    } 

  /* Enable this later
  if(ioctl(consfd,VT_LOCKSWITCH,1)){
    perror("ioctl(VT_LOCKSWITCH)");
    exit(-1);
  }
  */

  /* FRAMEBUFFER SETUP */
  fbfd = open("/dev/fb0",O_RDWR);
  if(!fbfd){
    perror("fb-open");
    exit(-1);
  }
  if(ioctl(fbfd,FBIOGET_FSCREENINFO,&finfo)){
    perror("finfo-ioctl");
    exit(-1);
  }
  if(ioctl(fbfd,FBIOGET_VSCREENINFO,&vinfo)){
    perror("vinfo-ioctl");
    exit(-1);
  }
  screensize=vinfo.xres * vinfo.yres * vinfo.bits_per_pixel / 8;
  framebuffer = (char *)mmap(0,screensize,PROT_READ|PROT_WRITE,MAP_SHARED,fbfd,0);
  if((int)framebuffer == -0){
    perror("mmap");
    exit(-1);
  }
  // Clear screen and VRAM
  bzero(framebuffer,screensize);
  bzero(VRAM,0x20000);

  // Get keyboard mode
  if(ioctl(consfd,KDGKBMODE,&iodata)){
    perror("ioctl(KDGKBMODE)");
    exit(-1);
  }

  // Request keyboard-raw mode
  if(ioctl(consfd,KDSKBMODE,K_RAW)){
    perror("ioctl(KDSKBMODE)");
    exit(-1);
  }

  {
    struct termios tio;
    tio.c_iflag = (IGNPAR|IGNBRK)&(~PARMRK)&(~ISTRIP);
    tio.c_oflag = 0;
    tio.c_cflag = CREAD|CS8;
    tio.c_lflag = 0;
    tio.c_cc[VTIME]=0;
    tio.c_cc[VMIN]=1;
    cfsetispeed(&tio,9600);
    cfsetospeed(&tio,9600);
    tcsetattr(consfd,TCSANOW,&tio);    
  }

  // Set signal process owner
  if(fcntl(consfd,F_SETOWN,getpid()) < 0){
    perror("fcntl(F_SETOWN)");
    exit(-1);
  }

  // Get keyboard messages
  if(fcntl(consfd,F_SETFL,O_NONBLOCK|O_ASYNC) < 0){
    perror("fcntl(F_SETFL)");
    exit(-1);
  }
}

void redraw_display(){
  // Translate to 8bpp and post result
  unsigned long FBaddr=0;
  unsigned long long x=1;
  unsigned int row,col;
  unsigned long VRAMaddr=0;
  
  while(VRAMaddr < 0x20000){
    col = VRAMaddr*8;
    row = (col/1024);
    col -= (row*1024); // Remove excess
    x=1;
    FBaddr = (row*1280)+col;
    // Invert video if required.
    // This is structured like this to cut down on bit tests required in one pass.
    if((sib_video_attr&0x02) == 0x02){
	// Reverse Video
      while(x < 0x100000000LL){
	if((VRAM[VRAMaddr]&x)==x){
	  framebuffer[FBaddr]=0x00; // ON
	}else{
	  framebuffer[FBaddr]=0x0F; // OFF
	}
	x = x << 1;
	FBaddr++; // Next pixel
      }
    }else{
      // Normal Video
      while(x < 0x100000000LL){
	if((VRAM[VRAMaddr]&x)==x){
	  framebuffer[FBaddr]=0x0F; // ON
	}else{
	  framebuffer[FBaddr]=0x00; // OFF
	}
	x = x << 1;
	FBaddr++; // Next pixel
      }
    }
    //      FBaddr = row*col;
    FBaddr = (row*1280)+col;
    VRAMaddr++;
  }
  // Redraw the whole screen
  msync(framebuffer,screensize,MS_SYNC);
}

void sib_nubus_io(){
  unsigned long NUbus_Addr = ldb(NUbus_Address,24,0);  

  if(sib_log_arm > 0){
    sprintf(msg,"SIB: Log: Op %ld for address %lX\n",NUbus_Request,NUbus_Addr); logmsg(msg); 
    sib_log_arm--;
  }

  // Exclude ROM here
  if(NUbus_Request == VM_READ && NUbus_Addr > 0xFF8000){
    unsigned int ROMaddr = (NUbus_Addr&0x1FFF);

    /* Word 0 = byte 0
       Word 1 = byte 4
       Word 2 = byte 8
    */
    // Construct word
    NUbus_Data = SIB_ROM[ROMaddr]; NUbus_Data &= 0xFF;
    /* NUbus_Data = NUbus_Data << 8;
    NUbus_Data |= SIB_ROM[ROMaddr+1];
    NUbus_Data = NUbus_Data << 8;
    NUbus_Data |= SIB_ROM[ROMaddr+2];
    NUbus_Data = NUbus_Data << 8;
    NUbus_Data |= SIB_ROM[ROMaddr+3]; */
    if(NUbus_Addr&0x03){
      sprintf(msg,"SIB: Word-Read %lX for odd address %lX (ROM address %X)\n",NUbus_Data,NUbus_Addr,ROMaddr); logmsg(msg);
      cpu_die_rq=1;
    }
    NUbus_acknowledge=1;
    return;
  }   
  
  if(NUbus_Request == VM_BYTE_READ && NUbus_Addr >= 0xFF8000){         
    unsigned int ROMaddr = ((NUbus_Addr >> 2)&0x1FFF);

    if((NUbus_Addr&0x3) != 0){ logmsg("SIB: ODD ROM ADDR\n"); }
    if(ROMaddr > 0x1FFF){ logmsg("SIB: TOOBIG ROM ADDR\n"); }

    NUbus_Data = SIB_ROM[ROMaddr];
    //    sprintf(msg,"SIB: Byte-Read %lX for address %lX (ROM address %X)\n",NUbus_Data,NUbus_Addr,ROMaddr); logmsg(msg);
    NUbus_acknowledge=1;
    // cpu_die_rq=1;
    return;
  }  

  // NVRAM
  if(NUbus_Request == VM_BYTE_READ && (NUbus_Addr >= 0xFA0000 && NUbus_Addr <= 0xFA1FFF)){
    // NVRAM
    unsigned int NVRAMaddr = ((NUbus_Addr >> 2)&0x7FF);    
    if((NUbus_Addr&0x3) != 0){ logmsg("SIB: ODD NVRAM ADDR\n"); }
    NUbus_Data = NVRAM[NVRAMaddr];
    NUbus_acknowledge=1;
    return;
  }
  if(NUbus_Request == VM_BYTE_WRITE && (NUbus_Addr >= 0xFA0000 && NUbus_Addr <= 0xFA1FFF)){
    // NVRAM
    unsigned int NVRAMaddr = ((NUbus_Addr >> 2)&0x7FF);    
    if((NUbus_Addr&0x3) != 0){ logmsg("SIB: ODD NVRAM ADDR\n"); }

    NVRAM[NVRAMaddr] = NUbus_Data ;
    NUbus_acknowledge=1;
    return;
  }

  // VRAM
  if(NUbus_Request == VM_READ && (NUbus_Addr >= 0xE80000 && NUbus_Addr <= 0xE9FFFF)){
    unsigned int VRAMaddr = (NUbus_Addr&0x1FFFF);

    switch(NUbus_Addr&0x03){
    case 0: // WORD IO
      NUbus_Data  = VRAM[VRAMaddr+3]; NUbus_Data <<= 8;
      NUbus_Data |= VRAM[VRAMaddr+2]; NUbus_Data <<= 8;
      NUbus_Data |= VRAM[VRAMaddr+1]; NUbus_Data <<= 8;
      NUbus_Data |= VRAM[VRAMaddr+0];
      break;

    case 1: // LOW HALFWORD READ
      NUbus_Data =  VRAM[VRAMaddr]; NUbus_Data <<= 8;
      NUbus_Data |= VRAM[VRAMaddr-1];
      break;

    case 3: // HIGH HALFWORD READ
      NUbus_Data =  VRAM[VRAMaddr]; NUbus_Data <<= 8;
      NUbus_Data |= VRAM[VRAMaddr-1];
      NUbus_Data <<= 16;
      break;

    default:
      logmsg("BAD NUPI WORD-READ TYPE\n");
      cpu_die_rq=1;
    }
    NUbus_acknowledge=1;
    return;
  }
  if(NUbus_Request == VM_BYTE_READ && (NUbus_Addr >= 0xE80000 && NUbus_Addr <= 0xE9FFFF)){
    unsigned int VRAMaddr = (NUbus_Addr&0x1FFFF);

    NUbus_Data = VRAM[VRAMaddr];
    NUbus_Data <<= (8*(NUbus_Addr&0x03));
    NUbus_acknowledge=1;
    return;
  }
  if(NUbus_Request == VM_WRITE && (NUbus_Addr >= 0xEC0000 && NUbus_Addr <= 0xEDFFFF) && (NUbus_Addr&0x03)==0){
    // Read-Modify-Write version of VRAM
    unsigned int VRAMaddr = (NUbus_Addr&0x1FFFF);
    unsigned long VRAMdata = NUbus_Data;
    unsigned long VRAMrslt=0;
    unsigned long VSRCdata=0;

    VSRCdata  = VRAM[VRAMaddr+3]; VSRCdata <<= 8;
    VSRCdata |= VRAM[VRAMaddr+2]; VSRCdata <<= 8;
    VSRCdata |= VRAM[VRAMaddr+1]; VSRCdata <<= 8;
    VSRCdata |= VRAM[VRAMaddr+0];

    // D means framebuffer-data
    // S means memory-bus-data

    // Modify VRAMdata according to the logical operation register
    switch(sib_opn_reg){
    case 0: // CLEAR
      VRAMrslt = 0; break;
    case 1: // D NOR S
      {
	unsigned long tmp=0;
	unsigned long x=0x80000000;
	while(x >= 1){ if((VSRCdata&x)==0 && (VRAMdata&x)==0){ tmp |= x; } x >>= 1; } VRAMrslt = tmp;
      }
      break;
    case 2: // S AND D-
      VRAMrslt = (VRAMdata&(0xFFFFFFFF^VSRCdata)); break;
    case 3: // D-
      VRAMrslt = (0xFFFFFFFF^VSRCdata); break;
    case 4:  // S- AND D
      VRAMrslt = ((VRAMdata^0xFFFFFFFF)&VSRCdata); break;
    case 5:  // S-
      VRAMrslt = (VRAMdata^0xFFFFFFFF); break;
    case 6:  // D XOR S
      VRAMrslt = (VSRCdata^VRAMdata); break;
    case 7:  // D NAND S
      {
	unsigned long tmp=0;
	unsigned long x=0x80000000;
	while(x > 0){ if(!((VSRCdata&x)==x && (VRAMdata&x)==x)){ tmp |= x; } x >>= 1; } VRAMrslt = tmp; }
      //	sprintf(msg,"SIB: 0x%lX NAND 0x%lX = 0x%lX\n",VSRCdata,VRAMdata,VRAMrslt); logmsg(msg);
      break;
    case 8:  // D AND S
      VRAMrslt = (VSRCdata&VRAMdata); break;
    case 9:  // D XNOR S
      {
	unsigned long tmp=0;
	unsigned long x=0x80000000;
	while(x > 0){ if((VSRCdata&x) == (VRAMdata&x)){ tmp |= x; } x >>= 1; } VRAMrslt = tmp; }
      //	sprintf(msg,"SIB: 0x%lX XNOR 0x%lX = 0x%lX\n",VSRCdata,VRAMdata,VRAMrslt); logmsg(msg);
      break;
    case 10: // NOP(S)
      VRAMrslt = VRAMdata; break;
    case 11: // S OR D-
      VRAMrslt = (VRAMdata|(VSRCdata^0xFFFFFFFF)); break;
    case 12: // D OR S
      VRAMrslt = (VSRCdata|VRAMdata); break;
    case 13: // SET
      VRAMrslt = 0xFFFFFFFF; break;

    default:
      sprintf(msg,"SIB: Unknown logical operation %lX\n",sib_opn_reg); logmsg(msg);
      cpu_die_rq=1;
    }

    // Ones in the mask means bits in that position (IN MEMORY) carry over to the new data.
    // First, turn off mask ones in the new data
    VRAMdata = VRAMrslt&(~sib_mask_reg);
    // Now load ones from the source
    VRAMdata |= (VSRCdata&sib_mask_reg);

    //    sprintf(msg,"SIB: 0x%lX mask = 0x%lX final-data\n",sib_mask_reg,VRAMdata); logmsg(msg);

    VRAM[VRAMaddr+0] = VRAMdata&0xFF; 
    VRAMdata = VRAMdata >> 8;
    VRAM[VRAMaddr+1] = VRAMdata&0xFF; 
    VRAMdata = VRAMdata >> 8;
    VRAM[VRAMaddr+2] = VRAMdata&0xFF; 
    VRAMdata = VRAMdata >> 8;
    VRAM[VRAMaddr+3] = VRAMdata&0xFF;

    NUbus_acknowledge=1;
    if(sib_video_attr&0x01){ // Blank Video Output
      return;
    }

    // Translate to 8bpp and post result
    {
      unsigned long FBaddr=0;
      unsigned long long x=1;
      unsigned int row,col;

      col = VRAMaddr*8;
      row = (col/1024);
      col -= (row*1024); // Remove excess

      //      FBaddr = row*col;
      FBaddr = (row*1280)+col;
      // Invert video if required.
      // This is structured like this to cut down on bit tests required in one pass.
      if((sib_video_attr&0x02) == 0x02){
	// Reverse Video
	while(x < 0x100000000LL){
	  if((NUbus_Data&x)==x){
	    framebuffer[FBaddr]=0x00; // ON
	  }else{
	    framebuffer[FBaddr]=0x0F; // OFF
	  }
	x = x << 1;
	FBaddr++; // Next pixel
	}
      }else{
	// Normal Video
	while(x < 0x100000000LL){
	  if((NUbus_Data&x)==x){
	    framebuffer[FBaddr]=0x0F; // ON
	  }else{
	    framebuffer[FBaddr]=0x00; // OFF
	  }
	  x = x << 1;
	  FBaddr++; // Next pixel
	}
      }
      //      FBaddr = row*col;
      FBaddr = (row*1280)+col;
      msync(framebuffer+FBaddr,256,MS_SYNC);
    }
    // sprintf(msg,"SIB: Word-Write %lX for address %lX (VRAM address %X)\n",NUbus_Data,NUbus_Addr,VRAMaddr); logmsg(msg);
    //    cpu_die_rq=1;
    return;
  }
  if(NUbus_Request == VM_WRITE && (NUbus_Addr >= 0xE80000 && NUbus_Addr <= 0xE9FFFF)){
    unsigned int VRAMaddr = (NUbus_Addr&0x1FFFF);
    unsigned long VRAMdata = NUbus_Data;

    switch(NUbus_Addr&0x03){
    case 0: // WORD IO
      VRAM[VRAMaddr+0] = VRAMdata&0xFF; 
      VRAMdata = VRAMdata >> 8;
      VRAM[VRAMaddr+1] = VRAMdata&0xFF; 
      VRAMdata = VRAMdata >> 8;
      VRAM[VRAMaddr+2] = VRAMdata&0xFF; 
      VRAMdata = VRAMdata >> 8;
      VRAM[VRAMaddr+3] = VRAMdata&0xFF;
      break;

    case 3: // High Halfword Write
      VRAMdata >>= 16;
    case 1: // Low Halfword Write
      VRAM[VRAMaddr-1] = VRAMdata&0xFF;
      VRAMdata >>= 8;
      VRAM[VRAMaddr] = VRAMdata&0xFF;
      break;
      
    default:
      logmsg("BAD NUPI WORD-WRITE TYPE\n");
      cpu_die_rq=1;
    }

    NUbus_acknowledge=1;
    if(sib_video_attr&0x01){ // Blank Video Output
      return;
    }

    // Translate to 8bpp and post result
    {
      unsigned long FBaddr=0;
      unsigned long long x=1;
      unsigned int row,col;

      col = VRAMaddr*8;
      row = (col/1024);
      col -= (row*1024); // Remove excess

      //      FBaddr = row*col;
      FBaddr = (row*1280)+col;
      // Invert video if required.
      // This is structured like this to cut down on bit tests required in one pass.
      if((sib_video_attr&0x02) == 0x02){
	// Reverse Video
	while(x < 0x100000000LL){
	  if((NUbus_Data&x)==x){
	    framebuffer[FBaddr]=0x00; // ON
	  }else{
	    framebuffer[FBaddr]=0x0F; // OFF
	  }
	x = x << 1;
	FBaddr++; // Next pixel
	}
      }else{
	// Normal Video
	while(x < 0x100000000LL){
	  if((NUbus_Data&x)==x){
	    framebuffer[FBaddr]=0x0F; // ON
	  }else{
	    framebuffer[FBaddr]=0x00; // OFF
	  }
	  x = x << 1;
	  FBaddr++; // Next pixel
	}
      }
      //      FBaddr = row*col;
      FBaddr = (row*1280)+col;
      msync(framebuffer+FBaddr,256,MS_SYNC);
    }
    // sprintf(msg,"SIB: Word-Write %lX for address %lX (VRAM address %X)\n",NUbus_Data,NUbus_Addr,VRAMaddr); logmsg(msg);
    //    cpu_die_rq=1;
    return;
  }

  if(NUbus_Request == VM_BYTE_WRITE && (NUbus_Addr >= 0xE80000 && NUbus_Addr <= 0xE9FFFF)){
    // sib_log_arm = 1;
    unsigned int VRAMaddr = (NUbus_Addr&0x1FFFF);    
    unsigned long VRAMdata=0;

    // Derotate
    VRAMdata = ((NUbus_Data>>(8*(NUbus_Addr&0x3)))&0xFF);
    VRAM[VRAMaddr] = VRAMdata;
    
    // Translate to 8bpp and post result
    {
      unsigned long FBaddr=0;
      int x=1;
      unsigned int row,col;

      col = VRAMaddr*8;
      row = (col/1024);
      col -= (row*1024); // Remove excess

      if(col > 1024){
	cpu_die_rq=1;
      }
      
      //      sprintf(msg,"VRAMaddr %d, row %d col %d\n",VRAMaddr,row,col); logmsg(msg);

      if(sib_video_attr&0x01){ // Blank Video Output
	return;
      }

      FBaddr = (row*1280)+col;
      
      while(x<0x100){
	if((sib_video_attr&0x02) == 0x02){
	  if((VRAMdata&x)==x){
	    framebuffer[FBaddr]=0x00; // ON
	  }else{
	    framebuffer[FBaddr]=0x0F; // OFF
	  }
	  x = x << 1;
	  FBaddr++; // Next pixel
	}else{
	  if((VRAMdata&x)==x){
	    framebuffer[FBaddr]=0x0F; // ON
	  }else{
	    framebuffer[FBaddr]=0x00; // OFF
	  }
	  x = x << 1;
	  FBaddr++; // Next pixel
	}
      }
      //      FBaddr = VRAMaddr*8;
      FBaddr = (row*1280)+col;
      msync(framebuffer+FBaddr,64,MS_SYNC);
    }
    NUbus_acknowledge=1;
    //    sprintf(msg,"SIB: Byte-Write %lX for address %lX (VRAM address %X)\n",NUbus_Data,NUbus_Addr,VRAMaddr); logmsg(msg);
    // cpu_die_rq=1;
    // sib_log_arm = 14;
    return;
  }  

  // Registers
  if(NUbus_Request == VM_READ){
    switch(NUbus_Addr){
    case 0xE00084: // Mask Register
      NUbus_Data = sib_mask_reg;
      NUbus_acknowledge=1;
      return;      
    case 0xE00088: // Operation Register
      NUbus_Data = sib_opn_reg;
      NUbus_acknowledge=1;
      return;      
    case 0xF00000: // RTC - Time Interrupt Event Vector Address
      NUbus_Data = sib_rtc_int_vector;
      NUbus_acknowledge=1;
      return;       
    case 0xF00004: // Itimer Short Interval Expired
      NUbus_Data = sib_sintv_expired_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF00008: // Itimer Long Interval Expired
      NUbus_Data = sib_lintv_expired_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF0000C: //
      NUbus_Data = sib_serio_status_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF00010: //
      NUbus_Data = sib_lpt_ack_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF00014: //
      NUbus_Data = sib_gcmd_ack_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF00018: //
      NUbus_Data = sib_kbd_rtt_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF0001C: //
      NUbus_Data = sib_psu_overtemp_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF00020: //
      NUbus_Data = sib_kbd_bootchord_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF00024: //
      NUbus_Data = sib_mouse_moved_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF00028: //
      NUbus_Data = sib_mouse_button_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF0002C: //
      NUbus_Data = sib_voxdata_present_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF00030: //
      NUbus_Data = sib_sound_parity_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF00034: //
      NUbus_Data = sib_fiber_link_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF00038: //
      NUbus_Data = sib_power_fail1_vector;
      NUbus_acknowledge=1;
      return;
    case 0xF0003C: //
      NUbus_Data = sib_power_fail2_vector;
      NUbus_acknowledge=1;
      return;

    case 0xF20001: // Mouse Y-Position Register, left-shifted 8 (see below)
      NUbus_Data = sib_mouse_y_pos << 8;
      NUbus_acknowledge=1;
      return;

    }
  }

  if(NUbus_Request == VM_BYTE_READ){
    switch(NUbus_Addr){
    case 0xE00088: // Operation Register
      NUbus_Data = sib_opn_reg;
      NUbus_acknowledge=1;
      return;      

    case 0xF00040: // Configuration Register
      /* BITS ARE:
	 1 = (W)  RESET 
	 2 = (RW) NUBUS MASTER ENABLE 
	 4 = (R)  SIB Test LED
	 8 = (RW) NUbus Test (UNUSED ON SIB)
	 100 = (RW) Monitor Self-Test LED
	 200 = (RW) Chassis Self-Test LED
	 400 = (R)  PSU Overtemperature Warning
      */
      NUbus_Data = sib_config_reg;
      NUbus_acknowledge = 1;
      return;

    case 0xF00041: // Configuration register, upper half
      NUbus_Data = (sib_config_reg&0xFF00);
      NUbus_acknowledge = 1;
      return;

    case 0xF10000: // LPT Data Register (R00)
      NUbus_Data = sib_lpt_r00;
      NUbus_acknowledge = 1;
      return;

    case 0xF20008: // Mouse Keyswitch/Motion Data Register
      /* BITS ARE:
	 0x03 = Mouse X movement quadrature (inverted) (positive means move right/down, negative means move left/up)
	 0x0C = Mouse Y movement quadrature (inverted)
	 0x10 = Right Mouse Key Down
	 0x20 = Middle Mouse Key Down
	 0x40 = Left Mouse Key Down
	 0x80 = Keyboard Serial Data
      */
      /* If diagnostic loopback is set, fake out. */
      if((sib_int_diag_reg&0x0C)!=0){
	if(sib_int_diag_data&0x100){
	  NUbus_Data = 0x00;
	}else{
	  NUbus_Data = 0xFF;
	}
      }else{
	// If we are in data loopback mode...
	if((sib_int_diag_reg&0x02) != 0){
	  // Return diagnostic byte instead
	  NUbus_Data = sib_int_diag_data&0xFF;
	}else{
	  NUbus_Data = sib_mouse_motion_reg;
	}
      }
      NUbus_acknowledge=1;
      return;	 
      
    case 0xF2000C: // Diagnostic and Interrupt Enable
    case 0xF2000D:
      /* BITS ARE:
	 0x0F = Set Diagnostic Register
	      0x01 = VOICE PARALLEL DATA PATH SELECTED
	             Replace fiber voice data with diag-data bits 0x7F.
		     Bit 0x80 means voice-data-recieved and will cause an interrupt if that's enabled.
		     After the video has been established, setting this will blank the display.
	      0x02 = MOUSE PARALLEL DATA PATH SELECTED
	             Replace fiber mouse/keyboard data with diag-data bits 0x7F.
	      0x04 = FIBER SERIO INTERNAL LOOPBACK SELECTED
	             Replace fiber data with complement of diag-data bit 0x80
	      0x08 = FIBER SERIO EXTERNAL LOOPBACK SELECTED
	             Same as INTERNAL loopback but done at the monitor.
	 0x10 = Voice Interrupt Enable
	 0x20 = Mouse Button Interrupt Enable
	 0x40 = Mouse Motion Interrupt Enable
	 0x80 = Monitor Sound Error Interrupt Enable
      */
      NUbus_Data = (sib_int_diag_reg&(0xFF<<(NUbus_Addr&0x03)));
      NUbus_acknowledge = 1;
      return;

    case 0xF20014: // Sound Control Register
      /* BITS ARE:
	 0x7F = Sound Data
	 0x80 = Odd Parity
      */	 
      NUbus_Data = sib_sound_control;
      NUbus_acknowledge = 1;
      return;
      
    case 0xF2001C: // VOX Data Register
      NUbus_Data = sib_vox_data_reg;
      NUbus_acknowledge=1;
      return;	 

    case 0xF80040: // RTC Interrupt Status
      /* For reads, a 1 means the selected timer set an interrupt condition.
	 Same format as control register below */
      NUbus_Data = sib_rtc_int_status;
      sib_rtc_int_status = 0;
      NUbus_acknowledge=1;
      return;

    case 0xFC0000: // Keyboard USART Status
      /* BITS ARE:
	 1  = TRANSMIT READY
	 2  = RECIEVE-READY
	 4  = TRANSMIT-BUFFER-EMPTY
	 8  = RECIEVE-PARITY-ERROR
	 10 = RECIEVE-OVERRUN-ERROR
	 20 = RECIEVE-FRAME-ERROR
	 40 = RECIEVE-BREAK-DETECT
	 80 = DATA-SET-READY (This is always 1 because it is HARD-WIRED that way!)
      */
      if(sib_int_diag_reg&0x0C){
	logmsg("USART STATUS READ WITH DEBUG-ANALOG-LOOPBACK ON!\n");
      }
      // If we're at the bottom of the usart buffer, and the transmitter is enabled
      if(keyboard_usart_rx_bot == keyboard_usart_rx_top && (keyboard_usart_command&0x04)==1){
	keyboard_usart_status |= 0x04; // the Transmit Buffer is Empty
      }else{
	// Otherwise, turn it off.
	keyboard_usart_status &= 0xFFFB;
      }
      
      NUbus_Data = keyboard_usart_status;
      NUbus_acknowledge = 1;
      return;

    case 0xFC0004: // Keyboard USART Recieve Data
      if((keyboard_usart_status&0x02) != 0 ){ // && keyboard_usart_rx_bot != keyboard_usart_rx_top){
	NUbus_Data = keyboard_usart_rx_fifo[keyboard_usart_rx_bot]; 
	keyboard_usart_rx_fifo[keyboard_usart_rx_bot]=0; 
	keyboard_usart_rx_bot++;
	if(keyboard_usart_rx_bot > 25){ keyboard_usart_rx_bot = 0; }
      }else{
	NUbus_Data = 0;
      }
      NUbus_acknowledge=1;
      //      sprintf(msg,"SIB: KBD Recieve Data 0x%lX\n",NUbus_Data); logmsg(msg);
      return;	 
    }
  }
  
  if(NUbus_Request == VM_WRITE){
    switch(NUbus_Addr){
    case 0xE00084: // Mask Register
      sib_mask_reg = NUbus_Data;
      NUbus_acknowledge=1;
      return;    
    case 0xE00088: // Operation Register
      sib_opn_reg = NUbus_Data;
      NUbus_acknowledge=1;
      return;        
    case 0xF00000: // RTC - Time Interrupt Event Vector Address
      sib_rtc_int_vector = NUbus_Data;
      //      sprintf(msg,"RTC: Time Interrupt Vector 0x%lX\n",NUbus_Data); logmsg(msg);
      NUbus_acknowledge=1;
      return;       
    case 0xF00004: // Itimer Short Interval Expired
      sib_sintv_expired_vector = NUbus_Data;
      //      sprintf(msg,"RTC: Short Interval Vector 0x%lX\n",NUbus_Data); logmsg(msg);
      NUbus_acknowledge=1;
      return;
    case 0xF00008: //
      sib_lintv_expired_vector = NUbus_Data;
      //      sprintf(msg,"RTC: Long Interval Vector 0x%lX\n",NUbus_Data); logmsg(msg);
      NUbus_acknowledge=1;
      return;
    case 0xF0000C: //
      sib_serio_status_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF00010: //
      sib_lpt_ack_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF00014: //
      sib_gcmd_ack_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF00018: //
      sib_kbd_rtt_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF0001C: //
      sib_psu_overtemp_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF00020: //
      sib_kbd_bootchord_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF00024: //
      sib_mouse_moved_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF00028: //
      sib_mouse_button_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF0002C: //
      sib_voxdata_present_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF00030: //
      sib_sound_parity_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF00034: //
      sib_fiber_link_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF00038: //
      sib_power_fail1_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF0003C: //
      sib_power_fail2_vector = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xF00041: // Configuration register, left-shifted 8 (see below)
      {
	unsigned int SIB_Data = NUbus_Data>>8;
	sib_config_reg &= 0x4FF;             // Turn off LEDs
	sib_config_reg |= (SIB_Data&0x300); // Turn on busmaster,nubus-test
	NUbus_acknowledge=1;
	return;       
      }
      return;

    case 0xF20001: // Mouse Y-Position Register, left-shifted 8 (see below)
      sib_mouse_y_pos &= 0xFF;
      sib_mouse_y_pos |= (NUbus_Data>>8);
      NUbus_acknowledge=1;
      return;

    case 0xF20010: 
    case 0xF20011: // Diagnostic Data
      sib_int_diag_data = NUbus_Data;
      // sprintf(msg,"SIB: Diagnostic Data: Recieved 0x%lX\n",NUbus_Data); logmsg(msg);
      // If this is a VOX loopback...
      if((sib_int_diag_reg&0x01) != 0 && sib_int_diag_data&0x100){
	// logmsg("SIB: Loading VOX Data\n");
	// we need to generate a PI.
	if(sib_config_reg&0x02  && (sib_int_diag_reg&0x10) != 0){
	  nubus_io_request(NB_BYTE_WRITE,sib_voxdata_present_vector,0xFF);
	}
	// and load the register
	sib_vox_data_reg = NUbus_Data&0xFF;
      }
      // Also, if we're in analog-loopback mode
      if(sib_int_diag_reg&0x0C){
	// we're faking out the serial data here. Reciever on?
	if((keyboard_usart_command&0x04)!=0){
	  // Yes, we should throw an error here.
	  logmsg("SIB: Error should throw?\n");
	  cpu_die_rq=1;
	}
      }
      NUbus_acknowledge=1;
      return;

    case 0xF90000: // Interval Timer, Counter 0
      switch(sib_itimer_ctl_0&0x30){
      case 0x10: // Load LSB
	sib_itimer_ctr_0 = NUbus_Data&0xFF;
	sib_itimer_lod_0 |= 0x01;
	NUbus_acknowledge=1;
	break;
      case 0x20: // Load MSB
	sib_itimer_ctr_0 = (NUbus_Data&0xFF00)>>8;
	sib_itimer_lod_0 |= 0x02;
	NUbus_acknowledge=1;
	break;
      case 0x30: // Load MSB and LSB
	sib_itimer_ctr_0 = NUbus_Data&0xFFFF;
	NUbus_acknowledge=1;
	sib_itimer_lod_0 = 3;
	break;
      }
      if(sib_itimer_lod_0 == 0x3){
	sib_itimer_ena_0 = 1;  logmsg("SIB: Interval Timer 0 Started\n");
	sib_itimer_lod_0 = 0;
      }
      return;

    case 0xF90004: // Interval Timer, Counter 1
      switch(sib_itimer_ctl_0&0x30){
      case 0x10: // Load LSB
	sib_itimer_ctr_1 = NUbus_Data&0xFF;
	sib_itimer_lod_1 |= 0x01;
	sib_itimer_ped_1 = sib_itimer_ctr_1;
	NUbus_acknowledge=1;
	break;
      case 0x20: // Load MSB
	sib_itimer_ctr_1 = (NUbus_Data&0xFF00)>>8;
	sib_itimer_ped_1 = sib_itimer_ctr_1;
	NUbus_acknowledge=1;
	sib_itimer_lod_1 |= 0x02;
	break;
      case 0x30: // Load MSB and LSB
	sib_itimer_ctr_1 = NUbus_Data&0xFFFF;
	sib_itimer_ped_1 = sib_itimer_ctr_1;
	sib_itimer_lod_1 = 3;
	NUbus_acknowledge=1;
	break;
      }
      if(sib_itimer_lod_1 == 0x3){
	sib_itimer_ena_1 = 1;
	if(sib_itimer_ctl_1&0xE){
	  logmsg("SIB: Interval Timer 1 Started in Square-Wave-Gen Mode\n");
	}else{
	  logmsg("SIB: Interval Timer 1 Started in Terminal-Count Mode\n");
	}
	sib_itimer_lod_1 = 0;
      }
      return;

    case 0xF90008: // Interval Timer, Counter 2
      switch(sib_itimer_ctl_0&0x30){
      case 0x10: // Load LSB
	sib_itimer_ctr_2 = NUbus_Data&0xFF;
	sib_itimer_lod_2 |= 0x01;
	NUbus_acknowledge=1;
	break;
      case 0x20: // Load MSB
	sib_itimer_ctr_2 = (NUbus_Data&0xFF00)>>8;
	sib_itimer_lod_2 |= 0x02;
	NUbus_acknowledge=1;
	break;
      case 0x30: // Load MSB and LSB
	sib_itimer_ctr_2 = NUbus_Data&0xFFFF;
	sib_itimer_lod_2 = 3;
	NUbus_acknowledge=1;
	break;
      }
      if(sib_itimer_lod_2 == 0x3){
	sib_itimer_ena_2 = 1;  logmsg("SIB: Interval Timer 2 Started\n");
	sib_itimer_lod_2 = 0;
      }
      return;

    }
  }

  if(NUbus_Request == VM_BYTE_WRITE){
    switch(NUbus_Addr){

    case 0xE00000: // CRT controller R00
      // Screen Format: Characters-per-horizontal-period
      sib_crt_r00 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00004: // CRT controller R01
      // Screen Format: Characters-per-data-row
      sib_crt_r01 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00008: // CRT controller R02
      // Screen Format: Horizontal Delay
      sib_crt_r02 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE0000C: // CRT controller R03
      // Screen Format: Horizontal Sync Width
      sib_crt_r03 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00010: // CRT controller R04
      // Screen Format: Vertical Sync Width
      sib_crt_r04 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00014: // CRT controller R05
      // Screen Format: Vertical Delay
      sib_crt_r05 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00018: // CRT controller R06
      // Screen Format: Skew, pin configuration
      sib_crt_r06 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE0001C: // CRT controller R07
      // Screen Format: Visible data rows per frame
      sib_crt_r07 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00020: // CRT controller R08
      // Screen Format: Scan Lines (frame, data row)
      sib_crt_r08 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00024: // CRT controller R09
      // Screen Format: Scan Lines per frame LSB
      sib_crt_r09 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00028: // CRT controller R0A
      // Auxiliary register
      sib_crt_r0a = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE0002C: // CRT controller R0B
      // Auxiliary register
      sib_crt_r0b = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00030: // CRT controller R0C
      // Auxiliary register
      sib_crt_r0c = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00034: // CRT controller R0D
      // Auxiliary register
      sib_crt_r0d = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00038: // CRT controller R0E
      // Auxiliary register
      sib_crt_r0e = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE0003C: // CRT controller R0F
      // Auxiliary register
      sib_crt_r0f = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00040: // CRT controller R10
      // Auxiliary register
      sib_crt_r10 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00044: // CRT controller R11
      // Auxiliary register
      sib_crt_r11 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00048: // CRT controller R12
      // Auxiliary register
      sib_crt_r12 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE0004C: // CRT controller R13
      // Auxiliary register
      sib_crt_r13 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00050: // CRT controller R14
      // Auxiliary register
      sib_crt_r14 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00054: // CRT controller R15
      // Auxiliary register
      sib_crt_r15 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00058: // CRT controller R16
      // CURSOR CONTROL: RESET COMMAND
      logmsg("SIB: CURSOR RESET\n");
      sib_crt_r16 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE0005C: // CRT controller R17
      // Auxiliary register
      sib_crt_r17 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00060: // CRT controller R18
      // Auxiliary register
      sib_crt_r18 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00064: // CRT controller R19
      // Auxiliary register
      sib_crt_r19 = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00068: // CRT controller R1A
      // Auxiliary register
      sib_crt_r1a = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00080: // Video Attributes Register
      /* BITS ARE:
	 0x01 = Blank Video (1 = blank)
	 0x02 = Video Polarity (1 = reverse-video)
	 0x04 = Fiber Channel B Disconnected (1 = disconnected)
	 0x08 = Interrupt On Channel B Disconnection
      */
      if(NUbus_Data&0x01 && !(sib_video_attr&0x01)){
	// ** Blank Video *
	logmsg("SIB: Video Blanked\n");
      }
      if(!(NUbus_Data&0x01) && sib_video_attr&0x01){
	// ** Unblank Video **
	logmsg("SIB: Video Unblanked\n");
      }
      if((NUbus_Data&0x02) != (sib_video_attr&0x02)){
	sprintf(msg,"Video State Changed to %lX\n",NUbus_Data&0x2); logmsg(msg);
	// State changed - Update Linux framebuffer
	sib_video_attr = NUbus_Data;
	redraw_display();
	NUbus_acknowledge=1;
	return;
      }
      sib_video_attr = NUbus_Data;
      NUbus_acknowledge=1;
      return;

    case 0xE00088: // Operation Register
      sib_opn_reg = NUbus_Data; 
      NUbus_acknowledge=1;
      return;      

    case 0xF00040: // SIB Configuration Register
      /* BITS ARE:
	 1 = (W)  RESET 
	 2 = (RW) NUBUS MASTER ENABLE 
	 4 = (R)  SIB Test LED
	 8 = (RW) NUbus Test (UNUSED ON SIB)
	 100 = (RW) Monitor Self-Test LED
	 200 = (RW) Chassis Self-Test LED
	 400 = (R)  PSU Overtemperature Warning
      */
      if(NUbus_Data&0x1){
	// Keyboard USART reset
	keyboard_usart_mode=0; // MODESET mode
	keyboard_usart_status =  0x80; 
	keyboard_usart_mode = 0; 
	keyboard_usart_command = 0;
	keyboard_usart_rx_fifo[0] = 0;
	keyboard_usart_rx_top=0;
	keyboard_usart_rx_bot=0;
	
	logmsg("SIB: RESET\n");
      }
      if(NUbus_Data&0x2){
	logmsg("SIB: BUS-MASTER-ENABLE\n");
      }
      // Handle writables
      sib_config_reg &= 0xFF0;             // Turn off low bits
      sib_config_reg |= (NUbus_Data&0x0A); // Turn on busmaster,nubus-test
      NUbus_acknowledge=1;
      return;

    case 0xF00041: // Configuration register, upper half
      sib_config_reg &= 0xFF; // Turn off high bits
      sib_config_reg |= (NUbus_Data&0xFF00);
      NUbus_acknowledge = 1;
      return;

    case 0xF10000: // LPT register 00
      // LPT Data Register
      sib_lpt_r00 = NUbus_Data;
      NUbus_acknowledge=1;
      return;      

    case 0xF2000C: // Diagnostic and Interrupt Enable
    case 0xF2000D:
      /* BITS ARE:
	 0x0F = Set Diagnostic Register
	      0x01 = VOICE PARALLEL DATA PATH SELECTED
	             Replace fiber voice data with diag-data bits 0x7F.
		     Bit 0x80 means voice-data-recieved and will cause an interrupt if that's enabled.
		     After the video has been established, setting this will blank the display.
	      0x02 = MOUSE PARALLEL DATA PATH SELECTED
	             Replace fiber mouse/keyboard data with diag-data bits 0x7F.
	      0x04 = FIBER SERIO INTERNAL LOOPBACK SELECTED
	             Replace fiber data with complement of diag-data bit 0x80
	      0x08 = FIBER SERIO EXTERNAL LOOPBACK SELECTED (ALSO LOOPS KEYBOARD)
	             Same as INTERNAL loopback but done at the monitor.
	 0x10 = Voice Interrupt Enable
	 0x20 = Mouse Button Interrupt Enable
	 0x40 = Mouse Motion Interrupt Enable
	 0x80 = Monitor Sound Error Interrupt Enable
      */
      sib_int_diag_reg &= ~(0xFF>>(8*(NUbus_Addr&0x3)));           // Clear byte to be written 
      sib_int_diag_reg |= (NUbus_Data>>(8*(NUbus_Addr&0x3))&0xFF); // and replace it
      if(sib_int_diag_reg&0xF0){ logmsg("SIB: Voice/Mouse/Monitor Interrupt Enabled\n"); }
      NUbus_acknowledge = 1;
      return;

    case 0xF20014: // Sound Control
      sib_sound_control = NUbus_Data;
      // if(sib_sound_control != 0){ sprintf(msg,"SIB: Sound Command 0x%lX Sent\n",NUbus_Data); logmsg(msg); }
      NUbus_acknowledge=1;
      return;

    case 0xF80000: // RTC 100ns count
      sib_rtc_100ns = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80004: 
      sib_rtc_10ms = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80008:
      sib_rtc_sec = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF8000C:
      sib_rtc_min = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80010:
      sib_rtc_hour = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80014:
      sib_rtc_dow = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80018:
      sib_rtc_date = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF8001C:
      sib_rtc_month = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80020:
      sib_ram_100ns = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80024: 
      sib_ram_10ms = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80028:
      sib_ram_sec = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF8002C:
      sib_ram_min = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80030:
      sib_ram_hour = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80034:
      sib_ram_dow = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF80038:
      sib_ram_date = NUbus_Data;
      NUbus_acknowledge=1;
      return;
    case 0xF8003C:
      sib_ram_month = NUbus_Data;
      NUbus_acknowledge=1;
      return;
      
    case 0xF80044: // RTC Interrupt Control
      /* A 1 means set an interrupt whenever the selected timer is updated. 
	 Bits are:
	 0x01 = Compare (Interrupt when RTC timers = RAM timers)
	 0x02 = 10Hz
	 0x04 = 1Hz
	 0x08 = 1 minute
	 0x10 = 1 hour
	 0x20 = 1 day
	 0x40 = 1 week
	 0x80 = 1 month
      */
      sib_rtc_int_control = NUbus_Data;
      if(sib_rtc_int_control != 0){ sprintf(msg,"SIB: RTC PI Enabled (0x%lX)\n",NUbus_Data); logmsg(msg); }
      NUbus_acknowledge=1;
      return;

    case 0xF80054: // RTC "GO" Command
      sib_rtc_100ns=0;
      sib_rtc_10ms=0;
      sib_rtc_min=0;
      if(sib_rtc_sec > 0x40){
	sib_rtc_min++;
      }
      sib_rtc_sec=0;
      NUbus_acknowledge=1;
      return;      

    case 0xF90000: // Interval Timer, Counter 0
      switch(sib_itimer_ctl_0&0x30){
      case 0x10: // Load LSB
	sib_itimer_ctr_0 &= 0x00FF;
	sib_itimer_ctr_0 |= NUbus_Data&0xFF;
	sib_itimer_lod_0 |= 0x01;
	NUbus_acknowledge=1;
	break;
      case 0x20: // Load MSB
	sib_itimer_ctr_0 &= 0xFF00LL;
	sib_itimer_ctr_0 |= (NUbus_Data<<8);
	sib_itimer_lod_0 |= 0x02;
	NUbus_acknowledge=1;
	break;
      case 0x30: // Load LSB then MSB
	switch(sib_itimer_lod_0){
	case 0: // Load LSB
	  sib_itimer_ctr_0 &= 0x00FF;
	  sib_itimer_ctr_0 |= NUbus_Data&0xFF;
	  sib_itimer_lod_0 |= 0x01;
	  NUbus_acknowledge=1;
	  break;
	case 1: // Load MSB
	  sib_itimer_ctr_0 &= 0xFF00LL;
	  sib_itimer_ctr_0 |= (NUbus_Data<<8);
	  sib_itimer_lod_0 |= 0x02;
	  NUbus_acknowledge=1;
	  break;
	}
	break;
      }
      if(sib_itimer_lod_0 == 0x3){
	sib_itimer_ena_0 = 1;  logmsg("SIB: Interval Timer 0 Started\n");
	sib_itimer_lod_0 = 0;
      }
      return;

    case 0xF90004: // Interval Timer, Counter 1
      switch(sib_itimer_ctl_1&0x30){
      case 0x10: // Load LSB
	sib_itimer_ctr_1 &= 0x00FF;
	sib_itimer_ctr_1 |= NUbus_Data&0xFF;
	sib_itimer_ped_1 = sib_itimer_ctr_1;
	sib_itimer_lod_1 |= 0x01;
	NUbus_acknowledge=1;
	break;
      case 0x20: // Load MSB
	sib_itimer_ctr_1 &= 0xFF00LL;
	sib_itimer_ctr_1 |= (NUbus_Data<<8);
	sib_itimer_ped_1 = sib_itimer_ctr_1;
	sib_itimer_lod_1 |= 0x02;
	NUbus_acknowledge=1;
	break;
      case 0x30: // Load LSB then MSB
	switch(sib_itimer_lod_1){
	case 0: // Load LSB
	  sib_itimer_ctr_1 &= 0x00FF;
	  sib_itimer_ctr_1 |= NUbus_Data&0xFF;
	  sib_itimer_ped_1 = sib_itimer_ctr_1;
	  sib_itimer_lod_1 |= 0x01;
	  NUbus_acknowledge=1;
	  break;
	case 1: // Load MSB
	  sib_itimer_ctr_1 &= 0xFF00LL;
	  sib_itimer_ctr_1 |= (NUbus_Data<<8);
	  sib_itimer_ped_1 = sib_itimer_ctr_1;
	  sib_itimer_lod_1 |= 0x02;
	  NUbus_acknowledge=1;
	  break;
	}
	break;
      }
      if(sib_itimer_lod_1 == 0x3){
	sib_itimer_ena_1 = 1;
	if(sib_itimer_ctl_1&0xE){
	  logmsg("SIB: Interval Timer 1 Started in Square-Wave-Gen Mode\n");
	}else{
	  logmsg("SIB: Interval Timer 1 Started in Terminal-Count Mode\n");
	}
	sib_itimer_lod_1 = 0;
      }
      return;

    case 0xF90008: // Interval Timer, Counter 2
      switch(sib_itimer_ctl_2&0x30){
      case 0x10: // Load LSB
	sib_itimer_ctr_2 &= 0x00FF;
	sib_itimer_ctr_2 |= NUbus_Data&0xFF;
	sib_itimer_lod_2 |= 0x01;
	NUbus_acknowledge=1;
	break;
      case 0x20: // Load MSB
	sib_itimer_ctr_2 &= 0xFF00LL;
	sib_itimer_ctr_2 |= (NUbus_Data<<8);
	sib_itimer_lod_2 |= 0x02;
	NUbus_acknowledge=1;
	break;
      case 0x30: // Load LSB then MSB
	switch(sib_itimer_lod_2){
	case 0: // Load LSB
	  sib_itimer_ctr_2 &= 0x00FF;
	  sib_itimer_ctr_2 |= NUbus_Data&0xFF;
	  sib_itimer_lod_2 |= 0x01;
	  NUbus_acknowledge=1;
	  break;
	case 1: // Load MSB
	  sib_itimer_ctr_2 &= 0xFF00LL;
	  sib_itimer_ctr_2 |= (NUbus_Data<<8);
	  sib_itimer_lod_2 |= 0x02;
	  NUbus_acknowledge=1;
	  break;
	}
	break;
      }
      if(sib_itimer_lod_2 == 0x3){
	sib_itimer_ena_2 = 1;  logmsg("SIB: Interval Timer 2 Started\n");
	sib_itimer_lod_2 = 0;
      }
      return;

    case 0xF9000C: // Interval timer mode control
      /* BITS ARE:
	 0x01 = BCD-Count-Select (1 = count in BCD, 0 = count in 16-bit binary)
	 0x0E = Mode-Select (0 = interrupt-on-terminal-count, 3 = square-wave-rate-generator)
	 0x30 = Read-Load-Control (0 = Counter-Latching, 1 = Read/Load LSB, 2 = Read/Load MSB, 3 = Read/Load LSB then MSB)
	 0xC0 = Select-Counter (0 thru 2 selects the numbered counter, 3 is illegal)
      */
      sprintf(msg,"SIB: ITimer-Control: 0x%lX\n",NUbus_Data); logmsg(msg);
      if(NUbus_Data&0x01){ logmsg("SIB: BCD-Mode is not supported!\n"); }
      switch(NUbus_Data&0xC0){
      case 0x00: // Load 0
	sib_itimer_ctl_0 = NUbus_Data;
	sib_itimer_ena_0 = 0; sib_itimer_lod_0 = 0;
	NUbus_acknowledge = 1;
	return;
      case 0x40: // Load 1
	sib_itimer_ctl_1 = NUbus_Data;
	sib_itimer_ena_1 = 0; sib_itimer_lod_1 = 0;
	sib_itimer_lvl_1 = 0; sib_itimer_edg_1 = 0;
	NUbus_acknowledge = 1;
	return;
      case 0x80: // Load 2
	sib_itimer_ctl_2 = NUbus_Data;
	sib_itimer_ena_2 = 0; sib_itimer_lod_2 = 0;
	NUbus_acknowledge = 1;
	return;
      }
      break;

    case 0xFC0000: // Keyboard USART Control
      /* COMMAND BITS ARE:
	 1  = TRANSMIT ENABLE (1 = enabled)
	 2  = DATA-TERMINAL-READY (1 = force low)
	 4  = RECIEVE-ENABLE (1 = enable, enable requires error-status reset)
	 8  = SEND-BREAK-CHARACTER (1 = send break)
	 10 = ERROR-STATUS-RESET (Clears error, overrun, frame-error flags)
	 20 = REQUEST-TO-SEND (1 = force low)
	 40 = INTERNAL-RESET (1 = Return USART to mode-set operation)
	 80 = MUST-BE-ZERO
	 MODE BITS ARE:
	 03 = BAUD-RATE SELECT (00 = SYNCH,01 = No Division, 02 = Clock/26, 03 = Clock/64)
	 0C = CHARACTER LENGTH (00,04,08,0C = 5,6,7,8 bits)
	 10 = PARITY ENABLE    (1 = Enable)
	 20 = PARITY SELECT    (1 = EVEN, 0 = ODD)
	 C0 = STOP BIT SELECT  (00 = ILLEGL, 40 = 1, 80 = 1 1/2, C0 = 2 stop bits)
      */
      if(keyboard_usart_mode==0){ // MODE SET mode
	sprintf(msg,"SIB: USART Mode Set: 0x%lX\n",NUbus_Data); logmsg(msg);
	keyboard_usart_control = NUbus_Data; // Store this
	keyboard_usart_mode = 1; // Go to COMMAND SET mode	
	keyboard_usart_status |= 0x01; // Transmitter Ready isn't dependent on enable or whatever.
      }else{
	// COMMAND SET mode
	sprintf(msg,"SIB: USART Command Set: 0x%lX\n",NUbus_Data); logmsg(msg);
	if(NUbus_Data&0x01){ keyboard_usart_status |= 0x01; }else{ keyboard_usart_status &= 0xFFFB; }
	if(NUbus_Data&0x04){    // RX enable
	  if((keyboard_usart_status&0x38)==0){  // requires error reset to turn on.
	  //	  if(NUbus_Data&0x10){
	    keyboard_usart_status |= 0x02;
	  }
	}else{ keyboard_usart_status &= 0xFFFD; }
	if(NUbus_Data&0x10){ keyboard_usart_status &= 0x87; }
	if(NUbus_Data&0x40){
	  // INTERNAL RESET
	  keyboard_usart_status =  0x80; 
	  keyboard_usart_mode = 0; 
	  keyboard_usart_command = 0;
	  keyboard_usart_rx_fifo[0] = 0;
	  keyboard_usart_rx_top=0;
	  keyboard_usart_rx_bot=0;
	}else{
	  keyboard_usart_command = NUbus_Data;
	}
	if(NUbus_Data&0x05){
	  // Sending a break - Transmitter on?
	  if((keyboard_usart_command&0x01)==1){
	    // In loopback?
	    if(sib_int_diag_reg&0x08){
	      logmsg("SIB: BREAK looped\n");
	      keyboard_usart_status |= 0x70; // BREAK detected
	    }
	  }
	}

      }
      NUbus_acknowledge=1;
      return;

    case 0xFC0004: // Keyboard USART Transmit Data
      keyboard_usart_tx_data = NUbus_Data;
      // Push transmitted data if loopback mode
      // Do nothing if transmitter disabled.
      if((keyboard_usart_command&0x01)==1){
	if(sib_int_diag_reg&0x08){
	  keyboard_usart_rx_fifo[keyboard_usart_rx_top] = (NUbus_Data&0xFF);
	  keyboard_usart_rx_top++;
	  if(keyboard_usart_rx_top > 25){ keyboard_usart_rx_top = 0; }
	}
	// HACK-HACK - 0x00 is apparently an initialization code.
	// Respond with 0x70      
	if(NUbus_Data == 0x00){
	  keyboard_usart_rx_fifo[keyboard_usart_rx_top] = 0x70;
	  keyboard_usart_rx_top++;
	  if(keyboard_usart_rx_top > 25){ keyboard_usart_rx_top = 0; }
	}
      }
      //      sprintf(msg,"SIB: KBD Transmit Data 0x%X\n",keyboard_usart_tx_data); logmsg(msg);
      NUbus_acknowledge=1;
      return;      
    }
  }

  sprintf(msg,"SIB: Unknown NUbus IO: Op %ld for address %lX\n",NUbus_Request,NUbus_Addr); logmsg(msg); 
  cpu_die_rq=1;
}

inline void sib_clock_pulse(){
  // The cycle time for a microinstruction is 8MHz.
  // That's a period of (1,000,000,000 / 8) 125,000,000 ns.
  // That means a 1ms (1,000,000) timer is plenty to update the RTC
  // updatetime(0);
}
