;-*-Mode:Lisp; Package:FILE-SYSTEM; Base:8-*-

;This type of stream is used for probe opens to remote machines.
;No chaosnet connection is maintained after the initial information
;is received.
(DEFFLAVOR REMOTE-PROBE-STREAM
	(STREAM-PLIST
	 (WHICH-OPERATIONS NIL)
	 (MERGED-WHICH-OPERATIONS NIL)
	 PLIST
	 TRUENAME)
	(SI:FILE-STREAM-MIXIN SI:STREAM)
  (:INITABLE-INSTANCE-VARIABLES WHICH-OPERATIONS
				PLIST STREAM-PLIST
				TRUENAME)
  (:GETTABLE-INSTANCE-VARIABLES TRUENAME PLIST STREAM-PLIST)
  (:DEFAULT-HANDLER REMOTE-PROBE-STREAM-PASS-ON-OPERATION))

(DEFMETHOD (REMOTE-PROBE-STREAM :DIRECTION) () ())

(DEFMETHOD (REMOTE-PROBE-STREAM :WHICH-OPERATIONS) ()
  (OR MERGED-WHICH-OPERATIONS
      (SETQ MERGED-WHICH-OPERATIONS
	    (SI:UNION
	      (SUBSET #'(LAMBDA (ELT) (NOT (MEMQ ELT STANDARD-STREAM-OPS)))
		      WHICH-OPERATIONS)
	      (WITH-SELF-ACCESSIBLE SI:VANILLA-FLAVOR
		(FUNCALL #'(:METHOD SI:VANILLA-FLAVOR :WHICH-OPERATIONS)
			 ':WHICH-OPERATIONS))))))

;If the operation can be handled, and it is the name of a property
;on the PLIST, then return the associated value.
;This is used to handle operations such as :AUTHOR
;(but again, only if the server said those operations are handled).
(DEFUN REMOTE-PROBE-STREAM-PASS-ON-OPERATION (OPERATION &REST ARGS &AUX TEM)
  ARGS
  (DECLARE-FLAVOR-INSTANCE-VARIABLES (REMOTE-PROBE-STREAM)
    (COND ((NOT (MEMQ OPERATION WHICH-OPERATIONS))
	   (FERROR NIL "Operation ~S not handled by ~S"
		   OPERATION SELF))
	  ((SETQ TEM (GET (LOCF PLIST) OPERATION))
	   (CDR TEM)))))

(DEFMETHOD (REMOTE-PROBE-STREAM :INFO) ()
  (GET (LOCF STREAM-PLIST) ':INFO))

;Temporary, to override old versions of REMOTE.
(DEFMETHOD (REMOTE-PROBE-STREAM :CREATION-DATE) ()
  (GET (LOCF PLIST) ':CREATION-DATE))

;Both PLIST and STREAM-PLIST have a :BYTE-SIZE property.
;The former's applies to the file; the latter's, to the stream.
(DEFMETHOD (REMOTE-PROBE-STREAM :BYTE-SIZE) ()
  (GET (LOCF STREAM-PLIST) ':BYTE-SIZE))

;:CHARACTERS properties work just like :BYTE-SIZE properties.
(DEFMETHOD (REMOTE-PROBE-STREAM :CHARACTERS) ()
  (GET (LOCF STREAM-PLIST) ':CHARACTERS))

(DEFMETHOD (REMOTE-PROBE-STREAM :LENGTH) ()
  (GET (LOCF STREAM-PLIST) ':LENGTH))

;;; Remote streams that transfer data.

(DEFFLAVOR REMOTE-DATA-STREAM
  ((STATE ':OPEN)
   READ-PATHNAME-HOST SERVER-LOCAL-HOST
   CHAOS-BYTE-SIZE
   FILE-STREAM-TYPE
   (CONNECTION NIL)
   ;; If not NIL, this is our current temporary lossage state,
   ;; as obtained from the last FILE-LOSSAGE-OPCODE packet.
   ;; The lossage makes input and output stream ops get errors
   ;; until a FILE-WINNAGE-OPCODE clears it out.
   (LOSSAGE NIL)
   ;; AUTO-NEXT-NODE is T if we should let the data from all
   ;; nodes run together with no indication of where
   ;; one stops and the next starts.
   ;; That is, ignore FILE-NEXT-NODE-OPCODE packets.
   (AUTO-NEXT-NODE T)
   (PREREAD-PKTS NIL)
   ;; PREREAD-PKTS contains a list of two kinds of things:
   ;; input data packets, and lists whose car is a value
   ;; taken from a file-next-node-opcode.
   ;; It serves as a buffer for things that we have to read past
   ;; to get command replies we are waiting for.
   (CONTINUED-PKTS NIL)
   (SAVE-REPLY-SEQNUMS NIL)
   (NEXT-SEQNUM 1)
   (SAVED-COMMAND-REPLIES NIL))
  (SI:FILE-DATA-STREAM-MIXIN REMOTE-PROBE-STREAM)
  (:INITABLE-INSTANCE-VARIABLES FILE-STREAM-TYPE CONNECTION
				READ-PATHNAME-HOST SERVER-LOCAL-HOST
				AUTO-NEXT-NODE CHAOS-BYTE-SIZE)
  (:SPECIAL-INSTANCE-VARIABLES SERVER-LOCAL-HOST READ-PATHNAME-HOST)
  (:DEFAULT-HANDLER REMOTE-STREAM-PASS-ON-OPERATION))

;For any operation that is not recognized, if the stream on the
;file computer said it can handle the operation, pass the operation to it.
;If the operation can be handled, and it is the name of a property
;on the PLIST, then return the associated value.
;This is used to handle operations such as :AUTHOR
;(but again, only if the server said those operations are handled).
;To handle operations such as :SET-AUTHOR, the :SETTING-OPERATIONS
;property on the STREAM-PLIST should be an alist matching the setting
;operations with the property names they should set, such as
;((:SET-AUTHOR . :AUTHOR) ...).
;The setting operations are passed on, so that the quantity is set
;in the server, and also the local property is set.
(DEFUN REMOTE-STREAM-PASS-ON-OPERATION (OPERATION &REST ARGS &AUX TEM)
  (DECLARE-FLAVOR-INSTANCE-VARIABLES (REMOTE-DATA-STREAM)
    (COND ((NOT (MEMQ OPERATION WHICH-OPERATIONS))
	   (FERROR NIL "Operation ~S not handled by ~S"
		   OPERATION SELF))
	  ((SETQ TEM (GET (LOCF PLIST) OPERATION))
	   (CDR TEM))
	  ((EQ (CAR (SETQ TEM (AWAIT-COMMAND-REPLY
				(LEXPR-FUNCALL 'FILE-SEND-COMMAND 'STREAM-OPERATION OPERATION ARGS) T)))
	       T)
	   ;; Is this op supposed to set a stream property??
	   (SETQ TEM (ASSQ OPERATION (GET (LOCF STREAM-PLIST) ':SETTING-OPERATIONS)))
	   (COND (TEM
		  ;; Yes => do so.  This way, :set-author
		  ;; can alter the locally recorded value which :author returns.
		  (PUTPROP (LOCF PLIST) (CAR ARGS) (CDR TEM))))
	   (VALUES-LIST (CADR TEM)))
	  (T (FERROR NIL "~S on remote stream ~S got error ~A"
		     OPERATION SELF (CADR TEM))))))

(DEFMETHOD (REMOTE-DATA-STREAM :AFTER :INIT) (&REST IGNORE)
  (OR (GET (LOCF STREAM-PLIST) ':SETTING-OPERATIONS)
      (PUTPROP (LOCF STREAM-PLIST)
	       '((:SET-AUTHOR . :AUTHOR) (:SET-CREATION-DATE . :CREATION-DATE))
	       ':SETTING-OPERATIONS)))

(DEFMETHOD (REMOTE-DATA-STREAM :AFTER :CLOSE) (&OPTIONAL ABORTP)
  (FUNCALL (FUNCALL SI:PATHNAME ':HOST) ':SEND-IF-HANDLES ':REMOVE-STREAM SELF)
  (COND ((NEQ STATE ':CLOSED)
	 (SETQ STATE ':CLOSED)
	 (AND (GET (LOCF STREAM-PLIST) ':MUST-EXPLICITLY-CLOSE)
	      (LET (INFO
		    (VAL (AWAIT-COMMAND-REPLY (FILE-SEND-COMMAND 'CLOSE ABORTP))))
		(COND ((EQ (CAR VAL) 'T)
		       (SETF (LIST TRUENAME INFO) (CADR VAL))))
		(AND INFO (PUTPROP (LOCF STREAM-PLIST) INFO ':INFO))))
	 (COND (CONNECTION
		(CHAOS:CLOSE CONNECTION)
		(SETQ CONNECTION NIL)))	
	 (DOLIST (P PREREAD-PKTS)
	   (OR (LISTP P) (CHAOS:RETURN-PKT P)))
	 (SETQ PREREAD-PKTS NIL)
	 T)))

(DEFMETHOD (REMOTE-DATA-STREAM :SAFE-TO-USE-P) ()
  (IF (AND (EQ STATE ':OPEN)
	   CONNECTION
	   (EQ (CHAOS:STATE CONNECTION) 'CHAOS:OPEN-STATE))
      T
    (SETQ STATE ':CLOSED)
    (DOLIST (P PREREAD-PKTS)
      (OR (LISTP P) (CHAOS:RETURN-PKT P)))
    (SETQ PREREAD-PKTS NIL)
    NIL))

(DEFMETHOD (REMOTE-DATA-STREAM :QFASLP) ()
  (GET (LOCF STREAM-PLIST) ':QFASLP))

(DEFMETHOD (REMOTE-DATA-STREAM :SET-BYTE-SIZE) (NEW-BYTE-SIZE)
  (OR (MEMQ ':SET-BYTE-SIZE WHICH-OPERATIONS)
      (FERROR NIL "Operation :SET-BYTE-SIZE not handled by remote stream"))
  (COND ((EQ STATE ':CLOSED)
	 (FERROR NIL "stream not open")))
  (FUNCALL-SELF ':FINISH)
  (COND ((EQ (CAR (AWAIT-COMMAND-REPLY
		    (FILE-SEND-COMMAND 'SET-BYTE-SIZE
				       NEW-BYTE-SIZE
				       (EQ FILE-STREAM-TYPE 'INPUT))
		    T))
	     T)
	 (SETQ CHAOS-BYTE-SIZE (IF (> NEW-BYTE-SIZE 8) 16. 8))
	 (PUTPROP (LOCF STREAM-PLIST) NEW-BYTE-SIZE ':BYTE-SIZE)
	 (SETQ STATE ':OPEN))
	(T (SETQ STATE ':CLOSED))))	;Is this right?

(DEFMETHOD (REMOTE-DATA-STREAM :RENAME) (NEW-PATHNAME &OPTIONAL (ERRORP T))
  (LET* ((TEM (AWAIT-COMMAND-REPLY
		(FILE-SEND-COMMAND 'RENAME-STREAM NEW-PATHNAME ERRORP) T))
	 (NEW-TRUENAME (CAR (CADR TEM))))
    (IF (CAR TEM)
	(SETQ TRUENAME NEW-TRUENAME))
    (VALUES-LIST (CADR TEM))))

(DEFFLAVOR REMOTE-INPUT-STREAM
	(PKT)
	(SI:BUFFERED-INPUT-STREAM SI:INPUT-POINTER-REMEMBERING-MIXIN REMOTE-DATA-STREAM))

(DEFMETHOD (REMOTE-INPUT-STREAM :BEFORE :SET-BYTE-SIZE) (&REST IGNORE)
  (SETQ SI:INPUT-POINTER-BASE 0))

(DEFMETHOD (REMOTE-INPUT-STREAM :NEXT-NODE) ()
  (DO () (())
    (OR PREREAD-PKTS (AWAIT-COMMAND-REPLY NIL NIL NIL T))
    (IF (LISTP (CAR PREREAD-PKTS))
	(RETURN (CAR (POP PREREAD-PKTS)))
      (CHAOS:RETURN-PKT (POP PREREAD-PKTS)))))

(DEFMETHOD (REMOTE-INPUT-STREAM :NEXT-INPUT-BUFFER) (&OPTIONAL IGNORE)
  (COND ((EQ STATE ':OPEN)
	 ;;Process packets until we get an input or eof package (or find lossage).
	 (DO ((P)) (NIL)
	   (AND PREREAD-PKTS
		(RETURN (SETQ PKT (POP PREREAD-PKTS))))
	   (DO () ((OR (CHAOS:DATA-AVAILABLE CONNECTION) (NOT LOSSAGE)))
	     (CERROR ':NO-ACTION NIL (CAR LOSSAGE) (CADR LOSSAGE)))
	   (SETQ P (CHAOS:GET-NEXT-PKT CONNECTION))
	   (SELECT (CHAOS:PKT-OPCODE P)
	     ((FILE-BINARY-OPCODE FILE-CHARACTER-OPCODE FILE-EOF-OPCODE)
	      (RETURN (SETQ PKT P)))
	     (FILE-CONTINUED-OPCODE
	      (SETQ CONTINUED-PKTS (NCONC CONTINUED-PKTS (NCONS P))))
	     ((FILE-LOSSAGE-OPCODE FILE-WINNAGE-OPCODE
				   FILE-REPLY-OPCODE FILE-NEXT-NODE-OPCODE
				   CHAOS:CLS-OP)
	      (WITH-SELF-VARIABLES-BOUND
	        (PROCESS-COMMAND-REPLY NIL
				     (PROG1 (NCONC CONTINUED-PKTS (NCONS P))
					    (SETQ CONTINUED-PKTS NIL)))))
	     (OTHERWISE (FERROR NIL "Unexpected packet type"))))
	 ;; Now look at the input or eof packet.
	 (IF (LISTP PKT)
	     ;; If we have arrived at a node-mark (FILE-NEXT-NODE-OPCODE pkt's data),
	     ;; then put it back on the list; wait for user to do :NEXT-NODE on the stream.
	     (PROGN (PUSH PKT PREREAD-PKTS) NIL)
	   (COND ((= (CHAOS:PKT-OPCODE PKT)
		     (IF (= CHAOS-BYTE-SIZE 8)
			 FILE-CHARACTER-OPCODE
		       FILE-BINARY-OPCODE))
		  (GET-PKT-BUFFER PKT CHAOS-BYTE-SIZE))
		 ((= (CHAOS:PKT-OPCODE PKT)
		     FILE-EOF-OPCODE)
		  (CHAOS:RETURN-PKT PKT)
		  (SETQ STATE ':EOF
			PKT NIL))
		 (T (FERROR NIL "unexpected packet type")))))))

(DEFMETHOD (REMOTE-INPUT-STREAM :DISCARD-INPUT-BUFFER) (IGNORE)
  (AND PKT (CHAOS:RETURN-PKT (PROG1 PKT (SETQ PKT NIL)))))

(DEFMETHOD (REMOTE-INPUT-STREAM :SET-BUFFER-POINTER) (NEW-POINTER)
  (COND ((NEQ STATE ':CLOSED)
	 (COND ((EQ (CAR (AWAIT-COMMAND-REPLY
			  (FILE-SEND-COMMAND 'SET-POINTER NEW-POINTER T)
			  T))
		    T)
		(SETQ STATE ':OPEN)
		NEW-POINTER)
	       (T (SETQ STATE ':CLOSED)
		  NEW-POINTER)))	;Is this right?
	(T (FERROR NIL "stream not open"))))

(DEFFLAVOR REMOTE-CHARACTER-INPUT-STREAM
	()
	(SI:BUFFERED-LINE-INPUT-STREAM REMOTE-INPUT-STREAM))

(DEFFLAVOR REMOTE-OUTPUT-STREAM
	(PKT BUFFER-MIN-INDEX)
	(SI:BUFFERED-OUTPUT-STREAM SI:OUTPUT-POINTER-REMEMBERING-MIXIN
				   REMOTE-DATA-STREAM))

(DEFMETHOD (REMOTE-OUTPUT-STREAM :BEFORE :SET-BYTE-SIZE) (&REST IGNORE)
  (SETQ SI:OUTPUT-POINTER-BASE 0))

(DEFMETHOD (REMOTE-OUTPUT-STREAM :DISCARD-OUTPUT-BUFFER) (IGNORE)
  (AND PKT (CHAOS:RETURN-PKT (PROG1 PKT (SETQ PKT NIL)))))

(DEFMETHOD (REMOTE-OUTPUT-STREAM :SEND-OUTPUT-BUFFER) (BUFFER END-INDEX)
  BUFFER
  ;; Detect any waiting lossage or winnage packets.
  (DO () (())
    (AWAIT-COMMAND-REPLY NIL NIL T)
    ;; Get error now if losing.
    (IF LOSSAGE
	(CERROR T NIL (CAR LOSSAGE) (CADR LOSSAGE))
      (RETURN)))
  (LET ((N (- END-INDEX BUFFER-MIN-INDEX)))
    (COND ((NULL PKT))
	  ((ZEROP N)
	   (CHAOS:RETURN-PKT PKT))
	  (T
	   (SELECTQ CHAOS-BYTE-SIZE
	     (8 (SETF (CHAOS:PKT-NBYTES PKT) N)
		(STORE-ARRAY-LEADER N (CHAOS:PKT-STRING PKT) 0)	;Necessary??
		(CHAOS:SEND-PKT CONNECTION PKT FILE-CHARACTER-OPCODE))	;Character??
	     (16. (SETF (CHAOS:PKT-NBYTES PKT) (* N 2))
		  (CHAOS:SEND-PKT CONNECTION PKT FILE-BINARY-OPCODE)))))
    (SETQ PKT NIL)))

(DEFMETHOD (REMOTE-OUTPUT-STREAM :NEW-OUTPUT-BUFFER) ()
  (SETQ PKT (CHAOS:GET-PKT))
  (MULTIPLE-VALUE-BIND (BUFFER BUFFER-MIN BUFFER-MAX)
      (GET-PKT-BUFFER PKT CHAOS-BYTE-SIZE T)
    (SETQ BUFFER-MIN-INDEX BUFFER-MIN)
    (VALUES BUFFER BUFFER-MIN BUFFER-MAX)))

(DEFMETHOD (REMOTE-OUTPUT-STREAM :NEXT-NODE) (PATHNAME)
  (FILE-SEND-COMMAND 'NEXT-NODE PATHNAME))

(DEFFLAVOR REMOTE-CHARACTER-OUTPUT-STREAM
	()
	(SI:LINE-OUTPUT-STREAM-MIXIN REMOTE-OUTPUT-STREAM))

(COMPILE-FLAVOR-METHODS REMOTE-CHARACTER-OUTPUT-STREAM REMOTE-OUTPUT-STREAM
			REMOTE-CHARACTER-INPUT-STREAM REMOTE-INPUT-STREAM
			REMOTE-PROBE-STREAM)

;; Thisonly needs to work for the byte sizes used for chaos connections: 8 and 16.
(DEFUN GET-PKT-BUFFER (PKT BYTE-SIZE &OPTIONAL OUTPUT-P)
  (DECLARE (RETURN-LIST BUFFER MIN MAX))
  (SELECTQ BYTE-SIZE
    (10 (VALUES (CHAOS:PKT-STRING PKT)	;Leader doesn't matter
		0
		(IF OUTPUT-P
		    CHAOS:MAX-DATA-BYTES-PER-PKT
		    (CHAOS:PKT-NBYTES PKT))))
    (20 (VALUES PKT
		CHAOS:FIRST-DATA-WORD-IN-PKT
		(IF OUTPUT-P
		    CHAOS:MAX-WORDS-PER-PKT	;Is this right??
		    (+ CHAOS:FIRST-DATA-WORD-IN-PKT
		       (// (CHAOS:PKT-NBYTES PKT) 2)))))
    (OTHERWISE (FERROR NIL "Chaos connection byte size not 8 or 16."))))

;;; Send file commands and process replies to them.

;Send a file command to the server.
;Returns the seqnum which the command had,
;so you can wait for the reply.

(DEFUN FILE-SEND-COMMAND (&REST COMMAND-AND-ARGS)
  (DECLARE-FLAVOR-INSTANCE-VARIABLES (REMOTE-DATA-STREAM)
    (LET ((USE-STANDARD-SYMBOL-ENCODING T)
	  (SEQNUM (WITHOUT-INTERRUPTS
		   (PROG1 NEXT-SEQNUM
			  (SETQ NEXT-SEQNUM (1+ NEXT-SEQNUM))))))
      ;; Detect any waiting lossage or winnage packets.
      (DO () (())
	(AWAIT-COMMAND-REPLY NIL NIL T)
	;; Get error now if losing.
	(IF LOSSAGE
	    (CERROR T NIL (CAR LOSSAGE) (CADR LOSSAGE))
	  (RETURN)))
      (FILE-SEND-MESSAGE CONNECTION FILE-COMMAND-OPCODE (CONS SEQNUM COMMAND-AND-ARGS))
      SEQNUM)))

;Save the reply for command number <seqnum> when it arrives,
;but don't wait for it now.
(DEFUN SAVE-COMMAND-REPLY (SEQNUM)
  (DECLARE-FLAVOR-INSTANCE-VARIABLES (REMOTE-DATA-STREAM)
    (PUSH SEQNUM SAVE-REPLY-SEQNUMS)))

;Wait for a reply for command number <seqnum>.
;This requires reading past all input until the reply arrives,
;unless you say to discard the input instead.
;Replies to other command numbers are discarded unless
;they are for numbers which are on save-reply-seqnums.
;returns list of values

;Uses WITH-SELF-VARIABLES-BOUND rather than DECLARE-FLAVOR-INSTANCE-VARIABLES
;so we can be called with those variables bound specially, with SELF irrelevant.
(DEFUN AWAIT-COMMAND-REPLY (SEQNUM &OPTIONAL DISCARD-INPUT DONT-HANG UNTIL-DATA-PKT)
  (WITH-SELF-VARIABLES-BOUND
   (LOCAL-DECLARE ((SPECIAL SAVED-COMMAND-REPLIES PREREAD-PKTS CONTINUED-PKTS
			    CONNECTION LOSSAGE))
    (LET ((TEM (ASSQ SEQNUM SAVED-COMMAND-REPLIES)))
      (COND (TEM (SETQ SAVED-COMMAND-REPLIES
		       (DELQ TEM SAVED-COMMAND-REPLIES))
		 (CDDR TEM))				;Return values, flush seqnum and command.
	    (T (COND (DISCARD-INPUT
		      (DOLIST (P PREREAD-PKTS)
			(CHAOS:RETURN-PKT P))
		      (SETQ PREREAD-PKTS NIL)))
	       (DO ((P)) (())
		 (IF (NOT (CHAOS:DATA-AVAILABLE CONNECTION))
		     (PROGN (COND (DONT-HANG (RETURN))
				  (LOSSAGE
				   (CERROR T NIL (CAR LOSSAGE) (CADR LOSSAGE))))
			    (PROCESS-WAIT "File Command" 'CHAOS:DATA-AVAILABLE CONNECTION)))
		 (SETQ P (CHAOS:GET-NEXT-PKT CONNECTION))
		 (SELECT (CHAOS:PKT-OPCODE P)
		   ((FILE-CHARACTER-OPCODE FILE-BINARY-OPCODE FILE-EOF-OPCODE)
		    (COND ((NOT DISCARD-INPUT)
			   (SETQ PREREAD-PKTS
				 (NCONC PREREAD-PKTS (NCONS P))))
			  (T (CHAOS:RETURN-PKT P)))
		    (IF UNTIL-DATA-PKT (RETURN T)))
		   (FILE-CONTINUED-OPCODE
		    (SETQ CONTINUED-PKTS
			  (NCONC CONTINUED-PKTS (NCONS P))))
		   ((FILE-LOSSAGE-OPCODE FILE-WINNAGE-OPCODE
					 FILE-REPLY-OPCODE FILE-NEXT-NODE-OPCODE
					 CHAOS:CLS-OP)
		    (SETQ TEM (PROCESS-COMMAND-REPLY
				SEQNUM
				(PROG1 (NCONC CONTINUED-PKTS (LIST P))
				       (SETQ CONTINUED-PKTS NIL))))
		    (IF TEM (RETURN TEM))
		    (AND UNTIL-DATA-PKT PREREAD-PKTS
			 (RETURN T)))
		   (OTHERWISE (FERROR NIL "Unexpected packet opcode"))))))))))

;Process one reply.  
;if SEQNUM is nil, we are processing a reply encountered while
;we were looking for data.  If SEQNUM is a number, then we were
;actively looking for a reply for that seqnum. If so, we return the list of values
;as per the reply, normally at least two, the success value and string for the user.
;If SEQNUM is a number, we return NIL if the packet we handled
;was not a reply for that seqnum.

(DEFUN PROCESS-COMMAND-REPLY (SEQNUM FILE-CONTINUED-PKTS)
  (DECLARE (SPECIAL STATE LOSSAGE SAVED-COMMAND-REPLIES SAVE-REPLY-SEQNUMS
		    PREREAD-PKTS AUTO-NEXT-NODE))
   (LET ((FIRST-PKT (CAR FILE-CONTINUED-PKTS))
	 (USE-STANDARD-SYMBOL-ENCODING T))
     (COND ((= (CHAOS:PKT-OPCODE FIRST-PKT) CHAOS:CLS-OP)
	    (LET ((STRING (STRING-APPEND (CHAOS:PKT-STRING FIRST-PKT))))
	      (CHAOS:RETURN-PKT FIRST-PKT)
	      (SETQ STATE ':CLOSED)
	      (FERROR NIL "LMFILE connection closed: ~A" STRING)))
	   ((= (CHAOS:PKT-OPCODE FIRST-PKT) FILE-WINNAGE-OPCODE)
	    (SETQ LOSSAGE NIL))
	   (T
	    ;; An opcode containing data which is a list,
	    ;; and using preceding continued packets.
	    ;; file-lossage-opcode or file-next-node-opcode or file-reply-opcode.
	    (MULTIPLE-VALUE-BIND (MSG OPCODE)
	         (FILE-EXTRACT-LIST)
	      (RETURN-CONTINUED-PKTS)		;any left over.
	      (SELECT OPCODE
		(FILE-NEXT-NODE-OPCODE
		 (IF (NOT AUTO-NEXT-NODE)
		     (SETQ PREREAD-PKTS
			   (NCONC PREREAD-PKTS
				  (NCONS (LIST MSG))))))
		(FILE-LOSSAGE-OPCODE
		 (SETQ LOSSAGE (CONS (CAR MSG) (CDDR MSG)))
		 NIL)
		(FILE-REPLY-OPCODE
		 (COND ((AND SEQNUM (= SEQNUM (CAR MSG)))
			(CDDR MSG))		;Return list of values.  Flush seqnum and command.
		       ((MEMQ (CAR MSG) SAVE-REPLY-SEQNUMS)
			(SETQ SAVE-REPLY-SEQNUMS
			     (DELQ (CAR MSG) SAVE-REPLY-SEQNUMS))
			(PUSH MSG SAVED-COMMAND-REPLIES)
			NIL)
		       ))))))))

;Open a connection to the file server and send the first command,
;either in the RFC packet, or in a separate FILE-COMMAND-OPCODE
;packet after the connection is open.
;We return the connection, or a string saying why it couldn't open.
;You must wait for the reply yourself.

(DEFVAR OPEN-SERVER-CONN-TEM-ARRAY
	(MAKE-ARRAY CHAOS:MAX-DATA-BYTES-PER-PKT ':TYPE 'ART-STRING ':LEADER-LIST '(0)))

(DEFVAR OPEN-SERVER-CONN-TEM-ARRAY-LOCK NIL)

(DEFUN OPEN-SERVER-CONNECTION (CHAOS-ADDRESS &REST ARGS &AUX CONN LOCK THROW)
  (SETQ LOCK (LOCF OPEN-SERVER-CONN-TEM-ARRAY-LOCK))
  (UNWIND-PROTECT
    (FILE-BIND-FOR-WRITE
      (PROCESS-LOCK LOCK)
      (LET ((FILE-RFC-STREAM-OUT-PKT-STRING OPEN-SERVER-CONN-TEM-ARRAY))
	(STORE-ARRAY-LEADER 0 FILE-RFC-STREAM-OUT-PKT-STRING 0)
	(MULTIPLE-VALUE (NIL THROW)
	  (*CATCH 'RFC-FAIL
	    (FORMAT 'FILE-RFC-STREAM-OUT "LMFILE ~A ~S" USER-ID ARGS)))
	(SETQ CONN
	      (CONDITION-CASE (ERROR)
	           (CHAOS:CONNECT CHAOS-ADDRESS
				  (IF THROW (FORMAT NIL "LMFILE ~A (-1 NIL)" USER-ID)
				      FILE-RFC-STREAM-OUT-PKT-STRING))
		(SYS:REMOTE-NETWORK-ERROR (SEND ERROR ':STRING-FOR-PRINTING))))))
    (PROCESS-UNLOCK LOCK))
  (AND (NOT (STRINGP CONN))
       THROW
       (FILE-SEND-MESSAGE CONN FILE-COMMAND-OPCODE ARGS))
  CONN)

;;; REMOTE-DATA-PATHNAME-MIXIN -- for pathnames which parse locally
;;; but which open remote streams (and use file-commands for the operations
;;; that act on the file).

(DEFFLAVOR REMOTE-DATA-PATHNAME-MIXIN (SERVER-CHAOS-ADDRESS SERVER-LOCAL-HOST)
	   (DIRECTORY-STREAM-MIXIN REMOTE-PATHNAME)
  (:SPECIAL-INSTANCE-VARIABLES SERVER-LOCAL-HOST)
  (:INITABLE-INSTANCE-VARIABLES SERVER-CHAOS-ADDRESS SERVER-LOCAL-HOST))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :READ-DIRECTORY-STREAM-ENTRY) (STREAM)
  (LET ((READTABLE FILE-READTABLE)
	(USE-STANDARD-SYMBOL-ENCODING T))
    (READ-DIRECTORY-STREAM-ENTRY STREAM SELF)))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :DIRECTORY-STREAM-DEFAULT-PARSER) ()
  #'REMOTE-PARSE-RANDOM-SEXP)

;Like PARSE-RANDOM-SEXP but uses FILE-READTABLE
;which defines #P and #S.
(DEFUN REMOTE-PARSE-RANDOM-SEXP (STRING START)
  (LET ((IBASE 10.)
	(READTABLE FILE-READTABLE)
	(PACKAGE SI:PKG-USER-PACKAGE))
    (READ-FROM-STRING STRING NIL START)))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :OPEN) (PATHNAME &REST KEYWORD-ARGS
					       &KEY &OPTIONAL (ERROR T)
					       (ACCESS-ERROR (NOT ERROR))
					       (AUTO-NEXT-NODE T)
					       &ALLOW-OTHER-KEYS
					       &AUX -TYPE- STREAM-PLIST WHICH-OPERATIONS
					       PLIST TEXT-FLAG FLAVOR STREAM)

  (MULTIPLE-VALUE-BIND (CONN ANS)
      (SEND-PATHNAME-COMMAND-1 ACCESS-ERROR ERROR ':OPEN HOST 'OPEN KEYWORD-ARGS)
    (IF (ERRORP CONN) CONN
      (SETF (LIST -TYPE- WHICH-OPERATIONS STREAM-PLIST 'NIL PLIST TEXT-FLAG)
	    (CDR ANS))
      (SETQ FLAVOR (IF TEXT-FLAG
		       (SELECTQ -TYPE-
			 (INPUT 'REMOTE-CHARACTER-INPUT-STREAM)
			 (OUTPUT 'REMOTE-CHARACTER-OUTPUT-STREAM)
			 (PROBE 'REMOTE-PROBE-STREAM))
		     (SELECTQ -TYPE-
		       (INPUT 'REMOTE-INPUT-STREAM)
		       (OUTPUT 'REMOTE-OUTPUT-STREAM)
		       (PROBE 'REMOTE-PROBE-STREAM))))
      (IF (EQ FLAVOR 'REMOTE-PROBE-STREAM)
	  (PROGN
	    (SETQ STREAM (MAKE-INSTANCE FLAVOR
					':WHICH-OPERATIONS WHICH-OPERATIONS
					':STREAM-PLIST STREAM-PLIST
					':PLIST PLIST
					':PATHNAME PATHNAME
					':TRUENAME (GET (LOCF STREAM-PLIST) ':TRUENAME)))
	    (CHAOS:CLOSE CONN))
	(SETQ STREAM (MAKE-INSTANCE FLAVOR
				    ':WHICH-OPERATIONS WHICH-OPERATIONS
				    ':READ-PATHNAME-HOST HOST
				    ':SERVER-LOCAL-HOST SERVER-LOCAL-HOST
				    ':CONNECTION CONN
				    ':AUTO-NEXT-NODE AUTO-NEXT-NODE
				    ':STREAM-PLIST STREAM-PLIST
				    ':PLIST PLIST
				    ':PATHNAME PATHNAME
				    ':TRUENAME (GET (LOCF STREAM-PLIST) ':TRUENAME)
				    ':CHAOS-BYTE-SIZE
				    (IF (<= (GET (LOCF STREAM-PLIST) ':BYTE-SIZE) 8)
					8 16.)
				    ':FILE-STREAM-TYPE -TYPE-))
	(FUNCALL HOST ':SEND-IF-HANDLES ':ADD-STREAM STREAM))
      STREAM)))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :RENAME) (NEW-NAME &OPTIONAL (ERROR-P T))
  (AND (STRINGP NEW-NAME)
       (SETQ NEW-NAME (PARSE-PATHNAME NEW-NAME HOST)))
  (SEND-PATHNAME-COMMAND HOST ERROR-P ':RENAME 'RENAME NEW-NAME))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :MAKE-LINK) (TO-NAME &OPTIONAL (ERROR-P T))
  (AND (STRINGP TO-NAME)
       (SETQ TO-NAME (PARSE-PATHNAME TO-NAME HOST)))
  (SEND-PATHNAME-COMMAND HOST ERROR-P ':MAKE-LINK 'MAKE-LINK TO-NAME))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :CHANGE-PROPERTIES)
           (&OPTIONAL (ERROR-P T) &REST PROPERTIES)
  (SEND-PATHNAME-COMMAND HOST ERROR-P ':CHANGE-PROPERTIES
			 'CHANGE-PROPERTIES ERROR-P PROPERTIES))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :PROPERTIES) (&OPTIONAL (ERROR-P T))
  (SEND-PATHNAME-COMMAND HOST ERROR-P ':PROPERTIES 'PROPERTIES ERROR-P))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :EXPUNGE) (&KEY &OPTIONAL (ERROR T))
  (SEND-PATHNAME-COMMAND HOST ERROR ':EXPUNGE 'EXPUNGE ERROR))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :DELETE) (&OPTIONAL (ERROR-P T))
  (SEND-PATHNAME-COMMAND HOST ERROR-P ':DELETE 'DELETE ERROR-P))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :UNDELETE) (&OPTIONAL (ERROR-P T))
  (SEND-PATHNAME-COMMAND HOST ERROR-P ':UNDELETE 'UNDELETE ERROR-P))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :DIRECTORY-STREAM) (&OPTIONAL OPTIONS)
  (LET* ((NOERROR (MEMQ ':NOERROR OPTIONS))
	 (CONN
	   (SEND-PATHNAME-COMMAND HOST (NOT NOERROR) ':DIRECTORY-STREAM
				  'DIRECTORY-STREAM OPTIONS))
	 STREAM)
    (IF (ERRORP CONN) CONN
      (SETQ STREAM (MAKE-INSTANCE 'REMOTE-CHARACTER-INPUT-STREAM
		     ':WHICH-OPERATIONS NIL
		     ':READ-PATHNAME-HOST HOST
		     ':SERVER-LOCAL-HOST SERVER-LOCAL-HOST
		     ':CONNECTION CONN
		     ':STREAM-PLIST '(:CHARACTERS T :BYTE-SIZE 8)
		     ':PLIST NIL
		     ':PATHNAME SELF
		     ':TRUENAME NIL
		     ':CHAOS-BYTE-SIZE 8
		     ':FILE-STREAM-TYPE ':INPUT))
      (FUNCALL HOST ':SEND-IF-HANDLES ':ADD-STREAM STREAM)
      STREAM)))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :HOMEDIR) ()
  (LET ((PATH (MAKE-PATHNAME-INTERNAL HOST NIL USER-ID NIL NIL NIL)))
    (OR (FUNCALL PATH ':GET ':HOMEDIR)
	(FUNCALL PATH ':PUTPROP
		 (SEND-PATHNAME-COMMAND HOST T ':HOMEDIR 'HOMEDIR)
		 ':HOMEDIR))))


(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :ALL-DIRECTORIES) (OPTIONS)
  (SEND-PATHNAME-COMMAND HOST (NOT (MEMQ ':NOERROR OPTIONS)) ':ALL-DIRECTORIES
			 'ALL-DIRECTORIES))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :COMPLETE-STRING) (STRING OPTIONS)
  (LET ((VALUES
	  (SEND-PATHNAME-COMMAND HOST T ':COMPLETE-STRING 'COMPLETE-STRING STRING OPTIONS)))
    ;; We expect a list (string success-flag) back from the server.
    (VALUES (STRING-APPEND (FUNCALL HOST ':NAME-AS-FILE-COMPUTER)
			   ": "
			   (CAR VALUES))
	    (CADR VALUES))))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :MULTIPLE-FILE-PLISTS) (PATHNAMES &OPTIONAL OPTIONS)
  OPTIONS
  (FUNCALL-SELF ':MULTIPLE-FILE-PROPERTY-LISTS NIL PATHNAMES))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :MULTIPLE-FILE-PROPERTY-LISTS)
	   (IGNORE PATHNAMES &AUX TRANSLATED-PATHNAMES ANS)
  (SETQ TRANSLATED-PATHNAMES
	(LOOP FOR PATHNAME IN PATHNAMES COLLECTING (FUNCALL PATHNAME ':TRANSLATED-PATHNAME)))
  (SETQ ANS (SEND-PATHNAME-COMMAND HOST T ':PROPERTIES
				   'MULTIPLE-FILE-PROPERTY-LISTS TRANSLATED-PATHNAMES))
  (LOOP FOR PATHNAME IN PATHNAMES
	FOR PLIST IN ANS
	DOING (RPLACA PLIST PATHNAME))
  ANS)

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :UNDELETE-MULTIPLE-FILES) (IGNORE PATHNAMES)
  (SEND-PATHNAME-COMMAND HOST T ':UNDELETE-MULTIPLE-FILES
			 'UNDELETE-MULTIPLE-FILES PATHNAMES))

(DEFMETHOD (REMOTE-DATA-PATHNAME-MIXIN :DELETE-MULTIPLE-FILES) (IGNORE PATHNAMES)
  (SEND-PATHNAME-COMMAND HOST T ':DELETE-MULTIPLE-FILES 'DELETE-MULTIPLE-FILES PATHNAMES))

;;; REMOTE-PARSING-MIXIN -- not only the data but the parsing and printing
;;; are done on the remote host.

(DEFFLAVOR REMOTE-PARSING-MIXIN () (REMOTE-DATA-PATHNAME-MIXIN))

(DEFFLAVOR REMOTE-PARSING-PATHNAME () (REMOTE-PARSING-MIXIN HOST-PATHNAME))

;; The next 4 messages are the PARSE messages, which request the remote machine
;; to do the parsing.  Normally this is done locally.
(DEFMETHOD (REMOTE-PARSING-PATHNAME :STRING-FOR-PRINTING) ()
  (STRING-APPEND HOST #/: (SUBSTRING-CHAR-AND-AFTER
			    #/: (FUNCALL-SELF ':STRING-FOR-PRINTING))))

(DEFUN SUBSTRING-CHAR-AND-AFTER (CHAR STRING &OPTIONAL (AREA NIL))
       (LET ((IDX (STRING-SEARCH-CHAR CHAR STRING)))
	 (COND ((NULL IDX) "")
	       (T (SUBSTRING STRING IDX NIL AREA)))))

(DEFMETHOD (REMOTE-PARSING-PATHNAME :STRING-FOR-EDITOR) ()
  (LET* ((STR 
	   (SEND-PATHNAME-COMMAND HOST T ':STRING-FOR-EDITOR 'PARSE ':STRING-FOR-EDITOR))
	 (COLON (STRING-REVERSE-SEARCH-CHAR #/: STR))
	 (SPACE (STRING-REVERSE-SEARCH-CHAR #\SP STR COLON)))
    (STRING-APPEND (SUBSTRING STR 0 (1+ SPACE))
		   HOST
		   (SUBSTRING STR COLON))))

(DEFMETHOD (REMOTE-PARSING-PATHNAME :STRING-FOR-DIRED) ()
  (SEND-PATHNAME-COMMAND HOST T ':STRING-FOR-DIRED 'PARSE ':STRING-FOR-DIRED))

(DEFMETHOD (REMOTE-PARSING-PATHNAME :PARSE-NAMESTRING) (HOST-SPECIFIED NAMESTRING
							&OPTIONAL (START 0) END)
  (SEND-PATHNAME-COMMAND HOST T ':PARSE-NAMESTRING 'PARSE ':PARSE-NAMESTRING
			 HOST-SPECIFIED NAMESTRING START END))

(DEFUN SEND-PATHNAME-COMMAND (READ-PATHNAME-HOST ERROR-P COMMAND-FOR-ERROR COMMAND &REST ARGS)
    (LEXPR-FUNCALL 'SEND-PATHNAME-COMMAND-1 T ERROR-P COMMAND-FOR-ERROR
		   READ-PATHNAME-HOST COMMAND ARGS))

(DEFUN SEND-PATHNAME-COMMAND-1 (ACCESS-ERROR-P ERROR-P COMMAND-FOR-ERROR
				READ-PATHNAME-HOST COMMAND &REST ARGS
				&AUX CONN ANS (USE-STANDARD-SYMBOL-ENCODING T))
  (DECLARE (:SELF-FLAVOR REMOTE-DATA-PATHNAME-MIXIN))
  (FILE-OPERATION-RETRY
    (PROG FUNCTION ()
	  (CONDITION-CASE-IF (NOT ACCESS-ERROR-P) (ERROR)
	      (SETQ CONN
		    (LEXPR-FUNCALL 'OPEN-SERVER-CONNECTION
				   SERVER-CHAOS-ADDRESS
				   0
				   COMMAND
				   SELF
				   ARGS))
	    (REMOTE-NETWORK-ERROR (RETURN ERROR)))
	  ;; Just bind the instance variables rather than cons up a stream.
	  ;; If we need a stream, we'll cons it up later.
	  (LOCAL-DECLARE ((SPECIAL CONNECTION SAVE-REPLY-SEQNUMS SAVED-COMMAND-REPLIES
				   CONTINUED-PKTS PREREAD-PKTS LOSSAGE))
	    (LET (SELF
		  (CONNECTION CONN)
		  (SAVE-REPLY-SEQNUMS NIL)
		  (SAVED-COMMAND-REPLIES NIL)
		  (CONTINUED-PKTS NIL)
		  (PREREAD-PKTS)
		  (LOSSAGE NIL))
	      (UNWIND-PROTECT
		(SETQ ANS (AWAIT-COMMAND-REPLY 0))
		(OR (EQ (FIRST ANS) 'STREAM)	;Close connection unless answer indicates
		    (CHAOS:CLOSE CONN)))))	;to keep it open.
	  (RETURN
	    (SELECTQ (FIRST ANS)
	      (STREAM (VALUES CONN ANS))
	      (ERROR (FILE-PROCESS-ERROR-NEW (SECOND ANS) SELF NIL
					     (NOT ERROR-P) COMMAND-FOR-ERROR))
	      (OTHERWISE (VALUES-LIST (SECOND ANS))))))))

;;; Define the host objects for various sorts of pathnames.

(DEFFLAVOR REMEMBER-STREAMS-MIXIN
	((OPEN-STREAMS NIL))
	()
  :GETTABLE-INSTANCE-VARIABLES)

(DEFMETHOD (REMEMBER-STREAMS-MIXIN :ADD-STREAM) (STREAM)
  (WITHOUT-INTERRUPTS
   (PUSH STREAM OPEN-STREAMS)))

(DEFMETHOD (REMEMBER-STREAMS-MIXIN :REMOVE-STREAM) (STREAM)
  (WITHOUT-INTERRUPTS
   (SETQ OPEN-STREAMS (DELQ STREAM OPEN-STREAMS))))

;; Support for LMFS.
(DEFFLAVOR RM-PATHNAME
	()
	(LM-PARSING-MIXIN REMOTE-DATA-PATHNAME-MIXIN REMOTE-PATHNAME))

(DEFMETHOD (RM-PATHNAME :AFTER :INIT) (IGNORE)
  (SETQ SERVER-CHAOS-ADDRESS HOST
	SERVER-LOCAL-HOST (FUNCALL HOST ':NAME-AS-FILE-COMPUTER)))

;; Support for LFS.
(DEFFLAVOR REMOTE-LISPM-LMFILE-HOST
	()
	(REMOTE-LISPM-FILE-MIXIN SI:LISPM-CHAOS-HOST))

(DEFFLAVOR REMOTE-LMFILE-PATHNAME ()
	   (LMFILE-PARSING-MIXIN REMOTE-DATA-PATHNAME-MIXIN))

(DEFFLAVOR FC-HOST
	(DEVICE-NAME
	  CHAOS-ADDRESS
	  DISTANT-DEVICE)
	(SI:BASIC-HOST REMEMBER-STREAMS-MIXIN)
  (:GETTABLE-INSTANCE-VARIABLES DEVICE-NAME CHAOS-ADDRESS DISTANT-DEVICE)
  (:INITABLE-INSTANCE-VARIABLES DEVICE-NAME CHAOS-ADDRESS DISTANT-DEVICE))

(DEFMETHOD (FC-HOST :NAME) NIL DEVICE-NAME)
(DEFMETHOD (FC-HOST :PATHNAME-FLAVOR) ()
  (VALUES 'REMOTE-LMFILE-PATHNAME `(:SERVER-LOCAL-HOST ,DISTANT-DEVICE
				    :SERVER-CHAOS-ADDRESS ,CHAOS-ADDRESS)))

(DEFMETHOD (FC-HOST :SYSTEM-TYPE) () ':LMFILE)

(DEFUN ADD-FC-HOST (NAME CHAOS-ADDRESS)
  (COND ((NULL (GET-PATHNAME-HOST NAME T))
	 (LET ((HOST (MAKE-INSTANCE 'FC-HOST ':DEVICE-NAME NAME
				    ':CHAOS-ADDRESS CHAOS-ADDRESS
				    ':DISTANT-DEVICE NAME)))
	   (PUSH HOST *PATHNAME-HOST-LIST*)))))

(COMPILE-FLAVOR-METHODS RM-PATHNAME REMOTE-LISPM-LMFILE-HOST
			REMOTE-LMFILE-PATHNAME FC-HOST)

(DEFUN ADD-SITE-FC-HOSTS ()
  (MAPC #'(LAMBDA (NAME-AND-ADDRESS) (APPLY #'ADD-FC-HOST NAME-AND-ADDRESS))
	(SI:GET-SITE-OPTION ':CHAOS-LMFILE-SERVER-HOSTS)))

(ADD-INITIALIZATION "Add Site FC Hosts" '(ADD-SITE-FC-HOSTS) '(:SITE :NOW))
