;;; -*- Mode:LISP; Package:FILE-SYSTEM; Base:10; Readtable:T -*-
;;; (c) 1984 Massachusetts Institute of Technology
;;;
;;; This is SYS: IO; FILE; ACCESS
;;;
;;; Flavor definitions for file access.  Also includes useful functions and variables that
;;; are needed by various kinds of access.
;;; Definitions include the FILE-HOST-MIXIN, basic access objects, and useful flavors of 
;;; host units.  Aids for error handling/signalling, password guessing (your own, silly !).
;;; ``Data Connection'' defstruct.  Host flavors for LispM-supported file systems.
;;; Dormant file service connection cleanup process.  For examples of using these functions
;;; and flavors, look at SYS: NETWORK; CHAOS; QFILE

(DEFMACRO FILE-OPERATION-RETRY (&BODY BODY)
  "Like ERROR-RESTART for condition FILE-ERROR, but uses proceed-type :RETRY-FILE-OPERATION."
  (LET* ((TAG (GENSYM)))
    `(BLOCK FILE-OPERATION-RETRY
       (TAGBODY
	RETRY
	   (RETURN-FROM FILE-OPERATION-RETRY
	     (CATCH-CONTINUATION-IF T ',TAG #'(LAMBDA (IGNORE) (GO RETRY)) NIL
	       (WITH-STACK-LIST* (EH:CONDITION-RESUME-HANDLERS
				   '(FILE-ERROR :RETRY-FILE-OPERATION T
						   ("Retry the operation on the same file.")
						   SI:CATCH-ERROR-RESTART-THROW ,TAG)
				   EH:CONDITION-RESUME-HANDLERS)
	         . ,BODY)))))))

;;; This is the ``library'' function that should be used by protocol interfaces
;;; CONDITION-NAME is a condition name, and ERROR-STRING is a string decribing just the
;;; that error, not any files involved.  The first of MAKE-CONDITION-ARGS is the operation
;;; name; extra arguments are dependent on what CONDITION-NAME provides.  If
;;; CONDITION-NAME is NIL, then the condition FS:FILE-OPERATION-FAILURE-1 is used.
;;; PROCEEDABLE can be a list of proceed types, NIL, or an non-NIL atom (usually T) meaning
;;; to use the proceed type :RETRY-FILE-OPERATION.
(DEFUN FILE-PROCESS-ERROR (CONDITION-NAME ERROR-STRING &OPTIONAL PATHNAME-OR-STREAM
			   PROCEEDABLE NOERROR &REST MAKE-CONDITION-ARGS &AUX WHO-FOR)
  (TYPECASE PATHNAME-OR-STREAM
    (PATHNAME (SETQ WHO-FOR PATHNAME-OR-STREAM))
    (SI:FILE-STREAM-MIXIN (SETQ WHO-FOR (SEND PATHNAME-OR-STREAM :PATHNAME)))
    (T (SETQ WHO-FOR PATHNAME-OR-STREAM)))
  (AND WHO-FOR
       (SETQ ERROR-STRING (STRING-APPEND ERROR-STRING " for " WHO-FOR)))
  (LET ((CONDITION (APPLY #'MAKE-CONDITION (OR CONDITION-NAME 'FILE-OPERATION-FAILURE-1)
			  "~A"  WHO-FOR MAKE-CONDITION-ARGS)))
    (SEND CONDITION :SET-FORMAT-ARGS (LIST ERROR-STRING))
    (IF NOERROR CONDITION
      (SIGNAL CONDITION :PROCEED-TYPES
	      (COND ((CONSP PROCEEDABLE) PROCEEDABLE)
		    (PROCEEDABLE '(:RETRY-FILE-OPERATION)))))))
  
(DEFFLAVOR FILE-HOST-MIXIN
	   ((ACCESS NIL)			;Contains HOST-UNIT's
	    (APPROPRIATE-ACCESS-FLAVORS NIL)	;In case of failure
	    (PATHNAME-HASH-TABLE))
	()
  (:REQUIRED-METHODS :HSNAME-INFORMATION :SYSTEM-TYPE ; for QFILE.
		     :PATHNAME-FLAVOR)
  (:REQUIRED-INSTANCE-VARIABLES SI:PROPERTY-LIST)
  (:REQUIRED-FLAVORS SI:PROPERTY-LIST-MIXIN)
  (:GETTABLE-INSTANCE-VARIABLES)
  (:REQUIRED-FLAVORS SI:BASIC-HOST))

;;; This method may be overidden by other flavors, or by site information to distinguish
;;; slightly differing versions of file systems.
(DEFMETHOD (FILE-HOST-MIXIN :FILE-SYSTEM-TYPE) ()
  (OR (GETF SI:PROPERTY-LIST 'FILE-SYSTEM-TYPE)
      (SEND SELF :SYSTEM-TYPE)))

(DEFMETHOD (FILE-HOST-MIXIN :FILE-HOST-P) () T)
(DEFMETHOD (FILE-HOST-MIXIN :MAX-DATA-CONNECTIONS) () 20.)

(DEFMETHOD (FILE-HOST-MIXIN :GET-ACCESS) ()
  (OR ACCESS (SETQ ACCESS (SEND SELF :DETERMINE-ACCESS))))

(DEFMETHOD (FILE-HOST-MIXIN :ACCESS-OPERATION) (OP &REST ARGS)
  (LEXPR-SEND (SEND SELF :GET-ACCESS) OP ARGS))

(DEFMETHOD (FILE-HOST-MIXIN :DETERMINE-ACCESS) ()
  (SETQ APPROPRIATE-ACCESS-FLAVORS (DETERMINE-FILE-ACCESS-FLAVORS SELF))
  (IF APPROPRIATE-ACCESS-FLAVORS
      (SETQ ACCESS (MAKE-INSTANCE (FIRST APPROPRIATE-ACCESS-FLAVORS) :HOST SELF))
    (FERROR () "No file access path to ~A available" SELF)))

(DEFMETHOD (FILE-HOST-MIXIN :RETRY-ACCESS) ()
  (DOLIST (F APPROPRIATE-ACCESS-FLAVORS)
    (UNLESS (EQ (TYPE-OF ACCESS) F)
      (RETURN (SETQ ACCESS (MAKE-INSTANCE F :HOST SELF))))))
  
(DEFMETHOD (FILE-HOST-MIXIN :PEEK-FILE-SYSTEM) ()
  (AND ACCESS (SEND ACCESS :PEEK-FILE-SYSTEM)))

(DEFMETHOD (FILE-HOST-MIXIN :PEEK-FILE-SYSTEM-HEADER) ()
  (AND ACCESS (SEND ACCESS :PEEK-FILE-SYSTEM-HEADER)))

(DEFVAR *FILE-ACCESS-PATHS* () "An alist of (ACCESS-FLAVOR DESIRABILITY . CONDITIONS)")

(DEFMACRO DEFINE-FILE-ACCESS (&WHOLE ENTRY ACCESS-FLAVOR DESIRABILITY &BODY CONDITIONS)
  "Define a file access path, to be possibly used by pathnames.
ACCESS-FLAVOR should be a flavor of file access (see FS:BASIC-ACCESS).  It should
accept :HOST as an init keyword.  The desirability is expressed as a flonum in the range
/(0, 1].  CONDITIONS can be any of the following:
 :LOCAL			The host must be local.  Usually the desirability of local access
	should be 1.0.
 (:FILE-SYSTEM-TYPE . types) The file system flavor must be one of the listed types
 (:NETWORK . network-types) The host must be on one of the networks of the listed types
 (:PROTOCOL . protocols) The host must support one of the listed protocols."
  CONDITIONS
  (CHECK-TYPE DESIRABILITY (FLOAT (0) 1))
  `(PROGN
     (SETQ *FILE-ACCESS-PATHS* (DELQ (ASSQ ',ACCESS-FLAVOR *FILE-ACCESS-PATHS*)
				     *FILE-ACCESS-PATHS*))
     ;; Don't use PUSH here - this is used before PUSH is loaded.
     (SETQ *FILE-ACCESS-PATHS* (CONS ',(CDR ENTRY) *FILE-ACCESS-PATHS*))))

(DEFUN PERMISSIBLE-ACCESS-PATH-FLAVOR-P (HOST ACCESS-D &AUX COND-TYPE COND-ARGS)
  (DOLIST (CONDITION (CDDR ACCESS-D) T)
    (UNLESS
      (IF (EQ CONDITION :LOCAL)
	  (OR (EQ SI:LOCAL-HOST HOST)
	      (LET ((CHAOS (SEND HOST :SEND-IF-HANDLES :CHAOS-ADDRESS))) ; KLUDGE!
		(AND CHAOS (= CHAOS (SEND SI:LOCAL-HOST :CHAOS-ADDRESS)))))
	(SETQ COND-TYPE (FIRST CONDITION) COND-ARGS (REST CONDITION))
	(SELECTQ COND-TYPE
	  (:FILE-SYSTEM-TYPE (MEMQ (SEND HOST :FILE-SYSTEM-TYPE) COND-ARGS))
	  (:NETWORK (DOLIST (NETWORK COND-ARGS)
		      (IF (SEND HOST :NETWORK-TYPEP NETWORK) (RETURN T))))
	  (:PROTOCOL T) ; not supported yet
	  (OTHERWISE T)))
      (RETURN NIL))))

(DEFUN DETERMINE-FILE-ACCESS-FLAVORS (HOST &AUX GOOD-FLAVORS)
  (DOLIST (ACCESS-D *FILE-ACCESS-PATHS*)
    (AND (PERMISSIBLE-ACCESS-PATH-FLAVOR-P HOST ACCESS-D)
	 (PUSH ACCESS-D GOOD-FLAVORS)))
  (SETQ GOOD-FLAVORS (SORT GOOD-FLAVORS #'> :KEY #'CADR))
  (MAPCAR #'CAR GOOD-FLAVORS))

(DEFMETHOD (FILE-HOST-MIXIN :HOST-UNITS) ()
  (AND ACCESS (SEND ACCESS :SEND-IF-HANDLES :HOST-UNITS)))

(DEFMETHOD (FILE-HOST-MIXIN :RESET) (&OPTIONAL FORGET-ACCESS-P)
  (WHEN ACCESS
    (SEND ACCESS :RESET)
    (AND FORGET-ACCESS-P (SETQ ACCESS NIL
			       APPROPRIATE-ACCESS-FLAVORS NIL))))

;;; This also frees up any slots marked as open
(DEFMETHOD (FILE-HOST-MIXIN :CLOSE-ALL-FILES) (&OPTIONAL (MODE :ABORT))
  (AND ACCESS (SEND ACCESS :CLOSE-ALL-FILES MODE)))

(DEFMETHOD (FILE-HOST-MIXIN :OPEN-STREAMS) ()
  (AND ACCESS (SEND ACCESS :OPEN-STREAMS)))

(DEFMETHOD (FILE-HOST-MIXIN :RESET-DORMANT-HOST-UNITS) ()
  (AND ACCESS (SEND ACCESS :SEND-IF-HANDLES :RESET-DORMANT-HOST-UNITS)))

(DEFMETHOD (FILE-HOST-MIXIN :ANY-DORMANT-HOST-UNITS-P) ()
  (AND ACCESS (SEND ACCESS :SEND-IF-HANDLES :ANY-DORMANT-HOST-UNITS-P)))

;;; This method is currently QFILE specific
(DEFMETHOD (FILE-HOST-MIXIN :HSNAME-INFORMATION) (UNIT STR IDX)
  (LET* ((HSNAME (SUBSTRING STR (INCF IDX)
			    	(SETQ IDX (STRING-SEARCH-CHAR #/NEWLINE STR IDX))))
	 (DEFAULT-CONS-AREA SYS:BACKGROUND-CONS-AREA)
	 (HSNAME-PATHNAME (SEND SELF :HSNAME-PATHNAME HSNAME (SEND UNIT :HOST)))
	 (PERSONAL-NAME (SUBSTRING STR (INCF IDX)
				       (SETQ IDX (STRING-SEARCH-CHAR #/NEWLINE STR IDX))))
	 (GROUP-AFFILIATION (IF (OR (NULL IDX) (= IDX (1- (STRING-LENGTH STR))))
				#/SP
			        (CHAR STR (1+ IDX)))))
    (SETQ IDX (STRING-SEARCH ", " PERSONAL-NAME)
	  STR (NSUBSTRING PERSONAL-NAME 0 IDX))
    (AND IDX (SETQ STR (STRING-APPEND (NSUBSTRING PERSONAL-NAME (+ IDX 2)) #/SP STR)))
    (VALUES HSNAME-PATHNAME PERSONAL-NAME GROUP-AFFILIATION STR)))

(DEFMETHOD (FILE-HOST-MIXIN :LOGIN-UNIT) (UNIT LOGIN-P)
  ;; Connection is used up when logging out
  (AND (SEND UNIT :VALID-CONTROL-CONNECTION-P)
       (IF LOGIN-P
	   (SEND UNIT :LOGIN LOGIN-P SELF)
	 (SEND UNIT :CLOSE-CONTROL-CONNECTION)))
  T)

;;; Fascist caste system support. Boo Hiss etc
(DEFFLAVOR CASTE-FILE-HOST-MIXIN
	   ()
	   ()
  (:REQUIRED-FLAVORS FILE-HOST-MIXIN)
  (:REQUIRED-METHODS :DEFAULT-CAPABILITIES))

(DEFMETHOD (CASTE-FILE-HOST-MIXIN :ENABLE-CAPABILITIES) (&REST CAPABILITIES)
  (SEND SELF :ACCESS-OPERATION :ENABLE-CAPABILITIES
	(OR CAPABILITIES (SEND SELF :DEFAULT-CAPABILITIES))))

(DEFMETHOD (CASTE-FILE-HOST-MIXIN :DISABLE-CAPABILITIES) (&REST CAPABILITIES)
  (SEND SELF :ACCESS-OPERATION :DISABLE-CAPABILITIES CAPABILITIES))

(DEFUN FORGET-PASSWORD (NEW-USER-ID HOST)
  (LET ((ALIST-ELEMENT (ASS #'EQUALP `(,NEW-USER-ID ,(SEND HOST :NAME))
			    USER-HOST-PASSWORD-ALIST)))
    (IF ALIST-ELEMENT
	(SETQ USER-HOST-PASSWORD-ALIST
	      (DELQ ALIST-ELEMENT USER-HOST-PASSWORD-ALIST)))))

(DEFUN GUESS-PASSWORD-MAYBE (NEW-USER-ID HOST &AUX PASSWORD)
  (OR (CADR (ASS #'EQUALP `(,NEW-USER-ID ,(SEND HOST :NAME)) USER-HOST-PASSWORD-ALIST))
      ;; None remembered => guess, except on Multics
      ;; since multics would hassle the guy if it's wrong.
      (IF (EQ (SEND HOST :SYSTEM-TYPE) :MULTICS) "")
      ;; Try guessing password same as on some other host.
      (CADR (CAR USER-HOST-PASSWORD-ALIST))
      ;; Try guessing password same as uname or last part of it.
      (PROG1
	(SETQ PASSWORD
	      (SUBSTRING NEW-USER-ID
			 (1+ (OR (STRING-REVERSE-SEARCH #/. NEW-USER-ID) -1))))
	(PUSH `((,NEW-USER-ID ,(SEND HOST :NAME)) ,PASSWORD) USER-HOST-PASSWORD-ALIST))))

;;; This function should be useful for the :LOGIN methods of host units.  UNAME-HOST and
;;; HOST are usually the same, unless UNAME-HOST is the symbol FS:ITS
(DEFUN DETERMINE-USER-ID-AND-PASSWORD (UNAME-HOST HOST NEED-PASSWORD
				       &AUX PASSWORD NEW-USER-ID ENABLE-CAPABILITIES)
  (DECLARE (VALUES NEW-USER-ID PASSWORD ENABLE-CAPABILITIES))
  (SETQ NEW-USER-ID (CDR (ASSQ UNAME-HOST USER-UNAMES)))
  (COND ((EQ UNAME-HOST 'ITS)
	 (UNLESS NEW-USER-ID
	   ;; This is an ITS; ask for the user name for all ITSes.
	   (FORMAT *QUERY-IO* "~&ITS uname (default ~A): " USER-ID)
	   (LET ((NID (READLINE-TRIM *QUERY-IO*)))
	     (SETQ NEW-USER-ID (IF (EQUAL NID "") USER-ID NID)))))
	;; Not an ITS: if we don't know user id or if password failed,
	;; ask for one or both.
	((OR NEED-PASSWORD
	     ;(NULL NEW-USER-ID)
	     )
	 (MULTIPLE-VALUE-SETQ (NEW-USER-ID PASSWORD ENABLE-CAPABILITIES)
	   (FILE-GET-PASSWORD USER-ID UNAME-HOST)))
	;; We know the user id; use remembered password if any.
	((NULL PASSWORD)
	 (SETQ PASSWORD (GUESS-PASSWORD-MAYBE (OR NEW-USER-ID USER-ID) HOST))))
  (OR NEW-USER-ID (SETQ NEW-USER-ID USER-ID))
  (FILE-HOST-USER-ID NEW-USER-ID HOST)
  (VALUES NEW-USER-ID PASSWORD ENABLE-CAPABILITIES))

(DEFUN SET-LOCAL-VARIABLES-FROM-HOST-INFO
       (HOST NEW-USER-ID HSNAME-PATHNAME PERSONAL-NAME GROUP PERSONAL-NAME-1 &AUX ITEM)
  ;; Record info about this user
  ;; only if host login name equals name given to LOGIN.
  (AND (EQUAL USER-ID NEW-USER-ID)
       (EQUAL USER-PERSONAL-NAME "")
       (SETQ USER-PERSONAL-NAME PERSONAL-NAME
	     USER-GROUP-AFFILIATION GROUP
	     USER-PERSONAL-NAME-FIRST-NAME-FIRST PERSONAL-NAME-1))
  (SETQ CHAOS::GIVE-FINGER-SAVED-USER-ID T)	;Clear cache
  ;; If this is the user's login host but the host user id is not the one specified in LOGIN,
  ;; do not accept the file server's suggested home dir
  ;; since it is based on the file server login id.
  (AND (EQ HOST USER-LOGIN-MACHINE)
       (NOT (EQUAL USER-ID NEW-USER-ID))
       (SETQ HSNAME-PATHNAME (QUIET-USER-HOMEDIR HOST)))
  ;; Record homedir for this host.
  (IF (SETQ ITEM (ASSQ HOST USER-HOMEDIRS))
      (SETF (CDR ITEM) HSNAME-PATHNAME)
      (PUSH (CONS HOST HSNAME-PATHNAME) USER-HOMEDIRS)))

(DEFFLAVOR BASIC-ACCESS
	   (HOST)
	   ()
  (:INITABLE-INSTANCE-VARIABLES HOST)
  (:GETTABLE-INSTANCE-VARIABLES HOST)
  (:REQUIRED-METHODS
    :RESET		; () Close all streams, reset host units
    :OPEN-STREAMS	; () List of open streams
    :CLOSE-ALL-FILES	; (&optional (mode :abort))
    :OPEN 		; (FILE PATHNAME &REST OPTIONS) The PATHNAME argument is for the
			; the PATHNAME which was originally requested; the usual ``different''
			; thing for this to be is a logical pathname.
			; FILE seems to be ignored, but it's being kept in for now
    :RENAME		; (FILE NEW-PATHNAME ERROR-P)
    :DELETE		; (FILE ERROR-P)
    :COMPLETE-STRING	; (FILE STRING OPTIONS) FILE is mostly for defaulting
    :CHANGE-PROPERTIES	; (FILE ERROR-P &REST PROPERTIES)
;Is :HOMEDIR really used??
    :HOMEDIR		; (USER) Ignored most of the time
    :CREATE-LINK	; (FILE LINK-TO ERROR)
    :EXPUNGE 		; (FILE ERROR)
    :REMOTE-CONNECT	; (FILE ERROR ACCESS-MODE &OPTIONAL UNIT) Connect to the directory
			; FILE.  If ACCESS-MODE is T, then do TOPS-20 access.  If UNIT
			; is given, connect for just that unit.  (This argument should be
			; be ignored if it does not make sense the access object.)
    :CREATE-DIRECTORY	; (FILE ERROR)
    :DIRECTORY-LIST	; (FILE OPTIONS)
    :DIRECTORY-LIST-STREAM ; (FILE OPTIONS)
;Required only by DIRECTORY-STREAM-ACCESS-MIXIN
;   :DIRECTORY-STREAM	; (FILE OPTIONS)
    :ACCESS-DESCRIPTION	; () Returns a string describing the access method (for :PRINT-SELF)
    ))

;;; Two other operations that an access can handle are :ENABLE/DISABLE-CAPABILITIES.
;;; The arglist is (CAPABILITIES &OPTIONAL UNIT).  If UNIT is given, the capabilities
;;; are changed only on that unit.  The UNIT argument should be ignored by access objects
;;; for which it does not make sense.

(DEFMETHOD (BASIC-ACCESS :PRINT-SELF) (STREAM &REST CRUFT)
  (DECLARE (IGNORE CRUFT))
  (IF *PRINT-ESCAPE*
      (SYS:PRINTING-RANDOM-OBJECT (SELF STREAM :TYPE)
	(FORMAT STREAM "~A access to ~A" (SEND SELF :ACCESS-DESCRIPTION) HOST))
    (FORMAT STREAM "~A access to ~A" (SEND SELF :ACCESS-DESCRIPTION) HOST)))

(DEFMETHOD (BASIC-ACCESS :PEEK-FILE-SYSTEM) () NIL)
(DEFMETHOD (BASIC-ACCESS :PEEK-FILE-SYSTEM-HEADER) () NIL)

;;; Some access methods may implement this as a separate subprotocol
(DEFMETHOD (BASIC-ACCESS :PROPERTIES) (PATHNAME ERROR-P)
  (FILE-OPERATION-RETRY
    (LET ((DIR (SEND SELF :DIRECTORY-LIST PATHNAME (IF ERROR-P '(:DELETED)
						     '(:NOERROR :DELETED)))))
      (IF (CONSP DIR)
	  (IF (CADR DIR)
	      (VALUES (CADR DIR) (GET (CAR DIR) :SETTABLE-PROPERTIES))
	    ;; It is possible for a nonexistent file to give no error
	    ;; but just return an empty directory.
	    (LET ((CONDITION (MAKE-CONDITION 'FILE-NOT-FOUND "~A" PATHNAME :PROPERTIES)))
	      (SEND CONDITION :SET-FORMAT-ARGS
		    (LIST (FORMAT NIL "File not found for ~A" PATHNAME)))
	      (IF ERROR-P (SIGNAL-CONDITION :PROCEED-TYPES ())
		CONDITION)))
	DIR))))

(DEFMETHOD (BASIC-ACCESS :DELETE-MULTIPLE-FILES) (ERROR-P FILES)
  (MAPCAR #'FUNCALL FILES (CIRCULAR-LIST :DELETE) ERROR-P))

;;; This is currently here to abstract the usual file access protocol maintenance issues.
;;; It is not neccessary to use this, but it will be easier in many cases.
(DEFFLAVOR HOST-UNIT-ACCESS-MIXIN ; usually for remote protocols
	   ((HOST-UNITS NIL))
	   ()
  (:GETTABLE-INSTANCE-VARIABLES HOST-UNITS)
  (:REQUIRED-FLAVORS BASIC-ACCESS)
  (:REQUIRED-METHODS
    :HOST-UNIT-FLAVOR	; () Returns a flavor name, accepts init keywords for the instance 
			; variables ACCESS and HOST
    ))

(DEFMETHOD (HOST-UNIT-ACCESS-MIXIN :RESET) ()
  (DOLIST (HOST-UNIT HOST-UNITS)
    (SEND HOST-UNIT :RESET)))

(DEFMETHOD (HOST-UNIT-ACCESS-MIXIN :ANY-DORMANT-HOST-UNITS-P) ()
  (DOLIST (HOST-UNIT HOST-UNITS)
    (WHEN (SEND HOST-UNIT :DORMANT-P)
      (RETURN T))))

(DEFMETHOD (HOST-UNIT-ACCESS-MIXIN :RESET-DORMANT-HOST-UNITS) ()
  (DOLIST (HOST-UNIT HOST-UNITS)
    (SEND HOST-UNIT :DORMANT-RESET)))

(DEFMETHOD (HOST-UNIT-ACCESS-MIXIN :CLOSE-ALL-FILES) (&OPTIONAL (MODE :ABORT) &AUX CLOSED)
  (DOLIST (UNIT HOST-UNITS)
    (SETQ CLOSED (NCONC CLOSED (SEND UNIT :CLOSE-ALL-FILES MODE))))
  CLOSED)

(DEFMETHOD (HOST-UNIT-ACCESS-MIXIN :NEW-HOST-UNIT)
	   (&OPTIONAL (NOERROR-P NIL) &AUX UNIT (DEFAULT-CONS-AREA SYS:BACKGROUND-CONS-AREA))
  (SETQ UNIT (MAKE-INSTANCE (SEND SELF :HOST-UNIT-FLAVOR) :ACCESS SELF :HOST HOST))
  (SETQ HOST-UNITS (NCONC HOST-UNITS (NCONS UNIT)))
  (AND (SEND UNIT :VALIDATE-CONTROL-CONNECTION NOERROR-P)
       UNIT))

(DEFMETHOD (HOST-UNIT-ACCESS-MIXIN :OPEN-STREAMS) (&AUX STREAMS)
  (DOLIST (UNIT HOST-UNITS)
    (SETQ STREAMS (NCONC STREAMS (SEND UNIT :OPEN-STREAMS))))
  STREAMS)

;;; Return a valid host unit.  If no units, make one.  If any unit is still open, use it.
;;; Errors if fails to connect unless noerror-p is T.
(DEFMETHOD (HOST-UNIT-ACCESS-MIXIN :GET-HOST-UNIT) (&OPTIONAL NOERROR-P)
  (COND ((NULL HOST-UNITS)
	 (SEND SELF :NEW-HOST-UNIT NOERROR-P))
	((LOOP FOR UNIT IN HOST-UNITS
	       WHEN (SEND UNIT :VALIDATE-CONTROL-CONNECTION T)
	       RETURN UNIT))
	((NOT NOERROR-P)
	 (LET ((UNIT (CAR HOST-UNITS)))
	   (SEND UNIT :VALIDATE-CONTROL-CONNECTION)
	   UNIT))))

(DEFMETHOD (HOST-UNIT-ACCESS-MIXIN :LOGIN-UNIT) (-UNIT- LOGIN-P UNAME-HOST)
  (SEND -UNIT- :LOGIN LOGIN-P UNAME-HOST))

(DEFCONST HOST-UNIT-LIFETIME 72000.)		;20 Minutes.

(DEFCONST DATA-CONNECTION-LIFETIME 3600.)

;;; This flavor should be used with CHAOS FILE, and perhaps FTP.  It assumes that the
;;; the protcol uses data-connections internally, and that the DATA-CONNECTION
;;; DEFSTRUCT is used.  The flavor of host unit to be used with this should incorporate
;;; DATA-CONNECTION-MIXIN.
(DEFFLAVOR DC-ACCESS-MIXIN
	   ()
  (HOST-UNIT-ACCESS-MIXIN))

;;; This can be redefined
(DEFMETHOD (DC-ACCESS-MIXIN :ACCESS-SPECIFIC-DATA-CONN-LIMIT) () #o37)

;;; Get a DATA-CONNECTION for use in DIRECTION.
;;; Make two passes over existing units, first trying open ones.
(DEFMETHOD (DC-ACCESS-MIXIN :GET-DATA-CONNECTION) (DIRECTION &OPTIONAL NOERROR-P)
  (DO-NAMED TOP ((ERROR-P NIL T)) (NIL)
    (DO ((UNITS HOST-UNITS (CDR UNITS))
	 (UNIT) (DATA-CONN))
	((NULL UNITS))
      (SETQ UNIT (CAR UNITS))
      (AND (SEND UNIT :VALIDATE-CONTROL-CONNECTION (OR NOERROR-P (NOT ERROR-P)))
	   (SETQ DATA-CONN (SEND UNIT :GET-DATA-CONNECTION DIRECTION))
	   (RETURN-FROM TOP DATA-CONN UNIT)))
    (AND NOERROR-P
	 (NOT (SEND SELF :GET-HOST-UNIT T))	;If you can't get a valid connection,
	 (RETURN-FROM TOP NIL NIL))		;then you have a loosing host.
    (AND ERROR-P
	 (LET* ((UNIT (SEND SELF :NEW-HOST-UNIT))
		(DATA-CONN (SEND UNIT :GET-DATA-CONNECTION DIRECTION)))
	   (OR DATA-CONN
	       (FERROR NIL "New unit failed to allocate data connection"))
	   (RETURN-FROM TOP DATA-CONN UNIT)))))

(DEFMETHOD (DC-ACCESS-MIXIN :PEEK-FILE-SYSTEM) ()
  (DC-ACCESS-PEEK-FILE-SYSTEM))

;;; For things such as QFILE-ACCESS which access directories through a :DIRECTORY-STREAM
(DEFFLAVOR DIRECTORY-STREAM-ACCESS-MIXIN () ()
  (:REQUIRED-FLAVORS BASIC-ACCESS)
  (:REQUIRED-METHODS
    :DIRECTORY-STREAM-DEFAULT-PARSER	; () Returns function that takes args like SUBSTRING
    :DIRECTORY-STREAM			; (PATHNAME OPTIONS) Stream be parsed
    :READ-DIRECTORY-STREAM-ENTRY	; (STREAM PATHNAME) Use PATHNAME for defaults
    ))

(DEFMETHOD (DIRECTORY-STREAM-ACCESS-MIXIN :DIRECTORY-LIST) (PATHNAME OPTIONS &AUX DIR-LIST)
  (WITH-OPEN-STREAM (STREAM (SEND SELF :DIRECTORY-STREAM PATHNAME (REMQ :SORTED OPTIONS)))
    (IF (ERRORP STREAM) STREAM
      (SETQ DIR-LIST
	    (LET ((PATHNAME (SEND STREAM :PATHNAME)))
	      (LOOP AS ENTRY = (SEND SELF :READ-DIRECTORY-STREAM-ENTRY STREAM PATHNAME)
		    UNTIL (NULL ENTRY)
		    COLLECTING ENTRY)))
      (IF (MEMQ :SORTED OPTIONS)
	  (LET ((NULL-ELEM (ASSQ NIL DIR-LIST)))
	    (AND NULL-ELEM (SETQ DIR-LIST (DELQ NULL-ELEM DIR-LIST)))
	    (SETQ DIR-LIST (SORTCAR DIR-LIST #'PATHNAME-LESSP))
	    (AND NULL-ELEM (PUSH NULL-ELEM DIR-LIST))))
      DIR-LIST)))

;; This tells READ-DIRECTORY-STREAM-ENTRY how to parse most lines of directory stream data.
(DEFMETHOD (DIRECTORY-STREAM-ACCESS-MIXIN :DEFAULT-DIRECTORY-STREAM-PARSER) ()
  #'SUBSTRING)

(DEFMETHOD (DIRECTORY-STREAM-ACCESS-MIXIN :DIRECTORY-LIST-STREAM) (PATHNAME OPTIONS)
  (LET ((STREAM (SEND SELF :DIRECTORY-STREAM PATHNAME OPTIONS)))
    (IF (ERRORP STREAM)
	STREAM
      (LET-CLOSED ((INTERNAL-STREAM STREAM) (DEFAULTING-PATHNAME (SEND STREAM :PATHNAME))
		   (ACCESS SELF))
	#'DIRECTORY-STREAM-DIRECTORY-LIST-STREAM))))

(LOCAL-DECLARE ((SPECIAL INTERNAL-STREAM DEFAULTING-PATHNAME ACCESS))
(DEFSELECT DIRECTORY-STREAM-DIRECTORY-LIST-STREAM
  (:ENTRY () (SEND ACCESS :READ-DIRECTORY-STREAM-ENTRY INTERNAL-STREAM DEFAULTING-PATHNAME))
  (:CLOSE (&OPTIONAL MODE) (SEND INTERNAL-STREAM :CLOSE MODE))) )

;;; This is for access methods that get the directory information all at once, with
;;; a :DIRECTORY-LIST option
(DEFFLAVOR DIRECTORY-LIST-MIXIN () ()
  (:REQUIRED-FLAVORS BASIC-ACCESS)
  (:REQUIRED-METHODS :DIRECTORY-LIST)) ; (PATHNAME OPTIONS) Defines :DIRECTORY-LIST-STREAM

(DEFMETHOD (DIRECTORY-LIST-MIXIN :DIRECTORY-LIST-STREAM)
	   (PATHNAME OPTIONS &AUX DIRECTORY-LIST)
  (DECLARE (SPECIAL DIRECTORY-LIST))
  (IF (ERRORP (SETQ DIRECTORY-LIST (SEND SELF :DIRECTORY-LIST PATHNAME OPTIONS)))
      DIRECTORY-LIST
    (CLOSURE '(DIRECTORY-LIST) #'DEFAULT-DIRECTORY-LIST-STREAM)))

(LOCAL-DECLARE ((SPECIAL DIRECTORY-LIST))
(DEFSELECT DEFAULT-DIRECTORY-LIST-STREAM
  (:ENTRY () (POP DIRECTORY-LIST))
  (:CLOSE . IGNORE)) )

(DEFFLAVOR BASIC-HOST-UNIT
	(HOST				;Host object
	 ACCESS				;Access object
	 CONTROL-CONNECTION		;May be the only connection if things are multiplexed
	 (LOCK NIL)			;Lock to insure no timing screws
	 (LAST-USE-TIME (TIME)))
	()
  (:INITABLE-INSTANCE-VARIABLES HOST ACCESS)
  (:GETTABLE-INSTANCE-VARIABLES HOST ACCESS CONTROL-CONNECTION LAST-USE-TIME)
  (:REQUIRED-METHODS
    :RESET	; (&optional dont-unlock-lock-p) close all data streams in abort mode,
		; close control connection
    :DORMANT-RESET ; () Reset self if dormant
    :DORMANT-P	; () Return T if dormant.
    :CLOSE-ALL-FILES ; (&optional (mode :abort))
		; close all files in this host unit, reporting to *ERROR-OUTPUT*
    		;  and returning a list of closed streams
    :VALIDATE-CONTROL-CONNECTION ; (&optional no-error-p) Check that connection hasn't
		; gone away, making a new one if necessary.  Return NIL when failure
		; and NO-ERROR-P equal to T, otherwise barf on failure
    :OPEN-STREAMS ; () Return a list of open streams
    :OPEN-CONTROL-CONNECTION-P ; () Returns T if the connection is in an open state
    :CLOSE-CONTROL-CONNECTION ; () Close connection, for logging out
    )
  (:OUTSIDE-ACCESSIBLE-INSTANCE-VARIABLES LOCK LAST-USE-TIME))

(DEFMETHOD (BASIC-HOST-UNIT :PRINT-SELF) (STREAM DEPTH ESCAPEP)
  (DECLARE (IGNORE DEPTH ESCAPEP))
  (SI:PRINTING-RANDOM-OBJECT (SELF STREAM :TYPE)
    (FORMAT STREAM "for ~A" ACCESS)))

;;; Restore connected/accessed directories, capabilities of a unit.  If ENABLE-CAPABILITIES
;;; is T, enable capabilities for the host as a whole
(DEFMETHOD (BASIC-HOST-UNIT :RESTORE-SERVER-STATE) (ENABLE-CAPABILITIES &AUX TEM)
  (IF (SETQ TEM (GET HOST 'CONNECTED-DIRECTORY))
      (SEND ACCESS :REMOTE-CONNECT TEM T NIL SELF))
  (IF (SETQ TEM (GET HOST 'ACCESSED-DIRECTORIES))
      (DOLIST (X TEM)
	(SEND ACCESS :REMOTE-CONNECT X T T SELF)))
  (IF (SETQ TEM (GET HOST 'CAPABILITIES-ALIST))
      (DOLIST (X TEM)
	(WITH-STACK-LIST (CAP (CAR X))
	  (SEND ACCESS (IF (CDR X) :ENABLE-CAPABILITIES :DISABLE-CAPABILITIES) CAP))))
  (IF ENABLE-CAPABILITIES (SEND HOST :ENABLE-CAPABILITIES)))

;;; Lock a host unit around BODY
(DEFMACRO LOCK-HOST-UNIT ((HOST-UNIT) &BODY BODY)
  (LET ((LOCK (GENSYM)) (LOCKED-P (GENSYM)) (HU (GENSYM)))
    `(LET* ((,HU ,HOST-UNIT)
	    (,LOCK (LOCF (BASIC-HOST-UNIT-LOCK ,HU)))
	    (,LOCKED-P NIL))
       (UNWIND-PROTECT
	 (PROGN
	   (COND ((NEQ (CAR ,LOCK) CURRENT-PROCESS)
		  (PROCESS-LOCK ,LOCK)
		  (SETQ ,LOCKED-P T)))
	   . ,BODY)
	 (SETF (BASIC-HOST-UNIT-LAST-USE-TIME ,HU) (TIME))
	 (AND ,LOCKED-P (PROCESS-UNLOCK ,LOCK))))))

(DEFMETHOD (BASIC-HOST-UNIT :VALID-CONTROL-CONNECTION-P) ()
  (AND CONTROL-CONNECTION
       (SEND SELF :OPEN-CONTROL-CONNECTION-P)))

(DEFMETHOD (BASIC-HOST-UNIT :PEEK-FILE-SYSTEM) () ())

;;; A DATA-CONNECTION is associated with each data connection implementing a file stream.
;;; The two directions in the connection itself are used independently.
;;; This structure should be useful for file protocols other than CHAOS FILE
;;; You must maintain the LAST-USE-TIME so that the data connection won't be considered
;;; dormant.
(DEFSTRUCT (DATA-CONNECTION :LIST*
			    (:CONC-NAME DATA-)
			    (:ALTERANT NIL)
			    (:CONSTRUCTOR MAKE-DATA-CONNECTION
					  (CONNECTION INPUT-HANDLE OUTPUT-HANDLE)))
  CONNECTION					;The chaos connection
  INPUT-HANDLE
  OUTPUT-HANDLE
  (LAST-USE-TIME (TIME))
  (STREAM-LIST (LIST :INPUT NIL :OUTPUT NIL))
  )

(DEFSUBST DATA-HANDLE (DATA-CONNECTION DIRECTION)
  (SELECTQ DIRECTION
    (:INPUT (DATA-INPUT-HANDLE DATA-CONNECTION))
    (:OUTPUT (DATA-OUTPUT-HANDLE DATA-CONNECTION))))

(DEFSUBST DATA-STREAM (DATA-CONNECTION DIRECTION)
  (CADR (MEMQ DIRECTION (DATA-STREAM-LIST DATA-CONNECTION))))

(DEFUN DATA-CONNECTION-DORMANT (DATA-CONNECTION)
  (AND (NULL (DATA-STREAM DATA-CONNECTION :INPUT))
       (NULL (DATA-STREAM DATA-CONNECTION :OUTPUT))
       (> (TIME-DIFFERENCE (TIME) (DATA-LAST-USE-TIME DATA-CONNECTION))
	  DATA-CONNECTION-LIFETIME)))

(DEFFLAVOR DATA-CONNECTION-MIXIN
	   (MAX-DATA-CONNECTIONS
	    DATA-CONNECTIONS)
	   ()
  :GETTABLE-INSTANCE-VARIABLES
  (:REQUIRED-FLAVORS BASIC-HOST-UNIT)
  (:REQUIRED-METHODS 
    :FREE-DATA-CONNECTION ; (DATA-CONNECTION DIRECTION) Called when done with a
		; DATA-CONNECTION for DIRECTION
    :CLOSE-DORMANT-DATA-CONNECTIONS ; ()
    :GET-DATA-CONNECTION ; (DIRECTION) Return a free data conn.  See the `directional' mixins
    ))

(DEFMETHOD (DATA-CONNECTION-MIXIN :INIT) (IGNORE)
  (SETQ MAX-DATA-CONNECTIONS (SEND HOST :MAX-DATA-CONNECTIONS)))

;;; Called when done with a DATA-CONNECTION for DIRECTION.
;;; If free in both directions for long enough, it is flushed for being dormant.
(DEFMETHOD (DATA-CONNECTION-MIXIN :FREE-DATA-CONNECTION) (DATA-CONNECTION DIRECTION)
  (SETF (DATA-STREAM DATA-CONNECTION DIRECTION) NIL)
  (SETF (DATA-LAST-USE-TIME DATA-CONNECTION) (TIME))
  (SETQ LAST-USE-TIME (TIME)))

(DEFMETHOD (DATA-CONNECTION-MIXIN :DORMANT-P) ()
  (AND (NOT LOCK)
       ;; Don't kill a host unit if it has more than one data connection still,
       (NULL (CDR DATA-CONNECTIONS))
       ;; or if that data connection is doing anything.
       (LET ((DATA-CONNECTION (CAR DATA-CONNECTIONS)))
	 (OR (NULL DATA-CONNECTION)
	     (AND (NULL (DATA-STREAM DATA-CONNECTION :INPUT))
		  (NULL (DATA-STREAM DATA-CONNECTION :OUTPUT)))))
       (> (TIME-DIFFERENCE (TIME) LAST-USE-TIME)
	  HOST-UNIT-LIFETIME)))

(DEFMETHOD (DATA-CONNECTION-MIXIN :DORMANT-RESET) ()
  ;; If host unit has extra data connections, close them.
  (IF (CDR DATA-CONNECTIONS)
      (SEND SELF :CLOSE-DORMANT-DATA-CONNECTIONS))
  (LET (DO-IT)
    (WITHOUT-INTERRUPTS
      (AND (SEND SELF :CONTROL-CONNECTION)	; Don't bother, if already reset.
	   (SEND SELF :DORMANT-P)	; Don't reset if being used or used recently.
	   (SETQ DO-IT T)
	   (SETF LOCK 'LOCKED-FOR-SUICIDE)))
    (IF DO-IT
	(SEND SELF :RESET))))

;;; This also frees up any slots marked as open
(DEFMETHOD (DATA-CONNECTION-MIXIN :CLOSE-ALL-FILES) (&OPTIONAL (MODE :ABORT) &AUX CLOSED)
  (DOLIST (DATA-CONN DATA-CONNECTIONS)
    (DO ((LIST (DATA-STREAM-LIST DATA-CONN) (CDDR LIST)))
	((NULL LIST))
      (LET ((STREAM (CADR LIST)))
	(COND ((NULL STREAM))
	      ((EQ STREAM T)
	       (SETF (CADR LIST) NIL))
	      (T
	       (FORMAT *ERROR-OUTPUT* "~%Closing ~S" STREAM)
	       (PUSH STREAM CLOSED)
	       (SEND STREAM :CLOSE MODE))))))
  CLOSED)

(DEFMETHOD (DATA-CONNECTION-MIXIN :OPEN-STREAMS) (&AUX STREAMS)
  (DOLIST (DATA-CONN DATA-CONNECTIONS)
    (DO ((LIST (DATA-STREAM-LIST DATA-CONN) (CDDR LIST)))
	((NULL LIST))
      (LET ((STREAM (CADR LIST)))
	(UNLESS (SYMBOLP STREAM)
	  (PUSH STREAM STREAMS)))))
  STREAMS)

;;; This is for winning protocols like CHAOS FILE (affectionately known as QFILE)
;;; A file stream can be placed on the data connection if the desired direction on a
;;; particular data connection is free.
(DEFFLAVOR BIDIRECTIONAL-DATA-CONNECTION-MIXIN ()
	   (DATA-CONNECTION-MIXIN)
  )

;;; Get a data connection for this unit.  Makes a new one if there is room in within the
;;; maximum number.  We are assumed to have recently been checked for validity.
(DEFMETHOD (BIDIRECTIONAL-DATA-CONNECTION-MIXIN :GET-DATA-CONNECTION) (DIRECTION)
  (LOCK-HOST-UNIT (SELF)
     (DO ((DATA-CONNS DATA-CONNECTIONS (CDR DATA-CONNS))
	 (DATA-CONN))
	(NIL)
      (SETQ DATA-CONN (COND (DATA-CONNS (CAR DATA-CONNS))
			    ((= (LENGTH DATA-CONNECTIONS)
				(MIN MAX-DATA-CONNECTIONS
				     (SEND ACCESS :ACCESS-SPECIFIC-DATA-CONN-LIMIT)))
			     (RETURN NIL))
			    (T (SEND SELF :NEW-DATA-CONNECTION))))
      (COND ((NULL (DATA-STREAM DATA-CONN DIRECTION))
	     (SETF (DATA-STREAM DATA-CONN DIRECTION) T)	;Mark as allocated
	     (RETURN DATA-CONN))))))

;;; This is for lusing protocols like TCP FTP
;;; A file stream can be placed on the data connection if the data connection is unused in
;;; both directions
(DEFFLAVOR UNIDIRECTIONAL-DATA-CONNECTION-MIXIN ()
	   (DATA-CONNECTION-MIXIN)
  )

;;; Get a data connection for this unit.  Makes a new one if there is room in within the
;;; maximum number.  We are assumed to have recently been checked for validity.  Since
;;; this is the unidirectional mixin, both directions must be free.
(DEFMETHOD (UNIDIRECTIONAL-DATA-CONNECTION-MIXIN :GET-DATA-CONNECTION) (DIRECTION)
  (LOCK-HOST-UNIT (SELF)
     (DO ((DATA-CONNS DATA-CONNECTIONS (CDR DATA-CONNS))
	 (DATA-CONN))
	(NIL)
      (SETQ DATA-CONN (COND (DATA-CONNS (CAR DATA-CONNS))
			    ((= (LENGTH DATA-CONNECTIONS)
				(MIN MAX-DATA-CONNECTIONS
				     (SEND ACCESS :ACCESS-SPECIFIC-DATA-CONN-LIMIT)))
			     (RETURN NIL))
			    (T (SEND SELF :NEW-DATA-CONNECTION))))
      (COND ((AND (NULL (DATA-STREAM DATA-CONN :INPUT))
		  (NULL (DATA-STREAM DATA-CONN :OUTPUT)))
	     (SETF (DATA-STREAM DATA-CONN DIRECTION) T)	;Mark as allocated
	     (RETURN DATA-CONN))))))


;;; The following takes care of the process to do the above stuff every minute.
;;; These functions take care of closing connections that have not been used for a while.
(DEFUN RESET-DORMANT-HOST-UNITS ()
  (ERRSET
    (DOLIST (HOST *PATHNAME-HOST-LIST*)
      (SEND HOST :SEND-IF-HANDLES :RESET-DORMANT-HOST-UNITS))
    NIL))

(DEFVAR DORMANT-HOST-CONNECTION-GC-WAIT-ARG-LIST (LIST NIL 3600.))

(DEFUN WAIT-UNTIL-TIME (TIME DELAY)
  (> (TIME-DIFFERENCE (TIME) TIME) DELAY))

(DEFUN ANY-DORMANT-HOST-UNITS ()
  (DOLIST (HOST *PATHNAME-HOST-LIST*)
    (WHEN (SEND HOST :SEND-IF-HANDLES :ANY-DORMANT-HOST-UNITS-P)
      (RETURN T))))

(DEFVAR DORMANT-HOST-GC-PROCESS)

(DEFUN DORMANT-HOST-CONNECTION-GC-TOP-LEVEL ()
  (IF (ANY-DORMANT-HOST-UNITS)
      (PROCESS-RUN-FUNCTION "Flush file connections" 'RESET-DORMANT-HOST-UNITS))
  (SETF (CAR DORMANT-HOST-CONNECTION-GC-WAIT-ARG-LIST) (TIME))
  (SI:SET-PROCESS-WAIT CURRENT-PROCESS
		       #'WAIT-UNTIL-TIME
		       DORMANT-HOST-CONNECTION-GC-WAIT-ARG-LIST)
  (SETF (SI:PROCESS-WHOSTATE DORMANT-HOST-GC-PROCESS) "Sleep"))


(DEFUN INIT-DORMANT-HOST-GC-PROCESS ()
  (OR (BOUNDP 'DORMANT-HOST-GC-PROCESS)
      (SETQ DORMANT-HOST-GC-PROCESS (MAKE-PROCESS "Dormant FILE connection GC" :SIMPLE-P T)))
  (PROCESS-PRESET DORMANT-HOST-GC-PROCESS 'DORMANT-HOST-CONNECTION-GC-TOP-LEVEL)
  (PROCESS-RESET-AND-ENABLE DORMANT-HOST-GC-PROCESS))

(ADD-INITIALIZATION 'DORMANT-HOST-GC
		    `(INIT-DORMANT-HOST-GC-PROCESS)
		    '(:NORMAL))


(DEFUN FILE-LOGIN (LOGINP &AUX TEM)
  "Log all open host units in or out.  LOGINP = NIL means log out, otherwise log in."
  (DOLIST (HOST *PATHNAME-HOST-LIST*)
    (DOLIST (PROP '(QFILE-CONNECTED-DIRECTORY QFILE-ACCESSED-DIRECTORY CAPABILITIES-ALIST))
      (AND (SETQ TEM (GET-LOCATION-OR-NIL HOST PROP)) (SETF (CONTENTS TEM) NIL)))
    (DOLIST (UNIT (SEND HOST :SEND-IF-HANDLES :HOST-UNITS))
      (SEND HOST :LOGIN-UNIT UNIT LOGINP))))

(ADD-INITIALIZATION "File Login" '(FILE-LOGIN T) '(LOGIN))
(ADD-INITIALIZATION "File Logout" '(FILE-LOGIN NIL) '(LOGOUT))

;;; *** Host flavor stuff.  Some flavors are going to go away
;;; Operating system particular host flavors
(DEFFLAVOR FILE-HOST-ITS-MIXIN () (FILE-HOST-MIXIN))

(DEFMETHOD (FILE-HOST-ITS-MIXIN :PATHNAME-FLAVOR) () 'ITS-PATHNAME)

(DEFMETHOD (FILE-HOST-ITS-MIXIN :MAX-DATA-CONNECTIONS) () 3)

(DEFMETHOD (FILE-HOST-ITS-MIXIN :LOGIN-UNIT) (UNIT LOGIN-P)
  (SEND UNIT :LOGIN LOGIN-P 'ITS))

(DEFMETHOD (FILE-HOST-ITS-MIXIN :HSNAME-INFORMATION) (UNIT STR IDX)
  (LET* ((HOST (SEND UNIT :HOST))
	 (DEFAULT-CONS-AREA SYS:BACKGROUND-CONS-AREA)
	 (HSNAME (SUBSTRING STR (INCF IDX)
			        (SETQ IDX (STRING-SEARCH-CHAR #/NEWLINE STR IDX))))
	 (HSNAME-PATHNAME (MAKE-PATHNAME :HOST HOST :DEVICE "DSK" :DIRECTORY HSNAME))
	 (PERSONAL-NAME (SUBSTRING STR (INCF IDX)
				       (SETQ IDX (STRING-SEARCH-CHAR #/NEWLINE STR IDX))))
	 (GROUP-AFFILIATION (CHAR STR (1+ IDX))))
    (SETQ IDX (STRING-SEARCH ", " PERSONAL-NAME)
	  STR (NSUBSTRING PERSONAL-NAME 0 IDX))
    (AND IDX (SETQ STR (STRING-APPEND (NSUBSTRING PERSONAL-NAME (+ IDX 2)) #/SP STR)))
    (VALUES HSNAME-PATHNAME PERSONAL-NAME GROUP-AFFILIATION STR)))

(DEFMETHOD (FILE-HOST-ITS-MIXIN :LOGICALLY-BACKTRANSLATE-HOST-DEV-DIR)
	   (PHYSICAL-HOST PHYSICAL-DEVICE PHYSICAL-DIRECTORY)
  (IF (AND (EQ PHYSICAL-HOST SELF)
	   (MEMQ PHYSICAL-DEVICE '(NIL :UNSPECIFIC)))
      (VALUES SELF "DSK" PHYSICAL-DIRECTORY)))

(DEFMETHOD (FILE-HOST-ITS-MIXIN :GENERIC-BASE-TYPE) (FILE-TYPE)
  (IF (ASSOC-EQUAL FILE-TYPE *GENERIC-BASE-TYPE-ALIST*)
      :UNSPECIFIC  ;on ITS, we cant distinguish base types, since name2 is frequently
    FILE-TYPE))	    ; used as a version number.

(DEFFLAVOR FILE-HOST-TOPS20-MIXIN () (CASTE-FILE-HOST-MIXIN FILE-HOST-MIXIN))

(DEFMETHOD (FILE-HOST-TOPS20-MIXIN :DEFAULT-CAPABILITIES) () '("OPERATOR" "WHEEL"))

(DEFMETHOD (FILE-HOST-TOPS20-MIXIN :PATHNAME-FLAVOR) () 'TOPS20-PATHNAME)

(DEFMETHOD (FILE-HOST-TOPS20-MIXIN :MAX-DATA-CONNECTIONS) () 8)

(DEFFLAVOR FILE-HOST-TENEX-MIXIN () (FILE-HOST-TOPS20-MIXIN))

(DEFMETHOD (FILE-HOST-TENEX-MIXIN :PATHNAME-FLAVOR) () 'TENEX-PATHNAME)

(DEFFLAVOR FILE-HOST-VMS-MIXIN () (CASTE-FILE-HOST-MIXIN FILE-HOST-MIXIN))

(DEFMETHOD (FILE-HOST-VMS-MIXIN :PATHNAME-FLAVOR) () 'VMS-PATHNAME)

(DEFMETHOD (FILE-HOST-VMS-MIXIN :DEFAULT-CAPABILITIES) () '("SYSPRV"))

;;; FOO, This could be any number at all, depending on the quota assigned.....
(DEFMETHOD (FILE-HOST-VMS-MIXIN :MAX-DATA-CONNECTIONS) () 10.)

(DEFFLAVOR FILE-HOST-UNIX-MIXIN () (FILE-HOST-MIXIN))

(DEFMETHOD (FILE-HOST-UNIX-MIXIN :PATHNAME-FLAVOR) () 'UNIX-PATHNAME)

(DEFFLAVOR FILE-HOST-MULTICS-MIXIN () (FILE-HOST-MIXIN))

(DEFMETHOD (FILE-HOST-MULTICS-MIXIN :PATHNAME-FLAVOR) () 'MULTICS-PATHNAME)

;;; Mixin for hosts that knows how to name itself.
(DEFFLAVOR FILE-HOST-LISPM-MIXIN () (FILE-HOST-MIXIN))

(DEFMETHOD (FILE-HOST-LISPM-MIXIN :PRIMARY-DEVICE) () "DSK")

;;;; FILE protocol support.
(DEFMETHOD (FILE-HOST-LISPM-MIXIN :MAX-DATA-CONNECTIONS) () #o37)

(DEFMETHOD (FILE-HOST-LISPM-MIXIN :LOGIN-UNIT)
	   (UNIT LOGINP &AUX TEM (DEFAULT-CONS-AREA SYS:BACKGROUND-CONS-AREA))
  ;; Don't confuse the user by asking for UNAME and PASSWORD if he's logged in elsewhere.
  (AND (SETQ TEM (COND ((NOT (EQUAL USER-ID "")) USER-ID)
		       ((CDR (ASSQ 'ITS USER-UNAMES)))
		       ((CDAR USER-UNAMES))))
       (PUSH (CONS SELF TEM) USER-UNAMES))
  ;; Connection is used up when logging out
  (AND (SEND UNIT :VALID-CONTROL-CONNECTION-P)
       (IF LOGINP
	   (SEND UNIT :LOGIN LOGINP SELF)
	   (SEND UNIT :CLOSE-CONTROL-CONNECTION)))
  T)

;;; The Fileserver doesn't supply the user name information, so might as well use
;;; whatever's hanging around.
(DEFMETHOD (FILE-HOST-LISPM-MIXIN :HSNAME-INFORMATION) (IGNORE STR IDX)
  (VALUES (PARSE-PATHNAME STR NIL *DEFAULT-PATHNAME-DEFAULTS*
			  (INCF IDX) (STRING-SEARCH-CHAR #/NEWLINE STR IDX))
	  USER-PERSONAL-NAME USER-GROUP-AFFILIATION
	  USER-PERSONAL-NAME-FIRST-NAME-FIRST))

(DEFMETHOD (FILE-HOST-LISPM-MIXIN :PATHNAME-FLAVOR) ()
  (GET (SEND SELF :FILE-SYSTEM-TYPE) 'LISPM-PATHNAME-FLAVOR))


;;;; Predefined host flavors
(DEFFLAVOR ITS-HOST () (SI:HOST-ITS-MIXIN FILE-HOST-ITS-MIXIN SI:HOST))
(DEFPROP :ITS ITS-HOST SI:HOST-FLAVOR)

(DEFFLAVOR TOPS20-HOST () (SI:HOST-TOPS20-MIXIN FILE-HOST-TOPS20-MIXIN SI:HOST))
(DEFPROP :TOPS-20 TOPS20-HOST SI:HOST-FLAVOR)
(DEFPROP :TOPS20 TOPS20-HOST SI:HOST-FLAVOR)

(DEFFLAVOR TENEX-HOST () (SI:HOST-TENEX-MIXIN FILE-HOST-TENEX-MIXIN SI:HOST))
(DEFPROP :TENEX TENEX-HOST SI:HOST-FLAVOR)

(DEFFLAVOR VMS-HOST () (SI:HOST-VMS-MIXIN FILE-HOST-VMS-MIXIN SI:HOST))
(DEFPROP :VMS VMS-HOST SI:HOST-FLAVOR)

(DEFFLAVOR UNIX-HOST () (SI:HOST-UNIX-MIXIN FILE-HOST-UNIX-MIXIN SI:HOST))
(DEFPROP :UNIX UNIX-HOST SI:HOST-FLAVOR)

(DEFFLAVOR MULTICS-HOST () (SI:HOST-MULTICS-MIXIN FILE-HOST-MULTICS-MIXIN SI:HOST))
(DEFPROP :MULTICS MULTICS-HOST SI:HOST-FLAVOR)

(DEFFLAVOR LISPM-HOST () (SI:HOST-LISPM-MIXIN FILE-HOST-LISPM-MIXIN SI:HOST))
(DEFPROP :LISPM LISPM-HOST SI:HOST-FLAVOR)
(DEFPROP :LISPM LISPM-HOST FILE-SYSTEM-HOST-FLAVOR)

(COMPILE-FLAVOR-METHODS ITS-HOST TOPS20-HOST TENEX-HOST VMS-HOST
			UNIX-HOST MULTICS-HOST LISPM-HOST)

(DEFUN SITE-PATHNAME-INITIALIZE ()
  ;; Flush all old hosts
  (SETQ *PATHNAME-HOST-LIST* (DEL-IF #'(LAMBDA (X) (SEND X :SEND-IF-HANDLES :FILE-HOST-P))
				     *PATHNAME-HOST-LIST*))
  ;; And add new ones
  (DOLIST (HOST (OR (SI:GET-SITE-OPTION :FILE-SERVER-HOSTS)
		    (SI:GET-SITE-OPTION :CHAOS-FILE-SERVER-HOSTS)))
    (ADD-FILE-COMPUTER HOST))
  ;; Add LMFILE hosts.
  (ADD-LMFILE-HOSTS)
  (DOLIST (SPEC (GET-SITE-OPTION :HOST-DEFAULT-DEVICE-ALIST))
    (LET ((HOST (GET-PATHNAME-HOST (CAR SPEC) T)))
      (AND HOST (SET-IN-INSTANCE HOST 'SI::PRIMARY-DEVICE (CDR SPEC))))))

(DEFUN ADD-FILE-COMPUTER (HOST)
  "Add HOST to the list of hosts that can act as file servers.
HOST can be either a host name or a list (<host-name> <file-system-type>)."
  (IF (CONSP HOST)
      (LET ((H (FS:GET-PATHNAME-HOST (CAR HOST))))
	(SETF (GET H 'FILE-SYSTEM-TYPE) (CADR HOST))
	H)
    (FS:GET-PATHNAME-HOST HOST)))
  
;;; Forget about the previous ways of getting to a file...
(DEFUN RESET-FILE-ACCESS ()
  (WHEN (VARIABLE-BOUNDP *PATHNAME-HOST-LIST*)
    (DOLIST (H *PATHNAME-HOST-LIST*)
      (SEND H :SEND-IF-HANDLES :RESET T))))

(ADD-INITIALIZATION "Forget old file access" '(RESET-FILE-ACCESS) '(SYSTEM))


;;;; Interface to the LMFILE file system.
;;; The host used for access to a remote machine's LMFILE
;;; is NOT a host table host.

(DEFFLAVOR LMFILE-HOST
	   (HOST-NAME
	    CHAOS-ADDRESS
	    REMOTE-HOST-NAME)
	   (FILE-HOST-MIXIN SI:BASIC-HOST SI:PROPERTY-LIST-MIXIN)
  (:GETTABLE-INSTANCE-VARIABLES HOST-NAME CHAOS-ADDRESS REMOTE-HOST-NAME)
  (:INITABLE-INSTANCE-VARIABLES HOST-NAME CHAOS-ADDRESS REMOTE-HOST-NAME))

(DEFPROP :LMFILE LMFILE-HOST FILE-SYSTEM-HOST-FLAVOR)

#|
;;; These three operations are only used locally.
(DEFMETHOD (LMFILE-HOST :ADD-STREAM) (STREAM)
  (WITHOUT-INTERRUPTS
    (PUSH STREAM OPEN-STREAMS)))

(DEFMETHOD (LMFILE-HOST :REMOVE-STREAM) (STREAM)
  (WITHOUT-INTERRUPTS
    (SETQ OPEN-STREAMS (DELQ STREAM OPEN-STREAMS))))

;(DECLARE (*EXPR CLEAR-FILE-SYSTEM) (*EXPR STOP-FILE-SYSTEM))

(DEFMETHOD (LMFILE-HOST :SHUT-DOWN) (&OPTIONAL DRASTIC)
  (SETQ OPEN-STREAMS NIL)
  (IF (NEQ (SEND SELF :PATHNAME-FLAVOR) 'LMFILE-PATHNAME)
      (IF DRASTIC
	  (CLEAR-FILE-SYSTEM)
	(STOP-FILE-SYSTEM))))
|#

;;; This is here to make compile-flavor-methods win before PEEK is loaded.
;;; The function is defined in PEEKCH.
(DEFMETHOD (LMFILE-HOST :PEEK-FILE-SYSTEM-HEADER)
	   CHAOS:HOST-CHAOS-PEEK-FILE-SYSTEM-HEADER)

(DEFMETHOD (LMFILE-HOST :NAME) NIL HOST-NAME)
(DEFMETHOD (LMFILE-HOST :PATHNAME-FLAVOR) () 'LMFILE-PATHNAME)
(DEFMETHOD (LMFILE-HOST :SYSTEM-TYPE) () :LISPM)
(DEFMETHOD (LMFILE-HOST :FILE-SYSTEM-TYPE) () :LMFILE)

(DEFMETHOD (LMFILE-HOST :MAX-DATA-CONNECTIONS) () 8)

;;; The Fileserver doesn't supply the user name information, so might as well use
;;; whatever's hanging around.  Also, use this host as the host in the homedir,
;;; since the fileserver may use its own machine's hostname.
(DEFMETHOD (LMFILE-HOST :HSNAME-INFORMATION) (IGNORE STR IDX)
  (VALUES (SEND (PARSE-PATHNAME STR NIL *DEFAULT-PATHNAME-DEFAULTS*
				(INCF IDX) (STRING-SEARCH-CHAR #/NEWLINE STR IDX))
		:NEW-PATHNAME :HOST SELF)
	  USER-PERSONAL-NAME USER-GROUP-AFFILIATION
	  USER-PERSONAL-NAME-FIRST-NAME-FIRST))

;; interim kludge since lmfile hosts can only be chaos hosts currently
(DEFMETHOD (LMFILE-HOST :NETWORK-TYPEP) (TYPE)
  (EQ TYPE :CHAOS))

(DEFINE-SITE-VARIABLE LMFILE-SERVER-HOSTS :LMFILE-SERVER-HOSTS)

(DEFUN ADD-LMFILE-HOST (NAME CHAOS-ADDRESS REMOTE-NAME)
  (DOLIST (OHOST *PATHNAME-HOST-LIST*
		 (LET ((HOST (MAKE-INSTANCE 'LMFILE-HOST :HOST-NAME NAME
					    :REMOTE-HOST-NAME REMOTE-NAME
					    :CHAOS-ADDRESS CHAOS-ADDRESS)))
		   (PUSH HOST *PATHNAME-HOST-LIST*)
		   HOST))
    (WHEN (AND (TYPEP OHOST 'LMFILE-HOST)
	       (EQ (SEND OHOST :CHAOS-ADDRESS) CHAOS-ADDRESS)
	       (EQ (SEND OHOST :HOST-NAME) NAME)
	       (EQ (SEND OHOST :REMOTE-HOST-NAME) REMOTE-NAME))
      (SETQ *PATHNAME-HOST-LIST*
	    (CONS OHOST (DELQ OHOST *PATHNAME-HOST-LIST*)))
      (RETURN OHOST))))

;;; Add a host to reach each machine that should have one,
;;; except for this machine, if it is one of them,
;;; add "-R" to the name of the host, so that the specified name
;;; will still work for local LMFILE.
;;; The name used should be present in the host table as a nickname
;;; but should not be that host's primary name.
;;; Also deletes any such hosts that no longer belong.

;;; It is assumed that this is called AFTER the ordinary chaos hosts are put on
;;; so that this host will override the ordinary chaos host for this host's name.

;;; Now this is called directly by SITE-CHAOS-PATHNAME-INITIALIZE.
(DEFUN ADD-LMFILE-HOSTS (&AUX HOSTS-WANTED)
  (DOLIST (HN LMFILE-SERVER-HOSTS)
    (LET ((HOST (SI:PARSE-HOST HN T)))
      (WHEN HOST
	;; If this is that host,
	;; add a second one which has "-R" in its name
	;; and says to go thru the net back to this host.
	(AND (EQ HOST SI:LOCAL-HOST)
	     (PUSH (ADD-LMFILE-HOST (STRING-APPEND HN "-R") (SEND HOST :CHAOS-ADDRESS) HN)
		   HOSTS-WANTED))
	(PUSH (ADD-LMFILE-HOST HN (SEND HOST :CHAOS-ADDRESS) HN)
	      HOSTS-WANTED))))
  ;; Discard any hosts of this type that aren't supposed to be there.
  (DOLIST (HOST *PATHNAME-HOST-LIST*)
    (AND (TYPEP HOST 'LMFILE-HOST)
	 (NOT (MEMQ HOST HOSTS-WANTED))
	 (SETQ *PATHNAME-HOST-LIST* (DELQ HOST *PATHNAME-HOST-LIST*)))))

(COMPILE-FLAVOR-METHODS LMFILE-HOST)

