;;; -*- Mode:LISP; Package:SYSTEM-INTERNALS; Base:8; Readtable:ZL -*-

;;; ** (C) Copyright 1981, Symbolics, Inc.
;;;    Enhancements (C) Copyright 1981, Massachusetts Institute of Technology
;;; The Massachusetts Institute of Technology has acquired the rights from Symbolics
;;; to include the Software covered by the foregoing notice of copyright with its
;;; licenses of the Lisp Machine System **

;;; Flavor basis for streams

;;; To make a buffered stream:
;;; For input, define :NEXT-INPUT-BUFFER &OPTIONAL NO-HANG-P => ARRAY START END
;;;		      :DISCARD-INPUT-BUFFER ARRAY
;;; For output, define :NEW-OUTPUT-BUFFER => ARRAY START END
;;;		       :SEND-OUTPUT-BUFFER ARRAY END
;;;		       :DISCARD-OUTPUT-BUFFER ARRAY

;;; To make :SET-POINTER work, define :SET-BUFFER-POINTER NEW-POINTER => REAL-NEW-POINTER
;;; where REAL-NEW-POINTER should be such that the next :NEXT-INPUT-BUFFER will access the
;;; desired position somewhere in the buffer.

;;; To make binary streams, mix with
;;; for input	SI:BUFFERED-INPUT-STREAM
;;; for output	SI:BUFFERED-OUTPUT-STREAM
;;; for both	SI:BUFFERED-STREAM

;;; To make character streams, mix with
;;; for input	SI:BUFFERED-INPUT-CHARACTER-STREAM
;;; for output 	SI:BUFFERED-OUTPUT-CHARACTER-STREAM
;;; for both	SI:BUFFERED-CHARACTER-STREAM

;;; To make a character stream that does character set translation or compression or something
;;; like that, define a wrapper on :TYI (:TYO) that does the translation.  Then mix with
;;; for input	SI:BUFFERED-TYI-INPUT-STREAM
;;; for output	SI:BUFFERED-TYO-OUTPUT-STREAM
;;; for both	SI:BUFFERED-TYI-TYO-STREAM

;;; There are mixins in this file (SI:ASCII-TRANSLATING-INPUT-STREAM-MIXIN and
;;;	SI:ASCII-TRANSLATING-OUTPUT-STREAM-MIXIN) for translating between ASCII and Lisp
;;;	machine character set.

;;; For file streams, the mixin SI:STREAM-WITH-PATHNAME-MIXIN handles the pathname and
;;; printing nicely.  Use this for :PROBE openings (since they cannot do I/O).
;;; For input file streams, use SI:INPUT-FILE-STREAM-MIXIN.
;;; For output file streams, use SI:OUTPUT-FILE-STREAM-MIXIN.
;;; These make you interact with the who-line correctly.

;;; Base flavors

(DEFFLAVOR STREAM () ()
  (:REQUIRED-METHODS :DIRECTION)
  (:DOCUMENTATION "All streams are built on this.
This flavor is mostly for TYPEP, but also provides default methods for messages
which all streams, input or output, are required to handle."))

(DEFMETHOD (STREAM :DEFAULT :CLOSE) (&OPTIONAL MODE)
  (DECLARE (IGNORE MODE))
  NIL)

;;; Streams are binary until proven otherwise
(DEFMETHOD (STREAM :DEFAULT :CHARACTERS) () NIL)

(DEFMETHOD (STREAM :DEFAULT :ELEMENT-TYPE) ()
  (LET ((VALUE (SEND SELF :SEND-IF-HANDLES :BYTE-SIZE)))
    (IF VALUE
	`(UNSIGNED-BYTE ,VALUE)
        'FIXNUM)))

(DEFFLAVOR CHARACTER-STREAM () ()
  (:INCLUDED-FLAVORS STREAM))

(DEFMETHOD (CHARACTER-STREAM :CHARACTERS) () T)

(DEFMETHOD (CHARACTER-STREAM :ELEMENT-TYPE) () 'CHARACTER)

(DEFFLAVOR INPUT-STREAM () (STREAM)
  (:REQUIRED-METHODS :TYI :UNTYI)
; (:SELECT-METHOD-ORDER :TYI :UNTYI)
  (:DOCUMENTATION "All input streams are built on this."))

(DEFMETHOD (INPUT-STREAM :DIRECTION) () :INPUT)

(DEFMETHOD (INPUT-STREAM :DEFAULT :LISTEN) ()
  (LET ((TEM (SEND SELF :TYI-NO-HANG NIL)))
    (COND (TEM
	   (SEND SELF :UNTYI TEM)
	   T))))

(DEFMETHOD (INPUT-STREAM :DEFAULT :TYIPEEK) (&OPTIONAL EOF)
  (LET ((TEM (SEND SELF :TYI EOF)))
    (COND (TEM
	   (SEND SELF :UNTYI TEM)
	   TEM))))

;; why are these methods of all streams rather than just ones which support these operations
;; in a meaningful fashion?
(DEFMETHOD (INPUT-STREAM :DEFAULT :TYI-NO-HANG) (&OPTIONAL EOF)
  (SEND SELF :TYI EOF))
(DEFMETHOD (INPUT-STREAM :DEFAULT :ANY-TYI) (&OPTIONAL EOF)
  (SEND SELF :TYI EOF))
(DEFMETHOD (INPUT-STREAM :DEFAULT :ANY-TYI-NO-HANG) (&OPTIONAL EOF)
  (SEND SELF :TYI EOF))
  

(DEFMETHOD (INPUT-STREAM :DEFAULT :CLEAR-INPUT) ()
  NIL)

(DEFMETHOD (INPUT-STREAM :DEFAULT :READ-UNTIL-EOF) ()
  (LOOP WHILE (SEND SELF :TYI)))

(DEFMETHOD (INPUT-STREAM :DEFAULT :STRING-IN) (EOF STRING &OPTIONAL (START 0) END)
  (OR END (SETQ END (ARRAY-LENGTH STRING)))
  (LOOP WHILE (< START END)
	AS CH = (SEND SELF :TYI)
	WHILE CH
	DO (ASET CH STRING (PROG1 START (INCF START)))
	FINALLY (AND (ARRAY-HAS-LEADER-P STRING)
		     (STORE-ARRAY-LEADER START STRING 0))
		(AND (NULL CH) EOF (FERROR 'END-OF-FILE-1 "End of file on ~S." SELF))
		(RETURN (VALUES START (NULL CH)))))


(defflavor input-character-stream () (character-stream input-stream)
  :abstract-flavor
; (:select-method-order :read-char :tyi)
  (:required-methods :tyi))

(defmethod (input-character-stream :default :read-char) ()
  (let ((tem (send self :tyi)))
    (if (fixnump tem)
	(int-char tem)
        tem)))

;; These are only here for consistency with any-tyi-no-hang being a method of input-stream
;;  Sigh
;(defmethod (input-character-stream :default :read-char-no-hang) ()
;  (send self :read-char)
;(defmethod (input-character-stream :default :any-read-char) ()
;  (send self :read-char))
;(defmethod (input-character-stream :any-read-char-no-hang) ()
;  (send self :read-char))

(defflavor input-binary-stream () (input-stream)
  :abstract-flavor
; (:select-method-order :read-byte :tyi)
  (:required-methods :tyi)

(defmethod (input-binary-stream :default :read-byte) ()
  (send self :tyi))

(DEFFLAVOR OUTPUT-STREAM () (STREAM)
  (:REQUIRED-METHODS :TYO)
  :ABSTRACT-FLAVOR
; (:SELECT-METHOD-ORDER :TYO)
  (:DOCUMENTATION "All output streams are built on this."))

(DEFMETHOD (OUTPUT-STREAM :DIRECTION) () :OUTPUT)

;;; why is this a method of output-streams rather than output-character-streams?
(DEFMETHOD (OUTPUT-STREAM :DEFAULT :FRESH-LINE) ()
  (SEND SELF :TYO #/NEWLINE)
  T)

;;; why is this a method of output-streams rather than output-character-streams?
(DEFMETHOD (OUTPUT-STREAM :DEFAULT :STRING-OUT) (STRING &OPTIONAL (START 0) END)
  (OR END (SETQ END (ARRAY-ACTIVE-LENGTH STRING)))
  (DO ((I START (1+ I)))
      (( I END))
    (SEND SELF :TYO (GLOBAL:AREF STRING I))))

(DEFMETHOD (OUTPUT-STREAM :DEFAULT :CLEAR-OUTPUT) ()
  NIL)

(DEFMETHOD (OUTPUT-STREAM :DEFAULT :FORCE-OUTPUT) ()
  NIL)

(DEFMETHOD (OUTPUT-STREAM :DEFAULT :FINISH) ()
  NIL)

(DEFMETHOD (OUTPUT-STREAM :BEFORE :FINISH) ()
  (SEND SELF :FORCE-OUTPUT))

(DEFMETHOD (OUTPUT-STREAM :EOF) ()
  (SEND SELF :FINISH))

(defflavor output-character-stream () (character-stream output-stream)
  :abstract-flavor
; (:select-method-order :write-char :tyo)
  (:required-methods :tyo))

(defmethod (input-character-stream :default :write-char) (character)
  (send self :tyo (char-int character)))

(defflavor output-binary-stream () (output-stream)
  :abstract-flavor
; (:select-method-order :write-byte :tyo)
  (:required-methods :tyo))

(defmethod (output-binary-stream :default :write-byte) (byte)
  (send self :tyo byte))

(DEFFLAVOR BIDIRECTIONAL-STREAM () ()
  (:INCLUDED-FLAVORS INPUT-STREAM OUTPUT-STREAM))

(DEFMETHOD (BIDIRECTIONAL-STREAM :DIRECTION) () :BIDIRECTIONAL)

(defflavor bidirectional-character-stream () ()
  (:included-flavors bidirectional-stream input-character-stream output-character-stream))

(defflavor bidirectional-binary-stream () ()
  (:included-flavors bidirectional-stream input-binary-stream output-binary-stream))


;;;; Buffered streams

(DEFFLAVOR BASIC-BUFFERED-INPUT-STREAM
	((STREAM-INPUT-BUFFER NIL)
	 (STREAM-INPUT-INDEX NIL)
	 STREAM-INPUT-LIMIT)
	(INPUT-STREAM)
  :GETTABLE-INSTANCE-VARIABLES
  (:REQUIRED-METHODS :NEXT-INPUT-BUFFER :DISCARD-INPUT-BUFFER)
  (:DOCUMENTATION "Input stream with a buffer.  Defines only a :TYI method.
Requires methods :NEXT-INPUT-BUFFER, which takes a no hang argument and returns three
values, an array, a starting index, and an ending index, or NIL at EOF or no input available
right away.  And :DISCARD-INPUT-BUFFER takes the array back and throws it away someplace."))

;;; Returns non-NIL if any input was to be found.
(DEFMETHOD (BASIC-BUFFERED-INPUT-STREAM :SETUP-NEXT-INPUT-BUFFER) (&OPTIONAL NO-HANG-P)
  (SEND SELF :DISCARD-CURRENT-INPUT-BUFFER)
  (MULTIPLE-VALUE (STREAM-INPUT-BUFFER STREAM-INPUT-INDEX STREAM-INPUT-LIMIT)
    (SEND SELF :NEXT-INPUT-BUFFER NO-HANG-P)))

(DEFMETHOD (BASIC-BUFFERED-INPUT-STREAM :DISCARD-CURRENT-INPUT-BUFFER) ()
  (COND (STREAM-INPUT-BUFFER
	 ;; Pretend that the index has reached the end, so that input remembering will work.
	 (SETQ STREAM-INPUT-INDEX STREAM-INPUT-LIMIT)
	 (SEND SELF :DISCARD-INPUT-BUFFER
		       (PROG1 STREAM-INPUT-BUFFER (SETQ STREAM-INPUT-BUFFER NIL))))))

(DEFUN BASIC-BUFFERED-INPUT-STREAM-TYI (MESSAGE &OPTIONAL EOF)
  (DECLARE (:SELF-FLAVOR BASIC-BUFFERED-INPUT-STREAM))
  (LOOP UNTIL (AND STREAM-INPUT-BUFFER
		   (< STREAM-INPUT-INDEX STREAM-INPUT-LIMIT))
	;; Out of input, get some more
	UNTIL (SEND SELF :SETUP-NEXT-INPUT-BUFFER (EQ MESSAGE :TYI-NO-HANG))
	;; Reached end of file
	RETURN (AND EOF (FERROR 'SYS:END-OF-FILE-1 "End of file on ~S." SELF))
	;; Here we have a character available
	FINALLY (RETURN (PROG1 (GLOBAL:AREF STREAM-INPUT-BUFFER STREAM-INPUT-INDEX)
			       (OR (EQ MESSAGE :TYIPEEK) (INCF STREAM-INPUT-INDEX))))))

(DEFMETHOD (BASIC-BUFFERED-INPUT-STREAM :TYI) BASIC-BUFFERED-INPUT-STREAM-TYI)
(DEFMETHOD (BASIC-BUFFERED-INPUT-STREAM :TYI-NO-HANG) BASIC-BUFFERED-INPUT-STREAM-TYI)
(DEFMETHOD (BASIC-BUFFERED-INPUT-STREAM :TYIPEEK) BASIC-BUFFERED-INPUT-STREAM-TYI)

(DEFMETHOD (BASIC-BUFFERED-INPUT-STREAM :UNTYI) (CH)
  (LET ((NEW-INDEX (AND STREAM-INPUT-BUFFER
			(1- STREAM-INPUT-INDEX))))
    (COND ((AND NEW-INDEX
		( NEW-INDEX 0)
		(EQ (GLOBAL:AREF STREAM-INPUT-BUFFER NEW-INDEX) CH))
	   (SETQ STREAM-INPUT-INDEX NEW-INDEX))
	  (T
	   (FERROR NIL "Attempt to :UNTYI something different from last :TYI'ed.")))))

(DEFMETHOD (BASIC-BUFFERED-INPUT-STREAM :LISTEN) ()
  (OR (AND STREAM-INPUT-BUFFER
	   (< STREAM-INPUT-INDEX STREAM-INPUT-LIMIT))
      (SEND SELF :SETUP-NEXT-INPUT-BUFFER T)))

(DEFMETHOD (BASIC-BUFFERED-INPUT-STREAM :READ-UNTIL-EOF) ()
  (LOOP WHILE (SEND SELF :SETUP-NEXT-INPUT-BUFFER)))

(DEFMETHOD (BASIC-BUFFERED-INPUT-STREAM :CLEAR-INPUT) ()
  (SEND SELF :DISCARD-CURRENT-INPUT-BUFFER))

(DEFMETHOD (BASIC-BUFFERED-INPUT-STREAM :BEFORE :CLOSE) (&OPTIONAL IGNORE)
  (SEND SELF :DISCARD-CURRENT-INPUT-BUFFER))


(DEFFLAVOR BUFFERED-INPUT-STREAM () (BASIC-BUFFERED-INPUT-STREAM)
  (:DOCUMENTATION "Buffered input stream with :READ-INPUT-BUFFER.
Usef for all buffered input streams for which :TYI doesn't have wrappers to do
translation or such."))

(DEFMETHOD (BUFFERED-INPUT-STREAM :GET-INPUT-BUFFER) (&OPTIONAL EOF-ERROR-P)
  (LOOP UNTIL (AND STREAM-INPUT-BUFFER
		   (< STREAM-INPUT-INDEX STREAM-INPUT-LIMIT))
	;; Out of input, get some more
	UNTIL (SEND SELF :SETUP-NEXT-INPUT-BUFFER)
	;; Reached end of file
	RETURN (AND EOF-ERROR-P (FERROR 'SYS:END-OF-FILE-1 "End of file on ~S." SELF))
	;; Here we have a non-empty available
	FINALLY (RETURN (VALUES STREAM-INPUT-BUFFER
				STREAM-INPUT-INDEX
				(- STREAM-INPUT-LIMIT STREAM-INPUT-INDEX)))))

(DEFMETHOD (BUFFERED-INPUT-STREAM :READ-INPUT-BUFFER) (&OPTIONAL EOF-ERROR-P)
  (LOOP UNTIL (AND STREAM-INPUT-BUFFER
		   (< STREAM-INPUT-INDEX STREAM-INPUT-LIMIT))
	;; Out of input, get some more
	UNTIL (SEND SELF :SETUP-NEXT-INPUT-BUFFER)
	;; Reached end of file
	RETURN (AND EOF-ERROR-P (FERROR 'SYS:END-OF-FILE-1 "End of file on ~S." SELF))
	;; Here we have a non-empty available
	FINALLY (RETURN (VALUES STREAM-INPUT-BUFFER STREAM-INPUT-INDEX STREAM-INPUT-LIMIT))))

(DEFMETHOD (BUFFERED-INPUT-STREAM :STRING-IN) (EOF-ERROR-P STRING &OPTIONAL (START 0) END)
  (OR END (SETQ END (ARRAY-LENGTH STRING)))
  (LOOP WHILE (< START END)
	WHILE (LOOP UNTIL (AND STREAM-INPUT-BUFFER
			       (< STREAM-INPUT-INDEX STREAM-INPUT-LIMIT))
		    ;; Out of input, get some more
		    UNTIL (SEND SELF :SETUP-NEXT-INPUT-BUFFER)
		    DO (AND EOF-ERROR-P (FERROR 'SYS:END-OF-FILE-1 "End of file on ~S." SELF))
		    RETURN NIL
		    FINALLY (RETURN T))
	AS AMT = (MIN (- END START) (- STREAM-INPUT-LIMIT STREAM-INPUT-INDEX))
	DO (COPY-ARRAY-PORTION STREAM-INPUT-BUFFER STREAM-INPUT-INDEX
			       (SETQ STREAM-INPUT-INDEX (+ STREAM-INPUT-INDEX AMT))
			       STRING START (SETQ START (+ START AMT)))
	FINALLY (AND (ARRAY-HAS-LEADER-P STRING)
		     (SETF (FILL-POINTER STRING) START))
		(RETURN (VALUES START
				(NULL STREAM-INPUT-BUFFER)))))

(DEFMETHOD (BUFFERED-INPUT-STREAM :ADVANCE-INPUT-BUFFER) (&OPTIONAL NEW-INDEX)
  (COND (NEW-INDEX
	 (OR (AND ( NEW-INDEX 0)
		  ( NEW-INDEX STREAM-INPUT-LIMIT))
	     (FERROR NIL "New index out of range"))
	 (SETQ STREAM-INPUT-INDEX NEW-INDEX))
	(T
	 (SEND SELF :DISCARD-CURRENT-INPUT-BUFFER))))

(DEFFLAVOR BASIC-BUFFERED-OUTPUT-STREAM
	((STREAM-OUTPUT-BUFFER NIL)
	 (STREAM-OUTPUT-INDEX NIL)
	 STREAM-OUTPUT-LIMIT)
	(OUTPUT-STREAM)
  :GETTABLE-INSTANCE-VARIABLES
  (:REQUIRED-METHODS :NEW-OUTPUT-BUFFER :SEND-OUTPUT-BUFFER :DISCARD-OUTPUT-BUFFER)
  (:DOCUMENTATION "Output stream with a buffer.  Only gives a :TYO method.
Required methods are :NEW-OUTPUT-BUFFER, which returns three values, an array, starting index,
and ending index into which characters can be stuffed.  And :SEND-OUTPUT-BUFFER takes the
array and the ending output index reached, and transmit to the particular device.
:DISCARD-OUTPUT-BUFFER takes the array and should forget about sending the buffered data."))

(DEFMETHOD (BASIC-BUFFERED-OUTPUT-STREAM :SETUP-NEW-OUTPUT-BUFFER) ()
  (SEND SELF :SEND-CURRENT-OUTPUT-BUFFER)
  (MULTIPLE-VALUE (STREAM-OUTPUT-BUFFER STREAM-OUTPUT-INDEX STREAM-OUTPUT-LIMIT)
    (SEND SELF :NEW-OUTPUT-BUFFER)))

(DEFMETHOD (BASIC-BUFFERED-OUTPUT-STREAM :SEND-CURRENT-OUTPUT-BUFFER) ()
  (COND (STREAM-OUTPUT-BUFFER
	 (SEND SELF :SEND-OUTPUT-BUFFER
		       ;; If aborted out of write, prefer losing data to
		       ;; getting links circular.
		       (PROG1 STREAM-OUTPUT-BUFFER (SETQ STREAM-OUTPUT-BUFFER NIL))
		       STREAM-OUTPUT-INDEX))))

(DEFMETHOD (BASIC-BUFFERED-OUTPUT-STREAM :DISCARD-CURRENT-OUTPUT-BUFFER) ()
  (AND STREAM-OUTPUT-BUFFER
       (SEND SELF :DISCARD-OUTPUT-BUFFER (PROG1 STREAM-OUTPUT-BUFFER
						    (SETQ STREAM-OUTPUT-BUFFER NIL)))))

(DEFMETHOD (BASIC-BUFFERED-OUTPUT-STREAM :TYO) (CH)
  (LOOP UNTIL (AND STREAM-OUTPUT-BUFFER
		   (< STREAM-OUTPUT-INDEX STREAM-OUTPUT-LIMIT))
	DO (SEND SELF :SETUP-NEW-OUTPUT-BUFFER)
	FINALLY (ASET CH STREAM-OUTPUT-BUFFER STREAM-OUTPUT-INDEX)
		(INCF STREAM-OUTPUT-INDEX)))

(DEFMETHOD (BASIC-BUFFERED-OUTPUT-STREAM :PAD-AND-SEND-BUFFER) (&OPTIONAL (PAD 0))
  (COND (STREAM-OUTPUT-BUFFER
	 (LOOP UNTIL (NOT (< STREAM-OUTPUT-INDEX STREAM-OUTPUT-LIMIT))
	       DO (PROGN (ASET PAD STREAM-OUTPUT-BUFFER STREAM-OUTPUT-INDEX)
			 (INCF STREAM-OUTPUT-INDEX))
	       FINALLY (SEND SELF :SEND-CURRENT-OUTPUT-BUFFER)))))


(DEFMETHOD (BASIC-BUFFERED-OUTPUT-STREAM :FORCE-OUTPUT) ()
  (SEND SELF :SEND-CURRENT-OUTPUT-BUFFER))

(DEFMETHOD (BASIC-BUFFERED-OUTPUT-STREAM :BEFORE :CLOSE) (&OPTIONAL ABORTP)
  (SEND SELF (IF ABORTP :DISCARD-CURRENT-OUTPUT-BUFFER
	                :SEND-CURRENT-OUTPUT-BUFFER)))

(DEFMETHOD (BASIC-BUFFERED-OUTPUT-STREAM :LAST-CHAR-OUTPUT) ()
  (AND STREAM-OUTPUT-BUFFER
       (PLUSP STREAM-OUTPUT-INDEX)
       (GLOBAL:AREF STREAM-OUTPUT-BUFFER (1- STREAM-OUTPUT-INDEX))))

(DEFFLAVOR BUFFERED-OUTPUT-STREAM () (BUFFERED-OUTPUT-STREAM-MIXIN
				      BASIC-BUFFERED-OUTPUT-STREAM)
  (:DOCUMENTATION "Buffered output stream with :STRING-OUT.
Should be used for all output streams which do not have wrappers on :TYO to do translation
or such like."))

(DEFFLAVOR BUFFERED-OUTPUT-STREAM-MIXIN () ()
  (:REQUIRED-FLAVORS BASIC-BUFFERED-OUTPUT-STREAM))

(DEFMETHOD (BUFFERED-OUTPUT-STREAM-MIXIN :STRING-OUT) (STRING &OPTIONAL (START 0) END)
  (OR END (SETQ END (ARRAY-ACTIVE-LENGTH STRING)))
  (LOOP WHILE (< START END)
	UNLESS STREAM-OUTPUT-BUFFER
	  DO (SEND SELF :SETUP-NEW-OUTPUT-BUFFER)
	AS AMT = (MIN (- END START) (- STREAM-OUTPUT-LIMIT STREAM-OUTPUT-INDEX))
	DO (COPY-ARRAY-PORTION STRING START (SETQ START (+ START AMT))
			       STREAM-OUTPUT-BUFFER STREAM-OUTPUT-INDEX
			       (SETQ STREAM-OUTPUT-INDEX (+ STREAM-OUTPUT-INDEX AMT)))
	WHEN (< START END)
	  DO (SEND SELF :SEND-CURRENT-OUTPUT-BUFFER)))

;;; Since this stream has enough knowledge to execute this message intelligently, let it.
;;; Note that this message can't go in BASIC-BUFFERED-OUTPUT-STREAM, because wrappers may
;;; be defined on :TYO messages which would cause it to lose.
(DEFMETHOD (BUFFERED-OUTPUT-STREAM-MIXIN :FRESH-LINE) ()
  (IF (EQ #/NEWLINE (OR (SEND SELF :LAST-CHAR-OUTPUT) 0))
      NIL
    (SEND SELF :TYO #/NEWLINE)
    T))


;;;; LINE input and output

;;; This comes in two different flavors, depending on whether the stream is buffered
;;; or unbuffered

(DEFFLAVOR UNBUFFERED-LINE-INPUT-STREAM () (INPUT-CHARACTER-STREAM)
  (:DOCUMENTATION "Character Input stream with :LINE-IN but no buffering.
Used with input streams which only support :READ-CHAR."))

(DEFMETHOD (UNBUFFERED-LINE-INPUT-STREAM :STRING-LINE-IN) (EOF-ERROR-P STRING
							   &OPTIONAL (START 0) END)
  (OR END (SETQ END (ARRAY-LENGTH STRING)))
  (LOOP WHILE (< START END)
	AS CH = (SEND SELF :READ-CHAR NIL NIL)
	WHILE (AND CH (NEQ CH #/NEWLINE))
	DO (SETF (CHAR STRING (PROG1 START (INCF START))) CH)
	FINALLY (AND (ARRAY-HAS-LEADER-P STRING)
		     (SETF (FILL-POINTER STRING) START))
		(AND (NULL CH) EOF-ERROR-P (FERROR 'END-OF-FILE-1 "End of file on ~S." SELF))
		(RETURN (VALUES START
				(NULL CH)
				(NEQ CH #/RETURN)))))

(DEFMETHOD (UNBUFFERED-LINE-INPUT-STREAM :LINE-IN) (&OPTIONAL LEADER)
  (LOOP WITH LENGTH-SO-FAR = 0
	 AND LINE = (MAKE-STRING #o80 :LEADER-LENGTH (AND (NUMBERP LEADER) LEADER))
	 AS CH = (SEND SELF :READ-CHAR NIL NIL)
	 UNTIL (NULL CH)		;i.e. EOF
	 UNTIL (EQ CH #/NEWLINE)
	 ;; We have an ordinary character, stick it on the end of the line
	 WHEN ( LENGTH-SO-FAR (ARRAY-LENGTH LINE))
	   DO (SETQ LINE (ADJUST-ARRAY-SIZE LINE (FLOOR (* LENGTH-SO-FAR 3) 2)))
	 DO (SETF (CHAR LINE LENGTH-SO-FAR) CH)
	    (INCF LENGTH-SO-FAR)
	 FINALLY
	  ;; Adjust size and active-length of line
	  (ADJUST-ARRAY-SIZE LINE LENGTH-SO-FAR)
	  (IF (ARRAY-HAS-LEADER-P LINE)
	      (SETF (FILL-POINTER LINE) LENGTH-SO-FAR))
	  (RETURN (VALUES LINE (NULL CH)))))

(DEFFLAVOR LINE-OUTPUT-STREAM-MIXIN () ()
  (:REQUIRED-METHODS :STRING-OUT)
  (:INCLUDED-FLAVORS OUTPUT-CHARACTER-STREAM)
  (:DOCUMENTATION "Output character stream with :LINE-OUT.
Used for buffered and unbuffered streams."))

(DEFMETHOD (LINE-OUTPUT-STREAM-MIXIN :LINE-OUT) (LINE &OPTIONAL (START 0) END)
  (SEND SELF :STRING-OUT LINE START END)
  (SEND SELF :WRITE-CHAR #/NEWLINE))


(DEFFLAVOR BUFFERED-LINE-INPUT-STREAM () (INPUT-CHARACTER-STREAM BUFFERED-INPUT-STREAM)
  (:DOCUMENTATION "Input character stream with buffering and :LINE-IN."))

;(DEFMETHOD (BUFFERED-LINE-INPUT-STREAM :LINE-IN) (&OPTIONAL LEADER)
;  (LOOP WITH LENGTH-SO-FAR = 0
;	 AND LINE = (MAKE-ARRAY 80 :TYPE 'ART-STRING
;				   :LEADER-LENGTH (AND (NUMBERP LEADER) LEADER))
;	UNLESS (AND STREAM-INPUT-BUFFER
;		    (< STREAM-INPUT-INDEX STREAM-INPUT-LIMIT))
;	  ;Out of buffer, get another one
;	  DO (SEND SELF :SETUP-NEXT-INPUT-BUFFER)
;	UNTIL (NULL STREAM-INPUT-BUFFER) 	;i.e. EOF
;	;We have a non-empty buffer, search for CR in it
;	AS CR-IDX = (%STRING-SEARCH-CHAR #\CR STREAM-INPUT-BUFFER STREAM-INPUT-INDEX
;					 STREAM-INPUT-LIMIT)
;	AS AMT = (- (OR CR-IDX STREAM-INPUT-LIMIT) STREAM-INPUT-INDEX)
;	;Nconc this many characters onto the end of the line
;	WHEN (> (+ AMT LENGTH-SO-FAR) (ARRAY-LENGTH LINE))
;	  DO (SETQ LINE (ADJUST-ARRAY-SIZE LINE (+ AMT LENGTH-SO-FAR)))
;	DO (COPY-ARRAY-PORTION STREAM-INPUT-BUFFER STREAM-INPUT-INDEX
;			         (SETQ STREAM-INPUT-INDEX (+ STREAM-INPUT-INDEX AMT))
;			       LINE LENGTH-SO-FAR (SETQ LENGTH-SO-FAR (+ LENGTH-SO-FAR AMT)))
;	UNTIL CR-IDX		;i.e. until we saw a CR
;	FINALLY
;	  ;If we terminated with a CR, advance over it
;	  (IF STREAM-INPUT-BUFFER (INCF STREAM-INPUT-INDEX))
;	  ;Adjust size and active-length of line
;	  (ADJUST-ARRAY-SIZE LINE LENGTH-SO-FAR)
;	  (IF (ARRAY-HAS-LEADER-P LINE) (STORE-ARRAY-LEADER LENGTH-SO-FAR LINE 0))
;	  (RETURN (VALUES LINE (NULL STREAM-INPUT-BUFFER)))))

(DEFMETHOD (BUFFERED-LINE-INPUT-STREAM :STRING-LINE-IN) (EOF STRING &OPTIONAL (START 0) END)
  (OR END (SETQ END (ARRAY-LENGTH STRING)))
  (LOOP WHILE (< START END)
	WHILE (LOOP UNTIL (AND STREAM-INPUT-BUFFER
			       (< STREAM-INPUT-INDEX STREAM-INPUT-LIMIT))
		    ;; Out of input, get some more
		    UNTIL (SEND SELF :SETUP-NEXT-INPUT-BUFFER)
		    DO (AND EOF (FERROR 'SYS:END-OF-FILE-1 "End of file on ~S." SELF))
		    RETURN NIL
		    FINALLY (RETURN T))
	AS LINE-END-INDEX = (%STRING-SEARCH-CHAR #/RETURN STREAM-INPUT-BUFFER
						 STREAM-INPUT-INDEX STREAM-INPUT-LIMIT)
	AS AMT = (MIN (- END START)
		      (- (OR LINE-END-INDEX STREAM-INPUT-LIMIT) STREAM-INPUT-INDEX))
	DO (COPY-ARRAY-PORTION STREAM-INPUT-BUFFER STREAM-INPUT-INDEX
			       (INCF STREAM-INPUT-INDEX AMT)
			       STRING START (INCF START AMT))
	   (WHEN (EQ STREAM-INPUT-INDEX LINE-END-INDEX)
	     (INCF STREAM-INPUT-INDEX)
	     (RETURN (VALUES START NIL NIL)))
	FINALLY (AND (ARRAY-HAS-LEADER-P STRING)
		     (SETF (FILL-POINTER STRING) START))
		(RETURN (VALUES START
				(NULL STREAM-INPUT-BUFFER)
				T))))

(DEFMETHOD (BUFFERED-LINE-INPUT-STREAM :LINE-IN) (&OPTIONAL LEADER)
  (BLOCK LINE-IN
    (LOOP
      ;; STRING is not made until needed to avoid calling ADJUST-ARRAY-SIZE except when
      ;; strings cross buffer boundaries.
      WITH STRING = NIL
      AND STRING-INDEX = 0
      DO (LOOP UNTIL (AND STREAM-INPUT-BUFFER
			  (< STREAM-INPUT-INDEX STREAM-INPUT-LIMIT))
	       ;Out of input, get some more
	       UNTIL (SEND SELF :SETUP-NEXT-INPUT-BUFFER)
	       ;EOF if none that way
	       WHEN (NULL STRING)
	       DO (SETQ STRING (MAKE-STRING STRING-INDEX
					    :LEADER-LENGTH (AND (NUMBERP LEADER) LEADER)))
	       WHEN (NUMBERP LEADER)
	         DO (SETF (FILL-POINTER STRING) STRING-INDEX)
	       DO (RETURN-FROM LINE-IN (VALUES STRING T)))
      ;; Now see if this buffer has a CR, and copy out the appropriate amount
      AS CR-INDEX = (%STRING-SEARCH-CHAR #/NEWLINE STREAM-INPUT-BUFFER STREAM-INPUT-INDEX
					 STREAM-INPUT-LIMIT)
      WITH NEW-STRING-INDEX AND NEW-BUFFER-INDEX
      WHEN (NOT (NULL CR-INDEX))
;	  WHEN (AND (NULL STRING) (NULL LEADER))
;	    DO (SETQ STRING (LET ((ARRAY STREAM-INPUT-BUFFER)
;				  (OFFSET STREAM-INPUT-INDEX))
;			      (AND (ARRAY-INDEXED-P ARRAY)
;				   (SETQ OFFSET (+ OFFSET (%P-CONTENTS-OFFSET ARRAY 3))
;					 ARRAY (%P-CONTENTS-OFFSET ARRAY 1)))
;			      (MAKE-ARRAY (- CR-INDEX STREAM-INPUT-INDEX)
;					  :TYPE 'ART-STRING
;					  :DISPLACED-TO ARRAY
;					  :DISPLACED-INDEX-OFFSET OFFSET))
;		     STREAM-INPUT-INDEX (1+ CR-INDEX))
;	       (RETURN (VALUES STRING NIL))
;	  ELSE
        DO (SETQ NEW-BUFFER-INDEX (1+ CR-INDEX)
		 NEW-STRING-INDEX (+ STRING-INDEX (- CR-INDEX STREAM-INPUT-INDEX)))
	ELSE DO (SETQ NEW-BUFFER-INDEX STREAM-INPUT-LIMIT
		      NEW-STRING-INDEX (+ STRING-INDEX
					  (- STREAM-INPUT-LIMIT STREAM-INPUT-INDEX)))
	WHEN (NULL STRING)
	  ;;Make a string to return or save the end of this packet in.
	  DO (SETQ STRING (MAKE-STRING NEW-STRING-INDEX
				       :LEADER-LENGTH (AND (NUMBERP LEADER) LEADER)))
	  ;;Was some stuff from previous packet, make room.
	  ELSE DO (ADJUST-ARRAY-SIZE STRING NEW-STRING-INDEX)
	DO (COPY-ARRAY-PORTION STREAM-INPUT-BUFFER STREAM-INPUT-INDEX NEW-BUFFER-INDEX
			       STRING STRING-INDEX NEW-STRING-INDEX)
	   (SETQ STREAM-INPUT-INDEX NEW-BUFFER-INDEX
		 STRING-INDEX NEW-STRING-INDEX)
	WHEN (NOT (NULL CR-INDEX))		;This buffer is enough to satisfy
	DO (AND (NUMBERP LEADER) (SETF (FILL-POINTER STRING) STRING-INDEX))
	   (RETURN (VALUES STRING NIL)))))


;;;; Less basic stream flavors
(DEFFLAVOR ASCII-TRANSLATING-INPUT-STREAM-MIXIN
	() ()
  (:REQUIRED-METHODS :TYI)
  (:INCLUDED-FLAVORS INPUT-CHARACTER-STREAM)
  (:DOCUMENTATION "A character input stream that translates characters from ASCII
into lisp machine character set for :TYI method."))

(DEFWRAPPER (ASCII-TRANSLATING-INPUT-STREAM-MIXIN :TYI) (IGNORE . BODY)
  `(PROGN
     .DAEMON-CALLER-ARGS.			;Prevent compiler warnings
     (TYI-FROM-ASCII-STREAM #'(LAMBDA (&REST .DAEMON-CALLER-ARGS.
				       &AUX (.DAEMON-MAPPING-TABLE. SELF-MAPPING-TABLE))
				.DAEMON-MAPPING-TABLE.
				. ,BODY))))

(DEFUN TYI-FROM-ASCII-STREAM (ASCII-STREAM &AUX CH)
  (CASE (SETQ CH (SEND ASCII-STREAM :TYI))
    (#o10 #/BS)
    (#o11 #/TAB)
    (#o12 #/LINE)
    (#o14 #/FF)
    (#o15 (LET ((CH1 (SEND ASCII-STREAM :TYI)))
	    (OR (= CH1 #o12) (SEND ASCII-STREAM :UNTYI CH1)))
	  #/NEWLINE)
    (#o177 #/RUBOUT)
    (T CH)))

(DEFFLAVOR ASCII-TRANSLATING-OUTPUT-STREAM-MIXIN
	() ()
  (:REQUIRED-METHODS :TYO)
  (:INCLUDED-FLAVORS OUTPUT-CHARACTER-STREAM)
  (:DOCUMENTATION "A character output stream that translates characters from
the lisp machine character set into ASCII for the :TYO method."))

(DEFWRAPPER (ASCII-TRANSLATING-OUTPUT-STREAM-MIXIN :TYO) ((CH) . BODY)
  `(TYO-TO-ASCII-STREAM #'(LAMBDA (&REST .DAEMON-CALLER-ARGS.
				   &AUX (.DAEMON-MAPPING-TABLE. SELF-MAPPING-TABLE))
			    .DAEMON-MAPPING-TABLE.
			    . ,BODY)
			CH))

(DEFUN TYO-TO-ASCII-STREAM (ASCII-STREAM CH)
  (SEND ASCII-STREAM :TYO
	(CASE CH
	  (#/BS #o10)
	  (#/TAB #o11)
	  (#/LINE #o12)
	  (#/FF #o14)
	  (#/NEWLINE (SEND ASCII-STREAM :TYO #o15)
		     #o12)
	  (#/RUBOUT #o177)
	  (T CH))))

(DEFFLAVOR INPUT-POINTER-REMEMBERING-MIXIN
	((INPUT-POINTER-BASE 0)
	 (STREAM-INPUT-LOWER-LIMIT 0))
	()
  (:INCLUDED-FLAVORS BASIC-BUFFERED-INPUT-STREAM)
  ;; :SET-BUFFER-POINTER is sent when a :SET-POINTER request goes beyond the current buffer.
  ;; It should return the real position set and arrange for the next :NEXT-INPUT-BUFFER
  ;; to contain the desired position in it someplace.
  (:REQUIRED-METHODS :SET-BUFFER-POINTER)
  (:DOCUMENTATION "Buffered input stream with :SET-POINTER and :READ-POINTER methods."))

(DEFMETHOD (INPUT-POINTER-REMEMBERING-MIXIN :SET-BUFFER-POINTER) (NEW-POINTER)
  (FERROR NIL "Cannot set pointer on ~S to ~D" SELF NEW-POINTER))

;;; Obsolete shorthand message, but in the manual, so keep for a while.
(DEFMETHOD (INPUT-POINTER-REMEMBERING-MIXIN :REWIND) ()
  (SEND SELF :SET-POINTER 0))

(DEFMETHOD (INPUT-POINTER-REMEMBERING-MIXIN :SET-POINTER) (NEW-POINTER)
  (LOOP AS NEW-RELATIVE-POINTER = (+ (- NEW-POINTER INPUT-POINTER-BASE)
				     STREAM-INPUT-LOWER-LIMIT)
	WHEN (AND STREAM-INPUT-INDEX (= NEW-RELATIVE-POINTER STREAM-INPUT-INDEX))
	RETURN T				;Fast check
	UNTIL (IF STREAM-INPUT-BUFFER
		  (AND ( NEW-RELATIVE-POINTER STREAM-INPUT-LOWER-LIMIT)
		       (< NEW-RELATIVE-POINTER STREAM-INPUT-LIMIT))
		  (= NEW-RELATIVE-POINTER STREAM-INPUT-LOWER-LIMIT))
	DO (SEND SELF :DISCARD-CURRENT-INPUT-BUFFER)
	   (SETQ INPUT-POINTER-BASE (SEND SELF :SET-BUFFER-POINTER NEW-POINTER))
	   (SEND SELF :SETUP-NEXT-INPUT-BUFFER)
	FINALLY (SETQ STREAM-INPUT-INDEX NEW-RELATIVE-POINTER)))

(DEFMETHOD (INPUT-POINTER-REMEMBERING-MIXIN :BEFORE :DISCARD-INPUT-BUFFER) (IGNORE)
  (INCF INPUT-POINTER-BASE (- STREAM-INPUT-INDEX STREAM-INPUT-LOWER-LIMIT))
  (SETQ STREAM-INPUT-LOWER-LIMIT STREAM-INPUT-INDEX))

(DEFMETHOD (INPUT-POINTER-REMEMBERING-MIXIN :AFTER :SETUP-NEXT-INPUT-BUFFER)
	   (&OPTIONAL IGNORE)
  (AND STREAM-INPUT-BUFFER
       (SETQ STREAM-INPUT-LOWER-LIMIT STREAM-INPUT-INDEX)))

(DEFMETHOD (INPUT-POINTER-REMEMBERING-MIXIN :READ-POINTER) ()
  (+ INPUT-POINTER-BASE (IF STREAM-INPUT-INDEX
			    (- STREAM-INPUT-INDEX STREAM-INPUT-LOWER-LIMIT)
			    0)))

(DEFFLAVOR OUTPUT-POINTER-REMEMBERING-MIXIN
	((OUTPUT-POINTER-BASE 0)
	 (STREAM-OUTPUT-LOWER-LIMIT 0))
	()
  (:SETTABLE-INSTANCE-VARIABLES OUTPUT-POINTER-BASE)
  (:INCLUDED-FLAVORS BASIC-BUFFERED-OUTPUT-STREAM)
  (:DOCUMENTATION "Buffered output stream with :READ-POINTER method."))

(DEFMETHOD (OUTPUT-POINTER-REMEMBERING-MIXIN :BEFORE :SEND-OUTPUT-BUFFER) (IGNORE IGNORE)
  (INCF OUTPUT-POINTER-BASE (- STREAM-OUTPUT-INDEX STREAM-OUTPUT-LOWER-LIMIT))
  (SETQ STREAM-OUTPUT-LOWER-LIMIT STREAM-OUTPUT-INDEX))

(DEFMETHOD (OUTPUT-POINTER-REMEMBERING-MIXIN :AFTER :SETUP-NEW-OUTPUT-BUFFER) ()
  (SETQ STREAM-OUTPUT-LOWER-LIMIT STREAM-OUTPUT-INDEX))

(DEFMETHOD (OUTPUT-POINTER-REMEMBERING-MIXIN :READ-POINTER) ()
  (+ OUTPUT-POINTER-BASE (IF STREAM-OUTPUT-INDEX
			    (- STREAM-OUTPUT-INDEX STREAM-OUTPUT-LOWER-LIMIT)
			    0)))

;;; We send a :GET-OLD-DATA (args are buffer and ending index) if it is
;;; necessary to fill the beginning of the output buffer with the current
;;; contents of the file in order to avoid clobbering data that we are
;;; supposed to be skipping over.
(DEFMETHOD (OUTPUT-POINTER-REMEMBERING-MIXIN :SET-POINTER) (NEW-POINTER)
  (DO ((BUFFER-RELOADED NIL T)) (())
    (LET ((NEW-RELATIVE-POINTER (+ (- NEW-POINTER OUTPUT-POINTER-BASE)
				   STREAM-OUTPUT-LOWER-LIMIT)))
      ;; Does the buffer we have now include the desired pointer?
      (IF (IF STREAM-OUTPUT-BUFFER
	      (AND ( NEW-RELATIVE-POINTER STREAM-OUTPUT-LOWER-LIMIT)
		   (< NEW-RELATIVE-POINTER STREAM-OUTPUT-LIMIT))
	    (= NEW-RELATIVE-POINTER STREAM-OUTPUT-LOWER-LIMIT))
	  (PROGN (SETQ STREAM-OUTPUT-INDEX NEW-RELATIVE-POINTER)
		 (AND BUFFER-RELOADED
		      (SEND SELF :GET-OLD-DATA
				    STREAM-OUTPUT-BUFFER STREAM-OUTPUT-LOWER-LIMIT))
		 (RETURN (+ OUTPUT-POINTER-BASE
			    (IF STREAM-OUTPUT-INDEX
				(- STREAM-OUTPUT-INDEX STREAM-OUTPUT-LOWER-LIMIT)
			      0)))))
      ;; No, get another buffer after specifying where in the file we want.
      (SEND SELF :SEND-CURRENT-OUTPUT-BUFFER)
      (SETQ OUTPUT-POINTER-BASE
	    (SEND SELF :SET-BUFFER-POINTER NEW-POINTER))
      (SEND SELF :SETUP-NEW-OUTPUT-BUFFER))))

(DEFMETHOD (OUTPUT-POINTER-REMEMBERING-MIXIN :DEFAULT :GET-OLD-DATA)
	   (BUFFER-ARRAY OUTPUT-LOWER-LIMIT)
  BUFFER-ARRAY OUTPUT-LOWER-LIMIT
  NIL)  


;;; Some useful combinations
(DEFFLAVOR BUFFERED-INPUT-CHARACTER-STREAM ()
	   (INPUT-POINTER-REMEMBERING-MIXIN BUFFERED-LINE-INPUT-STREAM)
  (:DOCUMENTATION "A buffered input character stream, gives :LINE-IN."))

(DEFFLAVOR BUFFERED-OUTPUT-CHARACTER-STREAM ()
	   (LINE-OUTPUT-STREAM-MIXIN CHARACTER-STREAM BUFFERED-OUTPUT-STREAM)
  (:DOCUMENTATION "A buffered output character stream, gives :LINE-OUT."))

(DEFFLAVOR BUFFERED-CHARACTER-STREAM ()
	   (BIDIRECTIONAL-STREAM INPUT-POINTER-REMEMBERING-MIXIN BUFFERED-LINE-INPUT-STREAM
	    LINE-OUTPUT-STREAM-MIXIN BUFFERED-OUTPUT-STREAM)
  (:DOCUMENTATION "A bidrection character stream, :LINE-IN and :LINE-OUT."))

(DEFFLAVOR BUFFERED-STREAM ()
	   (BIDIRECTIONAL-STREAM BUFFERED-INPUT-STREAM BUFFERED-OUTPUT-STREAM)
  (:DOCUMENTATION "A bidirection buffered stream."))

(DEFFLAVOR BUFFERED-TYI-INPUT-STREAM ()
	   (INPUT-POINTER-REMEMBERING-MIXIN UNBUFFERED-LINE-INPUT-STREAM
	    BASIC-BUFFERED-INPUT-STREAM)
  (:DOCUMENTATION "A buffered character input stream for use with :TYI wrappers."))

(DEFFLAVOR BUFFERED-TYO-OUTPUT-STREAM ()
	   (LINE-OUTPUT-STREAM-MIXIN CHARACTER-STREAM BASIC-BUFFERED-OUTPUT-STREAM)
  (:DOCUMENTATION "A buffered character output stream for use with :TYO wrappers."))

(DEFFLAVOR BUFFERED-TYI-TYO-STREAM ()
	   (BIDIRECTIONAL-STREAM INPUT-POINTER-REMEMBERING-MIXIN UNBUFFERED-LINE-INPUT-STREAM
	    BASIC-BUFFERED-INPUT-STREAM LINE-OUTPUT-STREAM-MIXIN BASIC-BUFFERED-OUTPUT-STREAM)
  (:DOCUMENTATION "A bidirectional buffered character stream, for use with
:TYI and :TYO wrappers."))


;;;; Flavors for use with file computers

;;; For use with :PROBE OPEN calls
(DEFFLAVOR FILE-STREAM-MIXIN
	(PATHNAME)
	()
  (:INCLUDED-FLAVORS STREAM)
  (:REQUIRED-METHODS :TRUENAME :PROPERTY-LIST)
  (:INITABLE-INSTANCE-VARIABLES PATHNAME)
  (:GETTABLE-INSTANCE-VARIABLES PATHNAME)
  (:DOCUMENTATION "Streams for use with file computers, as returned by OPEN."))

(DEFMETHOD (FILE-STREAM-MIXIN :PRINT-SELF) (STREAM IGNORE IGNORE)
  (SYS:PRINTING-RANDOM-OBJECT (SELF STREAM :TYPE)
    (PRIN1 (STRING PATHNAME) STREAM)))

(DEFMETHOD (FILE-STREAM-MIXIN :GET) (INDICATOR)
  (LET ((PLIST (SEND SELF :PROPERTY-LIST)))
    (GETF PLIST INDICATOR)))

(DEFMETHOD (FILE-STREAM-MIXIN :CREATION-DATE) ()
  (SEND SELF :GET :CREATION-DATE))

(DEFMETHOD (FILE-STREAM-MIXIN :INFO) ()
  (CONS (SEND SELF :TRUENAME) (SEND SELF :CREATION-DATE)))

(DEFMETHOD (FILE-STREAM-MIXIN :GENERIC-PATHNAME) ()
  (SEND PATHNAME :GENERIC-PATHNAME))

;;; For use with :READ and :WRITE OPEN calls
(DEFFLAVOR FILE-DATA-STREAM-MIXIN
	()
	(FILE-STREAM-MIXIN)
  (:INCLUDED-FLAVORS STREAM)
  (:REQUIRED-METHODS :READ-POINTER :LENGTH :QFASLP)
  (:DOCUMENTATION "Streams which can actually do file I/O."))

(DEFMETHOD (FILE-DATA-STREAM-MIXIN :AFTER :INIT) (IGNORE)
  (SEND TV:WHO-LINE-FILE-STATE-SHEET :ADD-STREAM SELF))

(DEFMETHOD (FILE-DATA-STREAM-MIXIN :AFTER :CLOSE) (&OPTIONAL IGNORE)
  (SEND TV:WHO-LINE-FILE-STATE-SHEET :DELETE-STREAM SELF))

(DEFMETHOD (FILE-DATA-STREAM-MIXIN :WHO-LINE-INFORMATION)
	   (&AUX COUNT LENGTH PERCENT DIRECTION)
  (SETQ COUNT (SEND SELF :READ-POINTER))
  (SETQ DIRECTION (SEND SELF :DIRECTION))
  (AND (SETQ LENGTH (SEND SELF :LENGTH))
       (NOT (ZEROP LENGTH))
       (SETQ PERCENT (ROUND (* 100. COUNT) LENGTH)))
  (LET ((MAYBE-NAME (SEND SELF :TRUENAME)))
    ;; directory streams (and maybe others) don't have truenames; show
    ;; their pathnames instead.
    (IF (NULL MAYBE-NAME) (SETQ MAYBE-NAME (SEND SELF :PATHNAME)))
    (VALUES MAYBE-NAME DIRECTION COUNT PERCENT)))

(DEFFLAVOR INPUT-FILE-STREAM-MIXIN
	()
	(FILE-DATA-STREAM-MIXIN)
  (:INCLUDED-FLAVORS INPUT-POINTER-REMEMBERING-MIXIN)
  (:DOCUMENTATION "Streams for use with input files."))

(DEFFLAVOR OUTPUT-FILE-STREAM-MIXIN
	()
	(FILE-DATA-STREAM-MIXIN)
  (:INCLUDED-FLAVORS OUTPUT-POINTER-REMEMBERING-MIXIN)
  (:DOCUMENTATION "Streams for use with output files."))

(DEFMETHOD (OUTPUT-FILE-STREAM-MIXIN :LENGTH) () NIL)


(DEFSTRUCT (INDENTING-STREAM (:ALTERANT ()))
  INDENTING-STREAM-BASE-STREAM
  INDENTING-STREAM-INDENTATION
  INDENTING-STREAM-BEGINNING-OF-LINE?)

(DEFUN MAKE-STREAM-INDENTABLE (STREAM)
  "Return an indenting stream which passes output to STREAM.
If STREAM is already an indenting stream, it is returned unchanged.
The indenting stream inserts a specified indentation before each line.
It supports the :SET-INDENTATION and :INDENT-RELATIVE operations
to specify the amount of indentation."
  (IF (SEND STREAM :OPERATION-HANDLED-P :INDENT-RELATIVE)
      STREAM
    (LET-CLOSED ((*SELF (MAKE-INDENTING-STREAM
			  :INDENTING-STREAM-BASE-STREAM STREAM
			  :INDENTING-STREAM-INDENTATION 0
			  :INDENTING-STREAM-BEGINNING-OF-LINE? NIL)))
      #'(LAMBDA (&REST STUFF)
	  (APPLY #'INDENTING-STREAM-INTERFACE *SELF STUFF)))))

(DEFUN INDENTING-STREAM-INTERFACE (SELF MESSAGE &REST ARGS)
  (LET ((HANDLER (AND (SYMBOLP MESSAGE)
		      (GET MESSAGE 'INDENTING-STREAM-OPERATIONS))))
    (IF (NULL HANDLER)
	(APPLY (INDENTING-STREAM-BASE-STREAM SELF) MESSAGE ARGS)
	(FUNCALL HANDLER SELF ARGS))))

(DEFUN INDENTING-STREAM-WHICH-OPERATIONS (SELF)
  (LET ((MY-OPS NIL)
	(BASE-OPS (SEND (INDENTING-STREAM-BASE-STREAM SELF) :WHICH-OPERATIONS)))
    (SETQ MY-OPS
	  (LIST* NIL
		 :WHICH-OPERATIONS :OPERATION-HANDLED-P :SEND-IF-HANDLES
		 :SET-INDENTATION :INDENT-RELATIVE
		 :TYO :STRING-OUT :FRESH-LINE
		 BASE-OPS))
    (DO ((TAIL MY-OPS (CDR TAIL)))
	((EQ (CDR TAIL) BASE-OPS)
	 (CDR MY-OPS))
      (COND ((MEMQ (CADR TAIL) BASE-OPS)
	     (RPLACD TAIL (CDDR TAIL))
	     (IF (EQ (CDR TAIL) BASE-OPS)
		 (RETURN (CDR MY-OPS))))))))

(DEFUN INDENTING-STREAM-INDENT (SELF)
  (IF (SEND (INDENTING-STREAM-BASE-STREAM SELF) :OPERATION-HANDLED-P :INCREMENT-CURSORPOS)
      (SEND (INDENTING-STREAM-BASE-STREAM SELF) :INCREMENT-CURSORPOS
	    					(INDENTING-STREAM-INDENTATION SELF)
						0 :CHARACTER)
    (DOTIMES (I (INDENTING-STREAM-INDENTATION SELF))
      (SEND (INDENTING-STREAM-BASE-STREAM SELF) :TYO #/SPACE))))


(DEFUN (:PROPERTY :WHICH-OPERATIONS INDENTING-STREAM-OPERATIONS) (SELF IGNORE)
  (INDENTING-STREAM-WHICH-OPERATIONS SELF))

(DEFUN (:PROPERTY :OPERATION-HANDLED-P INDENTING-STREAM-OPERATIONS) (SELF ARGS)
  (MEMQ (CAR ARGS) (INDENTING-STREAM-WHICH-OPERATIONS SELF)))

(DEFUN (:PROPERTY :SEND-IF-HANDLES INDENTING-STREAM-OPERATIONS) (SELF ARGS)
  (IF (MEMQ (CAR ARGS) (INDENTING-STREAM-WHICH-OPERATIONS SELF))
      (LEXPR-SEND SELF (CAR ARGS) (CDR ARGS))
    NIL))

(DEFUN (:PROPERTY :SET-INDENTATION INDENTING-STREAM-OPERATIONS) (SELF ARGS)
  (SETF (INDENTING-STREAM-BEGINNING-OF-LINE? SELF) NIL)
  (SETF (INDENTING-STREAM-INDENTATION SELF) (CAR ARGS)))

(DEFUN (:PROPERTY :INDENT-RELATIVE INDENTING-STREAM-OPERATIONS) (SELF ARGS)
  (SETF (INDENTING-STREAM-BEGINNING-OF-LINE? SELF) NIL)
  (INCF (INDENTING-STREAM-INDENTATION SELF) (CAR ARGS)))

(DEFUN (:PROPERTY :TYO INDENTING-STREAM-OPERATIONS) (SELF ARGS &AUX (CAR (CAR ARGS)))
; character lossage
  (IF (CHARACTERP CAR) (SETQ CAR (CHAR-INT CAR)))
  (LEXPR-SEND (INDENTING-STREAM-BASE-STREAM SELF) :TYO (CAR ARGS) (CDR ARGS))
  (SETF (INDENTING-STREAM-BEGINNING-OF-LINE? SELF) (EQ CAR #/NEWLINE))
  (IF (NOT (NULL (INDENTING-STREAM-BEGINNING-OF-LINE? SELF)))
      (INDENTING-STREAM-INDENT SELF)))

(DEFUN (:PROPERTY :STRING-OUT INDENTING-STREAM-OPERATIONS) (SELF ARGS)
  (LET ((STRING (CAR ARGS))
	(START (CADR ARGS))
	(END (CADDR ARGS)))
    (IF (NULL START) (SETQ START 0))
    (IF (NULL END) (SETQ END (ARRAY-ACTIVE-LENGTH STRING)))
    (COND ((> END START)
	   (SETF (INDENTING-STREAM-BEGINNING-OF-LINE? SELF) NIL)
	   (LET ((MARK START))
	     (DO ()
		 ((NOT (SETQ MARK (%STRING-SEARCH-CHAR #/NEWLINE STRING MARK END)))
		  (SEND (INDENTING-STREAM-BASE-STREAM SELF) :STRING-OUT STRING START END)
		  (RETURN STRING))
	       (SEND (INDENTING-STREAM-BASE-STREAM SELF) :STRING-OUT STRING START MARK)
	       (INDENTING-STREAM-INTERFACE SELF :TYO #/NEWLINE))
	       (COND ((= (SETQ MARK (1+ MARK)) END)
		      (SETF (INDENTING-STREAM-BEGINNING-OF-LINE? SELF) T)
		      (RETURN STRING)))
	       (SETQ START MARK)))))))

(DEFUN (:PROPERTY :FRESH-LINE INDENTING-STREAM-OPERATIONS) (SELF IGNORE)
  (IF (NULL (INDENTING-STREAM-BEGINNING-OF-LINE? SELF))
      (INDENTING-STREAM-INTERFACE SELF :TYO #/NEWLINE)))
