;;; -*- Mode:Lisp;Package:Tape;Base:8 -*-
;;; (c) 1984 Lisp Machine Incorporated
;;; Wesperco unit definition.  Unibus/CADR hardware
;;; Calls functions in MTRQB
;;; Most of the code is taken from MTDEFS

;;; Internal functions:
;;; WE-PRINT-STATUS  prints current status from hardware.
;;; EXECUTE-MT-RQB actually does it.  This normally done by microcode for DISK-RQBs.
;;; WE-WAIT-READY, WE-WAIT-UNIT-READY
;;; UNIBUS-MAP-MT-RQB,  UNIBUS-UNMAP-MT-RQB.
;;; WE-RUN rqb command &optional minus-byte-count unit density ibm-mode
;;; WE-RUN-SIMPLE command unit &optional count.  For commands that don't transfer data.

;;; Normal RQBs are now used for magtape, but
;;; PRINT-WE-RQB, WIRE-MT-RQB, UNWIRE-MT-RQB, UNIBUS-MAP-MT-RQB and UNIBUS-UNMAP-MT-RQB
;;; must be used with magtape RQBs.

(DEFUN SIGN-EXTEND-16 (X)
  (IF (ZEROP (LOGAND 100000 X)) X (LOGIOR X -100000)))

(DEFCONST %WE-RQ-DONE-FLAG 0)		;0 RQ ENTERED, -1 COMPLETED
	       ;; These are set up by the requester
(DEFCONST %WE-RQ-COMMAND 1)		;MT Command register
(DEFCONST %WE-BYTE-COUNT 2)		;NEGATIVE BYTE COUNT
(DEFCONST %WE-READ 3)			;ADDITIONAL COMMAND BITS AND STATUS
	      ;; These are stored when the operation completes.
	      ;; The order must agree with the order of the UNIBUS addresses.
(DEFCONST %WE-RQ-STATUS 4)		;MT STATUS REG
(DEFCONST %WE-COMMAND-AFTER 5)		;MT COMMAND REGISTER AFTER XFER
(DEFCONST %WE-BYTE-COUNT-AFTER 6)	;MT RECORD COUNT AFTER
(DEFCONST %WE-RQ-MEM-ADDRESS 7)		;LAST UNIBUS REF ADDR
(DEFCONST %WE-DATA-BUFFER 10)
(DEFCONST %WE-READ-AFTER 11)

(EVAL-WHEN (COMPILE LOAD EVAL)
;This says that the selected drive is ready.
(DEFSUBST WE-STATUS-READY ()
 (LDB-TEST 0001 (%UNIBUS-READ WE-UA-STATUS)))

;This says that the controller is ready.
(DEFSUBST WE-COMMAND-READY ()
  (LDB-TEST 0701 (%UNIBUS-READ WE-UA-COMMAND)))

;These are bits in the status stored after a request is executed.

;End of file mark reached when trying to read something.
(DEFSUBST WE-STATUS-EOF ()
 (LDB-TEST 1601 (AR-1 RQB %WE-RQ-STATUS)))

;End of tape reached.
(DEFSUBST WE-STATUS-EOT ()
 (LDB-TEST 1201 (AR-1 RQB %WE-RQ-STATUS)))

;At beginning of tape (rewound, or spaced back that far).
(DEFSUBST WE-STATUS-BOT ()
 (LDB-TEST 0501 (AR-1 RQB %WE-RQ-STATUS)))

;Any sort of error.
(DEFSUBST WE-STATUS-ERROR ()
 (NOT (ZEROP (LOGAND 115600 (AR-1 RQB %WE-RQ-STATUS)))))
)

;These are the OLD ways of accessing these flags.
(DEFCONST %%WE-STATUS-ILL-COM 1701)
(DEFCONST %%WE-STATUS-EOF 1601)
(DEFCONST %%WE-STATUS-ODD-LENGTH 1501)	;Last word filled with 0s
(DEFCONST %%WE-STATUS-PARITY-ERR 1401)	;parity error, LRC error, or postamble error
(DEFCONST %%WE-STATUS-GRANT-LATE 1301)
(DEFCONST %%WE-STATUS-EOT 1201)
(DEFCONST %%WE-STATUS-RLENGTH 1101)	;RECORD LENGTH ERROR
(DEFCONST %%WE-STATUS-BAD-TAPE 1001)
(DEFCONST %%WE-STATUS-NXM 0701)
(DEFCONST %%WE-STATUS-ON-LINE 0601)
(DEFCONST %%WE-STATUS-BOT 0501)
(DEFCONST %%WE-STATUS-7-TRACK 0401)
(DEFCONST %%WE-STATUS-SETTLE 0301)	;after rewinding
(DEFCONST %%WE-STATUS-WRITE-LOCK 0201)
(DEFCONST %%WE-STATUS-REWINDING 0101)
(DEFCONST %%WE-STATUS-READY 0001)
(DEFCONST %WE-STATUS-ERROR 115600)		 ;Mask for bits which are errors normally

;This is the bit to set to tell the controller to execute the command
;already set up in the various registers.
(DEFCONST %WE-COMMAND-GO 1)

;These are fields to set up in the RQB's %WE-RQ-COMMAND word
;to control a transfer or operation to be done.

(EVAL-WHEN (COMPILE LOAD EVAL)
(DEFSUBST WE-COMMAND-UNIT ()
  (LDB 1003 (AR-1 RQB %WE-RQ-COMMAND)))

(DEFSUBST WE-COMMAND-DENSITY ()
  (LDB 1502 (AR-1 RQB %WE-RQ-COMMAND)))

(DEFSUBST WE-COMMAND-XBA-BITS ()
  (LDB 0402 (AR-1 RQB %WE-RQ-COMMAND)))

(DEFSUBST WE-COMMAND-INTERRUPT-ENABLE ()
  (LDB 0601 (AR-1 RQB %WE-RQ-COMMAND)))

;This field's value is one of the command codes below.
(DEFSUBST WE-COMMAND ()
  (LDB 0103 (AR-1 RQB %WE-RQ-COMMAND)))
)

;Mag tape command codes.
(DEFCONST %WE-COMMAND-OFFLINE 0)	;unload tape
(DEFCONST %WE-COMMAND-READ 1)
(DEFCONST %WE-COMMAND-WRITE 2)
(DEFCONST %WE-COMMAND-WRITE-EOF 3)
(DEFCONST %WE-COMMAND-SPACE-FOR 4)
(DEFCONST %WE-COMMAND-SPACE-REV 5)
(DEFCONST %WE-COMMAND-WRITE-WITH-EXTENDED-GAP 6)
(DEFCONST %WE-COMMAND-REWIND 7)

(DEFCONST WE-COMMAND-NAMES
	  '(%WE-COMMAND-OFFLINE %WE-COMMAND-READ %WE-COMMAND-WRITE
	    %WE-COMMAND-WRITE-EOF %WE-COMMAND-SPACE-FOR %WE-COMMAND-SPACE-REV
	    %WE-COMMAND-WRITE-WITH-EXTENDED-GAP %WE-COMMAND-REWIND))

(DEFCONST %%WE-COMMAND 0103)
(DEFCONST %%WE-COMMAND-XBA-BITS 0402)	;EXTENDED UNIBUS ADR 17,16
(DEFCONST %WE-COMMAND-INTERRUPT-ENABLE 1_6.)
(DEFCONST %%WE-COMMAND-READY 0701)
(DEFCONST %%WE-COMMAND-UNIT 1003)
(DEFCONST %WE-COMMAND-POWER-CLEAR 1_12.)
(DEFCONST %%WE-COMMAND-DENSITY 1502)
(DEFCONST %WE-COMMAND-ERROR 100000)

;Unibus register addresses.
(DEFCONST WE-UA-STATUS  772520)
(DEFCONST WE-UA-COMMAND 772522)
(DEFCONST WE-UA-BYTEC   772524)
(DEFCONST WE-UA-CMA     772526)
(DEFCONST WE-UA-BFR     772530)
(DEFCONST WE-UA-DRD     772532)

;Error reporting.

(DEFUN WE-PRINT-STATUS NIL
  (FORMAT T "~%Status ")
  (WE-DECODE-STATUS (%UNIBUS-READ WE-UA-STATUS))
  (FORMAT T "~%Command ")
  (WE-DECODE-COMMAND (%UNIBUS-READ WE-UA-COMMAND)))

(DEFUN WE-DECODE-STATUS (STATUS)
  (CADR:CC-PRINT-SET-BITS STATUS
		     '(TAPE-UNIT-READY REWIND-STATUS WRITE-LOCK SETTLE-DOWN
		       SEVEN-TRACK BOT DRIVE-ON-LINE NXM
		       BAD-TAPE RECORD-LENGTH-ERROR EOT BUS-GRANT-LATE
		       PARITY-ERROR NOT-USED END-OF-FILE ILLEGAL-COMMAND)))

(DEFUN WE-DECODE-COMMAND (COM)
  (CADR:CC-PRINT-SET-BITS COM
		     '(GO FCN-0 FCN-1 FCN-2
		       XBA16 XBA17  INT-ENABLE CONTROLLER-READY
		       UNIT-SEL0 UNIT-SEL1 UNIT-SEL2 PARITY-EVEN
		       POWER-CLEAR DENSITY0 DENSITY1 ERROR)))

(DEFUN PRINT-WE-RQB (RQB)
  (FORMAT T "~%command before:")
  (WE-DECODE-COMMAND (AR-1 RQB %WE-RQ-COMMAND))
  (FORMAT T "~%Byte count before ~s" (AR-1 RQB %WE-BYTE-COUNT))
  (FORMAT T "~%command after:")
  (WE-DECODE-COMMAND (AR-1 RQB %WE-COMMAND-AFTER))
  (FORMAT T "~%status after:")
  (WE-DECODE-STATUS (AR-1 RQB %WE-RQ-STATUS))
  (FORMAT T "~%mem addr after ~S, byte-count-after ~S"
	  (AR-1 RQB %WE-RQ-MEM-ADDRESS) (AR-1 RQB %WE-BYTE-COUNT-AFTER))
  RQB)

(DEFFLAVOR WESPERCO-UNIT () (BASIC-LOCAL-MT-UNIT))

(DEFMETHOD (WESPERCO-UNIT :ACCESS) () ':WESPERCO)

(DEFMETHOD (WESPERCO-UNIT :ODD-BYTES-OK-P) () ())

(DEFMETHOD (WESPERCO-UNIT :STATUS-READY) () (WE-STATUS-READY))
(DEFMETHOD (WESPERCO-UNIT :COMMAND-READY) () (WE-COMMAND-READY))
(DEFMETHOD (WESPERCO-UNIT :STATUS-EOF) (RQB) (WE-STATUS-EOF))
(DEFMETHOD (WESPERCO-UNIT :STATUS-EOT) (RQB) (WE-STATUS-EOT))
(DEFMETHOD (WESPERCO-UNIT :STATUS-BOT) (RQB) (WE-STATUS-BOT))
(DEFMETHOD (WESPERCO-UNIT :STATUS-ERROR) (RQB) (WE-STATUS-ERROR))

(DEFMETHOD (WESPERCO-UNIT :DECODE-STATUS) (RQB THING)
  "RQBs serve as the STATUS object."
  (SELECTQ THING
    (:BYTES-TRANSFERRED
     (- (SIGN-EXTEND-16 (AR-1 RQB %WE-BYTE-COUNT-AFTER))
	(SIGN-EXTEND-16 (AR-1 RQB %WE-BYTE-COUNT))))
    (:EOT (WE-STATUS-EOT))
    (:EOF (WE-STATUS-EOF))
    (:BOT (WE-STATUS-BOT))
    (:ERROR (WE-STATUS-ERROR))
    (:ERROR-CODE (LOGAND 115600 (AR-1 RQB %WE-RQ-STATUS))) ; confirm this -- 1/11/84 RpK
    (:WRITE-ONLY (LDB-TEST 0201 (AR-1 RQB %WE-RQ-STATUS)))
    (:BAD-TAPE (LDB-TEST 1001 (AR-1 RQB %WE-RQ-STATUS)))
    (:ON-LINE (LDB-TEST 0601 (AR-1 RQB %WE-RQ-STATUS)))))


;Low-level command execution.
;Initialize the various header words of an RQB, such as %WE-RQ-COMMAND,
;then call EXECUTE-WE-RQB.

(DEFUN EXECUTE-MT-RQB (RQB &OPTIONAL SET-MODIFIED)
  (WIRE-MT-RQB RQB T SET-MODIFIED)
  (WE-WAIT-READY)
  (LET ((UA (UNIBUS-MAP-MT-RQB RQB)))
    (%UNIBUS-WRITE WE-UA-COMMAND (AR-1 RQB %WE-RQ-COMMAND))	;SELECT UNIT
    (WE-WAIT-UNIT-READY)
    (%UNIBUS-WRITE WE-UA-CMA UA)
    (%UNIBUS-WRITE WE-UA-BYTEC (AR-1 RQB %WE-BYTE-COUNT))
    (%UNIBUS-WRITE WE-UA-DRD (AR-1 RQB %WE-READ))
    (%UNIBUS-WRITE WE-UA-COMMAND (+ (AR-1 RQB %WE-RQ-COMMAND)
				    %WE-COMMAND-GO))	;TURN ON THE "GO" BIT.
    (WE-WAIT-READY)
    (DOTIMES (W 6)
      (AS-1 (%UNIBUS-READ (+ WE-UA-STATUS (* 2 W)))
	    RQB
	    (+ %WE-RQ-STATUS W)))
    (UNIBUS-UNMAP-MT-RQB RQB)
    (UNWIRE-MT-RQB RQB)
    RQB))



(DEFUN WE-WAIT-READY ()
  "Await controller ready"
  (OR (WE-COMMAND-READY)
      (PROCESS-WAIT "MagTape"
	#'(LAMBDA () (WE-COMMAND-READY))))
  NIL)

(DEFUN WE-WAIT-UNIT-READY ()
  "Await selected unit ready"
  (OR (WE-STATUS-READY)
      (PROCESS-WAIT "MT Unit"
	#'(LAMBDA () (WE-STATUS-READY))))
  NIL)

(defmethod (wesperco-unit :wait-unit-ready) (ignore) (we-wait-unit-ready))
(defmethod (wesperco-unit :wait-controller-ready) (ignore) (we-wait-ready))

(DEFUN UNIBUS-MAP-MT-RQB (RQB &OPTIONAL (FIRST-UMP 0)
		    &AUX (LONG-ARRAY-FLAG (%P-LDB %%ARRAY-LONG-LENGTH-FLAG RQB))
			 (HIGH (+ (%POINTER RQB) 1 LONG-ARRAY-FLAG
				  (FLOOR (ARRAY-LENGTH RQB) 2))))
  (DO ((VADR (+ (%POINTER RQB) PAGE-SIZE)
	     (+ VADR PAGE-SIZE)) ;Start with 2nd page of rqb array
       (UMP FIRST-UMP (1+ UMP))
       (NP 0 (1+ NP)))
      ((>= VADR HIGH))
    (COND ((> NP 14.)
	   (FERROR NIL "TOO MANY PAGES")))
    (SETUP-UNIBUS-MAP UMP VADR))
  (+ 140000 (* FIRST-UMP 2000)))

(DEFUN UNIBUS-UNMAP-MT-RQB (RQB &OPTIONAL (FIRST-UMP 0)
			   &AUX (LONG-ARRAY-FLAG (%P-LDB %%ARRAY-LONG-LENGTH-FLAG RQB))
				(LOW (- (%POINTER RQB) (ARRAY-DIMENSION RQB 0) 2))
				(HIGH (+ (%POINTER RQB) 1 LONG-ARRAY-FLAG
					 (FLOOR (ARRAY-LENGTH RQB) 2))))
  (DO ((VADR (+ LOW PAGE-SIZE) (+ VADR PAGE-SIZE)) ;Start with 2nd page of rqb array
       (UMP FIRST-UMP (1+ UMP)))
      ((>= VADR HIGH))
    (%UNIBUS-WRITE (+ 766140 (* 2 UMP)) 0)))

;;; The Unibus map is 16 words at 766140.  It consists of 14 address bits, write-ok, and valid
;;; It controls locations 140000-177777 (2000 byte locations per page).
(DEFUN SETUP-UNIBUS-MAP (UNIBUS-MAP-PAGE XBUS-ADR)
  (%UNIBUS-WRITE (+ 766140 (* 2 UNIBUS-MAP-PAGE))
		 (+ 140000 (LDB 1016 (%PHYSICAL-ADDRESS XBUS-ADR))))
  (+ 140000 (* UNIBUS-MAP-PAGE 2000) (* 4 (LOGAND 377 XBUS-ADR))))	; returns ubus-adr

;Use WE-RUN to transfer the contents of an RQB.
;Use WE-RUN-SIMPLE to do spacing operations.

(DEFVAR WE-RETRY-COUNT 5)
(DEFVAR WE-ATTEMPT-TO-WRITE-WITH-EXTENDED-GAP-COUNT 3)

(DEFUN WE-RUN (RQB COMMAND &OPTIONAL MINUS-BYTE-COUNT (UNIT 0) (DENSITY 0) IBM-MODE)
  (DECLARE (:VALUES RQB))
  (PROG ((RETRIES WE-RETRY-COUNT))
   AGAIN
    	(SETF (AREF RQB %WE-RQ-COMMAND) 0)
	(SETF (WE-COMMAND-UNIT) UNIT)
	(SETF (WE-COMMAND-DENSITY) DENSITY)
	(SETF (WE-COMMAND) COMMAND)
	(IF (NULL MINUS-BYTE-COUNT)
	    (SETQ MINUS-BYTE-COUNT
		  (MINUS (ARRAY-LENGTH (RQB-8-BIT-BUFFER RQB)))))
	(AS-1 MINUS-BYTE-COUNT RQB %WE-BYTE-COUNT)
	(AS-1 (IF IBM-MODE 1_10. 0)
	      RQB %WE-READ)
	(EXECUTE-MT-RQB RQB (= COMMAND %WE-COMMAND-READ))
	(COND ((WE-STATUS-EOT)
	       (CERROR ':NO-ACTION NIL 'END-OF-TAPE
		       "End of tape on unit ~D, command ~D, ~D bytes.
Density ~S, IBM-mode ~S, rqb ~S."
		       UNIT (NTH COMMAND WE-COMMAND-NAMES)
		       (- MINUS-BYTE-COUNT) DENSITY IBM-MODE RQB)))
	(COND ((NOT (WE-STATUS-ERROR))
	       (RETURN RQB))
	      (T (FORMAT T "~%MAGTAPE ERROR!")
		 (PRINT-WE-RQB RQB)
		 (WE-RUN-SIMPLE %WE-COMMAND-SPACE-REV UNIT 1)
		 (AND (= COMMAND %WE-COMMAND-WRITE)
		      ( RETRIES WE-ATTEMPT-TO-WRITE-WITH-EXTENDED-GAP-COUNT)
		      (SETQ COMMAND %WE-COMMAND-WRITE-WITH-EXTENDED-GAP))
		 (IF (>= (SETQ RETRIES (1- RETRIES)) 0)
		     (GO AGAIN)
		   (CATCH-ERROR-RESTART-EXPLICIT-IF T (MT-ERROR :RETRY "Retry magtape operation.")
		     (FERROR 'MT-ERROR "MagTape operation failed."))
		   (GO AGAIN))))))

(DEFMETHOD (WESPERCO-UNIT :I-WRITE-BUFFER) (BUFFER COUNT DENSITY IBM-MODE)
  (LET ((RESULT (WE-RUN BUFFER %WE-COMMAND-WRITE (MINUS COUNT) NUMBER DENSITY IBM-MODE)))
    (VALUES RESULT RESULT)))

(DEFMETHOD (WESPERCO-UNIT :I-READ-BUFFER) (BUFFER COUNT DENSITY)
  (LET((RESULT (WE-RUN BUFFER %WE-COMMAND-READ (MINUS COUNT) NUMBER DENSITY)))
    (VALUES RESULT RESULT)))

(DEFUN WE-RUN-SIMPLE (COMMAND UNIT &OPTIONAL COUNT (RQB (GET-DISK-RQB 0) DONT-RETURN-RQB))
  "Run a simple command on the Wesperco controller.
COUNT is made MINUS for the hardware."
  (UNWIND-PROTECT
    (PROGN (IF COUNT (AS-1 (MINUS COUNT) RQB %WE-BYTE-COUNT))
	   (SETF (AREF RQB %WE-RQ-COMMAND) 0)
	   (SETF (WE-COMMAND) COMMAND)
	   (SETF (WE-COMMAND-UNIT) UNIT)
	   (EXECUTE-WE-RQB RQB))
    (OR DONT-RETURN-RQB
	(RETURN-DISK-RQB RQB)))
  T)

;Convenient spacing operations.

;Note: NTIMES fed thru to hardware.  NTIMES of 0 means moby many.
(DEFMETHOD (WESPERCO-UNIT :SPACE) (NTIMES 1)
  (WE-RUN-SIMPLE %WE-COMMAND-SPACE-FOR NUMBER NTIMES))

(DEFMETHOD (WESPERCO-UNIT :SPACE-TO-EOF) (NTIMES 1)
  (DOTIMES (C NTIMES) (SEND SELF ':SPACE 0)))

(DEFMETHOD (WESPERCO-UNIT :SPACE-REV) (NTIMES)
  (DOTIMES (I NTIMES) (WE-RUN-SIMPLE %WE-COMMAND-SPACE-REV NUMBER 1)))

;; Reverse through the tape, positioning the tape at the beginning of a file.
;; If SKIP-N-BLOCKS is 0, this positions the tape at the beginning of this file.
;; If SKIP-N-BLOCKS is 1, this positions the tape at the beginning of the previous file, etc.
;; If this reaches the beginning of the tape prematurely, it stops there and returns NIL.
(DEFMETHOD (WESPERCO-UNIT :SPACE-REV-TO-BOF) (SKIP-N-FILES &AUX RQB)
  (UNWIND-PROTECT
      (PROG ()
	    (SETQ RQB (GET-DISK-RQB 0))
	 L  (WE-RUN-SIMPLE %WE-COMMAND-SPACE-REV NUMBER 1 RQB)
	    (COND ((WE-STATUS-EOF)
		   (COND ((ZEROP SKIP-N-FILES)
			  ;; If we stop at an EOF block, we must space forward over it.
			  (WE-RUN-SIMPLE %WE-COMMAND-SPACE-FOR NUMBER 1 RQB)
			  (RETURN T)))
		   (DECF SKIP-N-FILES))
		  ((WE-STATUS-BOT)
		   (RETURN (ZEROP SKIP-N-FILES))))
	    (GO L))
    (RETURN-DISK-RQB RQB)))

;; This function attempts to bypass all files on the tape until two
;; consecutive EOFs are found, then positions the tape over the last EOF.
;; The tape is now in a configuration allowing one to append new files.
(DEFMETHOD (WESPERCO-UNIT :SPACE-TO-APPEND) (&AUX RQB)
  (UNWIND-PROTECT
      (PROG ((EOF T))
	    (SETQ RQB (GET-DISK-RQB 0))
	 L  (WE-RUN-SIMPLE %WE-COMMAND-SPACE-FOR NUMBER 1 RQB)
	    (IF (WE-STATUS-EOF)
		(IF (NOT EOF)
		    (SETQ EOF T)
		  (WE-RUN-SIMPLE %WE-COMMAND-SPACE-REV NUMBER 1 RQB)
		  (RETURN T))
	      (SETQ EOF NIL))
	    ;; WE-RUN-SIMPLE doesn't have this..
	    (IF (WE-STATUS-EOT)
		(CERROR ':NO-ACTION NIL 'END-OF-TAPE
			"End of tape on unit ~D, command ~D, ~D bytes.
Density ~S, IBM-mode ~S, rqb ~S."
			NUMBER '%WE-COMMAND-SPACE-FOR 1 0 NIL RQB))
	    (GO L))
    (RETURN-DISK-RQB RQB))
  T)

(DEFMETHOD (WESPERCO-UNIT :REWIND) ()
  (WE-RUN-SIMPLE %WE-COMMAND-REWIND NUMBER))

(DEFMETHOD (WESPERCO-UNIT :WRITE-EOF) ()
  (WE-RUN-SIMPLE %WE-COMMAND-WRITE-EOF NUMBER))

(DEFMETHOD (WESPERCO-UNIT :OFFLINE) ()
  (WE-RUN-SIMPLE %WE-COMMAND-OFFLINE NUMBER))



