#ifndef lint
static char RCSid[] = "$Header: /work6/oldtapes/tools/read20/read20.c,v 1.23 2001/11/28 15:32:45 phil Exp brad $";
#endif
/*
 *  Program to read Tops-20 Dumper format tapes
 *
 *                   Jim Guyton,  Rand Corporation
 *                   Original       10/20/82
 *                   jdg:   -n added 6/11/83
 *                   jdg:    can now extract 8-bit-byte files  2/9/86
 *
 * Lot of mods by Jay Lepreau, Univ of Utah, 1-2/87.
 * See the RCS log for details.
 *
 * Phil Budne <phil@ultimate.com>;
 * 04-Dec-95
 *	Added fixes from Stu Grossman (Stanford) from July 1990
 * 13-Feb-96
 *	Added -r (raw) extraction for 36-bit byte files
 * 24-Mar-00
 *	Added -C (TPC tape format)
 * 28-Nov-01
 *	Fixed for little endian, gcc 2.95.2, FreeBSD compilation
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <ctype.h>
#include "dump.h"
#include <signal.h>
#include <errno.h>

#define LOGFILE "Logfile"		/* logfile should be changeable */

char *ctime(), *index(), *rindex(), *strcpy(), *strcat();
char *unixname();
long getfield(), unixtime();

int  fdTape;                    /* File handle for Dumper-20 format tape */
char tapeblocka[TAPEBLK];        /* One logical record from tape */
char *prevblock = 0;
int firstblock = 0;		/* We are reading first block of a file */
FILE *fpFile;                   /* Output file handle on extracts */
int debug = 0;
int default_bytesize = 7;	/* Default byte size for binary files */
int numflg = 0;                 /* Non-zero if using numeric filenames */
int keepcr = 0;			/* Keep CR's in CRLF pairs in text files */
int dodir = 0;				/* directory listing */
int xflg = 0;				/* extract */
int verbose = 0;
int genflg;				/* keep generation number */
int rawflg;				/* extract raw tape data -phil */
int tpcflg;				/* TPC format tape -phil */
int simhflg;				/* TPC format tape -brad */
int nselect;		/* number of files still to be selected by number */
int doallflag;		/* act on all files cause no args given */

int number;                     /* Current output file "number" */

#define TAPE "/dev/rmt8"        /* Default input tape */

int  bytesize;          /* Number of bits/byte in current file */
long  numbytes;          /* Number of bytes in current file */
int  pgcount;           /* Number of twenex pages in file */
long pageno, tapeno, ssno, filenum;

unsigned tprot;		/* Tops-20 protection */
char *timeptr;

long timep[2];
#define atime (timep[0])
#define mtime (timep[1])

int offline, archived, invisible;
int apgcount, tp1, tp2, ss1, ss2, tf1, tf2;

char topsname[130];
char sunixname[300];

struct want {
     unsigned short ssnum;
     unsigned short fnum;
} want[10000];				/* limited by 20000 char arglist */

int cursswant;
int compwant();

#define CONTAINS_BOF 1		/* Block contains the beginning of a file */
#define CONTAINS_EOF 2		/* Block contains the end of a file */
#define NUL_AT_END 4		/* Put a null at the end */
#define STRING (CONTAINS_BOF|CONTAINS_EOF|NUL_AT_END) /* Make strings easy */

char **patterns = 0;     /* Filename match patterns */
int numpats = 0;         /* Number of patterns */
char *expression = 0;
char *re_comp_error;     /* Error message from re_comp() */
extern char *re_comp();
extern void exit();
extern int errno;

main(argc, argv)
int argc;
char *argv[];
{
	char *tape = TAPE;              /* Pathname for tape device/file */
	char	*tapeblock;
	int rc;
	int rtype;

	/* Do switch parsing */

	signal(SIGINT, exit);	/* Allow ^C to cause profiling to finish */

	while(argc>1 && argv[1][0] == '-'){
		switch(argv[1][1]){
		case 'f':
			if (argc <= 2)
				punt(0, "Need filename after -f");
			tape = argv[2];
			argc--; argv++;
			break;
		case 'T':             /* Force text mode on "binary" files */
			default_bytesize = 7;
			break;
		case 'B':             /* Force byte mode on "binary" files */
			default_bytesize = 8;
			break;
		case 't':             /* directory listing */
			dodir = 1;
			break;
		case 'x':             /* extract */
			xflg = 1;
			break;
		case 'v':             /* verbosity */
			verbose++;
			break;
		case 'g':             /* keep gen number */
			genflg++;
			break;
		case 'd':
			debug = atoi(&argv[1][2]);
			fprintf(stderr, "Debug value set to %d\n", debug);
			break;
		case 'n':               /* numeric output filenames */
			if (argc <= 2)
				punt(0, "Need number after -n");
			number = atoi(argv[2]);         /* First file name */
			numflg = 1;
			argc--; argv++;
			break;
		case 'c':		/* keep CR`s in CR/LF pairs */
			keepcr++;
			break;
		case 'r':		/* extract raw tape data */
			rawflg++;
			break;
		case 'C':		/* TPC tape format */
			tpcflg++;
			break;
		case 'H':		/* TPC tapesimh tape format */
			simhflg++;
			break;
		case 'e':               /* regular expression */
			if (argc <= 2)
				punt(0, "Need expression after -e");
			if (expression)
				punt(0, "Only one regexp allowed");
			expression = argv[2];
			if ((re_comp_error = re_comp(expression)) != 0)
				punt(0, "re_comp: %s", re_comp_error);
			argc--; argv++;
			break;
		case 'S':		/* selected save set number */
			if (argc <= 2)
				punt(0, "Need save set number after -S");
			cursswant = atoi(argv[2]);
			argc--; argv++;
			break;
		case 'F':		/* selected file numbers */
			if (argc <= 2)
				punt(0, "Need file number(s) after -F");
			for (argc -= 2, argv += 2;
			  argc && isdigit(**argv);
			  argc--, argv++, nselect++) {
			      want[nselect].ssnum = cursswant;
			      want[nselect].fnum = atoi(*argv);
			}
			argc += 2; argv -= 2;
			break;
		default:
			punt(0, "unknown flag %s", argv[1]);
		}
		argc--;  argv++;
	}

	if (!xflg && !dodir)
		punt(0, "Need either '-x' or '-t' option.");

	if (argc > 1) {
		patterns = &argv[1];
		numpats = argc - 1;
	}
	doallflag = !(patterns || expression || nselect);
	if (nselect)
		qsort((char *)want, nselect, sizeof (struct want), compwant);

	if (!strcmp(tape, "-"))		/* stdin */
		fdTape = 0;
	else
		fdTape = open(tape, 0);         /* Open tape for read */
	if (fdTape == -1)
		punt(1, "Can't open 'tape' file %s", tape);

	rc = 0;
	for ( ; ; )             /* Loop till end of tape */
	{
					 /*** Read a block ***/
		if (rc == 0) {
			rc = getrec(fdTape, tapeblocka, TAPEBLK);
			if (debug > 99)
				printf("rc=%d\n", rc);
			if ((rc % (518*5)) != 0) {
				if (rc != 0)
				punt(1, "Oops.  Read block len = %d", rc);
			}
			if (rc == 0) {
				if (verbose)
					printf("\nEnd of tape.\n");
				exit(0);        /* Normal exit */
			}
			tapeblock = tapeblocka;
			rc = rc - 518*5;
		}
		else {
			tapeblock = tapeblock + 518*5;
			rc = rc - 518*5;
		}

		prevblock = 0;
					/*** Do something with it ***/
		switch(rtype = -getfield(tapeblock,
		    WdoffRectype, BtoffRectype, BtlenRectype))
		{
		  case RectypeData:             /* Data block */
			doDatablock(tapeblock);
			break;

		  case RectypeTphd:             /* Saveset header */
			doSaveset(tapeblock, 0);
			break;

		  case RectypeFlhd:             /* File header */
			doFileHeader(tapeblock);
			break;

		  case RectypeFltr:             /* File trailer */
			doFileTrailer(tapeblock);
			break;

		  case RectypeTptr:             /* Tape trailer */
			doTapeTrailer(tapeblock);
			break;

		  case RectypeUsr:              /* User directory info ? */
  			if (verbose >= 3)
				fprintf(stderr, "Directory record skipped\n");
			break;

		  case RectypeCtph:             /* Continued saveset hdr */
			doSaveset(tapeblock, 1);
			break;

		  case RectypeFill:             /* Fill record */
		  	if (verbose >= 3)
				fprintf(stderr, "Fill record skipped\n");
			break;

		  default:
			punt(0, "Unknown record type 0x%x", rtype);
			break;
		}
	}
}

int   masks[32] =       /* bitmasks for different length fields */
{              0x00000001, 0x00000003, 0x00000007,
   0x0000000f, 0x0000001f, 0x0000003f, 0x0000007f,
   0x000000ff, 0x000001ff, 0x000003ff, 0x000007ff,
   0x00000fff, 0x00001fff, 0x00003fff, 0x00007fff,
   0x0000ffff, 0x0001ffff, 0x0003ffff, 0x0007ffff,
   0x000fffff, 0x001fffff, 0x003fffff, 0x007fffff,
   0x00ffffff, 0x01ffffff, 0x03ffffff, 0x07ffffff,
   0x0fffffff, 0x1fffffff, 0x3fffffff, 0x7fffffff,
   0xffffffff
};

long
getfield(block, wordoff, bitoff, bitlen)
char *block;            /* Tape block record */
int wordoff;            /* 36-bit word offset */
register int bitoff;             /* Bit offset of field (from msb) */
register int bitlen;             /* Bit length of field */
{
	register char *p;                /* Used to point into record */
	unsigned long w32;      /* First 32 bits of the 36 bit word */
	int   w4;               /* Last 4 bits of the 36 bit word */
	long  w = 0;            /* the word to return */

				/* First, the "illegal" kludge */
	if (bitoff == 0 && bitlen == 36) {
		bitoff = 4;
		bitlen = 32;
	}
	if (bitlen > 32)
		punt(0, "Can't get that large a field = %d!", bitlen);

	/* A PDP-10 (or 20) 36-bit word is laid out with the first 32 bits
	   as the first 4 bytes and the last 4 bits are the low order 4 bits
	   of the 5th byte.   The high 4 bits of that byte should be zero */

	p = block + (5*wordoff);        /* Get ptr to word of interest */
	bcopy(p, (char *)&w32, 4);

	w32 = ntohl(w32);

	w4  = *(p+4);                               /* 5th */
	if (w4 > 017)
	  punt(0, "Not a PDP-10 tape!  w4 = octal %o", w4);

	/* Get the field right justified in the word "w".
	   There are three cases that I have to handle:
	      [1] field is contained in w32
	      [2] field crosses w32 and w4
	      [3] field is contained in w4
	*/

	if (bitoff+bitlen <= 32)        /* [1] field is contained in w32 */
	{
		w = w32 >> (32 - (bitoff+bitlen));
	}
	else if (bitoff <= 32)          /* [2] field crosses boundary */
	{
		w =  (w32 << (bitoff+bitlen-32))
		   | (w4  >> (36 - (bitoff+bitlen)));
	}
	else                            /* [3] field is contained in w4 */
	{
		w = w4 >> (36 - (bitoff+bitlen));
	}
	w = w & masks[bitlen-1];          /* Trim to proper size */
	return(w);
}


doDatablock(block)
char *block;
{
					    /* max is 5 bytes per word */
	static char buf[(512*5)+1];         /* A page of characters */
	int ct;
	int maxperblock;
	int nout, type;

	if (debug > 10)
		printf("*");
	if (fpFile == NULL)
		return;
					
	switch (bytesize) {
	   case 7:      maxperblock = 512*5; break;
	   case 8:      maxperblock = 512*4; break;
	   case 36:     maxperblock = 512*5; break;
	   default:     return;
	}

	if (firstblock) type = CONTAINS_BOF;
	else type = 0;

	if (numbytes > maxperblock)
		ct = maxperblock;
	else {
		ct = numbytes;
		type |= CONTAINS_EOF;
	}

	if (bytesize == 7) {
		nout = getstring(block, buf, 6, ct, type);
		fwrite(buf, 1, nout, fpFile);
	}
	else if (bytesize == 8) {
		getbytes(block, buf, 6, ct);
		fwrite(buf, 1, ct, fpFile);
	}
	else {				/* must be 36-bit bytes (raw) */
		fwrite(block+6*5, 1, ct, fpFile);
	}

	if (ferror(fpFile))
		punt(1, "Error writing %s", sunixname);

	firstblock = 0;
	numbytes -= ct;
}

/*ARGSUSED*/
doSaveset(block, contflag)
char *block;
int  contflag;
{
	static char name[102];
	static char ss[2];
	long t;

	if (debug > 10) printf("\nSaveset header:");
	tapeno = getfield(block, WdoffTapeNum, BtoffTapeNum, BtlenTapeNum);
	ssno = getfield(block, WdoffSaveSetNum, BtoffSaveSetNum,
	    BtlenSaveSetNum);
	getstring(block, name, WdoffSSName, sizeof(name), STRING);
	(void) strcat(name, ss);
	
	t = unixtime(block, WdoffSSDate);
	if (dodir || verbose)
		printf("%sSaveset '%s' %s\n", contflag ? "Continued " : "",
		    name, ctime(&t));

}

doFileHeader(block)
char *block;
{
    char *ts;
    static char prt_ar[2] = {'-', 'A'};
    static char prt_inv[2] = {'-', 'I'};
    static char prt_off[2] = {'-', 'O'};

    if (debug > 5)
	printf("File Header block:\n");

    filenum = getfield(block, WdoffFileNum, BtoffFileNum, BtlenFileNum);
    getstring(block, topsname, WdoffFLName, sizeof(topsname), STRING);
    ts = index(topsname, ';');		/* Chop off ;Pprotection;Aacct */
    if (ts)
	*ts = 0;
    folddown(topsname, topsname);

    fpFile = NULL;

    if ( doallflag ||
         (patterns && patternmatch()) ||
	 (expression && expmatch()) ||
	 (nselect && fmatch()) ) {
	getfdbinfo(block);
	pageno = getfield(block, WdoffPageNum, BtoffPageNum, BtlenPageNum);

	if (dodir || verbose) {
	    if (verbose)
		printf("%3d%6d ", ssno, filenum);
	    printf("%c%c%c", prt_ar[archived], prt_off[offline],
	      prt_inv[invisible]);
	    printf("%5d%9d %2d %o %s %s", offline ? apgcount : pgcount,
	      numbytes, bytesize, tprot, timeptr, topsname);
	    if (archived && verbose >= 2)
		printf(" %x%4d%5d %x%4d%5d", tp1, ss1, tf1, tp2, ss2, tf2);
	    if (pageno != 0)
		printf(" Split file, part 2");
	}

	if (xflg) {
	    if (bytesize == 0 || bytesize == 36) {
		if (rawflg) {
		    bytesize = 36;
		    numbytes *= 5;
		}
		else {
		    bytesize = default_bytesize;
		    numbytes *= 36/bytesize;
		}
	    }

	    if (!offline) {
		if (pageno != 0) {	/* continued file */
		    int missing = pageno * 512 * (36/bytesize);

		    numbytes -= missing;
		    if (!(dodir || verbose))
		    	printf("%s: Split file, part 2", topsname);
		    printf(": %d raw bytes missing.", missing);
		    if (!(dodir || verbose))
			putchar('\n');
		}
		firstblock = 1;
printf("try '%s'\n", unixname(topsname));
		fpFile = fopen(unixname(topsname), "w");
		if (fpFile == NULL) {
#if 0
		    punt(1, "Can't open %s for write", sunixname);
#else
		    perror(sunixname);
		    printf("Can't open %s for write\n", sunixname);
		    return;
#endif
		} else if (verbose)
		    printf(" Extracted.");
		if (fchmod(fileno(fpFile), t2uprot(tprot) & ~0111) < 0)
		    punt(1, "fchmod on %s", sunixname);
	    } else if (verbose)
		printf(" Skipping -- %s file.",
		  offline ? "offline" : "binary");
	}

	if (dodir || verbose)
	    putchar('\n');
    }
}

/* Return 1 if topsname matches any of the "extraction" strings. */
patternmatch()
{
	register int i;

	for (i = 0; i < numpats; i++)
		if (sindex(topsname, patterns[i]))
			return (1);
	return (0);
}

/* Return 1 if topsname matches the regular expression. */
expmatch()
{
	register int match;

	if (expression) {
		if ((match = re_exec(topsname)) == -1)
			punt(0, "re_exec: internal error on %s", topsname);
		else
			return (match);
	}
	return (0);
}

int widx;

/* Return 1 if current file number matches one selected by arg line. */
fmatch()
{

    while (want[widx].ssnum < ssno)
	widx++;
    if (want[widx].ssnum > ssno)
	return 0;
    while (want[widx].fnum < filenum)
	widx++;
    if (want[widx].fnum > filenum)
	return 0;
    return 1;
}

/*
 * Sets a bunch of global variables to info from the fdb.
 * For some reason the archive tape info is garbage.
 */
getfdbinfo(block)
char *block;
{
	
	mtime = unixtime(block, WdoffFDB_Wrt);
	atime = unixtime(block, WdoffFDB_Ref);
	timeptr = ctime(&mtime) + 4;		/* Skip over day-name field */
	timeptr[20] = '\0';			/* Chop off \n at end */

	bytesize = getfield(block, WdoffFDB_BSZ, BtoffFDB_BSZ, BtlenFDB_BSZ);
	numbytes = getfield(block, WdoffFDB_Size, BtoffFDB_Size,BtlenFDB_Size);
	pgcount  = getfield(block, WdoffFDB_PGC, BtoffFDB_PGC, BtlenFDB_PGC);
	tprot = getfield(block, WdoffFDB_PRT, BtoffFDB_PRT, BtlenFDB_PRT);

	archived = getfield(block, WdoffFDB_CTL, BtoffFDB_Arc, BtlenFDB_Arc);
	invisible = getfield(block, WdoffFDB_CTL, BtoffFDB_Inv, BtlenFDB_Inv);
	offline = getfield(block, WdoffFDB_CTL, BtoffFDB_Off, BtlenFDB_Off);
	apgcount = getfield(block, WdoffFDB_PGC_A, BtoffFDB_PGC, BtlenFDB_PGC);
	/* The rest is bogus. */
	tp1 = getfield(block, WdoffFDB_TP1, 0, 36);
	tp2 = getfield(block, WdoffFDB_TP2, 0, 36);
	ss1 = getfield(block, WdoffFDB_SS1, BtoffFDB_SS, BtlenFDB_SS);
	ss2 = getfield(block, WdoffFDB_SS2, BtoffFDB_SS, BtlenFDB_SS);
	tf1 = getfield(block, WdoffFDB_TF1, BtoffFDB_TF, BtlenFDB_TF);
	tf2 = getfield(block, WdoffFDB_TF2, BtoffFDB_TF, BtlenFDB_TF);
}

/*ARGSUSED*/
doFileTrailer(block)
char *block;
{
	if (debug > 10) printf(" File trailer\n");
	if (fpFile != NULL) {
		if (fclose(fpFile) == CONTAINS_EOF)
			punt(1, "fclose: write error on %s", sunixname);
		fpFile = NULL;
		utime(sunixname, timep);
		if (numbytes != 0)
			printf("%s: Split file, part 1: %d raw bytes left\n",
			    topsname, numbytes);
	}
}

/*ARGSUSED*/
doTapeTrailer(block)
char *block;
{
	if (debug > 10) printf("Tape Trailer");
}

/*VARARGS2*/
punt(prterrno, fmt, a1, a2)
char *fmt;
{
    extern int sys_nerr;

    fprintf(stderr, fmt, a1, a2);
    if (prterrno) {
	fprintf(stderr, ": ");
//	if (errno >= 0 && errno < sys_nerr)
//	    perror("");
//	else
	    fprintf(stderr, "ERRNO = %d %s\n", errno, strerror(errno));
    }
    else
	fprintf(stderr, "\n");
    exit(1);
}

/*
 * Unpack into buffer 's' the 7 bit string stored in 'block' and
 * append a null char.  Optionally strip CR's from CRLF pairs.  'max'
 * is the max number of 7-bit chars to unpack from 'block', not the
 * max to put into 's' (that's important!).  This only works if
 * getstring() is always called with 'max' mod 5 == 0, except for the
 * last call on "contiguous" blocks.
 * Returns number of chars stored in output buffer.
 */
int
getstring(block, s, wordoff, max, type)
register char *block;   /* Tape block */
register char *s;       /* Destination string buffer */
int  wordoff;           /* 36-bit offset from start of tape block */
int  max;               /* Max number of characters to xfer into s */
int type;		/* Info about block we are reading (first, last
			   or string) */
{
	register int ct;	/* Number of characters loaded so far */
	register char *op;	/* Output pointer for <CRLF> -> <LF> hacking */
	register int c;
	char *orig = s;         /* Save for debugging */
	unsigned long top4;
	static int state;
#define NORMAL 1
#define LAST_WAS_CR 2

	op = s;

	block += wordoff * 5;

	for (ct = 0; ct < max; ct += 5)
	{
		bcopy(block + ct, (char *)&top4, 4);
		top4 = ntohl(top4);
		
		*op = top4 >> (3 * 7 + 4);
		*(op+1) = (top4 >> (2 * 7 + 4)) & 0177;
		*(op+2) = (top4 >> 7 + 4) & 0177;
		*(op+3) = (top4 >> 4) & 0177;
		*(op+4) = ((top4 << 3) | (*(block + ct + 4) >> 1)) & 0177;

		op += 5;
	}

	if (keepcr) return (max);

	op = s;

	if (type & CONTAINS_BOF) state = NORMAL;

	for (ct = 0; ct < max; ct++)
	{
	  c = *s++;

	  switch (state) {
	  case NORMAL:
	    if (c != '\r')
	      *op++ = c;
	    else
	      state = LAST_WAS_CR;

	    continue;

	  case LAST_WAS_CR:
	    if (c == '\n') {
	      *op++ = c;
	      state = NORMAL;
	      continue;
	    }
	    if (c == '\r') {
	      *op++ = c;
	      continue;
	    }
	    *op++ = '\r';
	    *op++ = c;
	    state = NORMAL;
	  }
	}

	if (type & CONTAINS_EOF) {
	  if (state == LAST_WAS_CR)
	    *op++ = '\r';
	}
	if (type & NUL_AT_END) *s++ = 0;

	return (op - orig);
}

/* getbytes: like getstring, but ...
   1) uses 8 bit bytes
   2) doesn't stop on a zero
*/
getbytes(block, s, wordoff, max)
char *block;            /* Tape block */
register char *s;	/* Destination string buffer */
int  wordoff;           /* 36-bit offset from start of tape block */
int  max;               /* Max number of characters to xfer into s */
{
	register int i;         /* Counter for five characters per word */

	int ct = 0;             /* Number of characters loaded so far */
	char *orig = s;         /* Save for debugging */

	block += wordoff * 5;

	for (ct = 0; ct < max; ct += 4)
	{
		bcopy(block, s, 4);
		block += 5;
		s += 4;
	}
}

#define DayBaseDelta 0117213            /* Unix day 0 in Tenex format */

/*
 * This screws up on some of the atime's we see, including, yielding, e.g.
 * Fri Dec 23 23:28:16 1994
 * Fri Dec 23 23:28:16 1994
 * Tue Jan 13 07:57:03 1987
 */
long
unixtime(block, wordoff)
char *block;
int  wordoff;
{
	long int t, s;

	t = getfield(block, wordoff, 0, 18);    /* First half is day */
	
	if (t == 0) return(0);			/* 0 means never referenced */

	t -= DayBaseDelta;                      /* Switch to unix base */
						/* Now has # days since */
						/* Jan 1, 1970 */
	if (t < 0) {
	  fprintf(stderr, "ERROR - Date earlier than Jan 1,1970!!!\n");
	}

	s = getfield(block, wordoff, 18, 18);   /* 2nd half is fraction day */

/* Note that the following calculation must be performed in the order shown
   in order to preserve precision.  It should also be done in double prec.
   floating point to prevent overflows.
 */
	s = (s * (double)24. * 60. * 60.) / 01000000;	/* Turn into seconds */

	s += t*24*60*60;                        /* Add day base */
	return(s);
}

char *
unixname(name)
char *name;
{
	static FILE *log = NULL;
	register char *t, *p;
	static char lastdir[64];
	char nbuf[130];
	struct stat stb;
	int mask;
	register int newdir = 0;

	if (numflg) {             /* If numeric filenames */
		if (log == NULL) log = fopen(LOGFILE, "a");
		fprintf(log, "%d is %s\n", number, name);
		sprintf(sunixname, "%d", number++);
		return(sunixname);
	}

	strcpy(sunixname, index(name, '<') + 1); /* trim off device */
	t = rindex(sunixname, '>');        	 /* find end of directory */
	*t = '.';

	if (strncmp(lastdir, sunixname, t - sunixname)) {/* maybe new dir */
	    strncpy(lastdir, sunixname, t - sunixname);	/* remember it */
	    newdir = 1;
	}
if(newdir) printf("newdir; name '%s', lastdir '%s'\n", name, lastdir);

	for (p = sunixname; p <= t; p++)
	    if (*p == '.' || *p == '/') {
		if (newdir) {
		    *p = '\0';			/* temporarily null it off */
if(newdir) printf("sunixname '%s'\n", sunixname);
		    if (stat(sunixname, &stb) < 0) {
			mask = umask(2);
			if (mkdir(sunixname, 0777) < 0)
			    punt(1, "mkdir %s failed", sunixname);
			umask(mask);
		    }
		}
		*p = '/';
	    }
	
	for (p = t+1; *p; p++)
	  if (*p == '/') *p = '\\';

	if (!genflg) {
		t = rindex(sunixname, '.');	/* find last . */
		*t = 0;				/* zap it out */
	}
	return(sunixname);
}

int
t2uprot(prot)
register unsigned prot;
{
    register unsigned tprot, uprot;
    register int tshift;

#ifdef notdef
    if (f->FB_dir) {			/* THIS WON'T WORK! */
    	punt(0, "Can't handle directory %s", topsname);
	prot = gtdirprot(_dirnm(jfn));	/* returns 20 fmt protection */
	for (tshift=12, uprot=0; tshift >= 0; tshift -= 6) {
	    tprot = prot >> tshift;	/* pick up next field */
	    uprot <<= 3;
	    if (tprot & DP_rd)
		uprot |= WREAD|WEXEC;	/* world read, world execute */
	    if (tprot & (DP_cn|DP_cf))	/* allow write for either conn. */
		uprot |= WWRITE;	/*   access or add files access */
	}
    }
    else
#endif
    {  /* do it this way so easily modified-- i know it could be faster */
	for (tshift=12, uprot=0; tshift >= 0; tshift -= 6) {
	    tprot = prot >> tshift;
	    uprot <<= 3;
	    uprot |= (tprot >> 3) & 07;		/* just r,w,x */
	}
    }
    return uprot;
}

#ifdef notdef
#define HOUR 3600
#define DAY  (HOUR*24)
#define DAY0 40587     /* number of days between tops20 0 day and Unix 0 day */

#define makeword(l, r)	( ((l) << 18) | (r) )
#define getright(b)	( (b) & 0777777 )
#define getleft(b)	( (b) >> 18 )

/* Convert Tops-20 to Unix time -- curently incomplete due to 32 < 36 bits */
int
_t2utim(t)
unsigned t;
{
    register ticks, rh, secs;

    ticks = t - makeword(DAY0, 0);
    rh = getright(ticks) * DAY;
    secs = rh >> 18;
    if (rh % makeword(1,0) > 0400000)
	secs++;				/* round up */
    return (getleft(ticks) * DAY) + secs;
}
#endif

int
compwant(w1, w2)
register struct want *w1, *w2;
{
    int sdif;

    if (sdif = w1->ssnum - w2->ssnum)
	return sdif;
    return (w1->fnum - w2->fnum);
}

getrec(fd, addr, len)
     int len;
     char *addr;
{
  register int cc, remaining;

  if (tpcflg) {
    unsigned char count[2];

    if (read(fd, count, 2) != 2)
      return 0;

    /* XXX make sure less than len?? */
    remaining = count[0] + (count[1] << 8);
  }
  else
    if (simhflg) {
      unsigned char b[4];
      int mark;

      if (read(fd, b, 4) != 4)
	return 0;

      mark = (b[3] << 24) | (b[2] << 16) | (b[1] << 8) | b[0];
      if (mark == 0xffffffff)
	      mark = 0;
      remaining = mark;
    }
  else
    remaining = len;

  while (remaining > 0)
    {
      cc = read(fd, addr, remaining);

      if (simhflg) {
	 unsigned char b[4];
	 int mark;

	 if (read(fd, b, 4) != 4)
		 return 0;
      }

/*      fprintf(stderr, "read(0, %x, %d) = %d\n", addr, remaining, cc);*/

      if (cc == remaining) return(len);

      if (cc == 0) return(len - remaining);

      if (cc < 0) {
	fprintf(stderr, "Got an error, errno=%d\n", errno);
	exit(1);
      }

/* Hack hack, cough...  If we are reading from a tape (or a file), then don't
   try to pad the record...
 */
      if (fd != 0) return(cc);

      remaining -= cc;
      addr += cc;
    }
  return(len);
}
