;;; -*- Mode: Lisp; Package: FILE-SYSTEM; Base: 8.; Patch-File: T -*-
;;; Written 1/27/84 00:53:23 by LMFile,
;;; Reason: Fix failure to increment file open count on delete, rename, etc.
;;; while running on Lisp Machine Filecomputer from band 1
;;; with System 98.30, CADR 3.5, ZMail 53.9, MIT-Specific 22.0, Experimental Local-File 48.1, Experimental FILE-Server 8.2, Experimental LFS 3.1, Experimental MagTape 22.5, microcode 306, Xmntl FS.



; From file FSGUTS.LISP PS:<L.FILE> OZ:
#8R FILE-SYSTEM#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "FILE-SYSTEM")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: FILE; FSGUTS  "


(DEFSUBST LMFS-FILE-BEING-WRITTEN-OR-SUPERSEDED? (FILE)
  (OR (MINUSP (FILE-OPEN-COUNT FILE))
      (NOT (AND (FILE-CLOSED? FILE)
		(NULL (FILE-OVERWRITE-FILE FILE))))))

))

; From file FSGUTS.LISP PS:<L.FILE> OZ:
#8R FILE-SYSTEM#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "FILE-SYSTEM")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: FILE; FSGUTS  "

(DEFSUBST REQUIRE-CLOSED-FILE-NOT-BEING-SUPERSEDED (FILE)
  (IF (LMFS-FILE-BEING-WRITTEN-OR-SUPERSEDED? FILE)
      (LM-SIGNAL-ERROR 'OPEN-OUTPUT-FILE)))

))

; From file FSGUTS.LISP PS:<L.FILE> OZ:
#8R FILE-SYSTEM#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "FILE-SYSTEM")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: FILE; FSGUTS  "


(DEFSUBST REQUIRE-FILE-NOT-OPEN (FILE)
  (REQUIRE-CLOSED-FILE FILE)
  (UNLESS (ZEROP (FILE-OPEN-COUNT FILE))
    (LM-SIGNAL-ERROR 'FILE-LOCKED)))

))

; From file FSGUTS.LISP PS:<L.FILE> OZ:
#8R FILE-SYSTEM#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "FILE-SYSTEM")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: FILE; FSGUTS  "


(DEFUN LMFS-OPEN-OVERWRITE-FILE (FILE)
  (LOCKING (FILE-LOCK FILE)
    (REQUIRE-FILE-NOT-OPEN FILE)
    (DECF (FILE-OPEN-COUNT FILE))))

))

; From file FSGUTS.LISP PS:<L.FILE> OZ:
#8R FILE-SYSTEM#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "FILE-SYSTEM")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: FILE; FSGUTS  "


(DEFUN LMFS-CLOSE-FILE (FILE)
  "Guarantees that FILE has its Closed bit set."
  (LOCKING (FILE-LOCK FILE)
    (COND ((MINUSP (FILE-OPEN-COUNT FILE))
	   (UNLESS (= (FILE-OPEN-COUNT FILE) -1)
	     (FERROR NIL "File open count is less than -1."))
	   (INCF (FILE-OPEN-COUNT FILE)))
	  ((PLUSP (FILE-OPEN-COUNT FILE))
	   (DECF (FILE-OPEN-COUNT FILE)))
	  (T (FERROR NIL "Open count of file being closed is zero.")))
    (UNLESS (FILE-CLOSED? FILE)
      (REQUIRE-ZERO-OPEN-COUNT FILE)
      (SETF (FILE-CLOSED? FILE) T)
      (UNLESS (FILE-DELETED? FILE)
	(USING-PUT
	  (CHANGE-MAP-DISK-SPACE (FILE-MAP FILE) PUT-RESERVED PUT-USED)))
      (LET ((OVERWRITTEN-FILE (FILE-OVERWRITE-FILE FILE))
	    (DIRECTORY (FILE-DIRECTORY FILE)))
	(UNLESS (NULL OVERWRITTEN-FILE)
	  (LOCKING (FILE-LOCK OVERWRITTEN-FILE)
	    (SETF (FILE-OVERWRITE-FILE FILE) NIL)
	    (DECF (FILE-OPEN-COUNT OVERWRITTEN-FILE))
	    (SETF (FILE-OVERWRITE-FILE OVERWRITTEN-FILE) NIL)
	    (REPLACE-FILE-IN-DIRECTORY OVERWRITTEN-FILE FILE)
	    (COND ((FILE-DELETED? OVERWRITTEN-FILE)
		   (USING-PUT
		     (CHANGE-MAP-DISK-SPACE (FILE-MAP OVERWRITTEN-FILE)
					    PUT-RESERVED
					    PUT-FREE)))
		  (T
		   (REQUIRE-DELETABLE-FILE OVERWRITTEN-FILE)
		   (SETF (FILE-DELETED? OVERWRITTEN-FILE) T)
		   (USING-PUT
		     (CHANGE-MAP-DISK-SPACE (FILE-MAP OVERWRITTEN-FILE)
					    PUT-USED
					    PUT-FREE))))))
	(WRITE-DIRECTORY-FILES DIRECTORY)))))

))

; From file FSGUTS.LISP PS:<L.FILE> OZ:
#8R FILE-SYSTEM#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "FILE-SYSTEM")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: FILE; FSGUTS  "

(DEFUN LOOKUP-FILE (DIRSPEC NAME TYPE VERSION
		    &OPTIONAL IF-DOES-NOT-EXIST IF-EXISTS (REALLY-OPEN ':DIRECTORY-OK)
		    (DELETED? T))
  "The basic function for finding files.
If REALLY-OPEN is T, we increment the open count of the file,
and get an error if it is a directory.
If REALLY-OPEN is :DIRECTORY-OK (the default), we increment but allow directories.
If REALLY-OPEN is NIL, we do not increment the open count.
DELETED? non-NIL means deleted files can be opened."
  (%STORE-CONDITIONAL (LOCF DIRSPEC) ':WILD "*")
  (%STORE-CONDITIONAL (LOCF NAME) ':WILD "*")
  (%STORE-CONDITIONAL (LOCF TYPE) ':WILD "*")
  (%STORE-CONDITIONAL (LOCF TYPE) ':UNSPECIFIC "")
  (%STORE-CONDITIONAL (LOCF VERSION) ':UNSPECIFIC ':NEWEST)
  (LET ((DIRECTORY (IF (NAMED-STRUCTURE-P DIRSPEC) DIRSPEC
		     (LOOKUP-DIRECTORY DIRSPEC)))
	NO-NEW-VERSION
	USE-EXISTING
	OLD-FILE)
    (BLOCK WIN
      (LM-LOOKUP-ERROR
	;; Must not allow recursive locking -- see LMFS-OPEN-OUTPUT-FILE.
	(LOCKING (DIRECTORY-LOCK DIRECTORY)
	  (*CATCH 'LOOKUP-FILE-ERROR
	    (MULTIPLE-VALUE-BIND (FILE LOC LAST-VERSION-SEEN)
		(COND ((AND (EQ VERSION ':NEWEST) (EQ IF-EXISTS ':NEW-VERSION))
		       (MULTIPLE-VALUE-BIND (FILE LOC LAST-VERSION-SEEN)
			   (LOOKUP-NEWEST-FILE DIRECTORY NAME TYPE)
			 (VALUES FILE
				 (IF FILE (CDR LOC) LOC)
				 LAST-VERSION-SEEN)))
		      ((MEMQ VERSION '(:NEWEST 0))
		       (IF DELETED?
			   (LOOKUP-NEWEST-FILE DIRECTORY NAME TYPE)
			 (LOOKUP-NEWEST-NON-DELETED-FILE DIRECTORY NAME TYPE)))
		      ((EQ VERSION ':OLDEST)
		       (IF (OR DELETED? (EQ IF-EXISTS ':NEW-VERSION))
			   (LOOKUP-OLDEST-FILE DIRECTORY NAME TYPE)
			 (LOOKUP-OLDEST-NON-DELETED-FILE DIRECTORY NAME TYPE)))
		      ;; Depends on extra vars in MULTIPLE-VALUE-BIND
		      ;; becoming bound to NIL.
		      ((MINUSP VERSION)
		       (LET ((NEWEST (LOOKUP-NEWEST-FILE DIRECTORY NAME TYPE)))
			 (IF (NULL NEWEST)
			     (*THROW 'LOOKUP-FILE-ERROR 'FILE-NOT-FOUND))
			 (LOOKUP-NAMED-FILE DIRECTORY NAME TYPE
					    (+ (FILE-VERSION NEWEST) VERSION))))
		      (T
		       (LOOKUP-NAMED-FILE DIRECTORY NAME TYPE VERSION)))
	      (IF (AND FILE (DIRECTORY? FILE) REALLY-OPEN
		       (NEQ REALLY-OPEN ':DIRECTORY-OK))
		  (*THROW 'LOOKUP-FILE-ERROR 'FILE-IS-SUBDIRECTORY))
	      (IF (OR (NULL FILE)
		      (AND (NOT DELETED?)
			   (FILE-DELETED? FILE)))
		  ;; File "does not exist".
		  (ECASE IF-DOES-NOT-EXIST
		    ((NIL) (RETURN-FROM WIN NIL))
		    (:ERROR
		     (*THROW 'LOOKUP-FILE-ERROR
			     (IF FILE 'OPEN-DELETED-FILE 'FILE-NOT-FOUND)))
		    (:CREATE NIL))
		;; File "exists".  Should we use it?
		(ECASE IF-EXISTS
		  ((NIL)
		   (SETQ USE-EXISTING T))
		  (:NEW-VERSION
		   (UNLESS (MEMQ VERSION '(:NEWEST :OLDEST)) (SETQ USE-EXISTING T)))
		  (:SUPERSEDE (SETQ NO-NEW-VERSION T))
		  ((:OVERWRITE :TRUNCATE :APPEND)
		   (UNLESS (ZEROP (FILE-OPEN-COUNT FILE))
		     (*THROW 'LOOKUP-FILE-ERROR 'FILE-LOCKED))
		   (SETQ USE-EXISTING T))
		  (:ERROR
		   (*THROW 'LOOKUP-FILE-ERROR 'FILE-ALREADY-EXISTS))
		  ((:RENAME :RENAME-AND-DELETE)
		   (SETQ OLD-FILE FILE))))
	      (RETURN-FROM WIN
		(IF USE-EXISTING
		    (PROGN
		      (IF REALLY-OPEN
			  (IF (MEMQ IF-EXISTS '(:OVERWRITE :TRUNCATE :APPEND))
			      (LMFS-OPEN-OVERWRITE-FILE FILE)
			    (LMFS-OPEN-INPUT-FILE FILE)))
		      FILE)
		  (VALUES
		    (LMFS-OPEN-OUTPUT-FILE
		      DIRECTORY LOC NAME TYPE
		      (COND ((EQ VERSION ':NEWEST)
			     (IF (NULL LAST-VERSION-SEEN)
				 1
			       (IF NO-NEW-VERSION
				   LAST-VERSION-SEEN
				 (1+ LAST-VERSION-SEEN))))
			    ((EQ VERSION ':OLDEST)
			     (IF (NULL LAST-VERSION-SEEN)
				 1
			       (IF NO-NEW-VERSION
				   LAST-VERSION-SEEN
				 (1- LAST-VERSION-SEEN))))
			    ((NUMBERP VERSION)
			     (COND ((MINUSP VERSION)
				    (*THROW 'LOOKUP-FILE-ERROR 'FILE-NOT-FOUND))
				   ((NOT (< VERSION 1_16.))
				    (*THROW 'LOOKUP-FILE-ERROR 'VERSION-TOO-LARGE))
				   ((ZEROP VERSION)
				    (OR LAST-VERSION-SEEN 1))
				   (T VERSION))))
		      (UNLESS (AND (MEMQ VERSION '(:NEWEST :OLDEST))
				   (NOT NO-NEW-VERSION))
			FILE))
		    OLD-FILE))))))
	DIRECTORY NAME TYPE VERSION))))

;;;; Directory Lookup

))

; From file FSGUTS.LISP PS:<L.FILE> OZ:
#8R FILE-SYSTEM#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "FILE-SYSTEM")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: FILE; FSGUTS  "


(DEFUN LOOKUP-DIRECTORY-FILES (DIR NAME TYPE VERSION DELETED?)
  (LET ((FILES '()))
    (DOLIST (FILE (READ-DIRECTORY-FILES DIR))
      (IF (AND (WILDCARD-MATCH NAME (FILE-NAME FILE))
	       (WILDCARD-MATCH TYPE (FILE-TYPE FILE))
	       (OR (NOT (NUMBERP VERSION))
		   (NOT (PLUSP VERSION))
		   (= VERSION (FILE-VERSION FILE))))
	  (IF (OR DELETED? (LMFS-CLOSED-FILE? FILE))
	      (PUSH FILE FILES))))
    (SETQ FILES (NREVERSE FILES))
    (IF (OR (EQ VERSION ':WILD)
	    (AND (NUMBERP VERSION)
		 (> VERSION 0)))
	FILES
	(LOOP FOR FILE IN FILES BY #'NEXT-GENERIC-FILE
	      AS NEW-FILE = (LOOKUP-FILE DIR (FILE-NAME FILE) (FILE-TYPE FILE) VERSION
					 NIL NIL NIL DELETED?)
	      WHEN NEW-FILE COLLECT NEW-FILE))))

;;; Special Matcher for Wildcards

))

; From file FSGUTS.LISP PS:<L.FILE> OZ:
#8R FILE-SYSTEM#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "FILE-SYSTEM")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: FILE; FSGUTS  "

(DEFUN LMFS-OPEN-OUTPUT-FILE (DIRECTORY LOC NAME TYPE VERSION OVERWRITTEN-FILE)
  (WHEN (< (AREF PUT-USAGE-ARRAY PUT-FREE) PUT-MINIMUM-FREE-PAGES)
    (LM-SIGNAL-ERROR 'NO-MORE-ROOM))
  (REQUIRE-LOCK (DIRECTORY-LOCK DIRECTORY))
  (LET ((FILE (CREATE-NEW-FILE DIRECTORY NAME TYPE VERSION)))
    (SETF (FILE-OPEN-COUNT FILE) -1)
    (SETF (FILE-CLOSED? FILE) NIL)
    (SETF (FILE-DELETED? FILE) NIL)
    (COND ((NULL OVERWRITTEN-FILE)
	   (PUSH FILE (CDR LOC))
	   (PROCESS-UNLOCK (LOCF (DIRECTORY-LOCK DIRECTORY))))
	  (T
	   ;; This is critical; unlock directory before locking file.
	   (PROCESS-UNLOCK (LOCF (DIRECTORY-LOCK DIRECTORY)))
	   (LOCKING (FILE-LOCK OVERWRITTEN-FILE)
	     (REQUIRE-CLOSED-FILE-NOT-BEING-SUPERSEDED OVERWRITTEN-FILE)
	     (REQUIRE-DELETABLE-FILE OVERWRITTEN-FILE)
	     (SETF (FILE-OVERWRITE-FILE FILE) OVERWRITTEN-FILE)
	     (INCF (FILE-OPEN-COUNT OVERWRITTEN-FILE))
	     (SETF (FILE-OVERWRITE-FILE OVERWRITTEN-FILE) FILE))))
    (WRITE-DIRECTORY-FILES DIRECTORY)
    FILE))

))

; From file FSGUTS.LISP PS:<L.FILE> OZ:
#8R FILE-SYSTEM#:
(COMPILER-LET ((PACKAGE (PKG-FIND-PACKAGE "FILE-SYSTEM")))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: FILE; FSGUTS  "

(DEFUN LOOKUP-SUBDIRECTORY-STEP (NODE STEP OK-IF-NOT-THERE)
  (READ-DIRECTORY-FILES NODE)
  (LOCKING (DIRECTORY-LOCK NODE)
    (MULTIPLE-VALUE-BIND (FILE LOC)
	(LOOKUP-NAMED-FILE NODE
			   STEP
			   LMFS-DIRECTORY-TYPE
			   LMFS-DIRECTORY-VERSION)
      (COND ((NOT (NULL FILE))
	     (LOCKING (FILE-LOCK FILE)
	       (IF (NOT (DIRECTORY? FILE))
		   (FERROR NIL "~S: expected a Directory." FILE))
	       (IF (LMFS-FILE-BEING-WRITTEN-OR-SUPERSEDED? FILE)
		   (LM-LOOKUP-ERROR 'OPEN-UNFINISHED-DIRECTORY
				    NODE
				    STEP
				    LMFS-DIRECTORY-TYPE
				    LMFS-DIRECTORY-VERSION))
	       (IF (FILE-DELETED? FILE)
		   (LM-LOOKUP-ERROR 'OPEN-DELETED-DIRECTORY
				    NODE
				    STEP
				    LMFS-DIRECTORY-TYPE
				    LMFS-DIRECTORY-VERSION)))
	     (READ-DIRECTORY-FILES FILE)	;make sure files are read in.
	     FILE)
	    (LM-AUTOMATICALLY-CREATE-DIRECTORIES
	     (LET ((DIRECTORY (CREATE-NEW-DIRECTORY NODE STEP)))
	       (PUSH DIRECTORY (CDR LOC))
	       (PROCESS-UNLOCK (LOCF (DIRECTORY-LOCK NODE)))
	       (WRITE-DIRECTORY-FILES NODE)
	       DIRECTORY))
	    ((NOT OK-IF-NOT-THERE)
	     (LM-SIGNAL-ERROR 'DIRECTORY-NOT-FOUND))))))

))
