;;; -*- Mode:Lisp;Package:FS;Base:8;Fonts:CPTFONT-*-
;;; Robert P. Krajewski (RpK), 9 September 1983. (c) 1983 Lisp Machine Incorporated.
;;; Functions used at da machine wit da tape drive -- y'know ?
;;; Changes:
;;;  12/16/83 08:50:45  RpK, LMI-East		Made part of magtape system
;;;   1/05/84 19:24:18  RpK, LMI-East		BYTE-SIZE passed along too for dump server
;;;						:DUMP-RECORD-HOST site option

(DEFCONST DUMP-VERSION 4
  "Used for reading dump ``directories'' generated by dump functions.
Does not mean that the magtape format has changed !
/(Though perhaps the dump software may cause more properties to be remembered
on tape in the future.)")

(DEFINE-SITE-VARIABLE *DEFAULT-RECORD-HOST* :DUMP-RECORD-HOST)

;;; The format of the dump file consists mostly of records separated by newlines
;;; A record consists of
;;;  truename<nl>creation-date<nl>author<nl>properties
;;; where properties are : propname<sp>prop<nl>
;;;  truename is obtained by sending :STRING-FOR-PRINTING to the pathname
;;;  creation-date is printed with TIME:PRINT-UNIVERSAL-TIME
;;;  author is a string on a line by itself
;;;  each propname and prop are printed with PRIN1, with the following variables bound:
;;;   PACKAGE to "USER", BASE to 10, SI:PRINT-READABLY to T, 
;;;     and PRINLEVEL and PRINLENGTH to ()
;;;
;;; Version 4 is just like version 3, except an alist followed by a newline is the first thing
;;; in the file.  Each entry's CAR is a symbol in the keyword package.  The meanings are:
;;;
;;;  :VERSION       dump version
;;;  :DUMP-STARTED  universal time the dump started
;;;  :MACHINE       the machine on which the tape was written
;;;  :FILE-HOST     the file host whose files got dumped
;;;  :OPERATOR      the user who did the dump
;;;  :TYPE          usually :INCREMENTAL or :FULL
;;;  :DUMP-TAPE     the first tape name
;;;  :DUMP-TAPE-VOLUME which tape number in this dump (0 = first)
;;;  :THIS-TAPE     just to check
;;;  :REUSED-P      if this tape has been reused
;;;  :NEXT-TAPE     what tape, if any, the dump continues on.
;;;  :FIRST-PATHNAME the first file dumped on this tape
;;;  :LAST-PATHNAME the last file dumped on this tape
;;; So the first character in a version 4 file will always be a "(".

;;; Tape entries
(DEFSTRUCT (TAPE-ENTRY (:TYPE :ARRAY) (:CONC-NAME TE-))
  PATHNAME
  CREATION-DATE
  AUTHOR
  PROPERTIES  ; not used yet
  NEXT)       ; pointer to next TE

(DEFSUBST TE-GET (TAPE-ENTRY PROPERTY) (GET (LOCF (TE-PROPERTIES TAPE-ENTRY)) PROPERTY))
(DEFUN TE-PUT (TAPE-ENTRY PROPERTY VALUE)
  (PUTPROP (LOCF (TE-PROPERTIES TAPE-ENTRY)) VALUE PROPERTY))

(DEFVAR *TAPE-ENTRY-CONS-METER* 0)
(DEFVAR *FREE-TAPE-ENTRY* ())
(DEFUN GET-TAPE-ENTRY (PATHNAME CREATION-DATE AUTHOR &OPTIONAL PROPERTIES &AUX TE)
  (COND ((SETQ TE *FREE-TAPE-ENTRY*)
	 (WITHOUT-INTERRUPTS
	   (SETQ *FREE-TAPE-ENTRY* (TE-NEXT *FREE-TAPE-ENTRY*))
	   (ALTER-TAPE-ENTRY TE
			     PATHNAME PATHNAME CREATION-DATE CREATION-DATE
			     AUTHOR AUTHOR PROPERTIES PROPERTIES NEXT ()))
	 TE)
	(T (INCF *TAPE-ENTRY-CONS-METER*)
	   (MAKE-TAPE-ENTRY PATHNAME PATHNAME CREATION-DATE CREATION-DATE AUTHOR AUTHOR
			    PROPERTIES PROPERTIES NEXT ()))))

(DEFUN RETURN-TAPE-ENTRY (TE &OPTIONAL LINKED-ALSO-P)
  (WITHOUT-INTERRUPTS
    (IF LINKED-ALSO-P ; in this case, a tape entry -- look at WRITE-TAPE-ENTRIES and callers
	(SETF (TE-NEXT LINKED-ALSO-P) *FREE-TAPE-ENTRY*)
      (SETF (TE-NEXT TE) *FREE-TAPE-ENTRY*))
    (SETQ *FREE-TAPE-ENTRY* TE)))

(DEFUN CLEAR-TAPE-ENTRY-RESOURCE ()
  (SETQ *TAPE-ENTRY-CONS-METER* 0 *FREE-TAPE-ENTRY* ()))

(DEFMACRO DO-LINKED-TAPE-ENTRIES ((FIRST-TAPE-ENTRY TE) &BODY BODY)
  `(DO ((,TE ,FIRST-TAPE-ENTRY (TE-NEXT ,TE)))
       ((NULL ,TE))
     ,@BODY))

(DEFMACRO APPEND-TAPE-ENTRY (HEAD TAIL NEW &OPTIONAL NEW-TAIL)
  "Appends HEAD to NEW, using TAIL.
TAIL is updated to NEW-TAIL if supplied"
  (IF NEW-TAIL
      `(WHEN ,NEW
	 (COND (,HEAD ; now assume TAIL is non-()
		(SETF (TE-NEXT ,TAIL) ,NEW)
		(IF ,NEW-TAIL (SETQ ,TAIL ,NEW-TAIL)))
	       (T ; start up a new List, assume TAIL is ()
		(SETQ ,HEAD ,NEW)
		(SETQ ,TAIL ,NEW-TAIL)))) 
    `(WHEN ,NEW
       (COND (,HEAD ; now assume TAIL is non-()
	      (SETF (TE-NEXT ,TAIL) ,NEW)
	      (SETQ ,TAIL ,NEW))
	     (T ; start up a new List, assume TAIL is ()
	      (SETQ ,HEAD ,NEW)
	      (SETQ ,TAIL ,NEW))))))
	 
(DEFUN WRITE-TAPE-ENTRY (TE STREAM)
  "This does not bind any printing variables.  Use PRINTING-CLEARLY around it."
  (SEND STREAM ':LINE-OUT (SEND (TE-PATHNAME TE) ':STRING-FOR-printing))
  (TIME:PRINT-UNIVERSAL-TIME (TE-CREATION-DATE TE) STREAM)
  (SEND STREAM ':TYO #\RETURN)
  (SEND STREAM ':LINE-OUT  (TE-AUTHOR TE))
  (DO ((PLP (TE-PROPERTIES TE) (CDDR PLP)))
      ((NULL PLP))
    (PRIN1 (FIRST PLP) STREAM)
    (SEND STREAM ':TYO #\SPACE)
    (PRIN1 (SECOND PLP) STREAM)
    (SEND STREAM ':TYO #\RETURN))
  (SEND STREAM ':TYO #\RETURN))

(DEFUN WRITE-TAPE-ENTRIES (TE STREAM &optional deallocate-p)
  "Writes out entries starting with TE onto stream.
DEALLOCATE-P should be the tail TE."
  (PRINTING-CLEARLY
    (DO-LINKED-TAPE-ENTRIES (TE TTE)
      (WRITE-TAPE-ENTRY TTE STREAM)))
  (IF DEALLOCATE-P (RETURN-TAPE-ENTRY TE deallocate-p)))

(DEFUN GET-TAPE-NAME (TAPES)
  (LET ((TAPE-NAME (STRING-TRIM '(#\SPACE #\TAB)
				(PROMPT-AND-READ ':STRING "~&Enter Tape name : "))))
    (COND ((MEM #'STRING-EQUAL TAPE-NAME TAPES)
	   (FORMAT T "~&~A has already been used in this dump." TAPE-NAME)
	   (GET-TAPE-NAME TAPES))
	  ((STRING-EQUAL TAPE-NAME "")
	   (FORMAT T "~&Please type in a reasonable name.")
	   (GET-TAPE-NAME TAPES))
	  (T TAPE-NAME))))

(DEFUN TAPE-ENTRY-FILE (RECORD-HOST TAPE-NAME)
  (FS:MAKE-PATHNAME ':HOST RECORD-HOST
		    ':DIRECTORY "MagTape"
		    ':NAME (STRING-APPEND "DUMP-" TAPE-NAME)
		    ':TYPE ':TEXT
		    ':VERSION ':NEWEST))

(DEFUN GET-TAPE-ENTRY-FILE (tapes record-host tapes-to-use &AUX PROBE TAPE-NAME)
  (DECLARE (:VALUES PATHNAME TAPE-NAME REUSED-P new-tapes-to-use))
  (SETQ PROBE (TAPE-ENTRY-FILE RECORD-HOST
			       (SETQ TAPE-NAME (or (first tapes-to-use)
						   (GET-TAPE-NAME tapes)))))
  (COND ((PROBEF PROBE)
	 (FORMAT T "~&Tape ~A has been used.  " tape-name)
	 (IF (Y-OR-N-P "Want to reuse it ? ")
	     (VALUES PROBE TAPE-NAME T (cdr tapes))
	   (FORMAT T "~&Enter another tape name: ")
	   (GET-TAPE-ENTRY-FILE tapes record-host (cdr tapes-to-use))))
	(T
	 (VALUES PROBE TAPE-NAME () (cdr tapes-to-use)))))

(DEFVAR *DUMP-FILE-ALIST*
	'((:VERSION) 
	  (:DUMP-STARTED)  ; universal time at which the dump started
	  (:MACHINE)	   ; the machine on which the tape was written
	  (:FILE-HOST)	   ; the file system being dumped
	  (:RECORD-HOST)   ; place for database (will be a site option...)
	  (:OPERATOR)	   ; the user who did the dump
	  (:TYPE)	   ; usually :INCREMENTAL or :FULL
	  (:DUMP-TAPE)	   ; the first tape name
	  (:DUMP-TAPE-VOLUME) ; which tape number in this dump (0 = first)
	  (:REUSED-P)	   ; per tape
	  (:THIS-TAPE)	   ; just to check
	  (:NEXT-TAPE)
	  (:FIRST-PATHNAME)
	  (:LAST-PATHNAME)))

(DEFMACRO SET-DUMP-ATTRIBUTE (PROPERTY VALUE)
  `(SETF (CDR (ASSQ ',PROPERTY *DUMP-FILE-ALIST*)) ,VALUE))

(DEFUN WRITE-DUMP-ATTRIBUTES (STREAM)
  (PRINTING-CLEARLY (PRIN1 *DUMP-FILE-ALIST* STREAM))
  (SEND STREAM ':TYO #\RETURN))

(defsubst get-dump-attribute (property alist) (cdr (assq property alist)))

(DEFUN WRITE-TAPE-ENTRY-FILE (FIRST-TE LAST-TE PATHNAME FOLLOWING-TAPE REUSED-P)
  "Takes care of :NEXT-TAPE, :FIRST-PATHNAME, and :LAST-PATHNAME attribute before,
and then sets :THIS-TAPE attribute to FOLLOWING-TAPE afterwards, as well as REUSED-P"
  (SET-DUMP-ATTRIBUTE :NEXT-TAPE FOLLOWING-TAPE)
  (SET-DUMP-ATTRIBUTE :FIRST-PATHNAME (TE-PATHNAME FIRST-TE))
  (SET-DUMP-ATTRIBUTE :LAST-PATHNAME (TE-PATHNAME LAST-TE))
  (WITH-OPEN-FILE (STREAM PATHNAME ':DIRECTION ':OUTPUT)
    (WRITE-DUMP-ATTRIBUTES STREAM)
    (WRITE-TAPE-ENTRIES FIRST-TE STREAM last-te))
  (SET-DUMP-ATTRIBUTE :THIS-TAPE FOLLOWING-TAPE)
  (SET-DUMP-ATTRIBUTE :REUSED-P REUSED-P))

(DEFUN READ-TAPE-ENTRY-STREAM-BEGINNING (STREAM)
  "Returns two values: the version number and the info alist for this file."
  (DECLARE (:VALUES VERSION-NUMBER INFO-ALIST))
  (IF ( (SEND STREAM ':TYIPEEK) #/() (VALUES 3. ())
    (VALUES 4. (PROG1 (PRINTING-CLEARLY (READ STREAM)) (SEND STREAM ':TYI)))))

(DEFUN READ-SINGLE-TAPE-ENTRY (STREAM)
  "Returns either a tape entry or () (for end of file)."
  (LET ((C (SEND STREAM ':TYI)))
    (AND C
	 (SEND STREAM ':UNTYI C)
	 (LET ((TE (GET-TAPE-ENTRY (PARSE-PATHNAME (SEND STREAM ':LINE-IN))
				   (TIME:PARSE-UNIVERSAL-TIME (SEND STREAM ':LINE-IN))
				   (SEND STREAM ':LINE-IN))))
	   ;; If the next character is a NL, gobble it and return, else get the property
	   (DO ((C (SEND STREAM ':TYI) (SEND STREAM ':TYI)))
	       ((OR (NULL C) (CHAR-EQUAL C #\Return)) (IF C TE))
	     (SEND STREAM ':UNTYI C)
	     (PRINTING-CLEARLY
	       (SETF (TE-PROPERTIES TE)
		     (CONS (READ STREAM) (CONS (READ STREAM) (TE-PROPERTIES TE)))))
	     (SEND STREAM ':TYI)))))) ; gobble the Newline that follows
    
;;; The Tape directory is a list of ((type (tape1 ... tapen) utime-start host) completed)
;;; The most recent entries are up front.
(defconst tape-directory-file "MAGTAPE;TAPE.DUMP-DIRECTORY")
(defun tape-directory-file (record-host) (fs:parse-pathname tape-directory-file record-host))
(defmacro recdump-type (x) `(caar ,x))
(defmacro recdump-tapes (x) `(cadar ,x))
(defmacro recdump-start (x) `(caddar ,x))
(defmacro recdump-host (x) `(cadR (Cddar ,x)))
(defmacro recdump-completed (x) `(cadr ,x))

(DEFUN READ-DUMPS-OF-TYPE (RECORD-HOST FILE-HOST DUMP-TYPES &AUX RESULT)
  "Return a list of dump records that describe completed dumps of types TYPES on FILE-HOST.
The directory is read from RECORD-HOST (determined by the caller from the site option).
The list is in reverse chronological order of dump start times."
  (DOLIST (RD (READ-TAPE-DIRECTORY RECORD-HOST) (NREVERSE RESULT))
    (AND (RECDUMP-COMPLETED RD)
	 (EQ FILE-HOST (RECDUMP-HOST RD))
	 (MEMQ (RECDUMP-TYPE RD) DUMP-TYPES)
	 (PUSH RD RESULT))))
    
(defsubst find-tape-entry (type tapes utime host td)
  (assoc (list type tapes utime host) td))

(defun read-tape-directory (record-host)
  (with-open-file-case (s (tape-directory-file record-host))
    (fs:file-not-found ())
    (:no-error (printing-clearly (read s)))))

(defun write-tape-directory (new-td record-host)
  (with-open-file (s (tape-directory-file record-host) ':direction ':output)
    (printing-clearly  (print new-td s))))

(DEFMACRO MODIFYING-TAPE-DIRECTORY ((TAPE-DIRECTORY record-host) &BODY BODY)
  "Modify the TAPE-DIRECTORY on RECORD-HOST within BODY"
  `(LET ((,TAPE-DIRECTORY (READ-TAPE-DIRECTORY ,record-host)))
     ,@BODY
     (WRITE-TAPE-DIRECTORY ,TAPE-DIRECTORY ,record-host)))

(DEFUN DUMP-START-FOR-TAPE-DIRECTORY (TYPE FIRST-TAPE UTIME HOST record-host)
  (MODIFYING-TAPE-DIRECTORY (TD record-host)
    (PUSH (LIST (LIST TYPE (LIST FIRST-TAPE) UTIME HOST) ()) TD)))

(DEFUN DUMP-END-FOR-TAPE-DIRECTORY (TYPE FIRST-TAPE UTIME HOST record-host ALL-TAPES)
  (MODIFYING-TAPE-DIRECTORY (TD record-host)
    (LET ((ENTRY (FIND-TAPE-ENTRY TYPE (LIST FIRST-TAPE) UTIME HOST TD)))
      (SETF (RECDUMP-TAPES ENTRY) ALL-TAPES)
      (SETF (RECDUMP-COMPLETED ENTRY) T))))

(DEFUN EXPUNGE-UNCOMPLETED-DUMPS
       (&OPTIONAL (STARTED-BEFORE (TIME:PARSE-UNIVERSAL-TIME "one day ago"))
		  (RECORD-HOST *DEFAULT-RECORD-HOST*))
  "Forget about all uncompleted dumps started before STARTED-BEFORE."
  (MODIFYING-TAPE-DIRECTORY (TD RECORD-HOST)
    (SETQ TD (SUBSET #'(LAMBDA (RD) (OR (> (RECDUMP-START RD) STARTED-BEFORE)
					(RECDUMP-COMPLETED RD)))
		     TD))))

;;; Accessing locally and remotely  
(DEFPROP :LOCAL T SET-DUMPED-LOCALLY)

(DEFUN DIRECTORY-WALKER-LOCALLY (-TERMINAL-IO- DIRECTORIES PREDICATE)
  (LET ((TERMINAL-IO -TERMINAL-IO-))
    (DIRECTORY-WALKER DIRECTORIES PREDICATE)))

(DEFSELECT ((:PROPERTY :LOCAL FILE-WALKER))
  (:NEW-STATE (MODE IGNORE directories -TERMINAL-IO-)
    (LET ((SG (ALLOCATE-RESOURCE 'FILE-ITERATOR)))
      (STACK-GROUP-PRESET SG #'DIRECTORY-WALKER-LOCALLY
			  -TERMINAL-IO- directories (GET MODE 'INCLUDE-FILE-FUNCTION))
      SG))
  (:RETURN-STATE (S) (DEALLOCATE-RESOURCE 'FILE-ITERATOR S))
  (:OPEN-THING (FILE) (FILE-TRUENAME FILE))
  (:NEXT (STATE) (LOCAL-FILE-CORRECT-INFO STATE ':NEXT))
  (:RESET (STATE) (LOCAL-FILE-CORRECT-INFO STATE ':RESET))
  (:DUMP-OK (IGNORE) ()))

(DEFUN LOCAL-FILE-CORRECT-INFO (STATE MESSAGE &AUX (RESULT (FUNCALL STATE MESSAGE)))
  (IF RESULT
      (IF (DIRECTORY? RESULT) (VALUES T (DIRECTORY-FULL-NAME-KLUDGE RESULT) () 0)
	(VALUES () RESULT (FILE-AUTHOR-INTERNAL RESULT) (FILE-DEFAULT-BYTE-SIZE RESULT)))
    (values t () () 0))) ; Mimic server, () is a directory (& end of all files)

(DEFSELECT ((:PROPERTY :REMOTE FILE-WALKER))
  (:NEW-STATE (MODE HOST directories IGNORE)
    (let ((s (CHAOS:OPEN-STREAM HOST (STRING-APPEND "HKDUMP " MODE))))
      (printing-clearly (prin1 (MAPCAR #'DIRECTORY-FULL-NAME directories) s))
      (send s ':tyo #\Space)
      s))
  (:RETURN-STATE (S) (SEND S ':CLOSE ':raw)) ; don't finish
  (:OPEN-THING (FILE) FILE)
  (:NEXT (S) (REMOTE-FILE-CORRECT-INFO S ":NEXT "))
  (:RESET (S) (REMOTE-FILE-CORRECT-INFO S ":RESET "))
  (:DUMP-OK (S)
    (SEND S ':STRING-OUT ":OK ")))

(DEFUN REMOTE-FILE-CORRECT-INFO (STREAM REQUEST-STRING &AUX BYTE-SIZE)
  (SEND STREAM ':STRING-OUT REQUEST-STRING)
  (SEND STREAM ':FORCE-OUTPUT)
  (IF (= #/- (SEND STREAM ':TYI)) ; A file
      (VALUES ()
	      (FS:PARSE-PATHNAME (SEND STREAM ':LINE-IN)) ; pathname
	      (PROGN (SETQ BYTE-SIZE (SEND STREAM ':TYI)) (SEND STREAM ':LINE-IN))
	      BYTE-SIZE) ; make sure becaomes integer after common lisp
    (VALUES T (READ STREAM) (PROGN (SEND STREAM ':LINE-IN)
				   (SEND STREAM ':TYI) ; this should be 0 (byte size)
				   (SEND STREAM ':LINE-IN))
	    0)))

(DEFUN AWAIT-CARRIAGE-RETURN ()
  (SEND QUERY-IO ':FRESH-LINE)
  (FORMAT:OUTPUT QUERY-IO "Type " (format:ochar #\return ':lozenged) " to continue.")
  (do ((c (send query-io ':tyi) (send query-io ':tyi)))
      ((char-equal c #\return))
    (send query-io ':beep)))
  
;;; The actual dumping functions
(DEFVAR *DUMP-ONE-FILE-DUMMIED*
	() "Non-() if debugging
:YES Always proceed
:TYI Hand-down FERRORs out, Clear Input signals EOT, Hand-up set to :YES and proceed;
       anything else proceeds.")

(DEFUN DUMP-ONE-FILE-DUMMIED (FILE AUTHOR DUMP-PLIST BYTE-SIZE) DUMP-PLIST
  (FORMAT T "~&Supposedly dumping ~A by ~A. (byte size ~D)." FILE AUTHOR BYTE-SIZE)
  (IF (NEQ ':TYI *DUMP-ONE-FILE-DUMMIED*)
      (WITH-OPEN-FILE (PROBE FILE ':DIRECTION ())
	(GET-TAPE-ENTRY (SEND PROBE ':TRUENAME)
			(SEND PROBE ':GET ':CREATION-DATE)
			AUTHOR))
    (SEND STANDARD-OUTPUT ':STRING-OUT " ")
    (LET ((C (SEND STANDARD-INPUT ':TYI)))
      (COND ((CHAR-EQUAL C #\Hand-Down) (FERROR () "Foo !  Induced error"))
	    ((CHAR-EQUAL C #\Clear-Input) ())
	    (T
	     (IF (CHAR-EQUAL C #\Hand-Up) (SETQ *DUMP-ONE-FILE-DUMMIED* ':YES))
	     (WITH-OPEN-FILE (PROBE FILE ':DIRECTION ())
	       (GET-TAPE-ENTRY (SEND PROBE ':TRUENAME)
			       (SEND PROBE ':GET ':CREATION-DATE)
			       AUTHOR)))))))

;;; This is the real function.
;;; The dump-plist contains :TAPE, :DUMP-DATE, and :DUMP-TYPE
(DEFUN DUMP-ONE-FILE (FILE AUTHOR DUMP-PLIST BYTE-SIZE
		      &AUX CDATE TRUENAME CHARACTERS QFASLP LENGTH)
  "Returns either a tape entry, an error, or (), which means that we ran off the end of tape."
  (IF *DUMP-ONE-FILE-DUMMIED* (DUMP-ONE-FILE-DUMMIED FILE AUTHOR DUMP-PLIST BYTE-SIZE)
    (CONDITION-CASE (RESULT)
	(WITH-OPEN-FILE (S FILE ':DIRECTION ':INPUT ':CHARACTERS ':DEFAULT)
	  (FORMAT T
		  "~&~:[Binary~;Character~] ~:[~;QFASL ~]file ~A
 (byte size ~D, length ~D, created ~\time\) ... "
		  (SETQ CHARACTERS (SEND S ':CHARACTERS))
		  (SETQ QFASLP (SEND S ':QFASLP))
		  (SETQ TRUENAME (SEND S ':TRUENAME))
		  BYTE-SIZE
		  (SETQ LENGTH (SEND S ':LENGTH))
		  (SETQ CDATE (SEND S ':CREATION-DATE)))
	  (WITH-OPEN-STREAM (MT (MAKE-MT-FILE-STREAM
				  ':CHARACTERS CHARACTERS
				  ':DIRECTION ':OUTPUT
				  ':DEFAULTS-FROM-STREAM S
				  ':PLIST (LIST* ':CHARACTERS CHARACTERS
						 ':HOST (SEND (SEND TRUENAME ':HOST)
							      ':NAME-AS-FILE-COMPUTER)
						 ':DEVICE (SEND TRUENAME ':DEVICE)
						 ':DIRECTORY (SEND TRUENAME ':DIRECTORY)
						 ':NAME (SEND TRUENAME ':NAME)
						 ':TYPE (SEND TRUENAME ':TYPE)
						 ':VERSION (SEND TRUENAME ':VERSION)
						 ':AUTHOR AUTHOR
						 ':CREATION-DATE CDATE
						 ':QFASLP QFASLP
						 ':LENGTH-IN-BYTES LENGTH
						 ':LENGTH LENGTH
						 ':BYTE-SIZE BYTE-SIZE
						 DUMP-PLIST)))
	    (STREAM-COPY-UNTIL-EOF S MT))
	  (SEND STANDARD-OUTPUT ':LINE-OUT "done."))
      (FS:END-OF-TAPE ())
      (FS:FILE-ERROR
       (send error-output ':fresh-line)
       (SEND ERROR-OUTPUT ':BEEP 'FILE-DUMP-ERROR)
       (SEND ERROR-OUTPUT ':STRING-OUT " Error: ")
       (SEND RESULT ':REPORT ERROR-OUTPUT)
       (SEND ERROR-OUTPUT ':TYO #\RETURN)
       RESULT)
      (:NO-ERROR (GET-TAPE-ENTRY TRUENAME CDATE AUTHOR)))))

(DEFUN FILE-SYSTEM-DUMP (MODE &OPTIONAL DIRECTORIES
			 &KEY (HOST SI:LOCAL-HOST) (OPERATOR USER-ID)
			      (RECORD-HOST *DEFAULT-RECORD-HOST*)
			 QUERY-AT-END TAPES-TO-USE
			 &AUX ACCESS STATE
			 (ACCESS-METHOD
			   (GET (SETQ ACCESS (IF (EQ (SETQ HOST (SI:PARSE-HOST HOST))
						     SI:LOCAL-HOST)
						 ':LOCAL
					       ':REMOTE))
				'FILE-WALKER))
			 first-te last-te first-te-dir last-te-dir te-file DUMP-OK N-FILES
			 FILES-DUMPED-LOCAL FILES-DUMPED-LOCAL-DIR (TAPE-N 0) (TAPE-NAME)
			 TAPE-REUSED-P REAL-TYPE FIRST-TIME TAPES FIRST-TAPE
			 DUMP-PLIST BYTE-SIZE
			 (SET-DUMPED-LOCALLY (GET ACCESS 'SET-DUMPED-LOCALLY)))
  (SET-DUMP-ATTRIBUTE :VERSION DUMP-VERSION)
  (SET-DUMP-ATTRIBUTE :DUMP-STARTED (SETQ FIRST-TIME (TIME:GET-UNIVERSAL-TIME)))
  (SET-DUMP-ATTRIBUTE :MACHINE (SEND SI:LOCAL-HOST ':NAME))
  (SET-DUMP-ATTRIBUTE :RECORD-HOST (SEND (SETQ RECORD-HOST
					       (FS:GET-PATHNAME-HOST (OR RECORD-HOST
									 SI:LOCAL-HOST)))
					 ':NAME))
  (SET-DUMP-ATTRIBUTE :FILE-HOST (SEND HOST ':NAME))
  (SET-DUMP-ATTRIBUTE :OPERATOR OPERATOR)
  (SET-DUMP-ATTRIBUTE :TYPE (SETQ REAL-TYPE (REAL-DUMP-TYPE MODE)))
  (SETF (VALUES TE-FILE TAPE-NAME TAPE-REUSED-P)
	(GET-TAPE-ENTRY-FILE () RECORD-HOST TAPES-TO-USE))
  (SET-DUMP-ATTRIBUTE :DUMP-TAPE (SETQ FIRST-TAPE TAPE-NAME))
  (SET-DUMP-ATTRIBUTE :DUMP-TAPE-VOLUME 0)
  (SET-DUMP-ATTRIBUTE :THIS-TAPE TAPE-NAME)
  (SET-DUMP-ATTRIBUTE :REUSED-P TAPE-REUSED-P)
  ; The dump-plist contains :TAPE, :DUMP-DATE, and :DUMP-TYPE -- and look at order carefully!
  (SETQ DUMP-PLIST (LIST ':TAPE TAPE-NAME ':DUMP-DATE FIRST-TIME ':DUMP-TYPE REAL-TYPE))
  (OR DIRECTORIES (SETQ DIRECTORIES (NCONS (DC-ROOT-DIRECTORY))))
  (UNWIND-PROTECT
    (PROGN
      (SETQ STATE (FUNCALL ACCESS-METHOD ':NEW-STATE MODE HOST DIRECTORIES TERMINAL-IO))
      (DUMP-START-FOR-TAPE-DIRECTORY REAL-TYPE FIRST-TAPE FIRST-TIME HOST RECORD-HOST)
      (PUSH TAPE-NAME TAPES)
      (DO ((DUMP-RESULT) (DIRECTORY-P) (FILE T) (AUTHOR) (MESSAGE ':NEXT))
	  ((NULL FILE))
	(SETF (VALUES DIRECTORY-P FILE AUTHOR BYTE-SIZE)
	      (FUNCALL ACCESS-METHOD MESSAGE STATE))
	(SETQ MESSAGE ':NEXT) ; could have been :RESET
	(COND (DIRECTORY-P
	       (APPEND-TAPE-ENTRY FIRST-TE LAST-TE FIRST-TE-DIR LAST-TE-DIR)
	       (WHEN SET-DUMPED-LOCALLY
		 (IF FILES-DUMPED-LOCAL (NCONC FILES-DUMPED-LOCAL FILES-DUMPED-LOCAL-DIR)
		   (SETQ FILES-DUMPED-LOCAL FILES-DUMPED-LOCAL-DIR)))
	       (SETQ FIRST-TE-DIR () LAST-TE-DIR () FILES-DUMPED-LOCAL-DIR () N-FILES 0))
	      (FILE ; non-NIL
	       (SETQ DUMP-RESULT (DUMP-ONE-FILE (FUNCALL ACCESS-METHOD ':OPEN-THING FILE)
						AUTHOR
						DUMP-PLIST BYTE-SIZE))
	       (COND (DUMP-RESULT
		      (WHEN (NOT (ERRORP DUMP-RESULT)) ; Error message already printed
			(APPEND-TAPE-ENTRY FIRST-TE-DIR LAST-TE-DIR DUMP-RESULT)
			(INCF N-FILES)
			(WHEN SET-DUMPED-LOCALLY
			  (IF FILES-DUMPED-LOCAL-DIR
			      (NCONC FILES-DUMPED-LOCAL-DIR (NCONS FILE))
			    (SETQ FILES-DUMPED-LOCAL-DIR (NCONS FILE))))))
		     (T ; End of tape
		      (SEND TERMINAL-IO ':BEEP 'END-OF-DUMP-TAPE)
		      (FORMAT T "~&End of dump tape ~A."
			      (GET-DUMP-ATTRIBUTE ':THIS-TAPE *DUMP-FILE-ALIST*))
		      (SETQ FIRST-TE-DIR () LAST-TE-DIR () FILES-DUMPED-LOCAL-DIR ())
		      (FS:MT-SPACE-REV-TO-BOF 0)
		      (SETQ N-FILES 0)
		      (FS:MT-WRITE-EOF)
		      (FS:MT-REWIND)
		      ;; Write out the file (dump map)
		      (MULTIPLE-VALUE-BIND (NFILE NTAPE-NAME REUSED-P ttu)
			  (GET-TAPE-ENTRY-FILE tapes record-host tapes-to-use)
			(setq tapes-to-use ttu)
			(WRITE-TAPE-ENTRY-FILE first-te last-te TE-FILE NTAPE-NAME REUSED-P)
			(SETQ TE-FILE NFILE)
			(FORMAT T "~&Continuing ~A dump on ~:[new~;reused~] tape ~A."
				MODE REUSED-P NTAPE-NAME)
			(INCF TAPE-N)
			(SET-DUMP-ATTRIBUTE :DUMP-TAPE-VOLUME TAPE-N)
			(push ntape-name tapes)
			(SETF (SECOND DUMP-PLIST) NTAPE-NAME)
			(SETQ FIRST-TE () LAST-TE ()) ; keep only for one tape.
			)
		      (fs:mt-offline)
		      ;; type any key...
		      (FORMAT T "~&Type any key when ready to start again.")
		      (SEND TERMINAL-IO ':TYI)
		      (setq message ':reset)))) ; Will get send next time around
	      (t ; Last file encountered
	       (append-tape-entry first-te last-te first-te-dir last-te-dir)
	       (when set-dumped-locally
		 (if files-dumped-local (nconc files-dumped-local files-dumped-local-dir)
		   (setq files-dumped-local files-dumped-local-dir))))))
      ;; Write out last dump record.
      (fs:mt-write-eof)
      (fs:mt-rewind)
      (WRITE-TAPE-ENTRY-FILE FIRST-TE LAST-TE TE-FILE () ()) ; the END
      (SETQ DUMP-OK T)
      (when (OR (NOT QUERY-AT-END)
		(YES-OR-NO-P "Mark dump as successful ? "))
	(when sET-DUMPED-LOCALLY
	  (FORMAT T "~&Dumped ~D files locally." (LENGTH FILES-DUMPED-LOCAL))
	  (set-local-files-dumped files-dumped-local))
	(FUNCALL ACCESS-METHOD ':DUMP-OK STATE)
	(dump-end-for-tape-directory real-type first-tape first-time
				     host record-host (reverse tapes)))
	
      ) ; end protected form
    ;; cleanup forms
    (FUNCALL ACCESS-METHOD ':RETURN-STATE STATE)
    (IF DUMP-OK (FS:MT-OFFLINE))
    (FORMAT T "~&Dump ~:[aborted~:;completed~].~%" dump-ok)) ; end UNWIND-PROTECT

  )