;;; -*- Mode: Lisp; Package: File-System; Base: 8 -*-

;;; Management of disk units and packs

;;; Although we use the term "pack", it is really just a partition on the disk,
;;; and two partitions on the same pack can in fact be in use at once
;;; as two different "packs".

;;; The pack file-system header, which is the first block in the FILE partition.
;;; These symbols represent offsets within that block.


(DEFCONST HEADER-IDENTIFICATION 0)		;"FILE"
(DEFCONST HEADER-VERSION-NUMBER 1)		;1
(DEFCONST HEADER-VOLUME-NAME 2)			;32. characters
(DEFCONST HEADER-PACK-NUMBER-WITHIN-VOLUME 12)
(DEFCONST HEADER-NUMBER-OF-PACKS-IN-VOLUME 13)
(DEFCONST HEADER-SYSTEM-INCARNATION-NUMBER 14)	;Incremented each time pack is mounted,
						; used by the directory validater
(DEFCONST HEADER-PAGES-PER-BLOCK 15)
(DEFCONST HEADER-FREE-MAP-FIRST-BLOCK 16)	;Address of free block map
(DEFCONST HEADER-FREE-MAP-NUMBER-OF-BLOCKS 17)	;Size of free block map
(DEFCONST HEADER-AVAILABLE-TABLE-DANGER 20)	;If fewer than this many blks,
						;prohibit everything but deletion.
(DEFCONST HEADER-BLOCK-STARTS-FILE-MAP-FIRST-BLOCK 21)
(DEFCONST HEADER-LOCKED-BLOCK-MAP-FIRST-BLOCK 22)
(DEFCONST HEADER-ROOT-FIRST-BLOCK 23)		;Address of root directory if any
(DEFCONST HEADER-COMMENT 24)			;96. characters
	 ;54 next

(DEFCONST HEADER-PACK-CLEAN-FLAG 54)		;Nonzero means pack was dismounted
						;cleanly and doesn't need GC
						;when mounted next time.
						;2 means some files contain locked blocks,
						;1 means none do.
(DEFCONST HEADER-AVAILABLE-TABLE-MINIMUM 55)
(DEFCONST HEADER-AVAILABLE-TABLE-MAXIMUM 56)
(DEFCONST HEADER-AVAILABLE-TABLE-DESIRABLE 57)
(DEFCONST HEADER-AVAILABLE-TABLE-SIZE 60)
(DEFCONST HEADER-PARENT-VOLUME-NAME 61)		;32. chars.
;The following are present only if the header version is 2 or more.
(DEFCONST HEADER-NUMBER-OF-BLOCKS 71)		;Size of data in partition, in blocks.
	 ;72 next.
	 ;limit is 400

;; Default for value to initialize HEADER-AVAILABLE-TABLE-DANGER with.
(DEFVAR DEFAULT-DANGER-AVAILABLE-BLOCKS 200)

;; Default for value to initialize HEADER-AVAILABLE-TABLE-MINIMUM with.
(DEFVAR DEFAULT-MINIMUM-AVAILABLE-BLOCKS 300)

;; Default for value to initialize HEADER-AVAILABLE-TABLE-MAXIMUM with.
(DEFVAR DEFAULT-MAXIMUM-AVAILABLE-BLOCKS 2000)

;; Default for value to initialize HEADER-AVAILABLE-TABLE-DESIRABLE with.
(DEFVAR DEFAULT-DESIRABLE-AVAILABLE-BLOCKS 1000)

;; Default for value to initialize HEADER-AVAILABLE-TABLE-SIZE with.
;; The available table size of a partition should always be larger
;; than the desirable number of available blocks for that partition
;; to avoid a kind of thrashing.
(DEFVAR DEFAULT-AVAILABLE-TABLE-SIZE 1200)

(DEFUN VALIDATE-UNIT-NUMBER (UNIT)
  (OR (AND (FIXP UNIT)
	   ( UNIT 0)
	   (< UNIT N-UNITS))
      (FERROR NIL "Invalid unit number ~S" UNIT)))

(DEFUN INIT-FILE-PARTITION
       (UNIT PARTITION-NAME &OPTIONAL &KEY
	(VOLUME-NAME "FOO")
	(COMMENT "FOO")
	(PAGES-PER-BLOCK 1)
	(PACK-NUMBER 0)
	(PACKS-IN-VOLUME 1)
	(PARENT-VOLUME-NAME "ROOT")
	(DANGER-AVAILABLE-BLOCKS DEFAULT-DANGER-AVAILABLE-BLOCKS)
	(MINIMUM-AVAILABLE-BLOCKS DEFAULT-MINIMUM-AVAILABLE-BLOCKS)
	(MAXIMUM-AVAILABLE-BLOCKS DEFAULT-MAXIMUM-AVAILABLE-BLOCKS)
	(DESIRABLE-AVAILABLE-BLOCKS DEFAULT-DESIRABLE-AVAILABLE-BLOCKS)
	(AVAILABLE-TABLE-SIZE DEFAULT-AVAILABLE-TABLE-SIZE)
	&AUX RQB PART-BASE PART-SIZE
	(BLOCK-SIZE (* PAGE-SIZE PAGES-PER-BLOCK))
	(FREE-MAP-FIRST-BLOCK 1)
	FREE-MAP-N-BLOCKS)
  (OR (AND (FIXP PACK-NUMBER)
	   (FIXP PACKS-IN-VOLUME)
	   ( PACK-NUMBER 0)
	   (> PACKS-IN-VOLUME PACK-NUMBER))
      (FERROR NIL "Pack number ~S out of ~S packs is meaningless"
	      PACK-NUMBER PACKS-IN-VOLUME))
  (VALIDATE-UNIT-NUMBER UNIT)
  (CHECK-ARG VOLUME-NAME STRINGP "a string")
  (CHECK-ARG PARENT-VOLUME-NAME STRINGP "a string")
  (SETQ VOLUME-NAME (STRING-UPCASE VOLUME-NAME))
  (SETQ PARENT-VOLUME-NAME (STRING-UPCASE PARENT-VOLUME-NAME))
  (DISMOUNT-UNIT UNIT PARTITION-NAME)
  (UNWIND-PROTECT
    (PROG INIT ()
	  (SETQ RQB (GET-DISK-RQB))
	  (MULTIPLE-VALUE (PART-BASE PART-SIZE)
	    (FIND-DISK-PARTITION PARTITION-NAME NIL UNIT))
	  (OR PART-BASE (FERROR NIL "No partition named ~S on unit ~S" PARTITION-NAME UNIT))
	  (DISK-READ RQB UNIT PART-BASE)
	  (OR (YES-OR-NO-P
		(IF (STRING-EQUAL (GET-DISK-STRING RQB HEADER-IDENTIFICATION 4) "FILE")
		    (FORMAT NIL "Reinitialize unit ~S partition ~A, which is volume ~A pack ~S? "
			    UNIT PARTITION-NAME
			    (GET-DISK-STRING RQB HEADER-VOLUME-NAME 32.)
			    (GET-DISK-FIXNUM RQB HEADER-PACK-NUMBER-WITHIN-VOLUME))
		  (FORMAT NIL "Initialize unit ~S partition ~A, not formatted for LMFILE? "
			  UNIT PARTITION-NAME)))
	      (RETURN-FROM INIT NIL))
	  (FILLARRAY (RQB-BUFFER RQB) '(0))
	  (PUT-DISK-STRING RQB "FILE" HEADER-IDENTIFICATION 4)
	  (PUT-DISK-FIXNUM RQB 2 HEADER-VERSION-NUMBER)
	  (PUT-DISK-STRING RQB VOLUME-NAME HEADER-VOLUME-NAME 32.)
	  (PUT-DISK-FIXNUM RQB PACK-NUMBER HEADER-PACK-NUMBER-WITHIN-VOLUME)
	  (PUT-DISK-FIXNUM RQB PACKS-IN-VOLUME HEADER-NUMBER-OF-PACKS-IN-VOLUME)
	  (PUT-DISK-FIXNUM RQB 0 HEADER-SYSTEM-INCARNATION-NUMBER)
	  (PUT-DISK-FIXNUM RQB PAGES-PER-BLOCK HEADER-PAGES-PER-BLOCK)
	  (PUT-DISK-FIXNUM RQB (TRUNCATE PART-SIZE PAGES-PER-BLOCK) HEADER-NUMBER-OF-BLOCKS)
	  (SETQ FREE-MAP-N-BLOCKS
		(CEILING (TRUNCATE PART-SIZE PAGES-PER-BLOCK) (* BLOCK-SIZE 40)))
	  (PUT-DISK-FIXNUM RQB FREE-MAP-FIRST-BLOCK HEADER-FREE-MAP-FIRST-BLOCK)
	  (PUT-DISK-FIXNUM RQB FREE-MAP-N-BLOCKS HEADER-FREE-MAP-NUMBER-OF-BLOCKS)
	  (PUT-DISK-FIXNUM RQB (+ FREE-MAP-FIRST-BLOCK FREE-MAP-N-BLOCKS)
			   HEADER-BLOCK-STARTS-FILE-MAP-FIRST-BLOCK)
	  (PUT-DISK-FIXNUM RQB (+ FREE-MAP-FIRST-BLOCK FREE-MAP-N-BLOCKS FREE-MAP-N-BLOCKS)
			   HEADER-LOCKED-BLOCK-MAP-FIRST-BLOCK)
	  (PUT-DISK-FIXNUM RQB 0 HEADER-ROOT-FIRST-BLOCK)
	  (PUT-DISK-STRING RQB COMMENT HEADER-COMMENT 96.)
	  ;; Say we should GC this pack on first use.
	  (PUT-DISK-FIXNUM RQB 0 HEADER-PACK-CLEAN-FLAG)
	  (PUT-DISK-FIXNUM RQB DANGER-AVAILABLE-BLOCKS HEADER-AVAILABLE-TABLE-DANGER)
	  (PUT-DISK-FIXNUM RQB MINIMUM-AVAILABLE-BLOCKS HEADER-AVAILABLE-TABLE-MINIMUM)
	  (PUT-DISK-FIXNUM RQB MAXIMUM-AVAILABLE-BLOCKS HEADER-AVAILABLE-TABLE-MAXIMUM)
	  (PUT-DISK-FIXNUM RQB DESIRABLE-AVAILABLE-BLOCKS HEADER-AVAILABLE-TABLE-DESIRABLE)
	  (PUT-DISK-FIXNUM RQB AVAILABLE-TABLE-SIZE HEADER-AVAILABLE-TABLE-SIZE)
	  (PUT-DISK-STRING RQB PARENT-VOLUME-NAME HEADER-PARENT-VOLUME-NAME 32.)
	  (DISK-WRITE RQB UNIT PART-BASE)
; Clear out the block-starts-file map.
          (CLEAR-BITMAP UNIT (+ FREE-MAP-FIRST-BLOCK FREE-MAP-N-BLOCKS) FREE-MAP-N-BLOCKS
			PART-BASE PAGES-PER-BLOCK)
; Clear out the locked block map.
	  (CLEAR-BITMAP UNIT (+ FREE-MAP-FIRST-BLOCK FREE-MAP-N-BLOCKS FREE-MAP-N-BLOCKS)
			FREE-MAP-N-BLOCKS
			PART-BASE PAGES-PER-BLOCK)
; Clear out the free map.  Yes, say that no blocks are free!
; The first GC will correctly initialize the free map.
	  (CLEAR-BITMAP UNIT FREE-MAP-FIRST-BLOCK
			FREE-MAP-N-BLOCKS
			PART-BASE PAGES-PER-BLOCK))
    (RETURN-DISK-RQB RQB)))

;Change some of the parameters of a partition, without invalidating its contents.
;You must run this if you make the partition longer.
(DEFUN REINIT-FILE-PARTITION (UNIT PARTITION-NAME &OPTIONAL &KEY
				   VOLUME-NAME COMMENT PACK-NUMBER PACKS-IN-VOLUME
				   PARENT-VOLUME-NAME
				   DANGER-AVAILABLE-BLOCKS
				   MINIMUM-AVAILABLE-BLOCKS
				   MAXIMUM-AVAILABLE-BLOCKS
				   DESIRABLE-AVAILABLE-BLOCKS
				   AVAILABLE-TABLE-SIZE
				   &AUX RQB PART-BASE PART-SIZE PAGES-PER-BLOCK
				   BLOCK-SIZE
				   OLD-TOTAL-N-BLOCKS
				   TOTAL-N-BLOCKS
				   FREE-MAP-OLD-N-BLOCKS
				   FREE-MAP-N-BLOCKS)
  (VALIDATE-UNIT-NUMBER UNIT)
  (AND VOLUME-NAME (CHECK-ARG VOLUME-NAME STRINGP "a string"))
  (AND PARENT-VOLUME-NAME (CHECK-ARG PARENT-VOLUME-NAME STRINGP "a string"))
  (AND VOLUME-NAME (SETQ VOLUME-NAME (STRING-UPCASE VOLUME-NAME)))
  (AND PARENT-VOLUME-NAME (SETQ PARENT-VOLUME-NAME (STRING-UPCASE PARENT-VOLUME-NAME)))
  (DISMOUNT-UNIT UNIT PARTITION-NAME)
  (UNWIND-PROTECT
    (PROG INIT ()
	  (SETQ RQB (GET-DISK-RQB))
	  (MULTIPLE-VALUE (PART-BASE PART-SIZE)
	    (FIND-DISK-PARTITION PARTITION-NAME NIL UNIT))
	  (OR PART-BASE (FERROR NIL "No partition named ~A on unit ~S" PARTITION-NAME UNIT))
	  ;; Start from what's already in that partition's header.
	  (DISK-READ RQB UNIT PART-BASE)
	  (OR (AND (STRING-EQUAL (GET-DISK-STRING RQB HEADER-IDENTIFICATION 4) "FILE")
		   (MEMQ (GET-DISK-FIXNUM RQB HEADER-VERSION-NUMBER) '(1 2)))
	      (FERROR NIL "Unit ~S partition ~A not formatted for LMFILE"
		      UNIT PARTITION-NAME))
	  (OR (YES-OR-NO-P "Reinitialize unit ~S partition ~A, which is volume ~A pack ~S? "
			   UNIT PARTITION-NAME
			   (GET-DISK-STRING RQB HEADER-VOLUME-NAME 32.)
			   (GET-DISK-FIXNUM RQB HEADER-PACK-NUMBER-WITHIN-VOLUME))
	      (RETURN-FROM INIT NIL))
	  ;; Change pack identification if specified.
	  (AND VOLUME-NAME
	       (PUT-DISK-STRING RQB VOLUME-NAME HEADER-VOLUME-NAME 32.))
	  (AND PACK-NUMBER
	       (PUT-DISK-FIXNUM RQB PACK-NUMBER HEADER-PACK-NUMBER-WITHIN-VOLUME))
	  (AND PACKS-IN-VOLUME
	       (PUT-DISK-FIXNUM RQB PACKS-IN-VOLUME HEADER-NUMBER-OF-PACKS-IN-VOLUME))
	  (AND COMMENT
	       (PUT-DISK-STRING RQB COMMENT HEADER-COMMENT 96.))
	  (AND PARENT-VOLUME-NAME
	       (PUT-DISK-STRING RQB PARENT-VOLUME-NAME HEADER-PARENT-VOLUME-NAME 32.))
	  ;; Change available table parameters if specified.
	  (AND DANGER-AVAILABLE-BLOCKS
	       (PUT-DISK-FIXNUM RQB DANGER-AVAILABLE-BLOCKS HEADER-AVAILABLE-TABLE-DANGER))
	  (AND MINIMUM-AVAILABLE-BLOCKS
	       (PUT-DISK-FIXNUM RQB MINIMUM-AVAILABLE-BLOCKS HEADER-AVAILABLE-TABLE-MINIMUM))
	  (AND MAXIMUM-AVAILABLE-BLOCKS
	       (PUT-DISK-FIXNUM RQB MAXIMUM-AVAILABLE-BLOCKS HEADER-AVAILABLE-TABLE-MAXIMUM))
	  (AND DESIRABLE-AVAILABLE-BLOCKS
	       (PUT-DISK-FIXNUM RQB DESIRABLE-AVAILABLE-BLOCKS
				HEADER-AVAILABLE-TABLE-DESIRABLE))
	  (AND AVAILABLE-TABLE-SIZE
	       (PUT-DISK-FIXNUM RQB AVAILABLE-TABLE-SIZE HEADER-AVAILABLE-TABLE-SIZE))
	  ;; Change size of maps if they aren't big enough.
	  (SETQ PAGES-PER-BLOCK (GET-DISK-FIXNUM RQB HEADER-PAGES-PER-BLOCK))
	  (SETQ BLOCK-SIZE (* PAGE-SIZE PAGES-PER-BLOCK))
	  (SETQ FREE-MAP-N-BLOCKS
		(CEILING (TRUNCATE PART-SIZE PAGES-PER-BLOCK) (* BLOCK-SIZE 40)))
	  (SETQ FREE-MAP-OLD-N-BLOCKS (GET-DISK-FIXNUM RQB HEADER-FREE-MAP-NUMBER-OF-BLOCKS))
	  (SETQ OLD-TOTAL-N-BLOCKS
		(IF (= (GET-DISK-FIXNUM RQB HEADER-VERSION-NUMBER) 2)
		    (GET-DISK-FIXNUM RQB HEADER-NUMBER-OF-BLOCKS)
		  (* FREE-MAP-OLD-N-BLOCKS (* BLOCK-SIZE 40))))
	  (SETQ TOTAL-N-BLOCKS (TRUNCATE PART-SIZE PAGES-PER-BLOCK))
	  (PUT-DISK-FIXNUM RQB TOTAL-N-BLOCKS HEADER-NUMBER-OF-BLOCKS)
	  (PUT-DISK-FiXNUM RQB 2 HEADER-VERSION-NUMBER)
	  (COND ((> FREE-MAP-N-BLOCKS FREE-MAP-OLD-N-BLOCKS)
		 (OR (> TOTAL-N-BLOCKS (+ OLD-TOTAL-N-BLOCKS (* 3 FREE-MAP-N-BLOCKS)))
		     (FERROR NIL "Partition size increment is too small"))
		 ;; Size of maps must increase!
		 ;; Note it is ok to extend the free map with 0's,
		 ;; since the newly-created blocks will get freed up
		 ;; by the first GC.  Similarly, we can leave the old
		 ;; map blocks marked in use, and this way there is no
		 ;; need to do anything special about the new ones either.
		 ;; But we must mark the pack as dirty to cause GC!
		 (PUT-DISK-FIXNUM RQB 0 HEADER-PACK-CLEAN-FLAG)
		 (PUT-DISK-FIXNUM RQB FREE-MAP-N-BLOCKS HEADER-FREE-MAP-NUMBER-OF-BLOCKS)
		 (MOVE-AND-EXPAND-MAP UNIT RQB HEADER-FREE-MAP-FIRST-BLOCK
				      FREE-MAP-OLD-N-BLOCKS
				      OLD-TOTAL-N-BLOCKS
				      FREE-MAP-N-BLOCKS
				      PART-BASE PAGES-PER-BLOCK)
		 (MOVE-AND-EXPAND-MAP UNIT RQB HEADER-BLOCK-STARTS-FILE-MAP-FIRST-BLOCK
				      FREE-MAP-OLD-N-BLOCKS
				      (+ OLD-TOTAL-N-BLOCKS FREE-MAP-N-BLOCKS)
				      FREE-MAP-N-BLOCKS
				      PART-BASE PAGES-PER-BLOCK)
		 (MOVE-AND-EXPAND-MAP UNIT RQB HEADER-LOCKED-BLOCK-MAP-FIRST-BLOCK
				      FREE-MAP-OLD-N-BLOCKS
				      (+ OLD-TOTAL-N-BLOCKS
					 FREE-MAP-N-BLOCKS FREE-MAP-N-BLOCKS)
				      FREE-MAP-N-BLOCKS
				      PART-BASE PAGES-PER-BLOCK)))
	  (DISK-WRITE RQB UNIT PART-BASE))
    (RETURN-DISK-RQB RQB)))

(DEFUN MOVE-AND-EXPAND-MAP (UNIT HEADER-RQB RQB-INDEX OLD-SIZE NEW-POS NEW-SIZE
			    PART-BASE PAGES-PER-BLOCK)
  (LET ((RQB (GET-DISK-RQB PAGES-PER-BLOCK))
	(OLD-POS (GET-DISK-FIXNUM HEADER-RQB RQB-INDEX)))
    (PUT-DISK-FIXNUM HEADER-RQB NEW-POS RQB-INDEX)
    (DOTIMES (I NEW-SIZE)
      (IF (< I OLD-SIZE)
	  (DISK-READ RQB UNIT (+ I PART-BASE (* OLD-POS PAGES-PER-BLOCK)))
	(FILLARRAY (RQB-BUFFER RQB) '(0)))
      (DISK-WRITE RQB UNIT (+ I PART-BASE (* NEW-POS PAGES-PER-BLOCK))))
    (RETURN-DISK-RQB RQB))
  NIL)

(defun lmfile-error (error-type allow-errors format-string &rest control-args)
  "Signal a LMFILE-error message.  ERROR-TYPE currently is thrown away.
ALLOW-ERRORS says whether to signal an error.  But one have set up a 
*CATCH for fs:lmfile-error
FORMAT-STRING is how to format the text using the CONTROL-ARGS"
  (cond ((not (null allow-errors))
	 (format t "~%LMFile errors: ~&")
	 (funcall #'format t format-string control-args)
	 (*throw 'fs:lmfile-error (cons format-string control-args)))
	(t
	 (ferror error-type format-string control-args))))

;Look at a unit examine its header.
;We create the PACK-STRUCTURE object and store it in TENTATIVE-PACK-LIST.
;This does not actually make the unit available for use.
;MOUNT-TENTATIVE-UNITS does that, after verifying that all the tentative
;units make a consistent set.

(DEFUN CONSIDER-UNIT (UNIT &OPTIONAL (PARTITION-NAME "FILE") ALLOW-ERRORS
		    &AUX (RQB NIL)
		         (PACK (MAKE-PACK-STRUCTURE))
			 TB
			 PART-BASE
			 PART-SIZE)
  ;; See if this unit and partition are already on PACK-LIST or TENTATIVE-PACK-LIST.
  (DOLIST (PACK PACK-LIST)
    (AND (= (PACK-UNIT-NUMBER PACK) UNIT)
	 (STRING-EQUAL (PACK-PARTITION-NAME PACK) PARTITION-NAME)
	 (LMFILE-ERROR NIL ALLOW-ERRORS "Unit ~S partition ~A is already mounted" UNIT PARTITION-NAME)))
  (DOLIST (PACK TENTATIVE-PACK-LIST)
    (AND (= (PACK-UNIT-NUMBER PACK) UNIT)
	 (STRING-EQUAL (PACK-PARTITION-NAME PACK) PARTITION-NAME)	 (SETQ TENTATIVE-PACK-LIST
	       (DELQ PACK TENTATIVE-PACK-LIST))))
  (OR (and (not allow-errors) (SI:POWER-UP-DISK UNIT) ) ;;don't bother booting up if ERRORS
      (LMFILE-ERROR NIL ALLOW-ERRORS "Unit ~S is not on-line" UNIT))
  (UNWIND-PROTECT
    (PROGN (SETQ RQB (GET-DISK-RQB))
	   (MULTIPLE-VALUE (PART-BASE PART-SIZE)
		    (FIND-DISK-PARTITION PARTITION-NAME NIL UNIT))
	   (AND (NULL PART-BASE)
		(LMFILE-ERROR NIL ALLOW-ERRORS "No ~A partition on unit ~S" PARTITION-NAME UNIT))
	   (DISK-READ RQB UNIT PART-BASE)  ;Get pack file-system header
	   (OR (AND (STRING-EQUAL (GET-DISK-STRING RQB HEADER-IDENTIFICATION 4) "FILE")
		    (MEMQ (GET-DISK-FIXNUM RQB HEADER-VERSION-NUMBER) '(1 2)))
	       (LMFILE-ERROR NIL ALLOW-ERRORS "Unit ~S partition ~A not formatted for LMFILE"
		       UNIT PARTITION-NAME))
	   (SETF (PACK-UNIT-NUMBER PACK) UNIT)
	   (SETF (PACK-PARTITION-NAME PACK) PARTITION-NAME)
	   (SETF (PACK-VOLUME-NAME PACK) (GET-DISK-STRING RQB HEADER-VOLUME-NAME 32.))
	   (SETF (PACK-NUMBER-WITHIN-VOLUME PACK)
		 (GET-DISK-FIXNUM RQB HEADER-PACK-NUMBER-WITHIN-VOLUME))
	   (SETF (PACK-NUMBER-OF-PACKS-IN-VOLUME PACK)
		 (GET-DISK-FIXNUM RQB HEADER-NUMBER-OF-PACKS-IN-VOLUME))
	   (OR (AND (>= (PACK-NUMBER-WITHIN-VOLUME PACK) 0)
		    (< (PACK-NUMBER-WITHIN-VOLUME PACK)
		       (PACK-NUMBER-OF-PACKS-IN-VOLUME PACK))
		    (<= (PACK-NUMBER-OF-PACKS-IN-VOLUME PACK) N-UNITS))
	       (LMFILE-ERROR NIL ALLOW-ERRORS "Unit ~S partition ~A pack number information is inconsistent"
		       UNIT PARTITION-NAME))
	   (SETF (PACK-INCARNATION-NUMBER PACK)
		 (%24-BIT-PLUS (GET-DISK-FIXNUM RQB HEADER-SYSTEM-INCARNATION-NUMBER) 1))
	   (SETF (PACK-COMMENT PACK) (GET-DISK-STRING RQB HEADER-COMMENT 96.))
	   (SETF (PACK-PAGES-PER-BLOCK PACK)
		 (GET-DISK-FIXNUM RQB HEADER-PAGES-PER-BLOCK))
	   (SETF (PACK-FIRST-ABS-PAGE-NUMBER PACK) PART-BASE)
	   (SETF (PACK-NUMBER-OF-BLOCKS PACK)
		 (IF (= (GET-DISK-FIXNUM RQB HEADER-VERSION-NUMBER) 1)
		     (TRUNCATE PART-SIZE (PACK-PAGES-PER-BLOCK PACK))
		   (GET-DISK-FIXNUM RQB HEADER-NUMBER-OF-BLOCKS)))
	   (SETF (PACK-FREE-MAP-FIRST-BLOCK PACK)
		 (GET-DISK-FIXNUM RQB HEADER-FREE-MAP-FIRST-BLOCK))
	   (SETF (PACK-BLOCK-STARTS-FILE-MAP-FIRST-BLOCK PACK)
		 (GET-DISK-FIXNUM RQB HEADER-BLOCK-STARTS-FILE-MAP-FIRST-BLOCK))
	   (SETF (PACK-LOCKED-BLOCK-MAP-FIRST-BLOCK PACK)
		 (GET-DISK-FIXNUM RQB HEADER-LOCKED-BLOCK-MAP-FIRST-BLOCK))
	   (SETF (PACK-FREE-MAP-SCANNING-INDEX PACK) 0)
	   (SETF (PACK-FREE-MAP-NUMBER-OF-BLOCKS PACK)
		 (GET-DISK-FIXNUM RQB HEADER-FREE-MAP-NUMBER-OF-BLOCKS))
	   (OR (= (PACK-FREE-MAP-NUMBER-OF-BLOCKS PACK)
		  (CEILING (PACK-NUMBER-OF-BLOCKS PACK)
			   (* 32. (PACK-PAGES-PER-BLOCK PACK) PAGE-SIZE)))
	       (LMFILE-ERROR NIL ALLOW-ERRORS "Inconsistent free table size on unit ~S partition ~A"
		       UNIT PARTITION-NAME))
	   (SETF (PACK-DANGER-AVAILABLE-BLOCKS PACK)
		 (GET-DISK-FIXNUM RQB HEADER-AVAILABLE-TABLE-DANGER))
	   (SETF (PACK-MINIMUM-AVAILABLE-BLOCKS PACK)
		 (GET-DISK-FIXNUM RQB HEADER-AVAILABLE-TABLE-MINIMUM))
	   (SETF (PACK-MAXIMUM-AVAILABLE-BLOCKS PACK)
		 (GET-DISK-FIXNUM RQB HEADER-AVAILABLE-TABLE-MAXIMUM))
	   (SETF (PACK-DESIRABLE-AVAILABLE-BLOCKS PACK)
		 (GET-DISK-FIXNUM RQB HEADER-AVAILABLE-TABLE-DESIRABLE))
	   ;; Set up available table
	   (LET ((AVAILABLE-TABLE-SIZE (GET-DISK-FIXNUM RQB HEADER-AVAILABLE-TABLE-SIZE)))
	     (OR (AND (PLUSP (PACK-MINIMUM-AVAILABLE-BLOCKS PACK))
		      (< (PACK-MINIMUM-AVAILABLE-BLOCKS PACK)
			 (PACK-DESIRABLE-AVAILABLE-BLOCKS PACK))
		      (< (PACK-DESIRABLE-AVAILABLE-BLOCKS PACK)
			 (PACK-MAXIMUM-AVAILABLE-BLOCKS PACK))
		      (< (PACK-DESIRABLE-AVAILABLE-BLOCKS PACK)
			 AVAILABLE-TABLE-SIZE)
		      (< (PACK-DANGER-AVAILABLE-BLOCKS PACK)
			 (PACK-MINIMUM-AVAILABLE-BLOCKS PACK))
;		      (< (PACK-MAXIMUM-AVAILABLE-BLOCKS PACK)
;			 (PACK-NUMBER-OF-BLOCKS PACK))
		 )
		 (LMFILE-ERROR NIL ALLOW-ERRORS "Nonsensical available-table parameters on unit ~S partition ~A"
			 UNIT PARTITION-NAME))
	     (SETQ TB (MAKE-ARRAY (LIST AVAILABLE-TABLE-SIZE 2)
				  ':AREA PERMANENT-STORAGE-AREA))
	     (DOTIMES (I AVAILABLE-TABLE-SIZE) (ASET 0 TB I 0) (ASET 0 TB I 1)))
	   (SETF (PACK-AVAILABLE-TABLE PACK) TB)
	   (SETF (PACK-NUMBER-OF-AVAILABLE-BLOCKS PACK) 0)
	   (SETF (PACK-PARENT-VOLUME-NAME PACK)
		 (GET-DISK-STRING RQB HEADER-PARENT-VOLUME-NAME 32.))
	   (AND (STRING-EQUAL (PACK-PARENT-VOLUME-NAME PACK) "")
		(SETF (PACK-PARENT-VOLUME-NAME PACK) NIL))
	   (SETF (PACK-ROOT-FIRST-BLOCK PACK)
		 (GET-DISK-FIXNUM RQB HEADER-ROOT-FIRST-BLOCK))
	   (SELECTQ (GET-DISK-FIXNUM RQB HEADER-PACK-CLEAN-FLAG)
	     (0 (SETF (PACK-GC-BIT-MAP PACK)
		      (MAKE-ARRAY (* 2 PAGE-SIZE (PACK-PAGES-PER-BLOCK PACK)
				     (PACK-FREE-MAP-NUMBER-OF-BLOCKS PACK))
				  ':TYPE 'ART-16B)))
	     (2 (SETF (PACK-LOCKED-BLOCK-FILES PACK) T)))
	   ;; Read in the contents of the bit maps.
	   (SETF (PACK-BLOCK-STARTS-FILE-MAP-ARRAY PACK)
		 (READ-BITMAP-INTO-ARRAY PACK (PACK-BLOCK-STARTS-FILE-MAP-FIRST-BLOCK PACK)))
	   (SETF (PACK-LOCKED-BLOCK-MAP-ARRAY PACK)
		 (READ-BITMAP-INTO-ARRAY PACK (PACK-LOCKED-BLOCK-MAP-FIRST-BLOCK PACK)))
	   (SETF (PACK-FREE-MAP-RQB PACK)
		 (GET-DISK-RQB (* (PACK-FREE-MAP-NUMBER-OF-BLOCKS PACK)
				  (PACK-PAGES-PER-BLOCK PACK))))
	   (READ-BITMAP-INTO-ARRAY PACK (PACK-FREE-MAP-FIRST-BLOCK PACK)
				   (RQB-BUFFER (PACK-FREE-MAP-RQB PACK)))
	   (SETF (PACK-NUMBER-OF-FREE-BLOCKS PACK)
		 (COUNT-FREE-BLOCKS PACK))
	   (PUSH PACK TENTATIVE-PACK-LIST))		;Do this last
    (RETURN-DISK-RQB RQB)))

(DEFUN SET-ROOT-FIRST-BLOCK (PACK BLOCK &AUX RQB
				    (HEADER-PAGE (PACK-FIRST-ABS-PAGE-NUMBER PACK))
				    (UNIT (PACK-UNIT-NUMBER PACK)))
  (SETF (PACK-ROOT-FIRST-BLOCK PACK) BLOCK)
  (UNWIND-PROTECT
    (PROGN (SETQ RQB (GET-DISK-RQB))
	   (DISK-READ RQB UNIT HEADER-PAGE)  ;Get pack file-system header
	   (PUT-DISK-FIXNUM RQB BLOCK
			    HEADER-ROOT-FIRST-BLOCK)
	   (DISK-WRITE RQB UNIT HEADER-PAGE))	;Write back updated incarnation number
    (RETURN-DISK-RQB RQB)))

(DEFUN FIND-PARTITION-PACK (UNIT &OPTIONAL (PARTITION-NAME "FILE"))
  (DOLIST (PACK PACK-LIST)
    (AND (= (PACK-UNIT-NUMBER PACK) UNIT)
	 (STRING-EQUAL (PACK-PARTITION-NAME PACK) PARTITION-NAME)
	 (RETURN PACK))))

(DEFUN START-FILE-SYSTEM ()
  (AND FILE-SYSTEM-RUNNING (FERROR NIL "File system already running"))
  (SETQ OPEN-DIR-NODES NIL)
  (SETQ ROOT-NODE NIL ROOT-PACK NIL ROOT-VOLUME NIL)
  (COND ((MOUNT-TENTATIVE-UNITS)
	 (COND ((NULL ROOT-PACK)
		(SETQ ROOT-PACK (FIND-MOUNTED-PACK ROOT-VOLUME 0))
		;; If no root, probably pack just initialized, so GC it
		;; so that there is a free block to create a root in.
		(AND (PACK-GC-BIT-MAP ROOT-PACK)
		     (GC-PACK ROOT-PACK))))
	 (SETQ ROOT-NODE (OPEN-ROOT-NODE ROOT-PACK))
	 (SETQ FILE-SYSTEM-RUNNING T)
	 (GC-ALL-PACKS))
	(T (FORMAT T "~%File system cannot be started."))))

(DEFUN CLEAR-FILE-SYSTEM ()
;  (YES-OR-NO-P "Clear out file system?  This will clobber everything. ")
  (DOLIST (STR (FUNCALL LFS-HOST ':OPEN-STREAMS))
    (FUNCALL STR ':FORCE-CLOSE))
  (SETQ TENTATIVE-PACK-LIST NIL)
  (MAPC 'DIKE-OUT-PACK PACK-LIST)
  (AND (FBOUNDP 'FREE-ALL-FILE-AREAS) (FREE-ALL-FILE-AREAS))
  (SETQ ROOT-PACK NIL ROOT-NODE NIL ROOT-VOLUME NIL)
  (SETQ FILE-SYSTEM-RUNNING NIL))

(DEFUN STOP-FILE-SYSTEM ()
  (DISMOUNT-VOLUME ROOT-VOLUME)
  (AND (FBOUNDP 'FREE-ALL-FILE-AREAS) (FREE-ALL-FILE-AREAS)))

;See whether the tentative packs are consistent to mount,
;and mount them if they are.  If we get a root volume or root pack, record the fact.
(DEFUN MOUNT-TENTATIVE-UNITS (&AUX ROOTS RQB)
  (COND ((SETQ ROOTS (VERIFY-TENTATIVE-UNITS))
	 (UNWIND-PROTECT
	   (PROGN (SETQ RQB (GET-DISK-RQB))
		  (WITHOUT-INTERRUPTS
		   (DOLIST (PACK TENTATIVE-PACK-LIST)
		     (PUSH PACK PACK-LIST)
		     ;; and increment the mount count on the disk.
		     (DISK-READ RQB (PACK-UNIT-NUMBER PACK) (PACK-FIRST-ABS-PAGE-NUMBER PACK))
		     (PUT-DISK-FIXNUM RQB (PACK-INCARNATION-NUMBER PACK)
				      HEADER-SYSTEM-INCARNATION-NUMBER)
		     (PUT-DISK-FIXNUM RQB 0 HEADER-PACK-CLEAN-FLAG)
		     (DISK-WRITE RQB (PACK-UNIT-NUMBER PACK)
				 (PACK-FIRST-ABS-PAGE-NUMBER PACK)))
		   (SETQ TENTATIVE-PACK-LIST NIL)))
	   (RETURN-DISK-RQB RQB))
	 (OR ROOT-PACK (SETQ ROOT-PACK (CAR ROOTS)))
	 (OR ROOT-VOLUME (SETQ ROOT-VOLUME (CADR ROOTS)))
	 (AND FILE-SYSTEM-RUNNING (GC-ALL-PACKS))
	 T)))

;;; Make sure that we have a consistent set of packs mounted,
;;; counting both those already in use and those individually
;;; validated for use by CONSIDER-UNIT.
;;; 1. Each volume present must be complete.
;;; 2. In each volume, all packs agree on name of parent volume
;;;    and the parent volume, if any, is mounted also.
;;; 3. We have a root directory.
;;; Returns (root-pack root-volume) if you win and NIL if you lose.
(DEFUN VERIFY-TENTATIVE-UNITS ()
  (LET (ALL-VOLUMES
	RT-PACK
	RT-VOLUME
	(WIN T)		;Set to NIL if any problem is detected.
	(ALL-PACKS (APPEND TENTATIVE-PACK-LIST PACK-LIST)))
    ;; Find all the mounted volumes and their parents.
    (DOLIST (PACK ALL-PACKS)
      (OR (ASSOC (PACK-VOLUME-NAME PACK) ALL-VOLUMES)
	  (PUSH (LIST (PACK-VOLUME-NAME PACK)
		      (PACK-NUMBER-OF-PACKS-IN-VOLUME PACK)
		      (PACK-PARENT-VOLUME-NAME PACK))
		ALL-VOLUMES)))
    ;; Now check each volume: verify that all its packs are mounted,
    ;; that no duplicate or extra pack numbers are mounted,
    ;; and that its parent volume is mounted.
    (DOLIST (VOL ALL-VOLUMES)
      (LET ((VN (CAR VOL)) (NP (CADR VOL)) (PVN (CADDR VOL)))
	(COND ((AND (NULL PVN) RT-VOLUME)
	       (FORMAT T "~%Two root volumes, ~A and ~A." VN RT-VOLUME)
	       (SETQ WIN NIL))
	      ((NULL PVN) (SETQ RT-VOLUME VN)))
	;; Make sure all packs of the volume are mounted,
	;; and all agree on what our parent volume is.
	(DOTIMES (I NP)			;Find each pack within that volume
	  ;; Make sure no packs are missing.
	  (OR (DOLIST (PACK2 ALL-PACKS)
		(AND (= (PACK-NUMBER-WITHIN-VOLUME PACK2) I)
		     (STRING-EQUAL (PACK-VOLUME-NAME PACK2) VN)
		     (PROGN
		      (COND ((NOT (STRING-EQUAL (PACK-PARENT-VOLUME-NAME PACK2) PVN))
			     (FORMAT T "~%Packs of volume ~A disagree in parent volume name."
				     VN)
			     (SETQ WIN NIL)))
		      (RETURN T))))
	      (PROGN (SETQ WIN NIL)
		     (FORMAT T "~%Pack ~S of volume ~A is missing." I VN)))
	  ;; Make sure there are no extra packs of that volume
	  (LET ((NPACKS 0))
	    (DOLIST (PACK2 ALL-PACKS)
	      (AND (STRING-EQUAL (PACK-VOLUME-NAME PACK2) VN)
		   (INCF NPACKS)))
	    (OR (= NP NPACKS)
		(SETQ WIN NIL)
		(FORMAT T "~%Volume ~A has ~S extra packs mounted." VN (- NPACKS NP)))))
	;; Make sure the parent volume, if any, is mounted too.
	(OR (NULL PVN)
	    (ASSOC PVN ALL-VOLUMES)
	    (PROGN
	     (FORMAT T "~%Volume ~A is missing and needed as the parent of ~A." PVN VN)
	     (SETQ WIN NIL)))))
    ;; Find the root directory and the root pack.
    (DOLIST (PACK ALL-PACKS)
      (AND (EQUAL (PACK-VOLUME-NAME PACK) RT-VOLUME)
	   (NOT (ZEROP (PACK-ROOT-FIRST-BLOCK PACK)))
	   (COND (RT-PACK (FORMAT T "~%Packs ~S and ~S both have root directories."
				  PACK RT-PACK)
			  (SETQ WIN NIL))
		 (T (SETQ RT-PACK PACK)))))
    (COND (RT-VOLUME
	   (OR RT-PACK (FORMAT T "~%No root directory yet on volume ~A." RT-VOLUME)))
	  (T (FORMAT T "~%There is no root volume.")
	     (SETQ WIN NIL)))
    (AND WIN (LIST RT-PACK RT-VOLUME))))

(DEFUN DISMOUNT-UNIT (UNIT &OPTIONAL (PARTITION-NAME "FILE"))
  (LET ((PACK (FIND-PARTITION-PACK UNIT PARTITION-NAME)))
    (SETQ TENTATIVE-PACK-LIST (DELQ PACK TENTATIVE-PACK-LIST))
    (AND PACK
	 (DISMOUNT-VOLUME (PACK-VOLUME-NAME PACK)))))

;Dismount all the units in a volume, and all other volumes needing
;this one as a parent.
(DEFUN DISMOUNT-VOLUME (VOLUME-NAME &OPTIONAL NO-QUERY-FLAG)
  (SETQ VOLUME-NAME (STRING-UPCASE VOLUME-NAME))
  (COND ((FIND-MOUNTED-PACK VOLUME-NAME 0)
	 (COND ((NOT NO-QUERY-FLAG)
		(FORMAT T "~%Dismount volume ~A" VOLUME-NAME)
		(AND ROOT-VOLUME (STRING-EQUAL VOLUME-NAME ROOT-VOLUME)
		     (FORMAT T " and stop the file system"))
		(PRINC "? ")))
	 (COND ((OR NO-QUERY-FLAG (YES-OR-NO-P))
		(COND ((AND ROOT-VOLUME (STRING-EQUAL VOLUME-NAME ROOT-VOLUME))
		       (SETQ ROOT-VOLUME NIL
			     ROOT-PACK NIL
			     ROOT-NODE NIL
			     FILE-SYSTEM-RUNNING NIL)))
		(DOLIST (PACK PACK-LIST)
		  (AND (STRING-EQUAL (PACK-VOLUME-NAME PACK) VOLUME-NAME)
		       (DISMOUNT-PACK PACK)))
		(DOLIST (PACK PACK-LIST)
		  (AND (STRING-EQUAL (PACK-PARENT-VOLUME-NAME PACK) VOLUME-NAME)
		       (DISMOUNT-VOLUME (PACK-VOLUME-NAME PACK) T))))))
	(T (FORMAT T "~%Volume ~A not mounted" VOLUME-NAME))))

;;; Update data stored on disk and clear unit table
(DEFUN DISMOUNT-PACK (PACK &AUX RQB (UNIT (PACK-UNIT-NUMBER PACK)))
  (DOLIST (FILE (PACK-OPEN-FILES PACK))
    (FUNCALL (FILE-TOP-LEVEL-NODE FILE) 'FORCIBLY-CLOSE))
  (WITH-PACK-LOCK PACK
    ;;Flush any pending I/O, mapped-in dirs, and the like.
    ;;Empty the available table to update the free map.
    (FLUSH-AVAILABLE-BLOCK-TABLE PACK T)
    ;;Set flag that no GC is necessary when next mounted.
    (UNWIND-PROTECT
     (PROGN (SETQ RQB (GET-DISK-RQB))
	    (SI:DISK-READ RQB UNIT (PACK-FIRST-ABS-PAGE-NUMBER PACK))
	    (PUT-DISK-FIXNUM RQB
			     (IF (PACK-LOCKED-BLOCK-FILES PACK) 2 1)
			     HEADER-PACK-CLEAN-FLAG)
	    (DISK-WRITE RQB UNIT (PACK-FIRST-ABS-PAGE-NUMBER PACK)))
     (RETURN-DISK-RQB RQB))
    (DIKE-OUT-PACK PACK)))

(DEFUN DIKE-OUT-UNIT (UNIT &OPTIONAL (PARTITION-NAME "FILE"))
  (DIKE-OUT-PACK (FIND-PARTITION-PACK UNIT PARTITION-NAME)))

;;; Remove a unit from the system's tables without doing any disk I/O
(DEFUN DIKE-OUT-PACK (PACK)
  "Remove a PACK from the file system, without writing anything out!"
  (AND PACK (RETURN-DISK-RQB (PACK-FREE-MAP-RQB PACK)))
  (AND PACK (DOLIST (F (PACK-OPEN-FILES PACK))
	      (LET ((CORE-MAP (FILE-CORE-MAP F)))
		(DOLIST (MAPELT CORE-MAP)
		  (AND (MAPELT-DATA MAPELT)
		       (RETURN-DISK-RQB (MAPELT-DATA MAPELT)))))))
  (SETQ PACK-LIST (DELQ PACK PACK-LIST)))

;;; Should find all locks in the file system and reset them.  Used when debugging.
(DEFUN PUNT-ALL-LOCKS ()
  ;; Clear pack locks
  (DOLIST (PACK PACK-LIST)
    (SETF (PACK-LOCK PACK) NIL)))

(DEFUN PRINT-MOUNTED-UNITS ()
  (FUNCALL STANDARD-OUTPUT ':FRESH-LINE)
  (DOLIST (PACK PACK-LIST)
    (FORMAT T "Volume ~A pack ~S   Unit ~S partition ~A    ~A"
	    (PACK-VOLUME-NAME PACK) (PACK-NUMBER-WITHIN-VOLUME PACK)
	    (PACK-UNIT-NUMBER PACK) (PACK-PARTITION-NAME PACK)
	    (PACK-COMMENT PACK))))

;;; Given a volume name and a pack number, find its pack-structure, or
;;; NIL if not mounted.
(DEFUN FIND-MOUNTED-PACK (VOLUME-NAME PACK-NUMBER)
  (DOLIST (PACK PACK-LIST)
    (AND (= (PACK-NUMBER-WITHIN-VOLUME PACK) PACK-NUMBER)
	 (STRING-EQUAL (PACK-VOLUME-NAME PACK) VOLUME-NAME)
	 (RETURN PACK))))

;Return T if it is legitimate for FROM-VOLUME to contain a pointer
;to TO-VOLUME, and TO-PACK-NUMBER is the number of an actual pack.
(DEFUN CHECK-CROSS-VOLUME-REF (FROM-VOLUME TO-VOLUME &OPTIONAL (TO-PACK-NUMBER 0) &AUX PACK)
  (AND (SETQ PACK (FIND-MOUNTED-PACK TO-VOLUME TO-PACK-NUMBER))
       (OR (STRING-EQUAL FROM-VOLUME TO-VOLUME)
	   (CHECK-CROSS-VOLUME-REF FROM-VOLUME (PACK-PARENT-VOLUME-NAME PACK) 0))))

;; If we want to write on any pack of a specified volume, choose which one
;; (the one with the most free space).
;; Return the pack number.
(DEFUN FIND-PACK-TO-WRITE-ON (VOLUME-NAME &AUX FIRST-PACK N-PACKS BEST-PACK MAX-SO-FAR)
  ;; Find one pack of that volume that's sure to be mounted if any of them is.
  (SETQ FIRST-PACK (FIND-MOUNTED-PACK VOLUME-NAME 0))
  (COND (FIRST-PACK
	 ;; Now we can find out how many there are.
	 (SETQ N-PACKS (PACK-NUMBER-OF-PACKS-IN-VOLUME FIRST-PACK))
	 ;; Find  the pack with the most free space.
	 (SETQ MAX-SO-FAR 0)
	 (DOTIMES (I N-PACKS)
	   (LET ((PACK (FIND-MOUNTED-PACK VOLUME-NAME I)))
	     (COND ((> (+ (PACK-NUMBER-OF-FREE-BLOCKS PACK)
			  (PACK-NUMBER-OF-AVAILABLE-BLOCKS PACK))
		       MAX-SO-FAR)
		    (SETQ MAX-SO-FAR
			  (+ (PACK-NUMBER-OF-FREE-BLOCKS PACK)
			     (PACK-NUMBER-OF-AVAILABLE-BLOCKS PACK))
			  BEST-PACK I)))) )
	 (OR BEST-PACK 0))
	;; Volume not mounted => return 0 (as good as any other number).
	(T 0)))
