;;; -*-Mode:Lisp;Package:TAPE;Fonts:CPTFONT;Base:8-*-
;;; (c) 1984 Massachusetts Institute of Technology
;;;
;;; ``Without the letter U, units would be nits.''
;;;
;;; Tape unit scheme for Lisp Machines.  Designed to handle different pieces of hardware
;;; and remote access, too.  Keeps numbers of flavors of streams down to a minimum.
;;;

(DEFFLAVOR BASIC-UNIT (LOCK REWIND-MEMO NUMBER) ()
  :ABSTRACT-FLAVOR
  (:GETTABLE-INSTANCE-VARIABLES NUMBER LOCK)
  (:INITABLE-INSTANCE-VARIABLES NUMBER)
  ;; Standard operations follow.  If not documented, the operation has no arguments
  ;; and its meaning should be obvious.
  (:REQUIRED-METHODS
    :ACCESS			; returns hardware type or protocol name
    :HOST
    :LOCAL-P
    :DEVICE			; constant for local units, needs to be different for remotes
    :REWIND
    :UNLOAD
    :OFFLINE
    :WRITE-EOF			; done implicitly on normal close remotely (file stream)
    :BEGIN-INTERLEAVED-REWIND	; (id) start rewinding, remembering id
    :FINISH-INTERLEAVED-REWIND	; (id) finish rewinding
    :STREAM-FLAVOR		; (direction characters file-p &optional format)
    :SPACE			; (ntimes) }
    :SPACE-TO-EOF		; (ntimes) } ``0 means moby many''
    :SPACE-REV			; (ntimes need-status-p) }
    :SPACE-REV-TO-BOF		; (skip-n-files) 0 = tape at beginning of file; 1 = previous
    				; [CADR version returns () if premature.  Neccessary ?]
    :SPACE-TO-APPEND
    :STATUS-EOF			; (status)
    :STATUS-EOT			; (status)
    :STATUS-BOT			; (status)
    :STATUS-ERROR		; (status) usually not requested remotely
    :UNIT-EQUAL-P		; (device number host &optional protocol)
    :WAIT-CONTROLLER-READY
    :WAIT-UNIT-READY)
  (:DOCUMENTATION :ESSENTIAL-MIXIN "The basic flavor used by all tape units.
Does not define much on its own; it just insures that other flavors are defining the essential
operations themselves.
It defines three instance variables:
 LOCK (currently not used)
 REWIND-MEMO for interleaved rewind
 NUMBER for the controller"))

(DEFFLAVOR BASIC-REMOTE-UNIT (HOST) ()
  :ABSTRACT-FLAVOR
  (:INITTABLE-INSTANCE-VARIABLES HOST)
  (:GETTABLE-INSTANCE-VARIABLES HOST)
  (:REQUIRED-METHODS
    :PROTOCOL
    :ACTIVE-P))

(DEFMETHOD (BASIC-REMOTE-UNIT :ACCESS) () (SEND SELF ':PROTOCOL))

(DEFFLAVOR BASIC-LOCAL-UNIT () ()
  :ABSTRACT-FLAVOR
  ;; This flavor assumes a fairly standard mode of operation for a tape drive.
  ;; The model might break down for other bizarre things.
  (:REQUIRED-METHODS ; these usually implement the guts of the streams
    :STATUS			; returns a status ``object'' (might be a fixnum,
    				; or a structure) (instance variable for TM)
    :STATUS-READY		; this drive is ready
    :COMMAND-READY		; controller is ready
    ;; Locally, there must be a BUFFER, a STATUS object, block IO methods, and
    ;; ways to decode the status.  Note that the STATUS and BUFFER objects may
    ;; be the same (as on the CADR/Wesperco hardware).
    :I-WRITE-BUFFER		; (buffer count density ibm-mode) -> status
    :I-READ-BUFFER		; (buffer count density) -> status (containing bytes read)
    :DECODE-STATUS	; (status thing) return T, () or number for these interesting THINGs:
    			; :EOT, :EOF, :BOT, :WRITE-ONLY, :ERROR, :BYTES-TRANSFERRED, :ON-LINE,
    			; :ERROR-CODE, :BAD-TAPE
    :ODD-BYTES-OK-P		; happens to be () for the Wesperco
    )
  (:REQUIRED-FLAVORS BASIC-UNIT))

(DEFMETHOD (BASIC-LOCAL-UNIT :LOCAL-P) () T)
(DEFMETHOD (BASIC-LOCAL-UNIT :HOST) () SI:LOCAL-HOST)

(DEFMETHOD (BASIC-LOCAL-UNIT :UNLOAD) () (SEND SELF ':OFFLINE))

(defmethod (basic-local-unit :begin-interleaved-rewind) (memo)
  (setq rewind-memo memo)
  (send self ':rewind))

(DEFMETHOD (BASIC-LOCAL-UNIT :FINISH-INTERLEAVED-REWIND) (MEMO)
  (IF (NEQ MEMO REWIND-MEMO)
      (FERROR () "Somebody else was waiting for the tape to rewind: ~S" REWIND-MEMO)
    (DO ()
	((SEND SELF ':STATUS-BOT))
      (PROCESS-SLEEP 120. "MT Rewind"))))

(DEFFLAVOR BASIC-LOCAL-MT-UNIT-MIXIN () ()
  (:ABSTRACT-FLAVOR T)
  (:INCLUDED-FLAVORS BASIC-LOCAL-UNIT)
  (:DOCUMENTATION :ESSENTIAL-MIXIN "Used for MT-type devices.
Defines the streams to be used to be of the MT-family defined in MTSTR."))

(DEFMETHOD (BASIC-LOCAL-MT-UNIT-MIXIN :DEVICE) () ':MT)

(defmethod (basic-local-mt-unit-mixin :unit-equal-p) (device .number. host &optional protocol)
  protocol ; not used locally
  (and (eq device ':mt)
       (eq host si:local-host)
       (= number .number.)))

(DEFMETHOD (BASIC-LOCAL-MT-UNIT-MIXIN :STREAM-FLAVOR)
	   (DIRECTION CHARACTERS FILE-P &OPTIONAL FORMAT) ; might not need it here.
  format
  (IF FILE-P
      (IF CHARACTERS
	  (SELECTQ DIRECTION
	    (:INPUT 'MT-CHARACTER-INPUT-STREAM)
	    (:OUTPUT 'MT-CHARACTER-OUTPUT-STREAM)
	    (OTHERWISE ()))
	(SELECTQ DIRECTION
	  (:INPUT 'MT-INPUT-STREAM)
	  (:OUTPUT 'MT-OUTPUT-STREAM)
	  (OTHERWISE ()))
	(IF CHARACTERS
	    (SELECTQ DIRECTION
	      (:INPUT 'MT-FILE-CHARACTER-INPUT-STREAM)
	      (:OUTPUT 'MT-FILE-CHARACTER-OUTPUT-STREAM)
	      (OTHERWISE ()))
	  (SELECTQ DIRECTION
	    (:INPUT 'MT-FILE-INPUT-STREAM)
	    (:OUTPUT 'MT-FILE-OUTPUT-STREAM)
	    (OTHERWISE ()))))))

;;; Flavors defined in other files:
;;; WEUNIT: WESPERCO-UNIT (used on the CADR, Unibus)
;;; TMUNIT: TAPEMASTER-UNIT (used on the Lambda, multibus)
;;; RMUNIT: RM-UNIT (remote protocol, seems to be CHAOS-independent)

(DEFPROP :MT "Industry Standard Magnetic Tape" DEVICE-DESCRIPTION)
;; :MT must have a LOCAL-UNIT-FLAVOR property to work properly (should be set at initialisation)

(defprop :mta :mt device-alias)
(defprop :rmt :mt device-alias)

(defun set-local-mt-unit-flavor ()
  (setf (get ':mt 'local-unit-flavor)
	(select si:processor-type-code
	  (si:cadr-type-code 'wesperco-unit)
	  (si:lambda-type-code 'tapemaster-unit))))

(add-initialization "Set local MT unit flavor" '(set-local-mt-unit-flavor) '(:now))

(DEFINE-SITE-VARIABLE *DEFAULT-REMOTE-PROTOCOL* :DEFAULT-REMOTE-TAPE-PROTOCOL)
(SETQ *DEFAULT-REMOTE-PROTOCOL* ':MAGTAPE) ;1 for now, it's the RMS protocol0
;;; Protocol must have UNIT-FLAVOR property, take :DEVICE, :NUMBER, and :HOST args
;;; in MAKE-INSTANCE
(DEFVAR *DEFAULT-UNIT* :UNBOUND) ; initialised at warm boot
(DEFVAR *UNITS* ())

(DEFUN PARSE-UNIT-DEVICE-AND-NUMBER (THING DEFAULT-DEVICE)
  (DECLARE (:VALUES DEVICE NUMBER))
  (IF (ZEROP (STRING-LENGTH THING)) (VALUES DEFAULT-DEVICE 0)
    (LET ((IDX (STRING-SEARCH-SET "1234567890" THING)))
      (VALUES (PARSE-UNIT-DEVICE THING IDX DEFAULT-DEVICE)
	      (PARSE-UNIT-NUMBER (SUBSTRING THING (1+ IDX)))))))

(DEFUN PARSE-UNIT-DEVICE (THING IDX DEFAULT-DEVICE &AUX DEVICE)
  (PKG-BIND (PKG-FIND-PACKAGE "") ; actually, the keyword package
    (SETQ DEVICE (READ-FROM-STRING THING () 0 IDX)))
  (OR DEVICE (SETQ DEVICE DEFAULT-DEVICE))
  (DO ((ALIAS (GET DEVICE 'DEVICE-ALIAS) (GET DEVICE 'DEVICE-ALIAS)))
      ((NULL ALIAS) DEVICE)
    (SETQ DEVICE ALIAS)))

(DEFUN PARSE-UNIT-NUMBER (THING)
  (IF (ZEROP (STRING-LENGTH THING)) 0
    (LET ((IBASE 10.)) (READ-FROM-STRING THING))))

(defun make-local-default-unit (number device)
  (setq *default-unit* (make-instance (get device 'local-unit-flavor)
				      ':number number))
  (push *default-unit* *units*))
  
(DEFUN MAKE-UNIT (HOST DEVICE NUMBER PROTOCOL &AUX NIT)
  (dolist (unit *units*)
    (if (send unit ':unit-equal-p device number host protocol)
	(return (setq nit unit))))
  (if nit nit
    (setq nit (if (eq si:local-host host)
		  (make-instance (get device 'local-unit-flavor)
				 ':number number)
		(make-instance
		     (get protocol 'unit-flavor)
		     ':number number ':host host ':device device)))
    (push nit *units*)
    nit))

(DEFUN PARSE-UNIT (THING &OPTIONAL &KEY (DEFAULT-DEVICE ':MT)
					(DEFAULT-REMOTE-PROTOCOL *DEFAULT-REMOTE-PROTOCOL*))
  "Parse a THING into a tape unit.  A number, (), a unit object, or a string.
Acceptable strings: MT, MT0, 0, HOST:MT, HOST:MT0, HOST:0, HOST:"
  (COND ((NULL THING) *DEFAULT-UNIT*)
	((NUMBERP THING) (MAKE-UNIT SI:LOCAL-HOST DEFAULT-DEVICE THING ()))
	((STRINGP THING)
	 (SETQ THING (STRING-UPCASE (STRING-TRIM #\Space THING)))
	 (LET ((IDX (STRING-SEARCH-CHAR #/: THING)))
	   ;; see if there really is a host (i.e., a colon)
	   (COND (IDX
		  (MULTIPLE-VALUE-BIND (DEVICE NUMBER)
		      (PARSE-UNIT-DEVICE-AND-NUMBER (STRING-LEFT-TRIM #\Space
								      (SUBSTRING THING (1+ IDX)))
						    DEFAULT-DEVICE)
		    (MAKE-UNIT (SI:PARSE-HOST (STRING-RIGHT-TRIM #\Space
								 (SUBSTRING THING 0 IDX))
					      () T)
			       DEVICE NUMBER DEFAULT-REMOTE-PROTOCOL)))
		 (T
		  (MULTIPLE-VALUE-BIND (DEVICE NUMBER)
		      (PARSE-UNIT-DEVICE-AND-NUMBER THING DEFAULT-DEVICE)
		    (MAKE-UNIT SI:LOCAL-HOST DEVICE NUMBER ()))))))
	((TYPEP THING 'BASIC-UNIT) THING )
	(T (FERROR () "What is ~A ?" THING))))

