;;; -*- Mode:Common-Lisp; Package:SI; Base:8. -*-

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (c)(1)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1985- 1989 Texas Instruments Incorporated. All rights reserved.

;;;
;;; This file contains the core functions for the crash analyzer.
;;;   Report-Last-Shutdown writes analysis information from the last boot
;;; session.  Report-All-Shutdowns does the same for all boot session 
;;; information currently stored in the Crash Buffer in NVRAM.
;;;

;;; Package dependencies: must be loaded into whatever package rest of NVRAM
;;; system is in.  If that package is not SI, Load-Crash-Table should  be
;;; changed.  All other non-NVRAM names should carry explicit package prefixes.

;;; Edit History:
;;; -------------
;;; 8/85      ab   New file, August 1985.
;;; 9/85      ab   Minor cleanup.  Put debug functions here.  Prefix external names.
;;; 10/31/85  ab   NVRAM patch 2-5.
;;;                Add CA-Interpret-NuBus-Crash function.
;;; 12/09/85  ab   NVRAM patch 2-10.  
;;;                Fix "Invalid SCSI" code.
;;; 12/12/85  ab   NVRAM patch 2-9.
;;;                Redid Set-Microcode-Debug-Flags to use new 
;;;                Microcode-Debug-Flags-Bits list.
;;; 03-31-86  ab   NVRAM patch Rel 2, 2-2.
;;;                Add CA-Interpret-Page-Device-Crash.  Add functionality to
;;;                CA-Interpret-Nupi-Special-Event and CA-Interpret-RQB-Status-Word
;;; 08-18-86  RJF  Fixes bug 2552.  On nupi-special-event, address (rqb) pointed to
;;;                to status word instead of command word.
;;; 08-04-87  ab   Enhance DUMP-NVRAM-CONTENTS.
;;; 01-27-88  ab   Changes for MX.

;;;
;;; This file contains functions needed by the crash analyzer to interpret various
;;; crash codes.  Each function should begin with a comment indicating which 
;;; microcode file or files contain the CRASH-TABLE pseudo-ops that use the function.
;;; Correspondingly, the CRASH-TABLE pseudo-op using a special analysis function should
;;; indicate this in a comment, and point to this file (SYS: NVRAM; ANALYSIS-FUNCTIONS.LISP).
;;;
;;; All analysis functions should begin with the prefix "ca-" for "crash analyzer".
;;;

;;; Package dependencies: must be loaded into whatever package rest of NVRAM
;;; system is in, but beyond that should work.  (All names defined outside
;;; NVRAM system should carry explicit package prefix).



;;;
;;; Analysis Functions.
;;;

;;; Used at SG-ILLOP in uc-stack-group.  The pseudo op is:
;;; (Crash-Table "Illegal stack group state: ~a" (ca-interpret-illegal-sg-state M-1)

(Defun Ca-Interpret-Illegal-Sg-State (Saved-Crash-Register)
  (Let* ((State-Alist
	  '((#o0 . Global:Error) (#o1 . Active) (#o5 . Invoke-Call-On-Return)
	    (#o6 . Awaiting-Error-Recovery) (#o11 . Exhausted)))
	 (Key-Field (Ldb (Byte #o4 #o0) Saved-Crash-Register))
	 (Tem (Assoc Key-Field State-Alist :Test #'Eq)))
    (If Tem
      (Cdr Tem)
      (If (And (>= Key-Field #o12) (<= Key-Field #o17))
	'Other-Illegal-State
	'But-Legal-State-Found)))) 


;; Used in UR-Page-Fault (when swap RQB gets error) and in UR-Cold-Disk (when 
;; COLD-NUPI-ISSUE-COMMAND gets error.  RQB-STATUS contains the RQB status word.
;; Byte 1 of this word is the device or formatter error; byte 2 is NUPI error.
;; The code below uses new functions decode-nupi-controller-error and
;; decode-nupi-device-error.
;; Add new info saved in M-T, M-1 and M-2

(Defun Ca-Interpret-Rqb-Status-Word (Rqb-Status)
  (String-Append
   (If (Ldb-Test (Byte #o10 #o20) Rqb-Status);if byte 2 then controller error, else device error
     (Format () "controller error: ~%~@?~a ~%~@?Status Word = #x+~16r" Crec-Fmt-A ""
	     (Decode-Nupi-Controller-Error (Ldb (Byte #o10 #o20) Rqb-Status)) Crec-Fmt-A ""
	     Rqb-Status)
     (Format () "device error: ~%~@?~a ~%~@?Status Word = #x+~16r" Crec-Fmt-A ""
	     (Decode-Nupi-Device-Error (Ldb (Byte #o10 #o10) Rqb-Status)) Crec-Fmt-A ""
	     Rqb-Status))
   (Format () "~%~@?~a" Crec-Fmt-A ""
	   (Format () "Command Word: #x+~16r, Logical Unit: ~d." M-1
		   (Get-Logical-Unit (Ldb (Byte #o10 #o0) M-1))))
   (Format () "~%~@?~a" Crec-Fmt-A ""
	   (Format () "Block Address: ~d., Transfer Count: ~d." M-T M-2)))) 


;; Used in UR-NuBus on any kind of NuBus Error crash.
;; M-1 contains LVL1 map data in bits 0-15, LVL2 control in 16-31
;; MD will contain physical address if NuBus access was unmapped.  If
;; the access was mapped (ie, virtual address), M-2 will contain LVL2 map data
;; (ie physical page number) properly shifted into bits 10-31, and MD will
;; contain the original virtual address.

(Defun Ca-Interpret-Nubus-Crash (Md M-1 M-2)
 ;; Bit 15 in LVL1 control is 0 if mapped, 1 if unmapped.
  (Format ()
	  "~%~@?Physical Address: #x+~16r ~
               ~%~@?LVL1 Control:     #x+~16r ~
               ~%~@?LVL2 Control:     #x+~16r"
	  Crec-Fmt-A ""
	  (If (Ldb-Test (Byte #o1 #o17) M-1)
	   ;; unmapped
	    Md
	    ;; mapped
	    (Dpb (Ldb (Byte #o10 #o0) Md) (Byte #o10 #o0) M-2))
	  Crec-Fmt-A "" (Ldb (Byte #o20 #o0) M-1) Crec-Fmt-A "" (Ldb (Byte #o20 #o20) M-1))) 


;; Used in UR-Cold-Disk.  Crash-Table pseudo-op is:
;; (Crash-Table "NUPI Special Event Signaled: ~a" (ca-interpret-nupi-special-event M-1 M-2))
;; First argument will contain the 1st returned information word from NUPI-Status command
;; issued.  Second arg will be formatter number, if the overtemperature event was signaled.
;; Add support for displaying RQB address, command, and status words.  RQB address
;; is in VMA, RQB command word in M-T, and RQB status word in MD.

(Defun Ca-Interpret-Nupi-Special-Event (Special-Event-Bitmap Formatter-Number)
  (Format () "~%~@?~a~%~@?~a~%~@?~a~%~@?~a" Crec-Fmt-A ""
	  (If (Ldb-Test %%Nupi-Error-Status-Overtemperature Special-Event-Bitmap)
	    (Format () "Overtemperature on formatter ~d." Formatter-Number)
	    (Select Special-Event-Bitmap
	       (#o1 "Unrecoverable NuBus error encountered while fetching command block")
	       (#o2 "Overtemperature detected") (#o4 "Illegal access to NUPI control register")
	       (#o10 "Multiple commands issued to one device, formatter, or to NUPI")
	       (#o20 "Illegal command found in command block")
	       (#o40 "Invalid Special Event Type Found")
	       (#o100 "Command aborted with no command block updates")
	       (#o200 "Illegal interrupt encountered on NuBus access")
	       (#o400 "Invalid Special Event address") (#o1000 "Hardware Error")
	       (#o2000 "Invalid SCSI operation attempted")
	       (Otherwise (Format () "Unknown special event code = #x~16r" Special-Event-Bitmap))))
	  Crec-Fmt-A "" (Format () "RQB address:       #x~16r" (- Vma #o4))
			;;8-18-86, RJF
	  Crec-Fmt-A "" (Format () "RQB command word:  #x~16r" M-T) Crec-Fmt-A ""
	  (Format () "RQB status word:   #x~16r" Md))) 


;; Function for interpreting Invalid Page Device (not assigned) crashes.
(Defun Ca-Interpret-Page-Device-Crash (Vma)
  (Let* ((Va (Ldb %%Q-Pointer Vma))
	 (Ptr-Va (Convert-To-Signed Va))
	 (Reg (%Region-Number Ptr-Va))
	 (Area (%Area-Number Ptr-Va))
	 (Valid
	  (And Reg
	     (< Va
		(+ (Convert-To-Unsigned (Region-Origin Reg))
		   (Convert-To-Unsigned (Region-Free-Pointer Reg)))))))
    (Format () "~%~@?~a~a" Crec-Fmt-A ""
	    (Format () "Crash while trying to access virtual address #o+~o," Va)
	    (If Reg
	      (Format () "~%~@?~a~%~@?~a~a" Crec-Fmt-A ""
		      (Format () "  which is assigned to region #o+~o of area ~a (#o+~o)" Reg
			      (Area-Name Area) Area)
		      Crec-Fmt-A "" "  in the current environment, "
		      (If Valid
			"and is allocated to an object."
			"but is not allocated to an object."))
	      (Format () "~%~@?~a" Crec-Fmt-A ""
		      "  which is not assigned in the current environment."))))) 


;;;
;;; Hacks for debugging.  (Undocumented, not for general use)
;;;

;; NVRAM patch 2-9, -ab
;; Re-wrote this to use new Microcode-Debug-Flags-Bits list.

(Defun Set-Microcode-Debug-Flags (Debug-Flag &Key (Clear Nil) (Clear-All-Others Nil) (Reset Nil))
  "WARNING: Don't use this function unless you know what you're doing.  It allows
you to turn off or on various bits in the microcode's A-Debug-Flags register.
   The DEBUG-FLAG argument is the bit number of the bit to be set.  A list
of symbolic names to use as this argument can be found as the value of
si:Microcode-Debug-Flags-Bits.
   When CLEAR is non-nil, the specified DEBUG-FLAG-BIT is cleared rather than set.
   When CLEAR-ALL-OTHERS is non-nil, the specified DEBUG-FLAG is set and all others
are cleared.
   When RESET is non-nil, the debug flags are all reset to their normal state."
  (Check-Arg Debug-Flag
     (And (Integerp Debug-Flag) (>= Debug-Flag #o0) (<= Debug-Flag #o31)
	(Dolist (S Microcode-Debug-Flags-Bits Nil)
	  (If (= (Symbol-Value S) Debug-Flag)
	    (Return T))))
     "an integer corresponding to a valid microcode-debug-flag bit")
  (If Reset
    (Setq Microcode-Debug-Flags #o0)
    (Setq Microcode-Debug-Flags
	  (%Logdpb (If Clear
		     #o0
		     #o1)
		   (Byte #o1 Debug-Flag) (If Clear-All-Others
					   #o0
					   Microcode-Debug-Flags))))) 


;;AB new 8/4/87.
(DEFUN char-or-splat (int &aux chr)
  (SETQ chr (INT-CHAR int))
  (IF (GRAPHIC-CHAR-P chr) chr (INT-CHAR 0)))


(DEFINE-WHEN :NVRAM
  
(Defun Read-Nvram-Time ()
 ;; Displays current time as recorded in CREC.
  (Format () "~%Month ~d.  Day ~d.  Year ~d.  Hour ~d.  Minute ~d."
	  (Read-Current-Crash-Field Cro-Current-Month)
	  (Read-Current-Crash-Field Cro-Current-Day) (Read-Current-Crash-Field Cro-Current-Year)
	  (Read-Current-Crash-Field Cro-Current-Hour)
	  (Read-Current-Crash-Field Cro-Current-Minute))) 

(Defun Initialize-Crash-Record (Crash-Rec-Pointer)
  "Zeros out a crash record"
  (Do ((Addr Crash-Rec-Pointer (+ Addr #o4))
       (Lim (+ Crash-Rec-Pointer Crash-Rec-Len)))
      ((>= Addr Lim)
       #o0)
    (Write-Nvram Addr #o0)))

(Defun Dump-Crec-Hex (Crec)
 ;; Dump CREC's contents in hex.  For debugging.
  (If (Nvram-Functioning-P)
    (Do ((Offset Crec (+ Offset #o40)))
	((> Offset (+ Crec Crash-Rec-Len #o40))
	 Nil)
      (Format T "~&~4x -- ~2x  ~2x  ~2x  ~2x  ~2x  ~2x  ~2x  ~2x" Offset
	      (Read-Nvram Offset) (Read-Nvram (+ Offset #o4)) (Read-Nvram (+ Offset #o10))
	      (Read-Nvram (+ Offset #o14)) (Read-Nvram (+ Offset #o20))
	      (Read-Nvram (+ Offset #o24)) (Read-Nvram (+ Offset #o30))
	      (Read-Nvram (+ Offset #o34))))
    "Can't get to NVRAM")) 

;;AB improved this.
(Defun Dump-Nvram-Contents (&optional (start 0) (num-bytes (TRUNCATE SIB-NVRAM-Length 4.)))
 "Dump specified NVRAM locations in hex."
  (If (Nvram-Functioning-P)
    (Do ((Offset start (+ Offset #o40))
	 (ct 0 (+ ct 8.))
	 by0 by1 by2 by3 by4 by5 by6 by7)
	((OR (>= ct num-bytes)
	     (>= offset SIB-NVRAM-Length)))
      (Format T "~% ~4,'0,x  --  ~2,'0,x  ~2,'0,x  ~2,'0,x  ~2,'0,x    ~2,'0,x  ~2,'0,x  ~2,'0,x  ~2,'0,x"
	      Offset
	      (SETQ by0 (Read-Nvram (+ Offset #o00))) (SETQ by1 (Read-Nvram (+ Offset #o04)))
	      (SETQ by2 (Read-Nvram (+ Offset #o10))) (SETQ by3 (Read-Nvram (+ Offset #o14)))
	      (SETQ by4 (Read-Nvram (+ Offset #o20))) (SETQ by5 (Read-Nvram (+ Offset #o24)))
	      (SETQ by6 (Read-Nvram (+ Offset #o30))) (SETQ by7 (Read-Nvram (+ Offset #o34))))
      (FORMAT t "       [ ~c ~c ~c ~c  ~c ~c ~c ~c ]"
	      (char-or-splat by0) (char-or-splat by1)
	      (char-or-splat by2) (char-or-splat by3)
	      (char-or-splat by4) (char-or-splat by5)
	      (char-or-splat by6) (char-or-splat by7)))
    "Can't get to NVRAM"))
			   

(Defun Dump-Crec (Crec &Optional (Stream *Terminal-Io*))
 ;; Dumps out crec in semi-human-readable form.  For debugging.  If
 ;; STREAM is a string, it is interpreted as a filename, and output goes there.
  (When (Nvram-Functioning-P)
    (With-Open-Stream (S (If (Stringp Stream)
	  (Open (Fs:Parse-Pathname Stream) :Direction :Output)
	  Stream))
      (Setq Current-Crash-Record (Get-Crash-Record Crec))
      (Mapc
       #'(Lambda (Item)
	   (Format S "~%~a = ~25,0T~a" (First Item)
		   (If (Numberp (Second Item))
		     (Format () "~16r" (Second Item))
		     (Second Item))))
       Current-Crash-Record)
      ())))

;;End of DEFINE-WHEN
)


(DEFINE-UNLESS :NVRAM

(DEFPARAMETER crash-slots '("Load / Ucode Unit | Boot Slot  Progress"
			    "Ucode Partition Name"
			    "Load Partition Name"
			    "load version      | ucode version"
			    "Boot day  month   | Load revision"
			    "Cur month boot min| boot hour  year"
			    "Cur min   hour    | year       day"
			    "Halt Kind| Halt  Address   | report flgs"
			    "M-1"
			    "M-2"
			    "MD"
			    "VMA"
			    "M-FEF"
			    "UPC-2            | UPC-1"
			    "LC"
			    "M-T"))

(DEFUN dump-crash-rec (which)
  (let ((acb (add:get-acb #x40))
	(ch  (add:find-channel si:%Chan-Type-Misc)))
    (unwind-protect
	(progn
	  (add:init-acb acb
			si:%MC-readcrash
			si:%RC-Read-Crash-Record)
	  (add:set-parm-16b acb 0 which)
						; Execute
	  (add:transmit-packet-and-wait acb ch)
	  (add:check-error acb)
	  
	  (DOTIMES (i 16.)
	    (FORMAT t "~&Offset: ~2x  Data: ~8x  ~a~%" i (add:parm-32b acb i)
		    (NTH i crash-slots))))
      (setf (add:requestor-complete acb) t)
      (add:return-acb-fast acb) t)))

;;End of DEFINE-UNLESS
)